-- Update the UAB version if the version of this file is later than any other already loaded
uab.CheckVersion(tonumber(("$Revision: 43089 $"):match("%d+")), ("$Date: 2007-07-04 17:50:18 -0400 (Wed, 04 Jul 2007) $"):match("%d%d%d%d%-%d%d%-%d%d"))

local uabActivate = uab:NewModule("Activate", "AceConsole-2.0")

uabActivate.postponedactivate = {}
uabActivate.core = uab
uabActivate.dbDefaults = {
    startupopen = {
    }
}

--[[
Constant menu/slash commands.
--]]

uabActivate.options = {
    type = 'group',
    args = {},
}

function uabActivate:OnInitialize()
	if not self.db then
		self.core:RegisterDefaults(self.name, "profile", self.dbDefaults)
		self.db = self.core:AcquireDBNamespace(self.name)
	end

    self.timerFrame = CreateFrame("Frame", "uabActivateTimerFrame")
    self.timerFrame:SetScript("OnUpdate", function(self, elapsed) uabActivate:RunPostponedNonCombatActions(elapsed) end)
    self.timerFrame:Show()

    self:RegisterEvent("UnitActionBars_RefreshActivators")
    self:RegisterEvent("UnitActionBars_InitializationCompleted")
    self:RegisterEvent("UnitActionBars_FrameRegistered")

    self:RefreshActivatorOptions()

    self:RegisterChatCommand({ "/unitactivate", "/uaba" }, self.options)
end

function uabActivate:OnEnable()
end

function uabActivate:UnitActionBars_InitializationCompleted()
    self:RefreshActivatorOptions()
end

function uabActivate:UnitActionBars_FrameRegistered(frame)
    self:ScheduleNonCombatAction(self, "ActivateOnFrame", frame)
end

local nonCombatActions = {}
function uabActivate:PLAYER_REGEN_ENABLED()
	for i, v in ipairs(nonCombatActions) do
		v.p1(v.p2, v.p3)
        nonCombatActions[i] = nil
	end
end

function uabActivate:ScheduleNonCombatAction(p1, p2, p3)
	local func = p1
	if type(func) == "table" then
		local method = p2
		return self:ScheduleNonCombatAction(func[method], func, p3)
	end
	
    if InCombatLockdown() then
		nonCombatActions[#nonCombatActions+1] = {
            p1 = p1, p2 = p2, p3 = p3
        }
		return
	end
	
    p1(p2, p3)
end

local laterNonCombatActions = {}
local postponeCheck = 0

function uabActivate:RunPostponedNonCombatActions(elapsed)
    postponeCheck = postponeCheck + elapsed
    if postponeCheck > 0.2 then
        postponeCheck = 0
    	if InCombatLockdown() then
            return
        end

        if #laterNonCombatActions > 0 then
            local nowActions = laterNonCombatActions
            laterNonCombatActions = {}
            
        	for i, v in ipairs(nowActions) do
                self:ScheduleNonCombatAction(v.p1, v.p2, v.p3)
                nowActions[i] = nil
        	end
        end
    end
end

function uabActivate:PostponeNonCombatAction(p1, p2, p3)
    laterNonCombatActions[#laterNonCombatActions+1] = {
            p1 = p1, p2 = p2, p3 = p3
    }
end

function uabActivate:ActivateOnFrame(frame)
    if not self:AnyStartupForFrame(frame) then
        if self.postponedactivate[frame] then
            self.postponedactivate[frame] = nil
        end
        return
    end

    if not frame:IsVisible() then
        if not self.postponedactivate[frame] or self.postponedactivate[frame] < 5 then
            if not self.postponedactivate[frame] then
                self.postponedactivate[frame] = 0
            end
            self.postponedactivate[frame] = self.postponedactivate[frame] + 1
            self:PostponeNonCombatAction(self, "ActivateOnFrame", frame)
            return
        end
    end

    self.postponedactivate[frame] = nil
    self:StartupActivate(frame)
end

function uabActivate:AnyStartupForFrame(frame)
    local framename = frame:GetName()
    if framename then
        for name, info in pairs(self.db.profile.startupopen) do
            if info.frames then
                if info.frames[framename] then
                    return true
                end
            end
        end
    end
end

function uabActivate:UnitActionBars_RefreshActivators()
    self:RefreshActivatorOptions()
end

function uabActivate:RefreshActivatorOptions()
    self:FixupStartupActivate()

    if self.options.args then
        if self.options.args.startup then
            uab.ClearTable(self.options.args.startup.args)
            self.options.args.startup.args = {}
        end

        if self.options.args.open then
            uab.ClearTable(self.options.args.open.args)
            self.options.args.open.args = {}
        end
        
        if self.options.args.close then
            uab.ClearTable(self.options.args.close.args)
            self.options.args.close.args = {}
        end
    end

    if not self.options.args.startup then
        self.options.args.startup = {
            type = 'group',
            name = "Startup",
            desc = "Specify the action buttons to open on startup",
            disabled = function() return InCombatLockdown() end,
    		order = 5,
            args = {}
        }
    end

    if not self.options.args.open then
        self.options.args.open = {
            type = 'group',
            name = "Open",
            desc = "Open action buttons for an activator on a unit frame",
            disabled = function() return InCombatLockdown() end,
    		order = 10,
            args = {}
        }
    end
    
    if not self.options.args.openall then
        self.options.args.openall = {
            type = 'execute',
            name = "Open all startup",
            desc = "Open action buttons for all activators/unit frames configured to open at startup",
            disabled = function() return InCombatLockdown() end,
            func = function()
                self:OpenAllStartup()
            end,
    		order = 13,
        }
    end

    if not self.options.args.close then
        self.options.args.close = {
            type = 'group',
            name = "Close",
            desc = "Close action buttons for an activator on a unit frame",
            disabled = function() return InCombatLockdown() end,
    		order = 15,
            args = {}
        }
    end
    
    if not self.options.args.closeall then
        self.options.args.closeall = {
            type = 'execute',
            name = "Close all",
            desc = "Close any action buttons for all activators",
            disabled = function() return InCombatLockdown() end,
            func = function()
                self:CloseAllActivators()
            end,
    		order = 20,
        }
    end

    -- For each named activator, add an option to open the action buttons
    -- for that activator on a unit frame.

    local order = 20
	for key, val in pairs(self.core.db.profile.activators) do
        if val.name and val.inuse then
            self.options.args.startup.args[val.name] = self:CreateActivatorStartupOptions(val.name)
            self.options.args.startup.args[val.name].order = order
            self.options.args.open.args[val.name] = self:CreateActivatorOpenOptions(val.name)
            self.options.args.open.args[val.name].order = order
            self.options.args.close.args[val.name] = self:CreateActivatorCloseOptions(val.name)
            self.options.args.close.args[val.name].order = order
        end
        order = order + 10
	end

    if not self.core.options.args.activate then
        local prior_order = self.core.options.args.effectsettings.order
        self.core.options.args.activate = {
            type = 'group',
            name = "Activate",
            desc = "Directly activate action buttons on unit frames",
            order = prior_order + 5,
        }
    end
    
    self.core.options.args.activate.args = self.options.args
end

function uabActivate:FixupStartupActivate()
    local newnames = {}

    for name, info in pairs(self.db.profile.startupopen) do
        local activatorobj = uab:GetActivatorObj(info.code)
        if activatorobj then
            newnames[name] = self.core.db.profile.activators[info.code].name
        end
    end

    for name, newname in pairs(newnames) do
        if newname ~= name then
            local newdata = uab.CloneTable(self.db.profile.startupopen[name])
            self.db.profile.startupopen[newname] = newdata
            uab.ClearTable(self.db.profile.startupopen[name])
            self.db.profile.startupopen[name] = nil
        end
    end            
end

function uabActivate:CreateActivatorStartupOptions(activatorname)
    local options = {
        type = 'group',
        name = activatorname,
        desc = "Open action buttons for this activator on the specified unit frame at startup",
        disabled = function() return InCombatLockdown() end,
		order = 10,
        args = {}
    }

    options.args.add = {
        type = 'text',
        name = "Add",
        desc = "Open action buttons for this activator on the specified unit frame at startup",
        usage = "<unit frame name>",
        get = function() return end,
        set = function(unitframename)
            self:AddStartupActivator(activatorname, unitframename)
        end,
		order = 10,
        validate = function(unitframename)
            local found

            for name, info in pairs(self.db.profile.startupopen) do
                if name == activatorname and info.frames and info.frames[unitframename] then
                    found = true
                    break
                end
            end

            return not found            
        end,
    }

    options.args.remove = {
        type = 'group',
        name = "Remove",
        desc = "Remove startup activation for this activator on a unit frame",
        disabled = function() return InCombatLockdown() or not self:AnyStartupForActivator(activatorname) end,
		order = 20,
        args = {}
    }

    if self:AnyStartupForActivator(activatorname) then
        local frameorder = 10
        for unitframe, _ in pairs(self.db.profile.startupopen[activatorname].frames) do
            options.args.remove.args[unitframe] = {
                type = 'toggle',
                name = unitframe,
                desc = "Remove this unitframe from activation at startup",
                order = frameorder,
                get = function()
                    return true
                end,
                set = function(v)
                    if not v then
                        self:RemoveStartupActivator(activatorname, unitframe)
                    end
                end,
            }
            frameorder = frameorder + 10
        end
    end

    return options
end

function uabActivate:AnyStartupForActivator(activatorname)
    if self.db.profile.startupopen[activatorname] and self.db.profile.startupopen[activatorname].frames then
        for unitframe, _ in pairs(self.db.profile.startupopen[activatorname].frames) do
            return true
        end
    end
end

function uabActivate:AddStartupActivator(activatorname, unitframename)
    if not self.db.profile.startupopen[activatorname] then
        self.db.profile.startupopen[activatorname] = {}
        for code, data in pairs(self.core.db.profile.activators) do
            if data.name and data.name == activatorname then
                self.db.profile.startupopen[activatorname].code = code
                break
            end
        end
    end

    if not self.db.profile.startupopen[activatorname].frames then
        self.db.profile.startupopen[activatorname].frames = {}
    end

    self.db.profile.startupopen[activatorname].frames[unitframename] = true
    self:RefreshActivatorOptions()
end

function uabActivate:RemoveStartupActivator(activatorname, unitframename)
    if self.db.profile.startupopen[activatorname] then
        local found
    
        if self.db.profile.startupopen[activatorname] and self.db.profile.startupopen[activatorname].frames then
            self.db.profile.startupopen[activatorname].frames[unitframename] = nil
            for unitframe, _ in pairs(self.db.profile.startupopen[activatorname].frames) do
                found = true
                break
            end
        end
    
        if not found then
            -- no frames that need to be opened on startup for this activator.
            self.db.profile.startupopen[activatorname].frames = nil
            self.db.profile.startupopen[activatorname].code = nil
            self.db.profile.startupopen[activatorname] = nil
        end

        self:RefreshActivatorOptions()
    end
end

function uabActivate:OpenAllStartup()
    local registered = {}
    
    for frame, _ in pairs(uab.ccframes) do
        local name = frame:GetName()
        if name then
            registered[name] = true
        end
    end

    for name, info in pairs(self.db.profile.startupopen) do
        local activator = uab:GetActivatorObj(info.code)
        if activator then
            for code, data in pairs(self.core.db.profile.activators) do
                if data.name and tostring(data.name) == name then
                    local name = data.name
                    if info.frames then
                        for framename, _ in pairs(info.frames) do
                            if registered[framename] then
                                self:OpenActivatorActions(name, framename)
                            end
                        end
                    end
                end
            end
        end
    end
end

function uabActivate:StartupActivate(frame)
    local framename = frame:GetName()
    if framename then
        for name, info in pairs(self.db.profile.startupopen) do
            local activator = uab:GetActivatorObj(info.code)
            if activator then
                for code, data in pairs(self.core.db.profile.activators) do
                    if data.name and tostring(data.name) == name then
                        local name = data.name
                        if info.frames then
                            if info.frames[framename] then
                                self:OpenActivatorActions(name, framename)
                            end
                        end
                    end
                end
            end
        end
    end
end

function uabActivate:CreateActivatorOpenOptions(activatorname)
    local options
    
    options = {
        type = 'text',
        name = activatorname,
        desc = "Open action buttons for the '"..activatorname.."' activator on a unit frame",
        usage = "<unit frame name>",
        get = function() return end,
        set = function(unitframename)
            self:OpenActivatorActions(activatorname, unitframename)
        end,
        validate = function(unitframename)
            if getglobal(unitframename) then
                return true
            end
        end,
    }

    return options
end

function uabActivate:OpenActivatorActions(activatorname, unitframename)
    if getglobal(unitframename) then
--        uab.print("Opening activator actions for the "..activatorname.." activator on the unit frame '"..unitframename.."'")
    
        local afc = uab:FindActivatorFrameControlForFrame(activatorname, unitframename)
        if afc then
            -- afc is an ActivatorFrameControl for the desired activator and unit frame.
            -- Open the action buttons by simulating a click on that control.
            --
            -- Keep in mind that the modifiers (shift, ctrl, alt) are the values that
            -- the user currently has pressed when someone attempts to open the actions
            -- for an activator using this method.  This may affect whether the actions
            -- actually are shown.
            --
            -- Also keep in mind that the mouse is not necessarily positioned where it
            -- would be if the frame was clicked.  The activators that can be activated
            -- using this mechanism are currently restricted to named activators, which
            -- are not associated with a key-click combination.  However, if the layout
            -- associated with the unit frame (or actions) indicates cursor-based
            -- positioning, then the action buttons may not show up where you might expect
            -- them; instead they will show up relative to the current mouse cursor
            -- position.
    
            afc.frame:Click("LeftButton")        
--        else
--            uab.print("Unable to locate frameactivator frame control")
        end
--    else
--        uab.print("Unable to locate frame with the name : "..unitframename)
    end
end

function uabActivate:CreateActivatorCloseOptions(activatorname)
    local options
    
    options = {
        type = 'text',
        name = activatorname,
        desc = "Open action buttons for the '"..activatorname.."' activator on a unit frame",
        usage = "<unit frame name>",
        get = function() return end,
        set = function(unitframename)
            self:CloseActivatorActions(activatorname, unitframename)
        end,
        validate = function(unitframename)
            if getglobal(unitframename) then
                return true
            end
        end,
    }

    return options
end

function uabActivate:CloseActivatorActions(activatorname, unitframename)
    if getglobal(unitframename) then
--        uab.print("Closing activator actions for the "..activatorname.." activator on the unit frame '"..unitframename.."'")
    
        local afc = uab:FindActivatorFrameControlForFrame(activatorname, unitframename)
        if afc then
            -- We need to set the state of the secure header associated with the
            -- ActivatorFrameControl to 0, causing the actions to hide (disappear).
    
            uab.SetFrameAttribute(uab:GetSecureHeaderObj(afc.secureheadercode).frame, "state", "0")
        end
    end
end

function uabActivate:CloseAllActivators()
    uab.classes.SecureHeader:HideAllHeaders()
end

