-- Update the UAB version if the version of this file is later than any other already loaded
uab.CheckVersion(tonumber(("$Revision: 46816 $"):match("%d+")), ("$Date: 2007-08-17 22:21:12 -0400 (Fri, 17 Aug 2007) $"):match("%d%d%d%d%-%d%d%-%d%d"))

--[[
Identifies the action information.  This includes the visual behavior of the 
associated action button (hidden, low alpha, blinking, etc.) and the action to 
execute when the button is clicked.  This class sets the action attributes on 
the ActionButton, and tells the ActionButton how to draw itself based on the 
visual effect settings and current unit state.  This class also controls 
whether the ActionButton is in use or not.  If not in use, the ActionButton is 
told to hide itself.

Each Action has one or more ActionButton instances associated with it. 
Each ActionButton instance is associated with a single Action instance.
here is one ActionButton for each action of the ActionSet associated with each 
UnitGroupHeader.
--]]

uabActionWrapper = uab:NewModule("ActionWrapper")
uabActionWrapper.core = uab

uabActionWrapper.interfaces = {}

uabActionWrapper.interfaces.IActionData = uab.mixins.aceOO.Interface {
    ShowTooltip = "function",
    GetCount = "function",
	GetIcon = "function",
	GetAttributes = "function",
    IsUsable = "function",
    GetClassName = "function",
    IsActive = "function",
    GetCooldown = "function",
    GetSpellName = "function",
    GetHasRank = "function",
    GetRank = "function",
}

function uabActionWrapper:OnInitialize()
    uabActionWrapper.super.OnInitialize(self)
end

function uabActionWrapper:CreateActionWrapper(actioninfo, ishostile)
	local actionobj
	local actionwrapper

	if uab.classes[actioninfo.classname] then
		actionobj = uab.classes[actioninfo.classname]:new(actioninfo.data)
		actionwrapper = uab.classes.ActionWrapper:new(actionobj, actioninfo.effect, actioninfo.message, actioninfo.useplayerunit, actioninfo.sethideunusable, actioninfo.hideunusable, ishostile)
	end

	return actionwrapper
end

local function AnyChosenVisualEffect(visualeffects, includecooldown)
    if visualeffects then
		if visualeffects.highlight or
            visualeffects.lowalpha or
			visualeffects.noalpha or
            visualeffects.blink or
			visualeffects.notusable or
            (includecooldown and visualeffects.showcooldown) then
            return true
        end
    end
end
        
local defaultStr = "Default"
local oldDefaultStr = "default"

local function IsDefaultUnitEffect(effect)
    return effect == oldDefaultStr or effect == defaultStr
end

uab.classes.ActionWrapper = uab.mixins.aceOO.Class()

--[[
for each unit effect, the following visual effects are valid
    -- highlight
    -- lowalpha
    -- noalpha
    -- blink
    -- notusable
    -- showcooldown
    
effects is a table of unit effects, which each entry being
a table of visual effects.  The unit effects that can be set are
dynamically determined at run-time.
--]]        
function uab.classes.ActionWrapper.prototype:init(actionobj, effects, msginfo, useplayerunit, sethideunusable, hideunusable, ishostile)
	uab.classes.ActionWrapper.super.prototype.init(self)

    self.ishostile = ishostile
    self.actionobj = actionobj    
    self.msginfo = msginfo
    self.useplayerunit = useplayerunit
    self.sethideunusable = sethideunusable
    self.hideunusable = hideunusable
    self.buttons = {}

    self.effects = {}
    for uniteffect, visualeffects in pairs(effects) do
        self:SetUnitEffectVisualEffects(uniteffect, visualeffects, true)
    end
end

function uab.classes.ActionWrapper.prototype:Dump(dumper)
    dumper:Dump("Action Wrapper: ")
    dumper:IncIndent()
    dumper:Dump("Unit Visual Effects")
    dumper:IncIndent()
    
    for uniteffect, visualeffects in pairs(self.effects) do
        dumper:Dump("Unit Effect: "..uniteffect)
        dumper:IncIndent()
        dumper:Dump("Visual Effects")
        dumper:IncIndent()
        for visualeffect, _ in pairs(visualeffects) do
            dumper:Dump(visualeffect)
        end
        dumper:DecIndent()
        dumper:DecIndent()
    end

    dumper:DecIndent()
    dumper:Dump("Action")
    dumper:IncIndent()
    
    self.actionobj:Dump(dumper)
    
    dumper:DecIndent()
    dumper:DecIndent()
end

function uab.classes.ActionWrapper.prototype:SetUnitEffectVisualEffects(uniteffect, visualeffects, skipupdate)
    local effects = self.effects

    if not effects[uniteffect] then
        effects[uniteffect] = {}
    end

    uabUnitStatus:ClearEffectUse(uniteffect, self)

    for idx, _ in pairs(effects[uniteffect]) do
        effects[uniteffect][idx] = nil
    end
    
    effects[uniteffect] = visualeffects

    self:SetupEffectChecks()
    
    if not skipupdate then    
        self:UpdateVisualEffects()
    end
end

function uab.classes.ActionWrapper.prototype:ClearUnitEffectVisualEffects(uniteffect, skipupdate)
    local effects = self.effects
    if effects[uniteffect] then
        for idx, _ in pairs(effects[uniteffect]) do
            effects[uniteffect][idx] = nil
        end
        
        effects[uniteffect] = nil
    end
    
    uabUnitStatus:ClearEffectUse(uniteffect, self)

    self:SetupEffectChecks()
    
    if not skipupdate then
        self:UpdateVisualEffects()
    end
end

function uab.classes.ActionWrapper.prototype:SetupEffectChecks()
    if not self.cooldown then
        self.cooldown = {}
    end
    
    for k, _ in pairs(self.cooldown) do
        self.cooldown[k] = nil
    end
    
    if not self.enabledeffects then
        self.enabledeffects = {}
    end
    
    for k, _ in pairs(self.enabledeffects) do
        self.enabledeffects[k] = nil
    end
    
    for uniteffect, visualeffects in pairs(self.effects) do
        --[[
        We skip the default unit effect during this phase.  We apply
        the visual effects for the default inside the ApplyEffects
        method only if no other unit effects are enabled.
        --]]
        if not IsDefaultUnitEffect(uniteffect) then
            if visualeffects.showcooldown then
                self.cooldown[uniteffect] = true
                uabUnitStatus:SetEffectUse(uniteffect, self)
            end

            --[[
            We only look to see if a unit effect is "on" if there are
            visual effects to apply when that unit effect is on.
            --]]
            if AnyChosenVisualEffect(visualeffects) then
                self.enabledeffects[uniteffect] = true
                uabUnitStatus:SetEffectUse(uniteffect, self)
            end
        end
    end
end

function uab.classes.ActionWrapper.prototype:GetActionObj()
    return self.actionobj
end

function uab.classes.ActionWrapper.prototype:SetActionObj(actionobj)
    self.actionobj = actionobj
    if self.attributes then
        for attr, val in pairs(self.attributes) do
            self.attributes[attr] = nil
        end
        self.attributes = nil
    end
    
    self:SetAllButtonProperties()
end

function uab.classes.ActionWrapper.prototype:GetMessageInfo()
    return self.msginfo
end

function uab.classes.ActionWrapper.prototype:SetMessageInfo(msginfo)
    self.msginfo = msginfo
end

function uab.classes.ActionWrapper.prototype:ClearMsgInfo()
    self.msginfo = nil
end

function uab.classes.ActionWrapper.prototype:GetDefaultHideUnusable()
    return not self.sethideunusable
end

function uab.classes.ActionWrapper.prototype:SetDefaultHideUnusable(skipupdate)
    self.sethideunusable = nil
    
    if not skipupdate then    
        self:UpdateVisualEffects()
    end
end

function uab.classes.ActionWrapper.prototype:ClearDefaultHideUnusable(skipupdate)
    self.sethideunusable = true
    
    if not skipupdate then    
        self:UpdateVisualEffects()
    end
end

function uab.classes.ActionWrapper.prototype:GetHideUnusable()
    return self.hideunusable
end

function uab.classes.ActionWrapper.prototype:SetHideUnusable(skipupdate)
    self.hideunusable = true
    
    if not skipupdate then    
        self:UpdateVisualEffects()
    end
end

function uab.classes.ActionWrapper.prototype:ClearHideUnusable(skipupdate)
    self.hideunusable = nil
    
    if not skipupdate then    
        self:UpdateVisualEffects()
    end
end

function uab.classes.ActionWrapper.prototype:GetUsePlayerUnit()
    return self.useplayerunit
end

function uab.classes.ActionWrapper.prototype:SetUsePlayerUnit(skipupdate)
    self.useplayerunit = true
    
    if not skipupdate then    
        self:UpdateVisualEffects()
    end
end

function uab.classes.ActionWrapper.prototype:ClearUsePlayerUnit(skipupdate)
    self.useplayerunit = nil
    
    if not skipupdate then    
        self:UpdateVisualEffects()
    end
end

function uab.classes.ActionWrapper.prototype:AddButton(button)
    self.buttons[button] = true
    self:SetButtonAllProperties(button)
end

function uab.classes.ActionWrapper.prototype:RemoveButton(button)
    self.buttons[button] = nil
    self:ClearButtonAllProperties(button)
end

function uab.classes.ActionWrapper.prototype:SetAllButtonProperties()
    local attrs = self:GetAttributes()
    for button, _ in pairs(self.buttons) do
        self:SetButtonAllProperties(button, attrs)
    end
end

function uab.classes.ActionWrapper.prototype:SetButtonAllProperties(button, attributes)
    self:SetButtonAttributes(button, attributes)
    self:SetButtonIcon(button)
    self:SetButtonCount(button)
end

function uab.classes.ActionWrapper.prototype:SetAllButtonAttributes()
    local attrs = self:GetAttributes()
    for button, _ in pairs(self.buttons) do
        self:SetButtonAttributes(button, attrs)
    end
end

function uab.classes.ActionWrapper.prototype:SetButtonAttributes(button, attributes)
    button:SetAttributes(attributes or self:GetAttributes())
end

function uab.classes.ActionWrapper.prototype:ClearAllButtonProperties()
    local attrs = self:GetAttributes()
    for button, _ in pairs(self.buttons) do
        self:ClearButtonAllProperties(button, attrs)
    end
end

function uab.classes.ActionWrapper.prototype:ClearButtonAllProperties(button, attributes)
    self:ClearButtonAttributes(button, attributes)
    self:ClearButtonIcon(button)
    self:ClearButtonCount(button)
end

function uab.classes.ActionWrapper.prototype:ClearAllButtonAttributes()
    local attrs = self:GetAttributes()
    for button, _ in pairs(self.buttons) do
        self:ClearButtonAttributes(button, attrs)
    end
end

function uab.classes.ActionWrapper.prototype:ClearButtonAttributes(button, attributes)
    button:ClearAttributes(attributes or self:GetAttributes())
end

function uab.classes.ActionWrapper.prototype:SetButtonIcon(button)
    button:SetIcon(self:GetIcon())
end

function uab.classes.ActionWrapper.prototype:ClearButtonIcon(button)
    button:SetIcon(nil)
end

function uab.classes.ActionWrapper.prototype:SetButtonCount(button)
    button:SetCount(self:GetCount())
end

function uab.classes.ActionWrapper.prototype:ClearButtonCount(button)
    button:SetCount(nil)
end

function uab.classes.ActionWrapper.prototype:ShowTooltip(frame)
	if uab.db.profile.tooltips then
        self:ShowActionTooltip(frame)
    end
end

--[[
The following methods are wrappers around the functionality that we
expect the action object to provide.
--]]
function uab.classes.ActionWrapper.prototype:ShowActionTooltip(frame)
    return self.actionobj:ShowTooltip(frame)
end

function uab.classes.ActionWrapper.prototype:GetCount()
    return self.actionobj:GetCount()
end

function uab.classes.ActionWrapper.prototype:GetIcon()
    return self.actionobj:GetIcon()
end

function uab.classes.ActionWrapper.prototype:GetAttributes()
    if not self.attributes then
        local attribs = self.actionobj:GetAttributes()
    
        self.attributes = {}
        
        if not self.ishostile then
            for name, val in pairs(attribs) do
                self.attributes[name] = val
            end        
        else
            self.attributes["harmbutton"] = "hostile"
            
            --[[
            We have to make the attributes specific to hostile units only.
            --]]
    
            for name, val in pairs(attribs) do
                self.attributes[name.."-hostile"] = val
            end        
        end
    end
    
    return self.attributes
end

function uab.classes.ActionWrapper.prototype:IsUsable()
    return self.actionobj:IsUsable()
end

function uab.classes.ActionWrapper.prototype:GetClassName()
    return self.actionobj:GetClassName()
end

function uab.classes.ActionWrapper.prototype:GetSpellName()
    return self.actionobj:GetSpellName()
end

function uab.classes.ActionWrapper.prototype:IsActive()
    return self.actionobj:IsActive()
end

function uab.classes.ActionWrapper.prototype:GetActionCooldown()
    return self.actionobj:GetCooldown()
end

local function GetCooldownOverride(start, duration, new_start, new_duration)
    if not start or start <= 0 or not duration or duration <= 0 then
        -- no original start or duration, use new if set
        if new_start and new_start > 0 and new_duration and new_duration > 0 then
            return new_start, new_duration
        end
    else
        -- we have an original start and duration.  Check if we have new values
        if new_start and new_start > 0 and new_duration and new_duration > 0 then
            -- we have both original and new.  Check to see which ends the latest and use that one
            if start + duration < new_start + new_duration then
                return new_start, new_duration
            end
        end
    end
    
    return start, duration
end

function uab.classes.ActionWrapper.prototype:GetCooldown(button)
    local unit = self:GetEffectUnit(button)
    local start, duration, enable
    local effectstart, effectduration, effectenable

    -- Always start with the action's cooldown.
    -- Action cooldown is the cooldown before that action can be used again.
    -- Override if effect cooldown is longer
    start, duration, enable = self:GetActionCooldown()
    
    if unit and self.cooldown then
        local priority
        local havecooldown
    
        for effect, _ in pairs(self.cooldown) do
            local options = uabUnitStatus:GetUnitStatusConfig(effect)
            local newPriority = options.priority

            -- highest priority so far?
            if not priority or newPriority > priority then
                -- yes, check to see if this effect has a cooldown
                local hasbuff, tmp_enabled, tmp_start, tmp_duration = uabUnitStatus:IsStatusGained(effect, unit, self.data)
    
                if hasbuff and tmp_enabled and tmp_enabled > 0 and tmp_start and tmp_start > 0 and tmp_duration and tmp_duration > 0 then
                    -- use this cooldown as the effect cooldown, since it is the highest priority with a cooldown so far
                    effectenable = 1
                    effectstart = tmp_start
                    effectduration = tmp_duration
                    priority = newPriority
                end
            end
        end

        -- override action cooldown with the cooldown for the highest priority effect with a cooldown, if it ends later
        if effectenable and effectenable > 0 and effectstart and effectstart > 0 and effectduration and effectduration > 0 then
            enable = 1
            start, duration = GetCooldownOverride(start, duration, effectstart, effectduration)
        end        
    end

    return start, duration, enable
end

--[[--------------------------------------------------------------------
UpdateVisualEffects

This method updates the visual effects for the specified button based
on how the unit effects for this action are set.  It goes through each
of the unit effects, and for each one, if that unit effect is "on" then
the visual effects assigned to this action for that unit effect is
applied to the specified button.

The visual effects are not applied until all unit effects are checked.

This means that the behavior of a visual effect is that it is applied
to the button whenever ANY unit effect is on, and that unit effect
indicates that the visual effect should also be on.
----------------------------------------------------------------------]]

--[[--------------------------------------------------------------------
The following set of methods are used in the processing that updates
the Visual effects for a button
----------------------------------------------------------------------]]
function AnyEffects(button)
    local visualeffect = button.visualeffect

    if visualeffect then    
        for n, v in pairs(visualeffect) do
            return true
        end
    end
end

local function ClearEffects(button)
    for name, _ in pairs(button.visualeffect) do
        button.visualeffect[name] = nil
    end
end

local function CheckPriority(button, uniteffect, visualeffectstocheck)
    local visualeffect = button.visualeffect

    -- If highlighting, then use the highest priority highlight selection
    if visualeffectstocheck.highlight then
        local options = uabUnitStatus:GetUnitStatusConfig(uniteffect)
        local newPriority = options.priority
        local highlight = options.color

        if not visualeffect.highlight or newPriority > visualeffect.highlight.priority then
            visualeffect.highlight = visualeffect.highlight or {}
            visualeffect.highlight.color = highlight
            visualeffect.highlight.priority = newPriority
        end
    end

    visualeffect.lowalpha = visualeffect.lowalpha or visualeffectstocheck.lowalpha
    visualeffect.noalpha = visualeffect.noalpha or visualeffectstocheck.noalpha
    visualeffect.blink = visualeffect.blink or visualeffectstocheck.blink
    visualeffect.notusable = visualeffect.notusable or visualeffectstocheck.notusable
end

local function CheckItemIsUsable(usable)
    return not uab.db.profile.checkusable and not uab.db.profile.hideunusable
end

local function SetButtonUsable(button, usable)
    local frame = getglobal(button:GetFrame():GetName().."Icon")
    if usable then
        frame:SetVertexColor(1.0, 1.0, 1.0, 1.0)
    else
        frame:SetVertexColor(0.25, 0.25, 0.25, 1.0)
    end
end

local function ApplyEffects(button, usable, hideunusable, defaultvisualeffects)
    if not AnyEffects(button) and defaultvisualeffects then
        CheckPriority(button, defaultStr, defaultvisualeffects)
    end

    local visualeffect = button.visualeffect
    
    usable = usable or CheckItemIsUsable()

    if usable then
        usable = not visualeffect.notusable
    elseif hideunusable then
        visualeffect.noalpha = true
    end
    
    SetButtonUsable(button, usable)

    if visualeffect.noalpha then
        button:SetAlpha(0.0)
    else
        if visualeffect.highlight then
            local color = visualeffect.highlight.color

            button:BorderColor(color.r, color.g, color.b, color.a)
        else
            button:BorderColor(1.0, 1.0, 1.0, 0.0)
        end

        if visualeffect.lowalpha then
            button:SetAlpha(uab.db.profile.alpha)
        else
            button:SetAlpha(1.0)
        end

        if visualeffect.blink then
            button:StartFlash()
        else
            button:EndFlash()
        end
    end
end

function uab.classes.ActionWrapper.prototype:GetEffectUnit(button)
    if self.useplayerunit then
        return "player"
    else
        return uab.GetFrameUnit(button:GetFrame())
    end
end

function uab.classes.ActionWrapper.prototype:ShouldHideUnusable()
    if self.sethideunusable then
        return self.hideunusable
    else
        return uab.db.profile.hideunusable
    end
end

function uab.classes.ActionWrapper.prototype:UpdateVisualEffects(button)
    if not button then
        -- Update the visual effects for all buttons that have this action
        -- wrapper as their current action wrapper.  Provided that button
        -- is visible.
        uabActionButton:UpdateVisualEffects(self)
        return
    end

    button.visualeffect = button.visualeffect or {}

    --[[
    Clear any currently applied visual effects for this button.  The
    visual effects to apply are maintained by the button, instead of
    locally in this method so that the visual effect table is not
    reallocated on each call to this method.  This reduces memory
    usage, and garbage collection requirements.
    --]]
    ClearEffects(button)
    
    local unit = self:GetEffectUnit(button)

    if unit and self.enabledeffects then
        for uniteffect, _ in pairs(self.enabledeffects) do
            if uabUnitStatus:IsStatusGained(uniteffect, unit, self) then
                --[[
                Yes, check the visual effects for this unit effect
                against the visual effects already activated for
                this button to see if any of the visual effects
                should be enabled (or override the ones already there).
                --]]
                CheckPriority(button, uniteffect, self.effects[uniteffect])
            end
        end
    end

    -- Apply the visual effects that we identified
    ApplyEffects(button, self:IsUsable(), self:ShouldHideUnusable(), self.effects and self.effects[defaultStr])
end

function uab.classes.ActionWrapper.prototype:ShowButton(button)
    self:SetButtonIcon(button)
    self:SetButtonCount(button)
    
    self:UpdateVisualEffects(button)
    
    button:SetChecked(self:IsActive())
end

