-- Update the UAB version if the version of this file is later than any other already loaded
uab.CheckVersion(tonumber(("$Revision: 64958 $"):match("%d+")), ("$Date: 2008-03-18 08:43:00 -0400 (Tue, 18 Mar 2008) $"):match("%d%d%d%d%-%d%d%-%d%d"))

--[[------------------------------------------------------------------------------
The visual button for an instance of an action.  Based on settings, the button 
can look different.  It can look like a menu item, with the icon and spell name 
showing side by side, or look like a typical action button with just the icon 
showing.  Other appearances are possible as well.  This class controls 
implementing the standard appearance of the button, and how the visual effects 
actually appear (e.g., just the icon blinks, but the text does not, or the icon 
is highlighted, but the text is not).

An ActionButton instance is allocated for each action for a UnitGroupHeader.
Each ActionButton instance is associated with a single UnitGroupHeader and a 
single Action.

When an ActionButton is shown, it adjusts it's scale to the currently set scale,
then it tells the associated Action to identify the visual effects based on the 
state of the action and the unit the ActionButton is associated with.
--------------------------------------------------------------------------------]]
uabActionButton = uab:NewModule("ActionButton", "Metrognome-2.0", "AceEvent-2.0")

uabActionButton.core = uab
uabActionButton.minlevel = 0
uabActionButton.actionButtonFrames = {}
uabActionButton.visibleButtons = {}

local DogTag = AceLibrary("LibDogTag-2.0")

function uabActionButton:OnInitialize()
	uabActionButton.super.OnInitialize(self)
end
    
function uabActionButton:OnEnable()
    self:RegisterEvent("UnitActionBars_UnitStatus_Gained", "UpdateUnitVisualEffects") -- some unit status gained
    self:RegisterEvent("UnitActionBars_UnitStatus_Lost", "UpdateUnitVisualEffects")   -- some unit status lost
end

function uabActionButton:UpdateUnitVisualEffects(name, unit)
--[[
    for button, _ in pairs(self.visibleButtons) do
        local buttonaction = button:UnitStateModifiedAction()
        local buttonunit = button:GetCurrentUnit()
        if buttonaction and (not unit or buttonunit == unit) then
            buttonaction:UpdateVisualEffects(button)
        end
    end
--]]
end

--[[
local buttonevents = {
	"PLAYER_TARGET_CHANGED",
	"ACTIONBAR_UPDATE_STATE",
	"ACTIONBAR_UPDATE_USABLE",
	"ACTIONBAR_UPDATE_COOLDOWN",
	"CRAFT_SHOW",
	"CRAFT_CLOSE",
	"TRADE_SKILL_SHOW",
	"TRADE_SKILL_CLOSE",
	"PLAYER_ENTER_COMBAT",
	"PLAYER_LEAVE_COMBAT",
	"START_AUTOREPEAT_SPELL",
	"STOP_AUTOREPEAT_SPELL",
	"ACTIONBAR_SLOT_CHANGED",
	"PLAYER_ENTERING_WORLD",
	"ACTIONBAR_PAGE_CHANGED",
	"ACTIONBAR_SHOWGRID",
	"ACTIONBAR_HIDEGRID",
	"UPDATE_BINDINGS",
	"UPDATE_SHAPESHIFT_FORM",
    "UNIT_SPELLCAST_START",
    "UNIT_SPELLCAST_STOP",
    "UNIT_SPELLCAST_FAILED",
    "UNIT_SPELLCAST_INTERRUPTED",
    "UNIT_SPELLCAST_CHANNEL_START",
    "UNIT_SPELLCAST_CHANNEL_STOP",
}
--]]

local function OnActionButtonFrameUpdateFunc(button, elapsed)
	local self = button.uabobj
    if not self.isvisible then
        self.elapsed = 0
        return
    end

	self.elapsed = self.elapsed + elapsed
	if self.elapsed > 0.4 then
        local action = self:UnitStateModifiedAction()

   		self:TimerEventWithAction(self.elapsed, action)
       	self:UpdateWithAction(action)
		
        self.elapsed = 0
	end
end

--[[
local function OnActionButtonFrameEventFunc(button, event)
	local self = button.uabobj
    if not self.isvisible then return end
    if event == "UNIT_SPELLCAST_START" or event == "UNIT_SPELLCAST_STOP" or
        event == "UNIT_SPELLCAST_FAILED" or event == "UNIT_SPELLCAST_INTERRUPTED" or
        event == "UNIT_SPELLCAST_CHANNEL_START" or event == "UNIT_SPELLCAST_CHANNEL_STOP" then
        if arg1 ~= "player" then
            return
        end
    end

	self:Update()
end
--]]

--[[
The function that is called when some event occurs that requires us to
update the visual characteristics of the action buttons.
--]]
function uabActionButton:UpdateVisualEffects(action)
    for button, _ in pairs(self.visibleButtons) do
        local buttonaction = button:UnitStateModifiedAction()
        if buttonaction and buttonaction == action then
            action:UpdateVisualEffects(button)
        end
    end
end

function uabActionButton:SetVisible(button)
    button.isvisible = true
    self.visibleButtons[button] = true
    button:Update()
end

function uabActionButton:ClearVisible(button)
    button.isvisible = nil
    self.visibleButtons[button] = nil
end

function uabActionButton:GetActionButtons()
    return self.actionButtonFrames
end

function uabActionButton:RegisterActionButton(frame)
    if not self.actionButtonFrames[frame] then
        self:TriggerEvent("UnitActionBars_ActionButtonCreated", frame)
        self.actionButtonFrames[frame] = true
    end
end

local function IsInRaid()
    return GetNumRaidMembers() ~= 0
end

local function IsGrouped()
    return GetNumPartyMembers() ~= 0 or IsInRaid()
end

uab.classes.ActionButton = uab.mixins.aceOO.Class()

--[[
This class encapsulates access to all the action sets, and the action buttons within each of
those action sets, for a given activator, stance, and unit group.
--]]
function uab.classes.ActionButton.prototype:init(activatorcode, stancecode, unitgroupcode, actionsetcode, index)
	uab.classes.ActionButton.super.prototype.init(self)

    self.isenabled = true
    self.pendingattributes = {}
    self.flashing = {}
    self.activatorcode = activatorcode
    self.stancecode = stancecode
    self.unitgroupcode = unitgroupcode
    self.actionsetcode = actionsetcode
    self.index = index

    self.setattributes = {}
    self.showstates = {}
    self.posstates = {}

    self.actionobj = nil
    self.secureheadercode = -1
    
    self.framename = "UAB_ActionButton_"..self.activatorcode.."_"..self.stancecode.."_"..self.unitgroupcode.."_"..self.actionsetcode.."_"..self.index
    self.maxalpha = 1.0

    --[[
	Create the action button frame
	--]]
    self.frame = getglobal(self.framename)
    if not self.frame then
        self.frame = CreateFrame("CheckButton", self.framename, nil, "UAB_StandardActionButtonTemplate")
    end

	local frame = self.frame

    frame.uabobj = self
    uab.SetFrameLevelAndStrata(frame, 1)
    frame:Hide()
    
    frame.icon = getglobal(frame:GetName().."Icon")
    frame.itemcount = getglobal(frame:GetName().."Count")
    frame.cooldown = getglobal(frame:GetName().."Cooldown")

--    self.frame.border = getglobal(self.frame:GetName().."Border")

    frame.border = frame:CreateTexture(frame:GetName().."Border", "OVERLAY")
    frame.border:SetBlendMode("ADD")
    self:SetBorderTexture("Interface\\AddOns\\UnitActionBars\\images\\swborder2")

    frame.normaltexture = getglobal(frame:GetName().."NormalTexture")

    -- border fix-ups to smooth icon appearance a bit (borrowed from Bongos)
	frame.icon:SetTexCoord(0.06, 0.94, 0.06, 0.94)
        
    frame.cooldown:SetFrameLevel(2)

--    frame.cooldown:Show()
    frame.itemcount:Show()

    frame.border:Hide()

    self:HighlightBorderChanged()

    self:SetAttribute(frame, "useparent-unit", true)
    self:SetAttribute(frame, "useparent-unitsuffix", true)

    frame:SetScript("OnShow", function(self) self.uabobj:OnShow() end)
    frame:SetScript("PostClick", function(self, button) self.uabobj:PostClick(button) end)
    frame:SetScript("OnHide", function(self) self.uabobj:OnHide() end)
    frame:SetScript("OnEnter", function(self, motion) self.uabobj:ShowTooltip() end)
    frame:SetScript("OnLeave", function(self, motion) GameTooltip:Hide() end)
    
    self:PrepareForLayouts()
    self:LayoutsDone()

    self.elapsed = 0

    frame:SetScript("OnUpdate", OnActionButtonFrameUpdateFunc)

--    frame:SetScript("OnEvent", OnActionButtonFrameEventFunc)
--	for _, event in pairs(buttonevents) do
--		frame:RegisterEvent(event)
--	end
    
    uabActionButton:RegisterActionButton(frame)
end

function uab.classes.ActionButton.prototype:Dump(dumper)
    dumper:Dump("Action Button")
    dumper:IncIndent()
    dumper:Dump("Frame Name     : "..self.frame:GetName())
    dumper:Dump("Activator Code : "..self.activatorcode)
    dumper:Dump("Stance Code    : "..self.stancecode)
    dumper:Dump("Unit Group Code: "..self.unitgroupcode)
    dumper:Dump("Action Set Code: "..self.actionsetcode)
    dumper:Dump("Index          : "..self.index)
    dumper:Dump("Parent         : "..self.frame:GetParent():GetName())
    dumper:Dump("Attributes")

    dumper:IncIndent()
    dumper:DumpFrameAttribute(self.frame, "useparent-unit")
    dumper:DumpFrameAttribute(self.frame, "useparent-unitsuffix")
    dumper:DumpFrameAttribute(self.frame, "newstate")
    dumper:DumpFrameAttribute(self.frame, "ofspoint")
    dumper:DumpFrameAttribute(self.frame, "ofsx")
    dumper:DumpFrameAttribute(self.frame, "ofsy")
    dumper:DumpFrameAttribute(self.frame, "showstates")
    dumper:DumpFrameAttribute(self.frame, "hidestates")

    for name, _ in pairs(self.setattributes) do
		dumper:DumpFrameAttribute(self.frame, name)
    end
    
    dumper:DecIndent()
    dumper:DecIndent()
end

function uab.classes.ActionButton.prototype:GetFrame()
    return self.frame
end

function uab.classes.ActionButton.prototype:ApplyPendingAttributes()
	for name, val in pairs(self.pendingattributes) do
		for frame, data in pairs(val) do
    		uab.SetFrameAttribute(frame, name, data.val)
            val[frame] = nil
        end
        self.pendingattributes[name] = nil
	end
end

function uab.classes.ActionButton.prototype:SetAttribute(frame, attribute, val)
    if uab:IsPendingAttributes() then
        if not self.pendingattributes[attribute] then
            self.pendingattributes[attribute] = {}
        end
        self.pendingattributes[attribute][frame] = { val = val }
    else
        self:ApplyPendingAttributes()
        uab.SetFrameAttribute(frame, attribute, val)
    end
end

function uab.classes.ActionButton.prototype:ClearAttributes(attributes)
	for name, _ in pairs(attributes) do
		self.setattributes[name] = nil
        self:SetAttribute(self.frame, name, nil)
	end
end

function uab.classes.ActionButton.prototype:SetAttributes(attributes)
	for name, val in pairs(attributes) do
		self:SetAttribute(self.frame, name, val)
		self.setattributes[name] = true
	end
end

function uab.classes.ActionButton.prototype:ReApplyStayOpen(activatorcode)
    local setting = uab.db.profile.activators[activatorcode] and uab.db.profile.activators[activatorcode].stayopen
    local stayopen

    if not setting then
        stayopen = uab.db.profile.stayopen
    else
        stayopen = setting.enabled
    end
    
    if not stayopen then
        -- Set newstate so that the buttons always hide on a click
        self:SetAttribute(self.frame, "newstate", "*:0")
    else
        -- Set newstate so that the buttons stay visible on a click
        self:SetAttribute(self.frame, "newstate", nil)
    end
end

function uab.classes.ActionButton.prototype:SetSecureHeaderFromCode(secureheadercode)
    self.secureheadercode = secureheadercode
    if self.secureheadercode ~= -1 then
        local secureheaderobj = uab:GetSecureHeaderObj(self.secureheadercode)

        -- Setting the "addchild" attribute also causes the secureheader frame
        -- to be the parent frame of the attribute value (our frame), so we don't
        -- need to set that independently.
        self:SetAttribute(secureheaderobj:GetFrame(), "addchild", self.frame)
    end
end

function uab.classes.ActionButton.prototype:SetChecked(setit)
    if setit then
        self.frame:SetChecked(1)
    else
        self.frame:SetChecked(0)
    end
end

function uab.classes.ActionButton.prototype:BorderColor(r, g, b, a)
    local border = self.frame.border
    if a == 0 then
        border:Hide()
    else
        border:SetVertexColor(r,g,b,a)
        border:Show()
    end
end

function uab.classes.ActionButton.prototype:SetAlpha(alpha)
    self.maxalpha = alpha
end

function uab.classes.ActionButton.prototype:SetIcon(icon)
    if not self.icon or not icon or self.icon ~= icon then
        self.icon = icon
        if self.frame.icon then
            self.frame.icon:SetTexture(icon)
        end
    end
end

function uab.classes.ActionButton.prototype:SetCount(count)
    local btncount = self.frame.itemcount
    if count then
        btncount:Show()
        if not self.count or count ~= self.count then
            self.count = count
            btncount:SetText(count)
        end
    else
        self.count = nil
        btncount:Hide()
    end
end

function uab.classes.ActionButton.prototype:StartFlash()
    local flashing = self.flashing
    if not flashing.enabled then
        flashing.timer = uab.db.profile.blinkcycle/2
        flashing.direction = 1
        flashing.enabled = true
    end
end

function uab.classes.ActionButton.prototype:EndFlash()
    self.flashing.enabled = nil
end

function uab.classes.ActionButton.prototype:HighlightBorderChanged()
    local width = uab.db.profile.borderwidth
     
    if width > 5 then
        width = 5
    elseif width < 2 then
        width = 2
    end

    self:SetBorderTexture("Interface\\AddOns\\UnitActionBars\\images\\swborder" .. width)
end

function uab.classes.ActionButton.prototype:SetBorderTexture(texture)
    local border = self.frame.border

    border:SetTexture(texture)
    border:SetWidth(40)
    border:SetHeight(40)
    border:ClearAllPoints()
    border:SetPoint("CENTER", self.frame, "CENTER", 0, -1)
end

function uab.classes.ActionButton.prototype:SetFriendlyAction(actionobj)
    if self.actionobj then
        self.actionobj:RemoveButton(self)
    end
    self.actionobj = actionobj
    self.actionobj:AddButton(self)
    self:Update()
end

function uab.classes.ActionButton.prototype:ClearFriendlyAction()
    if self.actionobj then
        self.actionobj:RemoveButton(self)
    end
    self.actionobj = nil
    self:Update()
end

function uab.classes.ActionButton.prototype:AnyAction()
    if self.actionobj or self.hostileactionobj then
        return true
    end
end

function uab.classes.ActionButton.prototype:Release()
    self:ClearFriendlyAction()
    self:ClearHostileAction()
	for name, _ in pairs(self.setattributes) do
		self.setattributes[name] = nil
    end
    self:PrepareForLayouts()
    self:LayoutsDone()

    self.frame:SetParent(UIParent)  -- make sure the secure header does not consider this frame a child frame
    self.frame:Hide()
end

function uab.classes.ActionButton.prototype:SetHostileAction(actionobj)
    if self.hostileactionobj then
        self.hostileactionobj:RemoveButton(self)
    end
    self.hostileactionobj = actionobj
    self.hostileactionobj:AddButton(self)
    self:Update()
end

function uab.classes.ActionButton.prototype:ClearHostileAction()
    if self.hostileactionobj then
        self.hostileactionobj:RemoveButton(self)
    end
    self.hostileactionobj = nil
    self:Update()
end

function uab.classes.ActionButton.prototype:SetFrameLevel(level)
    local frame = self.frame

    frame.cooldown:SetFrameLevel(level+1)
    uab.SetFrameLevelAndStrata(frame, level)
end

function uab.classes.ActionButton.prototype:GetCurrentUnit()
    return uab.GetFrameUnit(self.frame)
end

function uab.classes.ActionButton.prototype:UnitStateModifiedAction()
    local unit = self:GetCurrentUnit()
    local actionobj = self.actionobj
    
    if unit and UnitCanAttack("player", unit) and self.hostileactionobj then
        actionobj = self.hostileactionobj
    end
    
    return actionobj
end

function uab.classes.ActionButton.prototype:Update(action)
    if not action then
        action = self:UnitStateModifiedAction()
    end

    self:UpdateWithAction(action)
end

function uab.classes.ActionButton.prototype:UpdateWithAction(action)
    self:UpdateCooldownWithAction(action)
    self:StatusChangedWithAction(action)
end
    
function uab.classes.ActionButton.prototype:UpdateCooldown(action)
    --[[
    This method just deals with updating the cooldown timer frame for
    this action button.
    --]]
    if not action then
        action = self:UnitStateModifiedAction()
    end

    self:UpdateCooldownWithAction(action)
end

function uab.classes.ActionButton.prototype:UpdateCooldownWithAction(action)
    if action then
        local start, duration, enable = action:GetCooldown(self)
    
        if enable and enable > 0 and start and start > 0 and duration and duration > 0 then
            CooldownFrame_SetTimer(self.frame.cooldown, start, duration, enable)
        else
            CooldownFrame_SetTimer(self.frame.cooldown, 0, 0, 0)
        end
    end
end

function uab.classes.ActionButton.prototype:StatusChanged(action)
    --[[
    Some visual aspect of the button might need to be updated, including
    making the button "hidden" (alpha = 0).
    --]]
    if not action then
        action = self:UnitStateModifiedAction()
    end

    self:StatusChangedWithAction(action)
end

function uab.classes.ActionButton.prototype:StatusChangedWithAction(action)
    if action then
        action:ShowButton(self)
    else
        self.frame:SetAlpha(0)
    end
end

function uab.classes.ActionButton.prototype:TimerEvent(elapsed, action)
    if not action then
        action = self:UnitStateModifiedAction()
    end

    self:TimerEventWithAction(elapsed, action)
end
    
function uab.classes.ActionButton.prototype:TimerEventWithAction(elapsed, action)
    --[[
    This method just deals with updating the alpha for the button
    frame (which includes flashing if appropriate).  It also shows
    the tooltip if needed.
    --]]
    
    if not self.isenabled then
        return
    end
    
    local a = self.maxalpha

    local flashing = self.flashing
    if action then
        local cycletime = uab.db.profile.blinkcycle
        if flashing.enabled then
            flashing.timer = flashing.timer - elapsed
            if flashing.timer < 0 then
                flashing.timer = cycletime/2
                if flashing.direction then
                    flashing.direction = nil
                else
                    flashing.direction = 1
                end
            end
        
            if flashing.direction then
                a = ((flashing.timer * 2)/cycletime)*self.maxalpha
            else
                a = (1 - ((flashing.timer * 2)/cycletime))*self.maxalpha
            end
        end
    else
        a = 0
        flashing.enabled = nil
    end

    self.frame:SetAlpha(a)

    --Tooltip stuff, probably for the cooldown timer
    if self.updateTooltip then
        self.updateTooltip = self.updateTooltip - elapsed
        if self.updateTooltip <= 0 then
            if GameTooltip:IsOwned(self) and action then
                self:ShowTooltip(action)
            end
            self.updateTooltip = TOOLTIP_UPDATE_TIME
        end
    end
end

function uab.classes.ActionButton.prototype:PrepareForLayouts()
    local info = self.showstates
    for idx, _ in pairs(info) do
        info[idx] = nil
    end

    info = self.posstates
    for x, _ in pairs(info) do
        info[x].pt = nil
        info[x].x = nil
        info[x].y = nil
        info[x] = nil
    end
end

function uab.classes.ActionButton.prototype:LayoutsDone()
    local ptattr, xattr, yattr
    local frame = self.frame
    
    for posstates, pos in pairs(self.posstates) do
        if not ptattr then
            ptattr = ""
            xattr = ""
            yattr = ""
        else
            ptattr = ptattr .. ";"
            xattr = xattr .. ";"
            yattr = yattr .. ";"
        end

        ptattr = ptattr .. posstates .. ":" .. pos.pt
        xattr = xattr .. posstates .. ":" .. pos.x
        yattr = yattr .. posstates .. ":" .. pos.y
    end

    self:SetAttribute(frame, "ofspoint", ptattr)
    self:SetAttribute(frame, "ofsx", xattr)
    self:SetAttribute(frame, "ofsy", yattr)

    local showstates

    for states, _ in pairs(self.showstates) do
        if not showstates then
            showstates = ""
        else
            showstates = showstates .. ","
        end

        showstates = showstates .. states
    end

    self:SetAttribute(frame, "showstates", showstates)
    if not showstates then
        self:SetAttribute(frame, "hidestates", "*")
    else
        self:SetAttribute(frame, "hidestates", nil)
    end
end
    
function uab.classes.ActionButton.prototype:AddShowStates(states)
    self.showstates[states] = true
end

function uab.classes.ActionButton.prototype:SetPointForStates(states, ofspoint, ofsx, ofsy)
    self.posstates[states] = { pt = ofspoint, x = ofsx, y = ofsy }
end

function uab.classes.ActionButton.prototype:OnShow()
    uabActionButton:SetVisible(self)
    
--[[
    -- parent starts as the header
    local parent = self.frame:GetParent()
    -- make parent be the control (parent of the header)
    parent = parent:GetParent()
    -- now, make it be the parent of the control (the unit frame)
    parent = parent:GetParent()

    self.frame:SetFrameLevel(parent:GetFrameLevel()+10)
--]]
    -- uab.SetFrameLevelAndStrata(self.frame, 1)
end

function uab.classes.ActionButton.prototype:OnHide()
    uabActionButton:ClearVisible(self)
end

function uab.classes.ActionButton.prototype:ShowTooltip(action)
    if not action then
        action = self:UnitStateModifiedAction()
    end
             
    self:ShowTooltipWithAction(action)
end

function uab.classes.ActionButton.prototype:ShowTooltipWithAction(action)
    if action then
        action:ShowTooltip(self.frame)
    end
end

function uab.classes.ActionButton.prototype:PrintAttr(attr)
    local fname = self.frame:GetName()
    if not fname then
        fname = "Unknown frame (no name)"
    end
    local hdr = "Attribute ("..fname..") - "..attr
    local val = self.frame:GetAttribute(attr)
    if not val then
        uab.print(hdr.." : nil")
    elseif type(val) == "string" then
        uab.print(hdr.." : "..val)
    elseif type(val) == "boolean" then
        uab.print(hdr.." : true")
    elseif type(val) == "table" and val.GetName then
        uab.print(hdr.." : "..val:GetName())
    else
        uab.print(hdr.." : raw value")
    end
end

function uab.classes.ActionButton.prototype:PostClick(button)
    local action = self:UnitStateModifiedAction()
    
    if action then
        local msgInfo = action:GetMessageInfo()
        if msgInfo then
            self:ShowActionMessage(msgInfo)
        end
    end
end

function uab.classes.ActionButton.prototype:ShowActionMessage(msgInfo)
    if msgInfo.enable and msgInfo.msg then
        local channel = nil
        local unit = uab.GetFrameUnit(self.frame)
        local msg

        if unit then
            msg = DogTag:Evaluate(unit, msgInfo.msg)
        
            if IsInRaid() then
                if msgInfo.groupch == "Raid" then
                    channel = "RAID"
                elseif msgInfo.groupch == "Party" then
                    channel = "PARTY"
                elseif msgInfo.groupch == "None" then
                    channel = "NONE"
                end
            end
            
            if not channel and IsGrouped() then
                if msgInfo.groupch == "Raid" or msgInfo.groupch == "Party" then
                    channel = "PARTY"
                elseif msgInfo.groupch == "None" then
                    channel = "NONE"
                end
            end
            
            if not channel then
                if msgInfo.defaultch == "Say" then
                    channel = "SAY"
                elseif msgInfo.defaultch == "Emote" then
                    channel = "EMOTE"
                elseif msgInfo.defaultch == "Channel" then
                    channel = "CHANNEL"
                elseif msgInfo.defaultch == "Self" then
                    channel = "SELF"
                else
                    channel = "NONE"
                end
            end
    
            if channel and channel ~= "NONE" then
                if channel == "SELF" then
                    uab.print(msg)
                elseif channel == "CHANNEL" then
                    SendChatMessage(msg, channel, "COMMON", msgInfo.channel)
                else
                    SendChatMessage(msg, channel)
                end
            end        
        end
    end
end

