local L = LibStub( "AceLocale-3.0" ):GetLocale( "UnderHood" )

local UnitHealth = UnitHealth
local UnitHealthMax = UnitHealthMax
local UnitMana = UnitMana
local UnitManaMax = UnitManaMax
local UnitPowerType = UnitPowerType
local UnitExists = UnitExists
local UnitCastingInfo = UnitCastingInfo
local UnitChannelInfo = UnitChannelInfo

UnderHood = LibStub( "AceAddon-3.0" ):NewAddon( "UnderHood", "AceEvent-3.0", "AceTimer-3.0", "AceConsole-3.0" )

UnderHood.Version = "1.3"
UnderHood.Revision = string.match( "$Revision: $", "%d+" ) or -1

UnderHood.playerLoggedIn = false

UnderHood:SetDefaultModuleState( false )

function UnderHood:OnInitialize()
	self.alpha = 1
	self.ParentFrame = CreateFrame( "Frame", "UnderHoodParent", UIParent )
	self.ParentFrame:SetWidth( 200 )
	self.ParentFrame:SetHeight( 200 )
	self.ParentFrame:SetAlpha( 1 )
	self.ParentFrame:SetPoint( "CENTER", UIParent, "CENTER" )
	self.ParentFrame:Show()

	self.OptionsHelper:Initialize()
	self:SetupConfig()
	self:SetupOptions()

	self:Print( string.format( L["Version %s.%d. Use |cff00ff00/uh|r to configure."], self.Version, self.Revision ) )
end

function UnderHood:OnEnable()
	self.inCombat = InCombatLockdown()

	self:RegisterEvent( "PLAYER_LOGIN" )
	self:RegisterEvent( "PLAYER_REGEN_DISABLED" )
	self:RegisterEvent( "PLAYER_REGEN_ENABLED" )
	self:RegisterEvent( "PLAYER_TARGET_CHANGED", "UpdateAlpha" )
	self:RegisterEvent( "PLAYER_FOCUS_CHANGED", "UpdateAlpha" )
	self:RegisterEvent( "UNIT_HEALTH", "UpdateAlphaIfPlayer" )
	self:RegisterEvent( "UNIT_MANA", "UpdateAlphaIfPlayer" )
	self:RegisterEvent( "UNIT_ENERGY", "UpdateAlphaIfPlayer" )
	self:RegisterEvent( "UNIT_RAGE", "UpdateAlphaIfPlayer" )
	self:RegisterEvent( "UNIT_PET", "UpdateAlphaIfPlayer" )

	self:RegisterEvent( "UNIT_SPELLCAST_START", "UpdateAlphaIfPlayer" )
	self:RegisterEvent( "UNIT_SPELLCAST_STOP", "UpdateAlphaIfPlayer" )
	self:RegisterEvent( "UNIT_SPELLCAST_FAILED", "UpdateAlphaIfPlayer" )
	self:RegisterEvent( "UNIT_SPELLCAST_DELAYED", "UpdateAlphaIfPlayer" )
	self:RegisterEvent( "UNIT_SPELLCAST_INTERRUPTED", "UpdateAlphaIfPlayer" )
	self:RegisterEvent( "UNIT_SPELLCAST_CHANNEL_START", "UpdateAlphaIfPlayer" )
	self:RegisterEvent( "UNIT_SPELLCAST_CHANNEL_UPDATE", "UpdateAlphaIfPlayer" )
	self:RegisterEvent( "UNIT_SPELLCAST_CHANNEL_STOP", "UpdateAlphaIfPlayer" )
	self:RegisterEvent( "UNIT_SPELLCAST_CHANNEL_INTERRUPTED", "UpdateAlphaIfPlayer" )

	self:UpdateModulesForProfile()
	self:UpdateBlizzardFrames()

	self.ParentFrame:SetScale( self.db.profile.general.scale )
	self:UpdateAlpha()

	self:ScheduleRepeatingTimer( "ProcessUpdate", 0.15 )
end

function UnderHood:PLAYER_LOGIN()
	self.playerLoggedIn = true

	self:SendMessage( "UnderHood_PlayerLogin" )
end

function UnderHood:PLAYER_REGEN_DISABLED()
	self.inCombat = true

	self:UpdateAlpha()
end

function UnderHood:PLAYER_REGEN_ENABLED()
	self.inCombat = nil

	self:UpdateAlpha()
end

function UnderHood:InConfigMode()
	return self.db.profile.general.config
end

function UnderHood:UpdateAlphaIfPlayer( event, uid )
	if uid ~= "player" then return end

	self:UpdateAlpha()
end

function UnderHood:UpdateAlpha()
	local alpha
	local pt = UnitPowerType( "player" )
	local as = self.db.profile.alpha

	if self.db.profile.general.config then
		alpha = 1
	elseif self.inCombat then
		alpha = as.combat
	elseif as.enableTarget and UnitExists( "target" ) then
		alpha = as.target
	elseif as.enableFocus and UnitExists( "focus" ) then
		alpha = as.focus
	elseif as.enablePet and UnitExists( "pet" ) then
		alpha = as.pet
	elseif as.enableRegen and
			(
				UnitHealth( "player" ) ~= UnitHealthMax( "player" ) or -- All classes
				(pt == 0 and UnitMana( "player" ) ~= UnitManaMax( "player" )) or -- Casters
				(pt == 1 and UnitMana( "player" ) > 0) or -- Warriors and Bear-druids
				(pt == 3 and UnitMana( "player" ) ~= UnitManaMax( "player" )) -- Rogues and Cat-druids
			) then
		alpha = as.regen
	elseif as.enableCasting and (UnitCastingInfo( "player" ) or UnitChannelInfo( "player" )) then
		alpha = as.casting
	else
		alpha = as.ooc
	end

	if alpha ~= self.alpha then
		self.alpha = alpha

		self.ParentFrame:SetAlpha( alpha )
	end
end

function UnderHood:GetAlpha()
	return self.alpha
end

--[[
	Regular Update handling.

	Each object that wants to participate in unified update should register in
	UnderHood and implement ProcessUpdate method.
--]]

local objectsToUpdate = {}

function UnderHood:RegisterUpdateTarget( target, method )
	if type( target ) ~= "table" then
		error( ("Bad argument #2 to 'RegisterUpdateTarget'. Expected %q, got %q."):format( "table", type( target ) ), 2 )
	end

	if not method then method = "ProcessUpdate" end

	if type( target[method] ) ~= "function" then
		error( ("Bad argument #2 to 'RegisterUpdateTarget'. No %q method declared."):format( method ), 2 )
	end

	if objectsToUpdate[target] then
		error( "Double registration detected in 'RegisterUpdateTarget'.", 2 )
	end

	objectsToUpdate[target] = target[method]
end

function UnderHood:UnregisterUpdateTarget( target )
	if type( target ) ~= "table" then
		error( ("Bad argument #2 to 'UnregisterUpdateTarget'. Expected %q, got %q."):format( "table", type( target ) ), 2 )
	end

	if objectsToUpdate[target] then
		objectsToUpdate[target] = nil
	end
end

function UnderHood:ProcessUpdate()
	if not next( objectsToUpdate ) then return end

	for target, method in pairs( objectsToUpdate ) do
		method( target )
	end
end

local UnitSendEvents = setmetatable({player=true, target=true, pet=true, mouseover=true}, {__index=function(self, unit)
	if not unit then
		error("Argument #1 to `UnitSendEvents' should be a string, got nil.", 2)
	end
	local party = unit:match("^party(%d)$") or unit:match("^partypet(%d)$")
	if party then
		party = tonumber(party)
		if party and party >= 1 and party <= 4 then
			self[unit] = true
			return true
		end
		self[unit] = false
		return false
	end
	local raid = unit:match("^raid(%d%d?)$")
	if raid then
		raid = tonumber(raid)
		if raid and raid >= 1 and raid <= 40 then
			self[unit] = true
			return true
		end
		self[unit] = false
		return false
	end
	self[unit] = false
	return false
end})

UnderHood.UnitSendEvents = UnitSendEvents

function UnderHood:GetUnitChangedEvent( unit )
	if not unit then
		error("Argument #1 to `GetUnitChangedEvent' should be a string, got nil.", 2)
	end

	if unit == "player" then return "PLAYER_ENTERING_WORLD"
	elseif unit == "target" then return "PLAYER_TARGET_CHANGED"
	elseif unit == "focus" then return "PLAYER_FOCUS_CHANGED"
	elseif unit == "pet" then return "UNIT_PET", true, "player"
	end

	local idx = unit:match( "^partypet(%d)$" )
	if idx then return "UNIT_PET", true, "party"..idx end

	if unit:match( "^party(%d)$" ) then return "PARTY_MEMBERS_CHANGED" end
end

function UnderHood:Debug( ... )
	if not( self.db ) or not( self.db.profile.general.debug ) then return end

	local count = select( "#", ... );
	local text = nil;

	if count == 0 then text = ""
	elseif count == 1 then text = tostring( select( 1, ... ) )
	elseif type( select( 1, ... ) ) == "string" and strfind( select( 1, ... ), "%%" ) then text = string.format( ... )
	else
		text = "";

		for i=1, count do
			if i > 1 then
				text = text..", "
			end

			text = text..tostring( select( i, ... ) )
		end
	end

	DEFAULT_CHAT_FRAME:AddMessage( "|cffffffffUnderHood |r|cff00ffff(Debug)|r|cffffffff: |r"..text );
end
