local tableCache = setmetatable( {}, { __mode='k' } )
local frameCache = {}
local frameCount = {}
local myFrames = {}

-- Table pools

function UnderHood.NewTable()
	local t = next( tableCache )
	
	if t then
		tableCache[t] = nil
		return t
	else
		return {}
	end
end

function UnderHood.NewList( ... )
	local t = next( tableCache )
	local count = select( "#", ... )
	
	if t then
		tableCache[t] = nil
		
		for i = 1, count do
		    t[i] = select( i, ... )
		end
		
		return t
	else
		return { ... }
	end
end

function UnderHood.ReleaseTable( t )
	if not t then return end
	
	for k in pairs( t ) do
		t[k] = nil
	end
	
	tableCache[t] = true
	
	return nil
end

-- Frame pools

function UnderHood:CreateFrame( frameType, parent, ... )
	if type( frameType ) ~= "string" then
		error( "Bad argument passed to CreateFrame" )
	end

	if not parent then parent = self.ParentFrame or UIParent end

	local frameTypeCache = frameCache[frameType]
	local frame

	if frameTypeCache then
		frame = next( frameTypeCache )
		frameTypeCache[frame] = nil

		if not next( frameTypeCache ) then
			frameCache[frameType] = nil
		end

		frame:SetParent( parent )
		frame:ClearAllPoints()
		frame:SetAlpha( 1 )

		if frameType == "Texture" then
			frame:SetTexture( nil )
			frame:SetVertexColor( 1, 1, 1, 1 )
		end

		frame:Show()
	else
		frameCount[frameType] = (frameCount[frameType] or 0) + 1
		local name = "UnderHood"..frameType..frameCount[frameType]

		if frameType == "Texture" then
			frame = parent:CreateTexture( name, ... )
		elseif frameType == "FontString" then
			frame = parent:CreateFontString( name, ... )
		elseif frameType == "Cooldown" then
		    frame = CreateFrame( "Cooldown", name, parent, "CooldownFrameTemplate" )
		else
			frame = CreateFrame( frameType, name, parent, ... )
		end

		myFrames[frame] = true
	end

	return frame
end

function UnderHood:ReleaseFrame( frame )
	if type( frame ) ~= "table" then
		error( "Bad argument passed to ReleaseFrame" )
	elseif type( frame.GetObjectType ) ~= "function" then
		error( "Attempt to release non-frame object." )
	end

	if not myFrames[frame] then
		error( "Attempt to release foreign frame." )
	end

	local frameType = frame:GetObjectType()

	if frameType == "FontString" then
		-- DogTag:RemoveFontString( frame )
		frame:SetText("")
		frame:SetJustifyH("CENTER")
		frame:SetJustifyV("MIDDLE")
		frame:SetNonSpaceWrap(true)
		frame:SetTextColor(1, 1, 1, 1)
		frame:SetFontObject(nil)
	elseif frameType == "Texture" then
		frame:SetTexture([[Interface\Buttons\WHITE8X8]])
		frame:SetVertexColor(0, 0, 0, 0)
		frame:SetBlendMode("BLEND")
		frame:SetDesaturated(false)
		frame:SetTexCoord(0, 1, 0, 1)
		frame:SetTexCoordModifiesRect(false)
	elseif frameType == "StatusBar" then
		frame:SetStatusBarColor(1, 1, 1, 1)
		frame:SetStatusBarTexture(nil)
		frame:SetValue(1)
		frame:SetOrientation("HORIZONTAL")
	end

	if frameType ~= "Texture" and frameType ~= "FontString" and not _G.OmniCC and _G.UnitClassBase then
		if frame:GetNumRegions() > 0 then
			error(("Deleting a frame of type %q that still has %d regions"):format(frameType, frame:GetNumRegions()), 2)
		elseif frame:GetNumChildren() > 0 then
			error(("Deleting a frame of type %q that still has %d children"):format(frameType, frame:GetNumChildren()), 2)
		end
	end

	frame:ClearAllPoints()
	frame:SetPoint("LEFT", UIParent, "RIGHT", 1e5, 0)
	frame:Hide()

	if frame.SetBackdrop then
		frame:SetBackdrop(nil)
	end

	frame:SetParent(UIParent)
	frame:SetAlpha(0)
	frame:SetHeight(0)
	frame:SetWidth(0)

	local frameTypeCache = frameCache[frameType]

	if not frameTypeCache then
		frameTypeCache = {}
		frameCache[frameType] = frameTypeCache
	end

	if frameTypeCache[frame] then
		error(("Double-free frame syndrome of type %q"):format(frameType), 2)
	end

	frameTypeCache[frame] = true

	return nil
end
