local UH = UnderHood
local UHT = UnderHood_Texts
local SML = LibStub( "LibSharedMedia-3.0" )
local DogTag = LibStub( "LibDogTag-3.0" )
local L = LibStub( "AceLocale-3.0" ):GetLocale( "UnderHood" )
local newTable, delTable = UH.NewTable, UH.ReleaseTable

local Text = {}
Text.__index = Text
UHT.FramePrototype = Text

function Text:Create( module, config )
	local self = {}
	setmetatable( self, Text )

	self.module = module
	self.config = config

	self.frame = UH:CreateFrame( "Button", nil, "SecureUnitButtonTemplate" ) -- parented to UnderHoodParentFrame
	self.frame:SetWidth( self.config.width )
	self.frame:SetHeight( self.config.height )
	self.frame:SetScale( self.config.scale )
	self.frame:SetFrameStrata( self.config.strata )
	self.frame:SetFrameLevel( self.config.level )
	self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.config.x, self.config.y )
	self.frame:SetAttribute( "unit", self.config.unit )

	self.texture = UH:CreateFrame( "Texture", self.frame, "BACKGROUND" )
	self.texture:SetTexture( 0, 0, 1, 0.4 )
	self.texture:SetAllPoints( true )
	self.texture:Hide()

	self.text = UH:CreateFrame( "FontString", self.frame )
	self.text:SetAllPoints( self.frame )
	self.text:SetJustifyH( self.config.hAlign or "LEFT" )
	self.text:SetJustifyV( self.config.vAlign or "MIDDLE" )
	self.text:SetShadowColor( 0, 0, 0, 1 )
	self.text:SetShadowOffset( 1, -1 )

	local fontFile = SML:Fetch( "font", self.config.fontName )
	self.text:SetFont( fontFile, self.config.fontSize )
	self.text:Hide()

	self:SetupOptions()

	if self.config.enabled then
		self:Enable()
	end

	return self
end

function Text:Destroy()
	self:Disable()

	if self.text then
		UH:ReleaseFrame( self.text )
		self.text = nil
	end

	if self.texture then
		UH:ReleaseFrame( self.texture )
		self.texture = nil
	end

	if self.frame then
		self.frame.menu = nil

		UH:ReleaseFrame( self.frame )
	end

	self.frame = nil
end

function Text:Enable()
	if self.enabled then return end

	self:UpdateInteractivity()

	self.frame:SetScript( "OnShow", function() self:OnShow() end )

	if self:InConfigMode() then
		self.text:SetText( self.config.name )
		self.text:Show()
		self.texture:Show()
		self.frame:Show()
	else
		local kwargs = newTable()
		kwargs.unit = self.config.unit
		DogTag:AddFontString( self.text, self.frame, self.config.text, "Items;Unit", kwargs )
		delTable( kwargs )

		self.text:Show()
		self.texture:Hide()
		RegisterUnitWatch( self.frame )
	end

	if not self:InConfigMode() and not UH.UnitSendEvents[self.config.unit] then
		UH:RegisterUpdateTarget( self )
	end

	self.enabled = true
end

function Text:Disable()
	if not self.enabled then return end

	self:UpdateInteractivity( true )

	if self:InConfigMode() then
		self.texture:Hide()
	else
		UH:UnregisterUpdateTarget( self )
		DogTag:RemoveFontString( self.text )
		UnregisterUnitWatch( self.frame )
	end

	self.frame:SetScript( "OnShow", nil )
	self.frame:Hide()

	self.enabled = nil
end

function Text:ProcessUpdate()
	if self.frame:IsVisible() and UH:GetAlpha() > 0 then
		DogTag:UpdateFontString( self.text )
	end
end

function Text:OnShow()
	if UH.UnitSendEvents[self.config.unit] then
		DogTag:UpdateFontString( self.text )
	end
end

function Text:UpdateInteractivity( turnOff )
	if self.config.interactive and not turnOff then
		self.frame:SetAttribute( "*type1", "target" )
		self.frame:SetAttribute( "*type2", "menu" )
		self.frame:EnableMouse( true )
		self.frame:RegisterForClicks( "AnyUp" )
	else
		self.frame:SetAttribute( "*type1", nil )
		self.frame:SetAttribute( "*type2", nil )
		self.frame:RegisterForClicks()
		self.frame:EnableMouse( false )
	end

	local unit = self.config.unit
	local num

	if self.config.interactive then
		if unit == "player" then
			self.frame.menu = function( self ) ToggleDropDownMenu( 1, nil, PlayerFrameDropDown, "cursor", 0, 0 ) end
		elseif unit == "target" then
			self.frame.menu = function( self ) ToggleDropDownMenu( 1, nil, TargetFrameDropDown, "cursor", 0, 0 ) end
		elseif unit == "focus" then
			self.frame.menu = function( self ) ToggleDropDownMenu( 1, nil, FocusFrameDropDown, "cursor", 0, 0 ) end
		elseif unit == "pet" then
			self.frame.menu = function( self ) ToggleDropDownMenu( 1, nil, PetFrameDropDown, "cursor", 0, 0 ) end
		else
			num = unit:match( "^party(%d)$" )

			if num then
				self.frame.menu = function( self ) ToggleDropDownMenu( 1, nil, _G["PartyMemberFrame"..num.."DropDown"], "cursor", 0, 0 ) end
			else
				self.frame.menu = nil
			end
		end
	else
		self.frame.menu = nil
	end

	ClickCastFrames = ClickCastFrames or {}

	if self.config.interactive then
		ClickCastFrames[self.frame] = true
	else
		ClickCastFrames[self.frame] = nil
	end
end

function Text:InConfigMode()
	return UH:InConfigMode()
end

function Text:SetConfigMode( mode )
	if not self.enabled then return end

	local fontFile = SML:Fetch( SML.MediaType.FONT, self.config.fontName )

	if mode then
		UH:UnregisterUpdateTarget( self )

		UnregisterUnitWatch( self.frame )
		self.frame:Show()
		self.texture:Show()

		DogTag:RemoveFontString( self.text )

		self.text:SetFont( fontFile, self.config.fontSize )

		self.text:SetText( self.config.name )
	else
		self.text:SetFont( fontFile, self.config.fontSize )

		local kwargs = newTable()
		kwargs.unit = self.config.unit
		DogTag:AddFontString( self.text, self.frame, self.config.text, "Items;Unit", kwargs )
		delTable( kwargs )

		if not UH.UnitSendEvents[self.config.unit] then
			UH:RegisterUpdateTarget( self )
		end

		self.texture:Hide()

		RegisterUnitWatch( self.frame )
	end
end

function Text:ChangeTextOrUnit( text, unit )
	if not self.enabled then
		self.config.text = text or self.config.text
		self.config.unit = unit or self.config.unit

		return
	end

	if text then
		self.config.text = text--DogTag:FixCodeStyle( text )
	end


	if unit then
		if not self:InConfigMode() and not UH.UnitSendEvents[unit] then
			UH:RegisterUpdateTarget( self )
		else
			UH:UnregisterUpdateTarget( self )
		end

		self.config.unit = unit
		self.frame:SetAttribute( "unit", unit )

		self:UpdateInteractivity()
	end

	if self:InConfigMode() then
		self.text:SetText( self.config.name )
	else
		local kwargs = newTable()
		kwargs.unit = self.config.unit
		DogTag:AddFontString( self.text, self.frame, self.config.text, "Items;Unit", kwargs )
		delTable( kwargs )
	end

	UH:NotifyOptionsChanged()
end

function Text:UpdateFont()
	if not self.enabled then return end

	local fontFile = SML:Fetch( SML.MediaType.FONT, self.config.fontName )

	self.text:SetFont( fontFile, self.config.fontSize )

	local kwargs = newTable()
	kwargs.unit = self.config.unit
	DogTag:AddFontString( self.text, self.frame, self.config.text, "Items;Unit", kwargs )
	delTable( kwargs )
end

local textCategories = {
	name = L["Name"],
	class = L["Class"],
	guild = L["Guild"],
	range = L["Range"],
	health = L["Health"],
	power = L["Power"],
	threat = L["Threat"],
	experience = L["Experience"],
	reputation = L["Reputation"],
	casting = L["Casting"],
	custom = L["Custom"],
}

--
-- Text styles registry: Thanks to Benumbed for assembling
-- Modified for LDT-3.0 syntax.
--

local textStyles = {
	name = {
		[L["Standard"]]				= "[Name] [AFK | DND]",
		[L["Hostility Colored"]]	= "[Name:HostileColor] [AFK | DND]",
		[L["Class Colored"]]		= "[Name:ClassColor] [AFK | DND]",
		[L["Long"]]					= "[Level:DifficultyColor:Bracket] [Name:ClassColor] [AFK | DND]",
		[L["Long with Class"]]		= "[Level:DifficultyColor:Bracket] [SmartClass:ClassColor:Bracket] [Name:ClassColor] [AFK | DND]",
		[L["Long with Race/Class"]]	= "[Level:DifficultyColor:Bracket] [SmartRace:Bracket] [SmartClass:ClassColor:Bracket] [Name:ClassColor] [AFK | DND]",
		[L["Long with Druid form"]]	= "[Level:DifficultyColor:Bracket] [Name:ClassColor] [DruidForm:Paren] [AFK | DND]",
	},
	class = {
		[L["Standard"]]				= "[Classification] [SmartRace:Bracket] [SmartClass:ClassColor] [DruidForm:Paren]",
		[L["Player Classes Only"]]	= "[Classification] [SmartRace:Bracket] [PlayerClass:ClassColor] [DruidForm:Paren]",
		[L["Short"]]				= "[DifficultyColor][Plus][White] [SmartRace]",
	},
	guild = {
		[L["Standard"]]				= "[Guild:Angle]",
	},
	range = {
		[L["Standard"]]				= "[Text(Range:):Gray] [Range?Text([Range]):Gray:Bracket][~Range?Text(28+):Gray:Bracket] [Text(yds):Gray]",
	},
	health = {
		[L["Absolute"]]				= "[SureHP:HPColor:PercentHP:Percent:HPColor]",
		[L["Absolute Short"]]		= "[SureHP:HPColor:Short:PercentHP:Percent:HPColor]",
		[L["Difference"]]			= "[SmartMissingHP:Color(ff7f7f)]",
		[L["Percent"]]				= "[PercentHP:Percent:HPColor]",
		[L["Mini"]]					= "[CurHP:VeryShort:HPColor]",
		[L["Smart"]] 				= "[CurHP:~IsLessEqual(1)?Text([CurHP:Short:HPColor] / [MaxHP:Short:HPColor] [PercentHP:Percent:Paren])!Status]",
		[L["Absolute and Percent"]]	= "[CurHP:Short:HPColor] / [MaxHP:Short:HPColor] [PercentHP:Percent:Paren]",
		[L["Informational"]]		= "[IsFriend?~Status?MissingHP:HideZero:Short:Color(ff7f7f)] || [CurHP:Short:HPColor] / [MaxHP:Short:HPColor] [PercentHP:Percent:Paren]"
	},
	power = {
		[L["Absolute"]]				= "[HasMP?FractionalMP:PowerColor]",
		[L["Absolute Short"]]		= "[HasMP?FractionalMP:Short:PowerColor]",
		[L["Difference"]]			= "[MissingMP:HideZero:Negate:PowerColor]",
		[L["Percent"]]				= "[PercentMP:Percent:PowerColor]",
		[L["Mini"]]					= "[HasMP?CurMP:VeryShort:PowerColor]",
		[L["Smart"]]				= "[CurMP:HideZero?Text([CurMP:Short:PowerColor] / [MaxMP:Short:PowerColor] [PercentMP:Percent:Paren])]",
	},
	threat = {
		[L["Absolute"]]				= "[HasThreat?Threat:Color(ff7f7f)]",
		[L["Absolute Short"]]		= "[HasThreat?Threat:Short:Color(ff7f7f)]",
		[L["Fractional"]]			= "[HasThreat?FractionalThreat:Color(ff7f7f)]",
		[L["Fractional Short"]]		= "[HasThreat?FractionalThreat:Short:Color(ff7f7f)]",
		[L["Difference"]]			= "[HasThreat?MissingThreat:Negate:Color(ff7f7f)]",
		[L["Percent"]]				= "[HasThreat?PercentThreat:Percent:Color(ff7f7f)]",
		[L["Mini"]]					= "[HasThreat?Threat:VeryShort:Color(ff7f7f)]",
		[L["Smart"]]				= "[Threat:HideZero?Text([Threat:Short:Color(ff7f7f)] / [MaxThreat:Short:Color(ff7f7f)] [PercentThreat:Percent:Paren])]",
	},
	experience = {
		[L["Absolute"]]				= "[FractionalXP:Fuchsia]",
		[L["Absolute Short"]]		= "[FractionalXP:Short:Fuchsia]",
		[L["Difference"]]			= "[MissingXP:Fuchsia]",
		[L["Percent"]]				= "[PercentXP:Percent:Fuchsia]",
		[L["Smart"]]				= "[CurXP:Short:Fuchsia] / [MaxXP:Short:Fuchsia] [PercentXP:Percent:Paren]",
		[L["Informational"]]		= "[CurXP:HideZero?Text([CurXP:Short:Fuchsia] / [MaxXP:Short:Fuchsia] [PercentXP:Percent:Paren])] [RestXP:HideZero?Text([RestXP:Short] [PercentRestXP:Percent:Paren]):Cyan:Bracket]",
	},
	reputation = {
		[L["Absolute"]]				= "[FractionalRep:Cyan]",
		[L["Absolute Short"]]		= "[FractionalRep:Short:Cyan]",
		[L["Difference"]]			= "[MissingRep:Cyan]",
		[L["Percent"]]				= "[PercentRep:Cyan]",
		[L["Smart"]]				= "[CurRep:HideZero?Text([CurRep:Short:Cyan] / [MaxRep:Short:Cyan] [PercentRep:Percent:Paren])]",
	},
	casting = {
		[L["Spell name"]]			= "[~CastStopDuration ? CastName:Abbreviate:Append( [CastRank:Romanize] )]",
		[L["Remaining cast time"]]	= "[~CastStopDuration ? CastEndDuration:Round(2)]",
	},
	custom = {
	},
}

local textStylesHelper = {}

-- Inverting style name / value

do
	for category, styles in pairs( textStyles ) do
		local tc = {}

		textStylesHelper[category] = tc

		for k, v in pairs( styles ) do
			tc[k] = k
		end
	end
end

local defaultStyle = {
	name = L["Standard"],
	class = L["Standard"],
	guild = L["Standard"],
	range = L["Standard"],
	health = L["Smart"],
	power = L["Smart"],
	threat = L["Percent"],
	experience = L["Smart"],
	reputation = L["Smart"],
	custom = "",
	casting = L["Spell name"],
}

function Text:SetupOptions()
	self.options = {
		name = function() return self.config.name end,
		type = "group",
		childGroups = "tab",
		args = {
			unitAndText = {
				name = L["Unit and Text"],
				type = "group",
				order = 100,
				args = {
					unit = {
						name = L["Unit"],
						type = "select",
						order = 1,
						values = UH.OptionsHelper.Unit,
						get = function() return self.config.unit end,
						set = function( info, value ) self:ChangeTextOrUnit( nil, value ) end,
					},
					interactive = {
						name = L["Interactive"],
						desc = L["Click to target unit, Right-click for popup menu"],
						type = "toggle",
						order = 2,
						get = function() return self.config.interactive end,
						set = function( info, value ) self.config.interactive = value; self:UpdateInteractivity() end,
					},
					descr1 = {
						name = "\n"..L["You can select one of the predefined styles or enter your own tag sequence below:"].."\n",
						type = "description",
						order = 3,
					},
					category = {
						name = L["Category"],
						type = "select",
						order = 4,
						values = textCategories,
						get = function() return self.config.category or "custom" end,
						set = function( info, value )
							self.config.category = value

							if value ~= "custom" then
								local style = defaultStyle[value]

								self.config.style = style

								self:ChangeTextOrUnit( textStyles[value][style], nil )
							else
								UH:NotifyOptionsChanged()
							end
						end,
					},
					style = {
						name = L["Style"],
						type = "select",
						order = 5,
						values = function() return textStylesHelper[self.config.category or "custom"] end,
						disabled = function() return not( self.config.category ) or self.config.category == "custom" end,
						get = function() return self.config.style or defaultStyle[self.config.category or "custom"] end,
						set = function( info, value )
							self.config.style = value

							self:ChangeTextOrUnit( textStyles[self.config.category][value], nil )
						end,
					},
					text = {
						name = L["Text"],
						type = "input",
						width = "full",
						order = 6,
						multiline = true,
						disabled = function() return self.config.category and self.config.category ~= "custom" end,
						get = function( info ) return self.config.text end,
						set = function( info, value ) self:ChangeTextOrUnit( value, nil ) end,
					},
					help = {
						name = L["Open DogTag Help"],
						type = "execute",
						order = 7,
						width = "full",
						func = function() DogTag:OpenHelp() end,
					},
				},
			},
			font = {
				name = L["Font"],
				type = "group",
				--inline = true,
				order = 101,
				args = {
					--[[
					name = {
						name = L["Name"],
						type = "select",
						order = 1,
						values = UH.OptionsHelper.Fonts,
						get = function() return self.config.fontName end,
						set = function( info, value ) self.config.fontName = value; self:UpdateFont() end,
					},
					--]]
					name = {
						name = L["Name"],
						type = "select",
						order = 1,
						dialogControl = "LSM30_Font",
						values = function() return AceGUIWidgetLSMlists.font end,
						get = function() return self.config.fontName end,
						set = function( info, value ) self.config.fontName = value; self:UpdateFont() end,
					},
					size = {
						name = L["Size"],
						type = "range",
						order = 2,
						min = 4,
						max = 38,
						step = 1,
						bigStep = 1,
						get = function() return self.config.fontSize end,
						set = function( info, value ) self.config.fontSize = value; self:UpdateFont() end,
					},
					hAlign = {
						name = L["Horizontal align"],
						type = "select",
						order = 3,
						values = UH.OptionsHelper.HAlign,
						get = function() return self.config.hAlign or "LEFT" end,
						set = function( info, value ) self.config.hAlign = value; self.text:SetJustifyH( value ) end,
					},
					vAlign = {
						name = L["Vertical align"],
						type = "select",
						order = 4,
						values = UH.OptionsHelper.VAlign,
						get = function() return self.config.vAlign or "MIDDLE" end,
						set = function( info, value ) self.config.vAlign = value; self.text:SetJustifyV( value ) end,
					},
				},
			},
			positionAndSize = {
				name = L["Position and Size"],
				type = "group",
				--inline = true,
				order = 102,
				args = {
					x = {
						name = L["X"],
						type = "range",
						order = 1,
						min = -800,
						max = 800,
						step = 1,
						bigStep = 1,
						get = function() return self.config.x end,
						set = function( info, value )
							self.config.x = value
							self.frame:ClearAllPoints()
							self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.config.x, self.config.y )
						end,
					},
					y = {
						name = L["Y"],
						type = "range",
						order = 2,
						min = -800,
						max = 800,
						step = 1,
						bigStep = 1,
						get = function() return self.config.y end,
						set = function( info, value )
							self.config.y = value
							self.frame:ClearAllPoints()
							self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.config.x, self.config.y )
						end,
					},
					width = {
						name = L["Width"],
						type = "range",
						order = 3,
						min = 1,
						max = 500,
						step = 1,
						bigStep = 1,
						get = function() return self.config.width end,
						set = function( info, value )
							self.config.width = value
							self.frame:SetWidth( value )
						end,
					},
					height = {
						name = L["Height"],
						type = "range",
						order = 4,
						min = 1,
						max = 500,
						step = 1,
						bigStep = 1,
						get = function() return self.config.height end,
						set = function( info, value )
							self.config.height = value
							self.frame:SetHeight( value )
						end,
					},
					scale = {
						name = L["Scale"],
						type = "range",
						order = 5,
						min = 0.01,
						max = 1,
						step = 0.01,
						bigStep = 0.01,
						get = function() return self.config.scale end,
						set = function( info, value )
							self.config.scale = value
							self.frame:SetScale( value )
						end,
					},
				},
			},
			zOrder = {
				name = L["Z-Order"],
				type = "group",
				order = 103,
				args = {
					strata = {
						name = L["Strata"],
						type = "select",
						order = 1,
						values = UH.OptionsHelper.Strata,
						get = function() return self.config.strata end,
						set = function( info, value ) self.config.strata = value; self.frame:SetFrameStrata( value ) end,
					},
					level = {
						name = L["Level"],
						type = "range",
						order = 2,
						min = 1,
						max = 10,
						step = 1,
						get = function() return self.config.level end,
						set = function( info, value ) self.config.level = value; self.frame:SetFrameLevel( value ) end,
					},
				},
			},
		},
		plugins = {},
	}
end
