local UH = UnderHood
local UHP = UnderHood_Portraits
local L = LibStub( "AceLocale-3.0" ):GetLocale( "UnderHood" )

local Portrait = {}
Portrait.__index = Portrait

UHP.FramePrototype = Portrait

local validStyles = {
	flat = "Flat",
	animated = "Animated",
	--class = "Class",
}

local borderDescr = {
	{ "UHE-TL", 8, { "TOPLEFT", "TOPLEFT", 0 } },
	{ "UHE-TR", 8, { "TOPRIGHT", "TOPRIGHT", 0 } },
	{ "UHE-BL", 8, { "BOTTOMLEFT", "BOTTOMLEFT", 0 } },
	{ "UHE-BR", 8, { "BOTTOMRIGHT", "BOTTOMRIGHT", 0 } },
	{ "UHE-T", 8, { { "TOPLEFT", "TOPRIGHT", 1 }, { "TOPRIGHT", "TOPLEFT", 2 } } },
	{ "UHE-B", 8, { { "BOTTOMLEFT", "BOTTOMRIGHT", 3 }, { "BOTTOMRIGHT", "BOTTOMLEFT", 4 } } },
	{ "UHE-L", 8, { { "TOPLEFT", "BOTTOMLEFT", 1 }, { "BOTTOMLEFT", "TOPLEFT", 3 } } },
	{ "UHE-R", 8, { { "TOPRIGHT", "BOTTOMRIGHT", 2 }, { "BOTTOMRIGHT", "TOPRIGHT", 4 } } },
}

function Portrait:Create( module, config )
	local self = setmetatable( {}, Portrait )

	LibStub( "AceEvent-3.0" ):Embed( self )

	self.module = module
	self.config = config

	self.frame = UH:CreateFrame( "Button", nil, "SecureUnitButtonTemplate" ) -- parented to UnderHoodParentFrame
	self.frame:SetWidth( self.config.width )
	self.frame:SetHeight( self.config.height )
	self.frame:SetScale( self.config.scale )
	self.frame:SetFrameStrata( self.config.strata )
	self.frame:SetFrameLevel( self.config.level )
	self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.config.x, self.config.y )
	self.frame:SetAttribute( "unit", self.config.unit )

	self.borderParts = {}

	for i, bd in ipairs( borderDescr ) do
		local tx = UH:CreateFrame( "Texture", self.frame, "BACKGROUND" )

		tx:SetTexture( "Interface\\AddOns\\UnderHood\\Media\\"..bd[1] )
		tx:SetWidth( bd[2] ); tx:SetHeight( bd[2] )
		self.borderParts[i] = tx

		local points = bd[3]

		if type( points[1] ) == "table" then
			for _, point in ipairs( points ) do
				tx:SetPoint( point[1], point[3] == 0 and self.frame or self.borderParts[point[3]], point[2] )
			end
		else
			tx:SetPoint( points[1], points[3] == 0 and self.frame or self.borderParts[points[3]], points[2] )
		end
	end

	self.background = UH:CreateFrame( "Texture", self.frame, "BACKGROUND" )
	self.background:SetTexture( 0, 0, 0, 0.25 )
	self.background:Show()

	if not self.config.showBorder then
		self:HideBorder()
	end

	self:SetupOptions()

	if self.config.enabled then
		self:Enable()
	end

	return self
end

function Portrait:Destroy()
	self:Disable()

	for _, bp in ipairs( self.borderParts ) do
		UH:ReleaseFrame( bp )
	end

	self.background = UH:ReleaseFrame( self.background )

	if self.texture then
		if self.activeStyle == "animated" then
			self.texture:SetScript( "OnShow", nil )
		end

		UH:ReleaseFrame( self.texture )
	end

	if self.frame then
		UH:ReleaseFrame( self.frame )
	end

	self.borderParts = nil
	self.texture = nil
	self.frame = nil
end

function Portrait:Enable()
	if self.enabled then return end

	self:UpdateInteractivity()

	if self:InConfigMode() then
		self.frame:Show()
	else
		self:DoUpdatePortrait( true )

		self:SetupEvents()
		RegisterUnitWatch( self.frame )
	end

	self.enabled = true
end

function Portrait:Disable()
	if not self.enabled then return end

	self:UnregisterAllEvents()
	self:UpdateInteractivity( true )

	if self:InConfigMode() then
		--self.texture:Hide()
	else
		UH:UnregisterUpdateTarget( self )
		UnregisterUnitWatch( self.frame )
	end

	self.frame:SetScript( "OnShow", nil )
	self.frame:Hide()

	self.enabled = nil
end

function Portrait:ShowBorder()
	for _, tx in ipairs( self.borderParts ) do
		tx:Show()
	end

	if self.texture then
		self.texture:ClearAllPoints()

		self.texture:SetPoint( "TOPLEFT", self.borderParts[1], "BOTTOMRIGHT", -4, 4 )
		self.texture:SetPoint( "BOTTOMRIGHT", self.borderParts[4], "TOPLEFT", 4, -4 )
		self.background:SetPoint( "TOPLEFT", self.borderParts[1], "BOTTOMRIGHT", -4, 4 )
		self.background:SetPoint( "BOTTOMRIGHT", self.borderParts[4], "TOPLEFT", 4, -4 )
	end
end

function Portrait:HideBorder()
	for _, tx in ipairs( self.borderParts ) do
		tx:Hide()
	end

	if self.texture then
		self.texture:ClearAllPoints(); self.texture:SetAllPoints( self.frame )
		self.background:ClearAllPoints(); self.background:SetAllPoints( self.frame )
	end
end

function Portrait:UpdatePortraitIfUnit( event, unit )
	if (self.unitOwner and unit == self.unitOwner) or (unit == self.config.unit) then
		self:DoUpdatePortrait( event == "UNIT_PORTRAIT_UPDATE" )
	end
end

function Portrait:UpdatePortrait()
	local guid = UnitGUID( self.config.unit )

	if guid ~= self.unitGUID then
		self:DoUpdatePortrait( true )
		self.unitGUID = guid
	end
end

function Portrait:ProcessUpdate()
	self:UpdatePortrait()
end

local classIcons = {
	["WARRIOR"] = {0, 0.25, 0, 0.25},
	["MAGE"] = {0.25, 0.49609375, 0, 0.25},
	["ROGUE"] = {0.49609375, 0.7421875, 0, 0.25},
	["DRUID"] = {0.7421875, 0.98828125, 0, 0.25},
	["HUNTER"] = {0, 0.25, 0.25, 0.5},
	["SHAMAN"] = {0.25, 0.49609375, 0.25, 0.5},
	["PRIEST"] = {0.49609375, 0.7421875, 0.25, 0.5},
	["WARLOCK"] = {0.7421875, 0.98828125, 0.25, 0.5},
	["PALADIN"] = {0, 0.25, 0.5, 0.75},
}

function Portrait:DoUpdatePortrait( force )
	local texture = self.texture

	if self.activeStyle ~= self.config.style or not( texture ) then
		if texture then
			if self.activeStyle == "animated" then
				texture:SetScript( "OnShow", nil )
			end

			UH:ReleaseFrame( texture )
		end

		self.activeStyle = self.config.style

		if self.activeStyle == "animated" then
			texture = UH:CreateFrame( "PlayerModel", self.frame )
			texture:SetScript( "OnShow", function() self:OnShowModel() end )
		else
			texture = UH:CreateFrame( "Texture", self.frame, "ARTWORK" )
		end

		self.texture = texture

		if texture then
			if self.config.showBorder then
				texture:SetPoint( "TOPLEFT", self.borderParts[1], "BOTTOMRIGHT", -4, 4 )
				texture:SetPoint( "BOTTOMRIGHT", self.borderParts[4], "TOPLEFT", 4, -4 )
			else
				texture:SetAllPoints( self.frame )
			end

			force = true
		else
			return
		end
	end

	if force then
		local unit = self.config.unit

		if self.activeStyle == "flat" then
			SetPortraitTexture( texture, unit )
			texture:SetTexCoord( 0.14644660941, 0.85355339059, 0.14644660941, 0.85355339059 )
		elseif self.activeStyle == "animated" then
			if not UnitExists( unit ) or not UnitIsConnected( unit ) or not UnitIsVisible( unit ) then
				texture:SetModelScale( 4.25 )
				texture:SetPosition( 0, 0, -1.5 )
				texture:SetModel( "Interface\\Buttons\\talktomequestionmark.mdx" )
			else
				texture:SetUnit( unit )
				texture:SetModelScale( 1 )
				texture:Hide()
			end
		elseif style == "class" then
			local className = select( 2, UnitClass( unit ) )

			if className then
				local class = classIcons[className]

				texture:SetTexture( "Interface\\Glues\\CharacterCreate\\UI-CharacterCreate-Classes" )
				texture:SetTexCoord( class[1], class[2], class[3], class[4] )
			else
				texture:SetTexture( "Interface\\Icons\\Ability_Hunter_BeastCall" )
				texture:SetTexCoord( 0, 1, 0, 1 )
			end
		end

		texture:Show()
	end
end

function Portrait:OnShowModel()
	if self.texture then
		self.texture:SetCamera( 0 )
	end
end

function Portrait:InConfigMode()
	return UH:InConfigMode()
end

function Portrait:SetConfigMode( mode )
	if not self.enabled then return end

	if mode then
		self:UnregisterAllEvents()
		UH:UnregisterUpdateTarget( self )
		UnregisterUnitWatch( self.frame )

		self.frame:Show()
	else
		self:SetupEvents()
		RegisterUnitWatch( self.frame )
	end
end

function Portrait:UpdateInteractivity( turnOff )
	if self.config.interactive and not turnOff then
		self.frame:SetAttribute( "*type1", "target" )
		self.frame:SetAttribute( "*type2", "menu" )
		self.frame:EnableMouse( true )
		self.frame:RegisterForClicks( "AnyUp" )
	else
		self.frame:SetAttribute( "*type1", nil )
		self.frame:SetAttribute( "*type2", nil )
		self.frame:RegisterForClicks()
		self.frame:EnableMouse( false )
	end

	local unit = self.config.unit
	local num

	if self.config.interactive then
		if unit == "player" then
			self.frame.menu = function( self ) ToggleDropDownMenu( 1, nil, PlayerFrameDropDown, "cursor", 0, 0 ) end
		elseif unit == "target" then
			self.frame.menu = function( self ) ToggleDropDownMenu( 1, nil, TargetFrameDropDown, "cursor", 0, 0 ) end
		elseif unit == "focus" then
			self.frame.menu = function( self ) ToggleDropDownMenu( 1, nil, FocusFrameDropDown, "cursor", 0, 0 ) end
		elseif unit == "pet" then
			self.frame.menu = function( self ) ToggleDropDownMenu( 1, nil, PetFrameDropDown, "cursor", 0, 0 ) end
		else
			num = unit:match( "^party(%d)$" )

			if num then
				self.frame.menu = function( self ) ToggleDropDownMenu( 1, nil, _G["PartyMemberFrame"..num.."DropDown"], "cursor", 0, 0 ) end
			else
				self.frame.menu = nil
			end
		end
	else
		self.frame.menu = nil
	end

	ClickCastFrames = ClickCastFrames or {}

	if self.config.interactive then
		ClickCastFrames[self.frame] = true
	else
		ClickCastFrames[self.frame] = nil
	end
end

function Portrait:SetupEvents()
	self:UnregisterAllEvents()

	local unit = self.config.unit
	local event, withUnit, owner = UH:GetUnitChangedEvent( unit )

	self:RegisterEvent( "UNIT_PORTRAIT_UPDATE", "UpdatePortraitIfUnit" )

	self.unitOwner = owner

	if event then
		if withOwner then
			self:RegisterEvent( event, "UpdatePortraitIfUnit" )
		else
			self:RegisterEvent( event, "UpdatePortrait" )
		end
	else
		UH:RegisterUpdateTarget( self )
	end

end

function Portrait:ChangeStyleOrUnit( style, unit )
	style = style or self.config.style
	unit = unit or self.config.unit

	self.config.unit = unit
	self.config.style = style

	self.frame:SetAttribute( "unit", unit )
	self:UpdateInteractivity()

	self.unitGUID = UnitGUID( unit )
	self:DoUpdatePortrait( true )
	self:SetupEvents()

	UH:NotifyOptionsChanged()
end

function Portrait:SetupOptions()
	self.options = {
		name = function() return self.config.name end,
		type = "group",
		childGroups = "tab",
		args = {
			unitAndType = {
				name = L["Unit and Style"],
				type = "group",
				order = 100,
				args = {
					unit = {
						name = L["Unit"],
						type = "select",
						order = 1,
						values = UH.OptionsHelper.Unit,
						get = function() return self.config.unit end,
						set = function( info, value ) self:ChangeStyleOrUnit( nil, value ) end,
					},
					icon = {
						name = L["Style"],
						type = "select",
						order = 2,
						values = validStyles,
						get = function() return self.config.style end,
						set = function( info, value ) self:ChangeStyleOrUnit( value, nil ) end,
					},
					interactive = {
						name = L["Interactive"],
						desc = L["Click to target unit, Right-click for popup menu"],
						type = "toggle",
						order = 3,
						get = function() return self.config.interactive end,
						set = function( info, value ) self.config.interactive = value; self:UpdateInteractivity() end,
					},
					showBorder = {
						name = L["Show border"],
						type = "toggle",
						order = 4,
						get = function() return self.config.showBorder end,
						set = function( info, value )
							self.config.showBorder = value

							if value then
								self:ShowBorder()
							else
								self:HideBorder()
							end
						end,
					},
				},
				plugins = {},
			},
			positionAndSize = {
				name = L["Position and Size"],
				type = "group",
				order = 102,
				args = {
					x = {
						name = L["X"],
						type = "range",
						order = 1,
						min = -800,
						max = 800,
						step = 1,
						bigStep = 1,
						get = function() return self.config.x end,
						set = function( info, value )
							self.config.x = value
							self.frame:ClearAllPoints()
							self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.config.x, self.config.y )
						end,
					},
					y = {
						name = L["Y"],
						type = "range",
						order = 2,
						min = -800,
						max = 800,
						step = 1,
						bigStep = 1,
						get = function() return self.config.y end,
						set = function( info, value )
							self.config.y = value
							self.frame:ClearAllPoints()
							self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.config.x, self.config.y )
						end,
					},
					width = {
						name = L["Width"],
						type = "range",
						order = 3,
						min = 1,
						max = 500,
						step = 1,
						bigStep = 1,
						get = function() return self.config.width end,
						set = function( info, value )
							self.config.width = value
							self.frame:SetWidth( value )
						end,
					},
					height = {
						name = L["Height"],
						type = "range",
						order = 4,
						min = 1,
						max = 500,
						step = 1,
						bigStep = 1,
						get = function() return self.config.height end,
						set = function( info, value )
							self.config.height = value
							self.frame:SetHeight( value )
						end,
					},
					scale = {
						name = L["Scale"],
						type = "range",
						order = 5,
						min = 0.01,
						max = 1,
						step = 0.01,
						bigStep = 0.01,
						get = function() return self.config.scale end,
						set = function( info, value )
							self.config.scale = value
							self.frame:SetScale( value )
						end,
					},
				},
			},
			zOrder = {
				name = L["Z-Order"],
				type = "group",
				order = 103,
				args = {
					strata = {
						name = L["Strata"],
						type = "select",
						order = 1,
						values = UH.OptionsHelper.Strata,
						get = function() return self.config.strata end,
						set = function( info, value ) self.config.strata = value; self.frame:SetFrameStrata( value ) end,
					},
					level = {
						name = L["Level"],
						type = "range",
						order = 2,
						min = 1,
						max = 10,
						step = 1,
						get = function() return self.config.level end,
						set = function( info, value ) self.config.level = value; self.frame:SetFrameLevel( value ) end,
					},
				},
			},
		},
		plugins = {},
	}
end
