local UH = UnderHood
local UHI = UnderHood_Icons
local L = LibStub( "AceLocale-3.0" ):GetLocale( "UnderHood" )

local IsResting = IsResting
local UnitAffectingCombat = UnitAffectingCombat
local UnitIsPVPFreeForAll = UnitIsPVPFreeForAll
local UnitFactionGroup = UnitFactionGroup
local UnitIsPVP = UnitIsPVP
local IsPartLeader = IsPartyLeader
local IsRaidLeader = IsRaidLeader
local IsRaidOfficer = IsRaidOfficer
local GetLootMethod = GetLootMethod
local GetRaidTargetIndex = GetRaidTargetIndex
local GetPetHappiness = GetPetHappiness
local SetRaidTargetIconTexture = SetRaidTargetIconTexture

local Icon = {}
Icon.__index = Icon

UHI.FramePrototype = Icon

local validIcons = {
	resting = L["Resting"],
	combat = L["Combat"],
	pvp = L["PvP"],
	leader = L["Leader"],
	ml = L["Master Looter"],
	rt = L["Raid Target"],
	happiness = L["Happiness"],
}

local unitIcons = {
	player = {
		resting = true,
		combat = true,
		pvp = true,
		leader = true,
		ml = true,
		rt = true,
	},
	target = {
		combat = true,
		pvp = true,
		rt = true,
	},
	focus = {
		combat = true,
		pvp = true,
		rt = true,
	},
	targettarget = {
		combat = true,
		pvp = true,
		rt = true,
	},
	pet = {
		combat = true,
		pvp = true,
		rt = true,
		happiness = true,
	},
}

local iconTextures = {
	resting = { "Interface\\CharacterFrame\\UI-StateIcon", 0.09, 0.43, 0.08, 0.42 },
	combat = { "Interface\\CharacterFrame\\UI-StateIcon", 0.57, 0.9, 0.08, 0.41 },
	pvp = {
		{ "Interface\\TargetingFrame\\UI-PVP-FFA", 0.05, 0.605, 0.015, 0.57 },
		{ "Interface\\TargetingFrame\\UI-PVP-Horde", 0.08, 0.58, 0.045, 0.545 },
		{ "Interface\\TargetingFrame\\UI-PVP-Alliance", 0.07, 0.58, 0.06, 0.57 },
	},
	leader = { "Interface\\GroupFrame\\UI-Group-LeaderIcon", 0.1, 0.84, 0.14, 0.88 },
	ml = { "Interface\\GroupFrame\\UI-Group-MasterLooter", 0.15, 0.9, 0.15, 0.9 },
	rt = { "Interface\\TargetingFrame\\UI-RaidTargetingIcons", 0, 1, 0, 1 },
	happiness = {
		{ "Interface\\PetPaperDollFrame\\UI-PetHappiness", 0.375, 0.5625, 0, 0.359375 },
		{ "Interface\\PetPaperDollFrame\\UI-PetHappiness", 0.1875, 0.375, 0, 0.359375 },
		{ "Interface\\PetPaperDollFrame\\UI-PetHappiness", 0, 0.1875, 0, 0.359375 },
	},
}

function Icon:Create( module, config )
	local self = {}
	setmetatable( self, Icon )

	self.module = module
	self.config = config
	self.rtiForConfig = math.random( 8 )

	self.frame = UH:CreateFrame( "Frame" ) -- parented to UnderHoodParentFrame
	self.frame:SetWidth( self.config.width )
	self.frame:SetHeight( self.config.height )
	self.frame:SetScale( self.config.scale )
	self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.config.x, self.config.y )

	self.texture = UH:CreateFrame( "Texture", self.frame, "ARTWORK" )
	self.texture:SetAllPoints( true )
	self.texture:Show()
	
	self:SetupOptions()

	if self.config.enabled then
	    self:Enable()
	end
	
	return self
end

function Icon:Destroy()
	self:Disable()
	
	if self.texture then
		UH:ReleaseFrame( self.texture )
	end
	
	if self.frame then
		UH:ReleaseFrame( self.frame )
	end

	self.texture = nil
	self.frame = nil
end

function Icon:Enable()
	if self.enabled then return end

	UH:RegisterUpdateTarget( self )
	self.frame:Show()
	
	self.enabled = true
end

function Icon:Disable()
	if not self.enabled then return end
	
	UH:UnregisterUpdateTarget( self )
	self.frame:Hide()
	
	self.enabled = nil
end

function Icon:ProcessUpdate()
	local icon = self.config.icon
	local unit = self.config.unit
	local iconIndex
	local configMode = self:InConfigMode()
	local visible, set = true, nil
	
	if self.config.visibility == "never" or (self.config.visibility == "ooc" and self.inCombat) then
		self.frame:Hide()
		return
	end

	if icon == "resting" then
		if not IsResting() then visible = false end
	elseif icon == "combat" then
		if not( UnitAffectingCombat( unit ) ) then visible = false end
	elseif icon == "pvp" then
		local pvpt
		
		if UnitIsPVPFreeForAll( unit ) then
			pvpt = 'FFA'
		else
			local faction = UnitFactionGroup( unit )
			
			if faction and UnitIsPVP( unit ) then
				pvpt = faction
			end
		end
		
		if pvpt or configMode then
			if pvpt == "Horde" then
				iconIndex = 2
			elseif pvpt == "Alliance" then
				iconIndex = 3
			else
				iconIndex = 1
			end
		else
			visible = false
		end
	elseif icon == "leader" then
		if not( IsPartyLeader() or IsRaidLeader() or IsRaidOfficer() ) then visible = false end
	elseif icon == "ml" then
		local _, lootmaster = GetLootMethod()
		
		if not( lootmaster == 0 ) then visible = false end
	elseif icon == "rt" then
		local rti = GetRaidTargetIndex( unit )
		
		if not configMode then
			if rti then
				iconIndex = rti
				set = true
			else
				visible = false
			end
		else
		    iconIndex = self.rtiForConfig
		    set = true
		end
	elseif icon == "happiness" then
		local happiness, damagePercentage, loyaltyRate = GetPetHappiness()

		if happiness == 3 or configMode then
			iconIndex = 3
		elseif happiness == 2 then
			iconIndex = 2
		elseif happiness == 1 then
			iconIndex = 1
		else
			visible = false
		end
	end

	if visible or configMode then
		local tex, l, r, t, b

		if iconIndex and not( set ) then
			tex, l, r, t, b = unpack( iconTextures[icon][iconIndex] )
		else
			tex, l, r, t, b = unpack( iconTextures[icon] )
		end
		
		self.texture:SetTexture( tex )
		self.texture:SetTexCoord( l, r, t, b )
		
		if set then
			SetRaidTargetIconTexture( self.texture, iconIndex )
		end

		if not self.frame:IsVisible() then
			self.frame:Show()
		end
	else
		self.frame:Hide()
	end
end

function Icon:InConfigMode()
	return UH:InConfigMode()
end

function Icon:SetConfigMode( mode )
end

function Icon:ChangeIconOrUnit( icon, unit )
	icon = icon or self.config.icon
	unit = unit or self.config.unit

	local needRefresh = false
	
	if unit ~= self.config.unit then
		if not( unitIcons[unit][icon] ) then
			for k, _ in pairs( unitIcons[unit] ) do
				icon = k
				needRefresh = true
				break
			end
		end
		
		self.config.unit = unit
	end
	
	self.config.icon = icon
	
	if needRefresh then
		UH:NotifyOptionsChanged()
	end
end

function Icon:SetupOptions()
	self.options = {
		name = function() return self.config.name end,
		type = "group",
		childGroups = "tab",
		args = {
			unitAndType = {
				name = L["Unit and Icon"],
				type = "group",
				order = 100,
				args = {
					unit = {
						name = L["Unit"],
						type = "select",
						order = 1,
						values = UH.OptionsHelper.Unit,
						get = function() return self.config.unit end,
						set = function( info, value ) self:ChangeIconOrUnit( nil, value ) end,
					},
					icon = {
						name = L["Icon"],
						type = "select",
						order = 2,
						values = function()
							local v = {}

							for icon, _ in pairs( unitIcons[self.config.unit] ) do
								v[icon] = validIcons[icon]
							end
							
							return v
						end,
						get = function() return self.config.icon end,
						set = function( info, value ) self:ChangeIconOrUnit( value, nil ) end,
					},
					tooltips = {
						name = L["Visibility"],
						type = "select",
						order = 3,
						values = UH.OptionsHelper.TooltipVisibility,
						get = function() return self.config.visibility end,
						set = function( info, value ) self.config.visibility = value end,
					},
				},
				plugins = {},
			},
			positionAndSize = {
				name = L["Position and Size"],
				type = "group",
				order = 102,
				args = {
					x = {
						name = L["X"],
						type = "range",
						order = 1,
						min = -800,
						max = 800,
						step = 1,
						bigStep = 1,
						get = function() return self.config.x end,
						set = function( info, value )
							self.config.x = value
							self.frame:ClearAllPoints()
							self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.config.x, self.config.y )
						end,
					},
					y = {
						name = L["Y"],
						type = "range",
						order = 2,
						min = -800,
						max = 800,
						step = 1,
						bigStep = 1,
						get = function() return self.config.y end,
						set = function( info, value )
							self.config.y = value
							self.frame:ClearAllPoints()
							self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.config.x, self.config.y )
						end,
					},
					width = {
						name = L["Width"],
						type = "range",
						order = 3,
						min = 1,
						max = 500,
						step = 1,
						bigStep = 1,
						get = function() return self.config.width end,
						set = function( info, value )
							self.config.width = value
							self.frame:SetWidth( value )
						end,
					},
					height = {
						name = L["Height"],
						type = "range",
						order = 4,
						min = 1,
						max = 500,
						step = 1,
						bigStep = 1,
						get = function() return self.config.height end,
						set = function( info, value )
							self.config.height = value
							self.frame:SetHeight( value )
						end,
					},
					scale = {
						name = L["Scale"],
						type = "range",
						order = 5,
						min = 0.01,
						max = 1,
						step = 0.01,
						bigStep = 0.01,
						get = function() return self.config.scale end,
						set = function( info, value )
							self.config.scale = value
							self.frame:SetScale( value )
						end,
					},
				},
			},
			zOrder = {
				name = L["Z-Order"],
				type = "group",
				order = 103,
				args = {
					strata = {
						name = L["Strata"],
						type = "select",
						order = 1,
						values = UH.OptionsHelper.Strata,
						get = function() return self.config.strata end,
						set = function( info, value ) self.config.strata = value; self.frame:SetFrameStrata( value ) end,
					},
					level = {
						name = L["Level"],
						type = "range",
						order = 2,
						min = 1,
						max = 10,
						step = 1,
						get = function() return self.config.level end,
						set = function( info, value ) self.config.level = value; self.frame:SetFrameLevel( value ) end,
					},
				},
			},
		},
		plugins = {},
	}
end
