local UH = UnderHood
local L = LibStub( "AceLocale-3.0" ):GetLocale( "UnderHood" )

local GetComboPoints = GetComboPoints

UnderHood_ComboPoints = UnderHood:NewModule( "ComboPoints", "AceConsole-3.0", "AceEvent-3.0" )

UnderHood_ComboPoints.configDefaults = {
	profile = {
	    x = 180,
	    y = 85,
	    layout = "D",
		size = 16,
		space = 0,
		scale = 1,
		strata = "LOW",
		level = 6,
		style = "diamond",
	},
}

local Module = UnderHood_ComboPoints

function Module:OnInitialize()
	self.db = UnderHood.db:RegisterNamespace( self.moduleName, self.configDefaults )

	self.db.RegisterCallback( self, "OnProfileReset", "HandleProfileChanges" )
	self.db.RegisterCallback( self, "OnProfileChanged", "HandleProfileChanges" )
	self.db.RegisterCallback( self, "OnProfileCopied", "HandleProfileChanges" )

	self.frame = UH:CreateFrame( "Frame" )
	self.points = {}
	
	for i = 1, 5 do
	    local point = UH:CreateFrame( "Texture", self.frame )
	    
		table.insert( self.points, point )
	end
	
	self:SetupOptions()
end

function Module:HandleProfileChanges()
	if UnderHood.playerLoggedIn then
	    self:RefreshProfile()
	end
end

function Module:OnEnable()
	if UnderHood.playerLoggedIn then
	    self:RefreshProfile()
	else
		self:RegisterMessage( "UnderHood_PlayerLogin",
			function()
				self:RefreshProfile();
				self:UnregisterMessage( "UnderHood_PlayerLogin" )
			end )
	end
	
	self:RegisterEvent( "PLAYER_COMBO_POINTS" )
	self:RegisterEvent( "PLAYER_TARGET_CHANGED", "PLAYER_COMBO_POINTS" )
end

function Module:OnDisable()
	self.frame:Hide()
end

function Module:GetOptions()
	if not self.Options then
		self:SetupOptions()
	end

	return self.Options
end

function Module:SetConfigMode( mode )
	self:UpdateComboPoints()
end

function Module:UpdateStyle()
	for i, point in ipairs( self.points ) do
	    if self.db.profile.style == "colored" then
			point:SetTexture( "Interface\\Addons\\UnderHood\\Media\\ColoredComboPoint-"..i )
	    else
			point:SetTexture( "Interface\\Addons\\UnderHood\\Media\\ComboPoint" )
		end
	end
end

function Module:RefreshProfile()
	if not self.db or not self.db.profile then return end

	for i, point in ipairs( self.points ) do
	    point:Hide()
	end

	self.frame:ClearAllPoints()
	self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.db.profile.x, self.db.profile.y )
	self.frame:SetFrameStrata( self.db.profile.strata )
	self.frame:SetFrameLevel( self.db.profile.level )
	self.frame:SetScale( self.db.profile.scale )

	self:UpdateStyle()
	self:PerformLayout()
	self:UpdateComboPoints()
	
	self.frame:Show()

	UnderHood:NotifyOptionsChanged()
end

local layouts = {
	D = {
	    size = function( size, space ) return size, size * 5 + space * 4 end,
	    point = "TOP", relative = "BOTTOM", dx = 0, dy = -1,
	},
	U = {
	    size = function( size, space ) return size, size * 5 + space * 4 end,
	    point = "BOTTOM", relative = "TOP", dx = 0, dy = 1,
	},
	L = {
	    size = function( size, space ) return size * 5 + space * 4, size end,
	    point = "RIGHT", relative = "LEFT", dx = -1, dy = 0,
	},
	R = {
	    size = function( size, space ) return size * 5 + space * 4, size end,
	    point = "LEFT", relative = "RIGHT", dx = 1, dy = 0,
	},
}

function Module:PerformLayout()
	local layout = layouts[self.db.profile.layout]
	local size = self.db.profile.size
	local space = self.db.profile.space
	local width, height = layout.size( self.db.profile.size, self.db.profile.space )
	
	self.frame:SetWidth( width )
	self.frame:SetHeight( height )
	
	for i, point in ipairs( self.points ) do
	    point:SetWidth( size )
	    point:SetHeight( size )
	    point:ClearAllPoints()
	    
	    if i == 1 then
	        point:SetPoint( layout.point, self.frame, layout.point, 0, 0 )
	    else
	        point:SetPoint( layout.point, self.points[i-1], layout.relative, space * layout.dx, space * layout.dy )
	    end
	end
end

function Module:PLAYER_COMBO_POINTS()
	self:UpdateComboPoints()
end

function Module:UpdateComboPoints()
	local points = UH:InConfigMode() and 5 or GetComboPoints()
	
	for i, point in ipairs( self.points ) do
	    if points >= i then
	        point:Show()
	    else
	        point:Hide()
	    end
	end
end

local validLayouts = {
	D = L["Down"],
	U = L["Up"],
	R = L["Right"],
	L = L["Left"],
}

local validStyles = {
	diamond = L["Diamond"],
	colored = L["Colored"],
}

function Module:SetupOptions()
	if self.Options then return end

	self.Options = {
		name = self.moduleName,
		type = "group",
		childGroups = "tab",
		args = {
		    style = {
		        name = L["Style"],
		        type = "select",
		        values = validStyles,
		        get = function() return self.db.profile.style end,
		        set = function( info, value ) self.db.profile.style = value; self:UpdateStyle() end,
			},
			positionAndSize = {
				name = L["Position and Size"],
				type = "group",
				order = 102,
				args = {
					x = {
						name = L["X"],
						type = "range",
						order = 1,
						min = -800,
						max = 800,
						step = 1,
						bigStep = 1,
						get = function() return self.db.profile.x end,
						set = function( info, value )
							self.db.profile.x = value
							self.frame:ClearAllPoints()
							self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.db.profile.x, self.db.profile.y )
						end,
					},
					y = {
						name = L["Y"],
						type = "range",
						order = 2,
						min = -800,
						max = 800,
						step = 1,
						bigStep = 1,
						get = function() return self.db.profile.y end,
						set = function( info, value )
							self.db.profile.y = value
							self.frame:ClearAllPoints()
							self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.db.profile.x, self.db.profile.y )
						end,
					},
					layout = {
						name = L["Layout"],
						type = "select",
						order = 3,
						values = validLayouts,
						get = function() return self.db.profile.layout end,
						set = function( info, value ) self.db.profile.layout = value; self:PerformLayout() end,
					},
					size = {
					    name = L["Combopoint size"],
					    type = "range",
					    order = 4,
					    min = 10,
					    max = 48,
					    step = 1,
					    bigStep = 1,
					    get = function() return self.db.profile.size end,
					    set = function( info, value )
					        self.db.profile.size = value
					        self:PerformLayout()
					    end,
					},
					spacing = {
						name = L["Spacing"],
						type = "range",
						order = 7,
						min = 0,
						max = 40,
						step = 1,
						bigStep = 1,
						get = function() return self.db.profile.space end,
						set = function( info, value )
							self.db.profile.space = value
							self:PerformLayout()
						end,
					},
					scale = {
						name = L["Scale"],
						type = "range",
						order = 9,
						min = 0.01,
						max = 1,
						step = 0.01,
						bigStep = 0.01,
						get = function() return self.db.profile.scale end,
						set = function( info, value )
							self.db.profile.scale = value
							self.frame:SetScale( value )
						end,
					},
				},
			},
			zOrder = {
				name = L["Z-Order"],
				type = "group",
				--inline = true,
				order = 103,
				args = {
					strata = {
						name = L["Strata"],
						type = "select",
						order = 1,
						values = UH.OptionsHelper.Strata,
						get = function() return self.db.profile.strata end,
						set = function( info, value ) self.db.profile.strata = value; self.frame:SetFrameStrata( value ) end,
					},
					level = {
						name = L["Level"],
						type = "range",
						order = 2,
						min = 1,
						max = 10,
						step = 1,
						get = function() return self.db.profile.level end,
						set = function( info, value ) self.db.profile.level = value; self.frame:SetFrameLevel( value ) end,
					},
				},
			},
		},
		plugins = {},
	}
end
