local UH = UnderHood
local UHB = UnderHood_Buffs
local UHBS = UHB.Spells
local L = LibStub( "AceLocale-3.0" ):GetLocale( "UnderHood" )

local table_sort = table.sort

local UnitBuff = UnitBuff
local UnitDebuff = UnitDebuff
local UnitIsUnit = UnitIsUnit
local UnitIsFriend = UnitIsFriend
local GetPlayerBuff = GetPlayerBuff
local GetPlayerBuffName = GetPlayerBuffName
local GetPlayerBuffTexture = GetPlayerBuffTexture
local GetPlayerBuffApplications = GetPlayerBuffApplications
local GetPlayerBuffDispelType = GetPlayerBuffDispelType
local GetPlayerBuffTimeLeft = GetPlayerBuffTimeLeft
local GetNumPartyMembers = GetNumPartyMembers
local GetNumRaidMembers = GetNumRaidMembers
local math_min, math_max, math_floor = math.min, math.max, math.floor
local newList, newTable, releaseTable = UH.NewList, UH.NewTable, UH.ReleaseTable

local MAX_AURAS = 40

local BuffFrame = {}
BuffFrame.__index = BuffFrame
UHB.FramePrototype = BuffFrame

local validTypes = {
	buffs = L["Buffs"],
	debuffs = L["Debuffs"],
}

local validLayouts = {
	LTD = L["Left, then Down"],
	RTD = L["Right, then Down"],
	LTU = L["Left, then Up"],
	RTU = L["Right, then Up"],
	DTL = L["Down, then Left"],
	DTR = L["Down, then Right"],
	UTL = L["Up, then Left"],
	UTR = L["Up, then Right"],
}

function BuffFrame:Create( module, config )
	local self = {}
	setmetatable( self, BuffFrame )

	LibStub( "AceEvent-3.0" ):Embed( self )
	
	self.module = module
	self.config = config
	
	self.frame = UH:CreateFrame( "Frame" ) -- parented to UnderHoodParentFrame
	self.frame:SetScale( self.config.scale )
	self.frame:SetFrameStrata( self.config.strata )
	self.frame:SetFrameLevel( self.config.level )
	self.frame:SetWidth( 1 )
	self.frame:SetHeight( 1 )
	self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.config.x, self.config.y )

	self.buttons = {}
	
	for i = 1, MAX_AURAS do
		local button = UH:CreateFrame( "Button", self.frame )

		button:SetWidth( self.config.size )
		button:SetHeight( self.config.size )
		button:SetFrameLevel( self.config.level + 1 )
		button:RegisterForClicks( "RightButtonUp" )
		
		button:SetID( i )
		
		button:SetBackdrop( {
			bgFile = "Interface\\Buttons\\WHITE8X8", tile = true, tileSize = 8,
			edgeFile = "Interface\\Buttons\\WHITE8X8", edgeSize = 1,
			insets = { left = 1, right = 1, top = 1, bottom = 1 },
		} )
		
		button:SetBackdropColor( 0, 0, 0, 0 )
		button:SetBackdropBorderColor( 0, 0, 0, 1 )

		button.Icon = UH:CreateFrame( "Texture", button, "BACKGROUND" )
		button.Icon:SetTexture( "Interface\\Icons\\INV_Misc_Ear_Human_02" )
		button.Icon:SetTexCoord( 0.07, 0.93, 0.07, 0.93 )
		button.Icon:SetPoint( "TOPLEFT", button, "TOPLEFT", 1, -1 )
		button.Icon:SetPoint( "BOTTOMRIGHT", button, "BOTTOMRIGHT", -1, 1 )
		button.Icon:Show()
		
		button.Count = UH:CreateFrame( "FontString", button, "OVERLAY", "NumberFontNormalSmall" )
		button.Count:SetJustifyH( "RIGHT" )
		button.Count:SetJustifyV( "BOTTOM" )
		button.Count:SetPoint( "BOTTOMRIGHT", button, "BOTTOMRIGHT", -1, 1 )
		button.Count:Show()

		button.cooldown = UH:CreateFrame( "Cooldown", button )
		button.cooldown:SetReverse( true )
		button.cooldown:SetPoint( "TOPLEFT", button, "TOPLEFT", 1, -1 )
		button.cooldown:SetPoint( "BOTTOMRIGHT", button, "BOTTOMRIGHT", -1, 1 )
		button.cooldown:Hide()
		
		button.cooldownText = UH:CreateFrame( "FontString", button.cooldown, "OVERLAY", "NumberFontNormalSmall" )
		button.cooldownText:SetTextColor( 1, 1, 1, 1 )
		button.cooldownText:SetShadowColor( 0, 0, 0, 1 )
		button.cooldownText:SetShadowOffset( 0.8, -0.8 )
		button.cooldownText:SetPoint( "TOPLEFT", button, "TOPLEFT", 1, -1 )
		button.cooldownText:SetPoint( "BOTTOMRIGHT", button, "BOTTOMRIGHT", -1, 1 )
		button.cooldownText:Hide()

		button.owner = self

		button:SetScript( "OnEnter", function()
			if this.owner.config.tooltips == "always"
				or (this.owner.config.tooltips == "ooc" and not( InCombatLockdown() )) then
				if not( this:IsVisible() and this.id ) or UH:GetAlpha() == 0 then return end

				GameTooltip:SetOwner( this, "ANCHOR_BOTTOMRIGHT" )

				if this.id == 0 then
					GameTooltip:SetText( this.isBuff and "Test buff" or "Test debuff" )
				elseif this.isPlayer then
					if this.isBuff then
					    local id = GetPlayerBuff( this.id, "HELPFUL" )
						GameTooltip:SetPlayerBuff( id, "HELPFUL" )
					else
					    local id = GetPlayerBuff( this.id, "HARMFUL" )
						GameTooltip:SetPlayerBuff( id, "HARMFUL" )
					end
				else
					if this.isBuff then
						GameTooltip:SetUnitBuff( this.unit, this.id )
					else
						GameTooltip:SetUnitDebuff( this.unit, this.id )
					end
				end

				GameTooltip:Show()
			end
		end )

		button:SetScript( "OnLeave", function()
			if GameTooltip:IsOwned( this ) then
				GameTooltip:Hide()
			end
		end )

		button:SetScript( "OnClick", function()
			if not this.isPlayer then return end

			CancelPlayerBuff( this.id )
		end )

		button:Hide()

		table.insert( self.buttons, button )
	end

	self.frame:SetAttribute( "unit", self.config.unit )

	self:SetupOptions()

	if self.config.enabled then
	    self:Enable()
	end
	
	return self
end

function BuffFrame:Destroy()
	self:Disable()
	
	for _, button in ipairs( self.buttons ) do
		UH:ReleaseFrame( button.cooldownText )
		UH:ReleaseFrame( button.cooldown )
		UH:ReleaseFrame( button.Count )
		UH:ReleaseFrame( button.Icon )

		button.cooldownText = nil
		button.cooldown = nil
		button.Count = nil
		button.Icon = nil

		button.owner = nil

		UH:ReleaseFrame( button )
	end

	self.buttons = nil

	if self.frame then
		self.frame:SetAttribute( "unit", nil )
		
		UH:ReleaseFrame( self.frame )
		self.frame = nil
	end
end

function BuffFrame:Enable()
	if self.enabled then return end

	if not self:InConfigMode() then
		RegisterUnitWatch( self.frame )
	else
		self.frame:Show()
	end

	self:PerformLayout()

	self:RegisterEvent( "UNIT_AURA" )
	self:RegisterEvent( "PLAYER_TARGET_CHANGED" )
	self:RegisterEvent( "PLAYER_FOCUS_CHANGED" )

	self:ForceUpdate()

	UH:RegisterUpdateTarget( self )

	self.enabled = true
end

function BuffFrame:Disable()
	if not self.enabled then return end

	UH:UnregisterUpdateTarget( self )
	self:UnregisterAllEvents()

	UnregisterUnitWatch( self.frame )
	self.frame:Hide()

	self.enabled = nil
end

local function getUnitAura( unit, index, isBuff )
	local name, rank, iconTexture, count, debuffType, duration, timeLeft
	
	if isBuff then
		name, rank, iconTexture, count, duration, timeLeft = UnitBuff( unit, index )
	else
		name, rank, iconTexture, count, debuffType, duration, timeLeft = UnitDebuff( unit, index )
	end
	
	return name, iconTexture, count, debuffType, duration, timeLeft
end

-- From PitBull_Aura, thanks once again :)

local getPlayerAuraDuration
do
	local buffData = {}
	local debuffData = {}
	
	function getPlayerAuraDuration( isBuff, key, providedDuration )
		local data = isBuff and buffData or debuffData
		local duration = data[key]
		if not duration then
			for i = 1, 40 do
				local name, _, time
				if isBuff then
					name, _, _, _, time = UnitBuff("player", i)
				else
					name, _, _, _, _, time = UnitDebuff("player", i)
				end
				if not name then
					break
				end
				if name == key then
					duration = time
					break
				end
			end
		end
		if not duration or duration < providedDuration then
			duration = providedDuration
		end
		data[key] = duration
		return duration
	end
end

local auraSort__isFriend
local auraSort__isBuff
local function auraSort(alpha, bravo)
	-- id, name, iconTexture, count, debuffType, duration, timeLeft, itemSlot, quality

	if not alpha then
		return false
	elseif not bravo then
		return true
	end

	-- show your own buffs/debuffs first
	local alpha_timeLeft, bravo_timeLeft = alpha[7], bravo[7]
	if not alpha_timeLeft ~= not bravo_timeLeft then
		if alpha_timeLeft then
			return true
		else
			return false
		end
	end

	if auraSort__isBuff then
		-- item buffs first
		local alpha_itemSlot, bravo_itemSlot = alpha[8], bravo[8]
		if alpha_itemSlot and not bravo_itemSlot then
			return true
		elseif not alpha_itemSlot and bravo_itemSlot then
			return false
		elseif alpha_itemSlot and bravo_itemSlot then
			return alpha_itemSlot < bravo_itemSlot
		end
	else
		if auraSort__isFriend then
			-- sort by dispel type
			local alpha_dispelType, bravo_dispelType = alpha[5], bravo[5]
			if alpha_dispelType ~= bravo_dispelType then
				if not alpha_dispelType then
					return false
				elseif not bravo_dispelType then
					return true
				end
				local canDispel_alpha_dispelType = UHBS.canDispel[alpha_dispelType]
				if not canDispel_alpha_dispelType ~= not UHBS.canDispel[bravo_dispelType] then
					-- show debuffs you can dispel first
					if canDispel_alpha_dispelType then
						return true
					else
						return false
					end
				end
				return alpha_dispelType < bravo_dispelType
			end
		end
	end

	-- sort by name
	local alpha_name, bravo_name = alpha[2], bravo[2]
	if alpha_name ~= bravo_name then
		if not alpha_name then
			return false
		elseif not bravo_name then
			return true
		end
		if auraSort__isBuff then
			if auraSort__isFriend then
				-- show buffs you can cast first
				local alpha_isMyBuff = UHBS.allFriendlyBuffs[alpha_name]
				local bravo_isMyBuff = UHBS.allFriendlyBuffs[bravo_name]
				if not alpha_isMyBuff ~= not bravo_isMyBuff then
					if alpha_isMyBuff then
						return true
					else
						return false
					end
				end
			end
		else
			if auraSort__isFriend then
				-- show debuffs you can cast first
				local alpha_isMyDebuff = UHBS.allFriendlyDebuffs[alpha_name]
				local bravo_isMyDebuff = UHBS.allFriendlyDebuffs[bravo_name]
				if not alpha_isMyDebuff ~= not bravo_isMyDebuff then
					if alpha_isMyDebuff then
						return true
					else
						return false
					end
				end
			else
				local alpha_isMyDebuff = UHBS.enemyDebuffs[alpha_name]
				local bravo_isMyDebuff = UHBS.enemyDebuffs[bravo_name]
				if not alpha_isMyDebuff ~= not bravo_isMyDebuff then
					if alpha_isMyDebuff then
						return true
					else
						return false
					end
				end
			end
		end
		return alpha_name < bravo_name
	end

	-- keep ID order
	local alpha_id, bravo_id = alpha[1], bravo[1]
	if not alpha_id then
		return false
	elseif not bravo_id then
		return true
	end
	return alpha_id < bravo_id
end

function BuffFrame:CollectAuras()
	local config = self.config
	local uid = config.unit
	local isBuff = config.type == "buffs"
	local auras = newList()
	local maxAuras = self.config.maxAuras
	local isPlayer = UnitIsUnit( uid, "player" )
	local isPet = not isPlayer and UnitIsUnit( uid, "pet" )
	local isFriend = isPlayer or isPet or UnitIsFriend( "player", uid )
	local filtering = self.config.filter
	local filteredSpells, extraFilteredSpells
	
	if filtering then
		if isBuff then
			if isFriend then
				filteredSpells = self.module.db.profile.filter.friendBuffs
				extraFilteredSpells = self.module.db.profile.filter.extraFriendBuffs
			else
				filtering = false
			end
		else
			if isFriend then
				filteredSpells = self.module.db.profile.filter.friendDebuffs
				extraFilteredSpells = self.module.db.profile.filter.extraFriendDebuffs
			else
				filteredSpells = self.module.db.profile.filter.enemyDebuffs
				extraFilteredSpells = self.module.db.profile.filter.extraEnemyDebuffs
			end
		end
	end
	
	if isPlayer then --uid == "player" then
		local id
		local i = 1

		while true do
			-- if #auras > maxAuras then break end

			id = GetPlayerBuff( i, isBuff and "HELPFUL" or "HARMFUL" )
			if id == 0 then break end
			
			local name, iconTexture, count, debuffType, timeLeft = GetPlayerBuffName( id ), GetPlayerBuffTexture( id ), GetPlayerBuffApplications( id ), GetPlayerBuffDispelType( id ), GetPlayerBuffTimeLeft( id )
			local duration = getPlayerAuraDuration( isBuff, name, timeLeft )

			local filtered = false

			if filtering then
			    if extraFilteredSpells and extraFilteredSpells[name] ~= nil then
			        filtered = not extraFilteredSpells[name]
				elseif not filteredSpells[name] then
					filtered = true
				elseif isBuff then
					filtered = not UHBS.totalSelfBuffs[name]
				else
					filtered = not UHBS.canDispel[debuffType] and not UHBS.totalSelfDebuffs[name]
				end
			end
			
			if not filtered then
				auras[#auras+1] = newList( i, name, iconTexture, count, debuffType, duration, timeLeft )
			end
			
			i = i + 1
		end
	else
		local id = 1

		while true do
			--if #auras > maxAuras then break end

			local name, iconTexture, count, debuffType, duration, timeLeft = getUnitAura( uid, id, isBuff )
			
			if not name then break end

			local filtered = false

			if filtering then
				if extraFilteredSpells and extraFilteredSpells[name] ~= nil then
			        filtered = not extraFilteredSpells[name]
				elseif not filteredSpells[name] or (extraFilteredSpells and extraFilteredSpells[name] ~= nil and (not extraFilteredSpells[name])) then
					filtered = true
				elseif isBuff then
					if isPet then
						filtered = not UHBS.totalPetBuffs[name]
					elseif isFriend then
						filtered = not UHBS.friendBuffs[name]
					end
				else
					if isPet then
						filtered = not UHBS.canDispel[debuffType] and not UHBS.totalPetDebuffs[name]
					elseif isFriend then
						filtered = not UHBS.canDispel[debuffType] and not UHBS.friendDebuffs[name]
					else
						filtered = not UHBS.enemyDebuffs[name]
					end
				end
			end
			
			if not filtered then
				auras[#auras+1] = newList( id, name, iconTexture, count, debuffType, duration, timeLeft )
			end

			id = id + 1
		end
	end

	auraSort__isFriend = isFriend
	auraSort__isBuff = isBuff
	
	table_sort( auras, auraSort )
	
	for i = maxAuras+1, #auras do
		auras[i] = releaseTable( auras[i] )
	end
	
	return auras
end

function BuffFrame:UpdateButtons()
	local auras = self:CollectAuras()
	local maxButtons = self.config.maxAuras
	local unit = self.config.unit
	local isBuff = self.config.type == "buffs"
	local isPlayer = UnitIsUnit( unit, "player" )
	local inConfigMode = self:InConfigMode()
	local groupCount = GetNumPartyMembers() + GetNumRaidMembers()

	if inConfigMode then
		for i = #auras+1, maxButtons do
			auras[i] = newList( 0, "Test", isBuff and "Interface\\Icons\\INV_Misc_Ear_Human_02" or "Interface\\Icons\\INV_Misc_Bone_HumanSkull_01", i )
		end
	end

	for i, button in ipairs( self.buttons ) do
		if i > maxButtons then
			if not button:IsVisible() then break end
			
			button.id = nil
			button.isBuff = nil
			button.isPlayer = nil
			button.unit = nil
			button.count = nil
			button.duration = nil
			button.isLarge = nil
			button.startTime = nil
			
			button:Hide()
		else
			if i > #auras then
				button.isBuff = nil
				button.isPlayer = nil
				button.unit = nil
				button.count = nil
				button.duration = nil
				button.isLarge = nil
				button.startTime = nil

				if inConfigMode then
					button.id = 0
					button.Icon:SetTexture( "Interface\\Icons\\INV_Misc_Ear_Human_02" )
					button.Count:SetText( i )
					button.Count:Show()

					button:Show()
				else
					button.id = nil
					if not button:IsVisible() then break end
					
					button:Hide()
				end
			else
				local id, name, texture, count, debuffType, duration, timeLeft = unpack( auras[i] )
				local startTime = nil
				local startTimeRounded = nil
				
				if duration and duration > 0 then
					startTime = GetTime()+timeLeft-duration
					startTimeRounded = math_floor( startTime )
				end
				
				if button.id ~= id or button.unit ~= unit or
					button.isBuff ~= isBuff or button.isPlayer ~= isPlayer or
					button.count ~= count or button.duration ~= duration or
					button.startTime ~= startTimeRounded or
					button.name ~= name or
					not( button:IsVisible() ) or self.forceUpdate then

					--[[
					if self.config.unit == "targettarget" then
						UH:Debug( ("Old:: %d: [%d] %s(%d) %f/%f @ %f"):format( i, button.id or -1, button.name or "nil", count or -1, timeLeft or -1, duration or -1, button.startTime or -1 ) )
						UH:Debug( ("New:: %d: [%d] %s(%d) %f/%f @ %f"):format( i, id or -1, name or "nil", count or -1, timeLeft or -1, duration or -1, startTimeRounded or -1 ) )
					end
					--]]

					button.id = id
					button.unit = unit
					button.count = count
					button.duration = duration
					button.startTime = startTimeRounded
					button.isBuff = isBuff
					button.isPlayer = isPlayer
					button.name = name

					button.Icon:SetTexture( texture )

					if count and count > 1 then
						button.Count:SetText( count )
						button.Count:Show()
					else
						button.Count:Hide()
					end

					if duration and (groupCount > 0 or self.doubleWhenSolo) then
					    button.isLarge = ((isBuff and self.doubleBuffs) or (not( isBuff ) and self.doubleDebuffs)) and true or nil
					else
					    button.isLarge = nil
					end

					if startTime then
						button.cooldown:Show()
						button.cooldown:SetCooldown( startTime, duration )
					else
						button.cooldown:Hide()
					end

					if isBuff or not self.colorDebuffBorders then
						button:SetBackdropBorderColor( 0, 0, 0, 1 )
					else
						local color = DebuffTypeColor[debuffType or "none"]

						button:SetBackdropBorderColor( color.r, color.g, color.b, 1 )
					end

					button:Show()
				end
			end
		end
	end
	
	self:PerformLayout()
	
	for i, v in ipairs( auras ) do
		auras[i] = releaseTable( auras[i] )
	end
	
	auras = releaseTable( auras )
end

function BuffFrame:ProcessUpdate()
	if self.needUpdate or self.config.unit == "targettarget" then
		self.needUpdate = nil
		self:UpdateButtons()
		self.forceUpdate = nil
	end
end

function BuffFrame:ForceUpdate()
	self.colorDebuffBorders = self.module.db.profile.colorDebuffBorders
	self.doubleBuffs = self.module.db.profile.doubleSelfBuffs
	self.doubleDebuffs = self.module.db.profile.doubleSelfDebuffs
	self.doubleWhenSolo = self.module.db.profile.doubleWhenSolo

	self.forceUpdate = true
	self.needUpdate = true
end

function BuffFrame:UNIT_AURA( eventName, unitId )
	if unitId == self.config.unit then
		self.needUpdate = true
	end
end

function BuffFrame:PLAYER_TARGET_CHANGED()
	if self.config.unit == "target" or self.config.unit == "targettarget" then
		self.forceUpdate = true
		self.needUpdate = true
	end
end

function BuffFrame:PLAYER_FOCUS_CHANGED()
	if self.config.unit == "focus" then
		self.needUpdate = true
	end
end

function BuffFrame:InConfigMode()
	return UH:InConfigMode()
end

function BuffFrame:SetConfigMode( mode )
	if mode then
		UnregisterUnitWatch( self.frame )
		
		self.frame:Show()
	else
		RegisterUnitWatch( self.frame )
	end
	
	self.needUpdate = true
	self.forceUpdate = true
end

function BuffFrame:ChangeUnit( unit )
	if unit and unit == self.config.unit then return end
	
	UnregisterUnitWatch( self.frame )
	
	self.config.unit = unit

	self.frame:SetAttribute( "unit", unit )

	if not self:InConfigMode() then
		RegisterUnitWatch( self.frame )
	else
		self.frame:Show()
	end

	self.needUpdate = true
	self.forceUpdate = true
end

function BuffFrame:SetupOptions()
	self.options = {
		name = function() return self.config.name end,
		type = "group",
		childGroups = "tab",
		args = {
			unitAndType = {
				name = L["Unit and Type"],
				type = "group",
				order = 100,
				args = {
					unit = {
						name = L["Unit"],
						type = "select",
						order = 1,
						values = UH.OptionsHelper.Unit,
						get = function() return self.config.unit end,
						set = function( info, value ) self:ChangeUnit( value ) end,
					},
					type = {
						name = L["Type"],
						type = "select",
						order = 2,
						values = validTypes,
						get = function( info ) return self.config.type end,
						set = function( info, value ) self.config.type = value; self.needUpdate = true end,
					},
					tooltips = {
						name = L["Show Tooltips"],
						type = "select",
						order = 3,
						values = UH.OptionsHelper.TooltipVisibility,
						get = function() return self.config.tooltips end,
						set = function( info, value ) self.config.tooltips = value end,
					},
					filter = {
						name = L["Filter auras"],
						desc = L["Filter certain auras based on your class"],
						type = "toggle",
						order = 6,
						get = function() return self.config.filter end,
						set = function( info, value ) self.config.filter = value; self.needUpdate = true end,
					},
				},
			},
			positionAndSize = {
				name = L["Position and Size"],
				type = "group",
				order = 102,
				args = {
					x = {
						name = L["X"],
						type = "range",
						order = 1,
						min = -800,
						max = 800,
						step = 1,
						bigStep = 1,
						get = function() return self.config.x end,
						set = function( info, value )
							self.config.x = value
							self.frame:ClearAllPoints()
							self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.config.x, self.config.y )
						end,
					},
					y = {
						name = L["Y"],
						type = "range",
						order = 2,
						min = -800,
						max = 800,
						step = 1,
						bigStep = 1,
						get = function() return self.config.y end,
						set = function( info, value )
							self.config.y = value
							self.frame:ClearAllPoints()
							self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.config.x, self.config.y )
						end,
					},
					layout = {
						name = L["Layout"],
						type = "select",
						order = 3,
						values = validLayouts,
						get = function() return self.config.layout end,
						set = function( info, value ) self.config.layout = value; self:PerformLayout() end,
					},
					size = {
						name = L["Button size"],
						type = "range",
						order = 4,
						min = 10,
						max = 48,
						step = 1,
						bigStep = 1,
						get = function() return self.config.size end,
						set = function( info, value )
							self.config.size = value
							self:PerformLayout()
						end,
					},
					maxAuras = {
						name = L["Auras to show"],
						desc = L["Maximum number of auras to show"],
						order = 5,
						type = "range",
						min = 1,
						max = MAX_AURAS,
						step = 1,
						bigStep = 1,
						get = function() return self.config.maxAuras end,
						set = function( info, value )
							if self.config.columns > value then
								self.config.columns = value
							end
							
							self.config.maxAuras = value
							self.needUpdate = true
						end,
					},
					columns = {
						name = L["Columns"],
						type = "range",
						order = 6,
						min = 1,
						max = MAX_AURAS,
						step = 1,
						bigStep = 1,
						get = function() return self.config.columns end,
						set = function( info, value )
							self.config.columns = math_min( value, self.config.maxAuras )
							self:PerformLayout()
						end,
					},
					spacing = {
						name = L["Space between buttons"],
						type = "range",
						order = 7,
						min = 0,
						max = 40,
						step = 1,
						bigStep = 1,
						get = function() return self.config.spacing end,
						set = function( info, value )
							self.config.spacing = value
							self:PerformLayout()
						end,
					},
					scale = {
						name = L["Scale"],
						type = "range",
						order = 9,
						min = 0.01,
						max = 1,
						step = 0.01,
						bigStep = 0.01,
						get = function() return self.config.scale end,
						set = function( info, value )
							self.config.scale = value
							self.frame:SetScale( value )
						end,
					},
				},
			},
			zOrder = {
				name = L["Z-Order"],
				type = "group",
				order = 103,
				args = {
					strata = {
						name = L["Strata"],
						type = "select",
						order = 1,
						values = UH.OptionsHelper.Strata,
						get = function() return self.config.strata end,
						set = function( info, value ) self.config.strata = value; self.frame:SetFrameStrata( value ) end,
					},
					level = {
						name = L["Level"],
						type = "range",
						order = 2,
						min = 1,
						max = 10,
						step = 1,
						get = function() return self.config.level end,
						set = function( info, value ) self.config.level = value; self.frame:SetFrameLevel( value ) end,
					},
				},
			},
		},
		plugins = {},
	}
end
