local UH = UnderHood
local UHB = UnderHood_Bars
local LSM = LibStub( "LibSharedMedia-3.0" )

local math_floor = math.floor

--[[
	Simple texture: consists of only one texture
--]]

local function anchorFrame( frame, parent, points )
    if points then
        if type( points ) == "table" then
            for _, point in ipairs( points ) do
                local pt, rpt, x, y = unpack( point )

                frame:SetPoint( pt, parent, rpt, x, y )
            end
        else
            frame:SetAllPoints( parent )
        end
    end
end

local SimpleTexture = {}
SimpleTexture.__index = SimpleTexture

function SimpleTexture:Create( parent, descriptor, settings )
	local self = setmetatable( {}, SimpleTexture )
	
	self.frame = UH:CreateFrame( "Frame", parent )
	self.texture = UH:CreateFrame( "Texture", self.frame )
	self.descriptor = descriptor
	self.settings = settings

	if descriptor.strata then
	    self.frame:SetFrameStrata( descriptor.strata )
	end
	
	if descriptor.lsmName and LSM then
		self.texture:SetTexture( LSM:Fetch( LSM.MediaType.STATUSBAR, settings[descriptor.lsmName] ) )
	else
		self.texture:SetTexture( descriptor.path )
	end
	
	self.texture:SetAllPoints( self.frame )
	
	if descriptor.blendMode then
	    self.texture:SetBlendMode( descriptor.blendMode )
	end
	
	anchorFrame( self.frame, parent, descriptor.points )
	
	if descriptor.color then
	    local r, g, b, a

	    if type( descriptor.color ) == "table" then
	        r, g, b, a = unpack( descriptor.color )
		elseif type( descriptor.color ) == "string" then
		    local color = settings[descriptor.color]

		    if type( color ) == "table" then
		        r, g, b, a = unpack( color )
		    end
		end

		if r then
			self:SetColor( r, g, b, a )
		end
	end

	self.texture:Show()

	if descriptor.visibility then
	    local value = settings[descriptor.visibility]

	    if value then
	        self.frame:Show()
		else
		    self.frame:Hide()
		end
	else
        self.frame:Show()
	end
	
	return self
end

function SimpleTexture:Destroy()
	UH:ReleaseFrame( self.texture )
	UH:ReleaseFrame( self.frame )
	
	self.texture = nil
	self.frame = nil
end

function SimpleTexture:SetTexture( tx )
	self.texture:SetTexture( tx )
end

function SimpleTexture:SetColor( r, g, b, a )
	self.texture:SetVertexColor( r, g, b, a )
end

function SimpleTexture:Show()
	self.frame:Show()
end

function SimpleTexture:Hide()
	self.frame:Hide()
end

function SimpleTexture:SetPoint( localPoint, relativeTo, relativePoint, x, y )
	self.frame:SetPoint( localPoint, relativeTo, relativePoint, x, y )
end

function SimpleTexture:SetAllPoints( relativeTo )
	self.frame:SetAllPoints( relativeTo )
end

function SimpleTexture:ClearAllPoints()
	self.frame:ClearAllPoints()
end

function SimpleTexture:SetOrientation( vertical, reversed )
	if self.vertical == vertical and self.reversed == reversed then return end
	
	self.vertical = vertical
	self.reversed = reversed

	local tx = self.texture
	local frame = self.frame

	tx:ClearAllPoints()

	if vertical then
		tx:SetPoint( "LEFT", frame, "LEFT" )
		tx:SetPoint( "RIGHT", frame, "RIGHT" )

		if reversed then
			tx:SetPoint( "BOTTOM", frame, "BOTTOM" )
		else
			tx:SetPoint( "TOP", frame, "TOP" )
		end
	else
		tx:SetPoint( "TOP", frame, "TOP" )
		tx:SetPoint( "BOTTOM", frame, "BOTTOM" )

	    if reversed then
			tx:SetPoint( "RIGHT", frame, "RIGHT" )
	    else
			tx:SetPoint( "LEFT", frame, "LEFT" )
	    end
	end
end

function SimpleTexture:SetStatusBarValue( p )
    local l, r, t, b, w, h = 0, 1, 0, 1, self.frame:GetWidth(), self.frame:GetHeight()
	local tx = self.texture
	local frame = self.frame
	
	if self.vertical then
		if self.reversed then
		    t = 1 - p
		else
		    b = p
		end
		
		h = h * p
	else
	    if self.reversed then
	        l = 1 - p
	    else
			r = p
	    end
	    
	    w = w * p
	end
	
	if math_floor( w * 100 ) == 0 or math_floor( h * 100 ) == 0 then
	    self.texture:Hide()
	else
		self.texture:SetWidth( w )
		self.texture:SetHeight( h )
		self.texture:SetTexCoord( l, r, t, b )
	    self.texture:Show()
	end
end

--[[
	Compound texture: consists of 8 or 9 textures
	
	descriptor:
	    .parts = {...}: - indexed table of parts
			{ "path to tga", width, height }
			Indexes are (from 1 to 9): TL, TR, BL, BR, T, B, L, R, C
--]]

local CompoundTextureAnchors = {
	{ "TOPLEFT", "TOPLEFT", false, 0, 0 }, -- TL
	{ "TOPRIGHT", "TOPRIGHT", false, 0, 0 }, -- TR
	{ "BOTTOMLEFT", "BOTTOMLEFT", false, 0, 0 }, -- BL
	{ "BOTTOMRIGHT", "BOTTOMRIGHT", false, 0, 0 }, -- BR
	{ -- T
		{ "TOPLEFT", "TOPRIGHT", 1, 0, 0 },
		{ "TOPRIGHT", "TOPLEFT", 2, 0, 0 },
	},
	{ -- B
		{ "BOTTOMLEFT", "BOTTOMRIGHT", 3, 0, 0 },
		{ "BOTTOMRIGHT", "BOTTOMLEFT", 4, 0, 0 },
	},
	{ -- L
		{ "TOPLEFT", "BOTTOMLEFT", 1, 0, 0 },
		{ "BOTTOMLEFT", "TOPLEFT", 3, 0, 0 },
	},
	{ -- R
		{ "TOPRIGHT", "BOTTOMRIGHT", 2, 0, 0 },
		{ "BOTTOMRIGHT", "TOPRIGHT", 4, 0, 0 },
	},
	{ -- C
		{ "TOPLEFT", "BOTTOMRIGHT", 1, 0, 0 },
		{ "BOTTOMRIGHT", "TOPLEFT", 4, 0, 0 },
	},
}

local CompoundTexture = {}
CompoundTexture.__index = CompoundTexture

function CompoundTexture:Create( parent, descriptor, settings )
	local self = setmetatable( {}, CompoundTexture )

	self.parts = descriptor.parts
	self.frame = UH:CreateFrame( "Frame", parent )

	if descriptor.strata then
	    self.frame:SetFrameStrata( descriptor.strata )
	end
	
	local textures = {}

	local path = descriptor.path
	
	for i, td in ipairs( descriptor.parts ) do
	    local name, size, gap = unpack( td )
	    local tx
	    
	    if name then
	    	tx = UH:CreateFrame( "Texture", self.frame )

			tx.usize = size
			tx.ugap = gap or 0
		    tx:SetWidth( size )
		    tx:SetHeight( size )

		    if descriptor.blendMode then
		        tx:SetBlendMode( descriptor.blendMode )
			end

			local pPath = path and path.."\\"..name or name
			
			tx:SetTexture( pPath )
			
			textures[i] = tx
			
			local ta = CompoundTextureAnchors[i]
			local lp, rp, rt, dx, dy
			
			if type( ta[1] ) == "table" then
			    for _, a in ipairs( ta ) do
				    lp, rp, rt, dx, dy = unpack( a )

				    tx:SetPoint( lp, rt and textures[rt] or self.frame, rp, dx, dy )
			    end
			else
			    lp, rp, rt, dx, dy = unpack( ta )
			    
			    tx:SetPoint( lp, rt and textures[rt] or self.frame, rp, dx, dy )
			end
			
			tx:Show()
		end
	end

	self.textures = textures

	anchorFrame( self.frame, parent, descriptor.points )

	if descriptor.color then
	    local r, g, b, a

	    if type( descriptor.color ) == "table" then
	        r, g, b, a = unpack( descriptor.color )
		elseif type( descriptor.color ) == "string" then
		    local color = settings[descriptor.color]

		    if type( color ) == "table" then
		        r, g, b, a = unpack( color )
		    end
		end

		if r then
			self:SetColor( r, g, b, a )
		end
	end

	if descriptor.visibility then
	    local value = settings[descriptor.visibility]

	    if value then
	        self.frame:Show()
		else
		    self.frame:Hide()
		end
	else
        self.frame:Show()
	end

	return self
end

function CompoundTexture:Destroy()
	for _, tx in ipairs( self.textures ) do
	    if tx then
	        tx.usize = nil
	        tx.ugap = nil
			UH:ReleaseFrame( tx )
		end
	end

	UH:ReleaseFrame( self.frame )

	self.textures = nil
	self.frame = nil
end

function CompoundTexture:SetTexture( tx )
	error( "CompoundTexture is not intended to be changeable.", 2 )
end

function CompoundTexture:SetColor( r, g, b, a )
	for _, tx in ipairs( self.textures ) do
		tx:SetVertexColor( r, g, b, a )
	end
end

function CompoundTexture:Show()
	self.frame:Show()
end

function CompoundTexture:Hide()
	self.frame:Hide()
end

function CompoundTexture:SetPoint( localPoint, relativeTo, relativePoint, x, y )
	self.frame:SetPoint( localPoint, relativeTo, relativePoint, x, y )
end

function CompoundTexture:SetAllPoints( relativeTo )
	self.frame:SetAllPoints( relativeTo )
end

function CompoundTexture:ClearAllPoints()
	self.frame:ClearAllPoints()
end

function CompoundTexture:SetOrientation( vertical, reversed )
	if self.vertical == vertical and self.reversed == reversed then return end

	self.vertical = vertical
	self.reversed = reversed

	local frame = self.frame
	local txs = self.textures
	local tx

	for _, tx in ipairs( txs ) do
	    tx:ClearAllPoints()
	end
	
	if vertical then
		txs[1]:SetPoint( "LEFT", frame, "LEFT" )
		txs[2]:SetPoint( "RIGHT", frame, "RIGHT" )
		txs[3]:SetPoint( "LEFT", frame, "LEFT" )
		txs[4]:SetPoint( "RIGHT", frame, "RIGHT" )
		txs[5]:SetPoint( "LEFT", txs[1], "RIGHT" )
		txs[5]:SetPoint( "RIGHT", txs[2], "LEFT" )
		txs[6]:SetPoint( "LEFT", txs[3], "RIGHT" )
		txs[6]:SetPoint( "RIGHT", txs[4], "LEFT" )
		txs[7]:SetPoint( "LEFT", frame, "LEFT" )
		txs[8]:SetPoint( "RIGHT", frame, "RIGHT" )

		if txs[9] then
			txs[9]:SetPoint( "LEFT", txs[7], "RIGHT" )
			txs[9]:SetPoint( "RIGHT", txs[8], "LEFT" )
		end
		
		if reversed then
			txs[3]:SetPoint( "BOTTOM", frame, "BOTTOM" )
			txs[4]:SetPoint( "BOTTOM", frame, "BOTTOM" )
			txs[6]:SetPoint( "BOTTOM", frame, "BOTTOM" )

			txs[7]:SetPoint( "BOTTOM", txs[3], "TOP" )
			txs[8]:SetPoint( "BOTTOM", txs[4], "TOP" )
			
			if txs[9] then
				txs[9]:SetPoint( "BOTTOM", txs[6], "TOP" )
			end

			txs[1]:SetPoint( "BOTTOM", txs[7], "TOP" )
			txs[2]:SetPoint( "BOTTOM", txs[8], "TOP" )
			txs[5]:SetPoint( "BOTTOMLEFT", txs[7], "TOPRIGHT" )
		else
			txs[1]:SetPoint( "TOP", frame, "TOP" )
			txs[2]:SetPoint( "TOP", frame, "TOP" )
			txs[5]:SetPoint( "TOP", frame, "TOP" )

			txs[7]:SetPoint( "TOP", txs[1], "BOTTOM" )
			txs[8]:SetPoint( "TOP", txs[2], "BOTTOM" )

			if txs[9] then
				txs[9]:SetPoint( "TOP", txs[5], "BOTTOM" )
			end

			txs[3]:SetPoint( "TOP", txs[7], "BOTTOM" )
			txs[4]:SetPoint( "TOP", txs[8], "BOTTOM" )
			txs[6]:SetPoint( "TOPLEFT", txs[7], "BOTTOMRIGHT" )
		end
	else
		txs[1]:SetPoint( "TOP", frame, "TOP" )
		txs[2]:SetPoint( "TOP", frame, "TOP" )
		txs[3]:SetPoint( "BOTTOM", frame, "BOTTOM" )
		txs[4]:SetPoint( "BOTTOM", frame, "BOTTOM" )
		txs[5]:SetPoint( "TOP", frame, "TOP" )
		txs[6]:SetPoint( "BOTTOM", frame, "BOTTOM" )
		txs[7]:SetPoint( "TOP", txs[1], "BOTTOM" )
		txs[7]:SetPoint( "BOTTOM", txs[3], "TOP" )
		txs[8]:SetPoint( "TOP", txs[2], "BOTTOM" )
		txs[8]:SetPoint( "BOTTOM", txs[4], "TOP" )

		if txs[9] then
			txs[9]:SetPoint( "TOP", txs[5], "BOTTOM" )
			txs[9]:SetPoint( "BOTTOM", txs[6], "TOP" )
		end

		if reversed then
			txs[2]:SetPoint( "RIGHT", frame, "RIGHT" )
			txs[4]:SetPoint( "RIGHT", frame, "RIGHT" )
			txs[8]:SetPoint( "RIGHT", frame, "RIGHT" )

			txs[5]:SetPoint( "RIGHT", txs[2], "LEFT" )
			txs[6]:SetPoint( "RIGHT", txs[4], "LEFT" )

			if txs[9] then
				txs[9]:SetPoint( "RIGHT", txs[8], "LEFT" )
			end

			txs[1]:SetPoint( "RIGHT", txs[5], "LEFT" )
			txs[3]:SetPoint( "RIGHT", txs[6], "LEFT" )
			txs[7]:SetPoint( "TOPRIGHT", txs[5], "BOTTOMLEFT" )
		else
			txs[1]:SetPoint( "LEFT", frame, "LEFT" )
			txs[3]:SetPoint( "LEFT", frame, "LEFT" )
			txs[7]:SetPoint( "LEFT", frame, "LEFT" )

			txs[5]:SetPoint( "LEFT", txs[1], "RIGHT" )
			txs[6]:SetPoint( "LEFT", txs[3], "RIGHT" )

			if txs[9] then
				txs[9]:SetPoint( "LEFT", txs[7], "RIGHT" )
			end

			txs[2]:SetPoint( "LEFT", txs[5], "RIGHT" )
			txs[4]:SetPoint( "LEFT", txs[6], "RIGHT" )
			txs[8]:SetPoint( "TOPLEFT", txs[5], "BOTTOMRIGHT" )
		end
	end
end

function CompoundTexture:SetStatusBarValue( p )
	local tvP, cvP, bvP, lhP, chP, rhP
	local tH, bH, cH, lW, cW, rW, fW, fH
	--local tG, bG, lG, rG
	local textures = self.textures

	--[[
	tG = textures[5].ugap
	bG = textures[6].ugap
	lG = textures[7].ugap
	rG = textures[8].ugap
	--]]
	
	fH = self.frame:GetHeight()
	tH = textures[5].usize
	bH = textures[6].usize
	cH = fH - tH - bH -- textures[7].uhh
	fW = self.frame:GetWidth()
	lW = textures[1].usize
	rW = textures[2].usize
	cW = fW - lW - rW -- textures[5].uhw
	
	-- Still not sure about taking gaps into account...
	--tvP = (tH - tG) / (fH - tG - bG) -- tH / fH
	--bvP = (bH - bG) / (fH - tG - bG) -- bH / fH
	--cvP = 1 - tvP - bvP --cH / fH

	tvP = tH / fH
	bvP = bH / fH
	cvP = cH / fH
	
	lhP = lW / fW
	chP = cW / fW
	rhP = rW / fW
	
	--cW = fW - lW - rW
	--cH = fH - tH - bH
	
	local lL, lR, cL, cR, rL, rR = 0, 1, 0, 1, 0, 1
	local tT, tB, cT, cB, bT, bB = 0, 1, 0, 1, 0, 1

	local vertical = self.vertical
	local reversed = self.reversed
	
	if vertical then
	    if reversed then
	        if p < bvP then
				p = p / bvP
	            bT = 1 - p; bH = bH * p
				cT = 0; cH = 0
				tT = 0; tH = 0
	        else
	            p = p - bvP
	            
	            if p < cvP then
	                p = p / cvP
	                cT = 1 - p; cH = cH * p
					tT = 0; tH = 0
	            else
					p = (p - cvP) / tvP
					
	                tT = 1 - p; tH = tH * p
	            end
	        end
	    else
	        if p < tvP then
	            tB = p / tvP; tH = tH * tB
				cB = 0; cH = 0
				bB = 0; bH = 0
			else
			    p = p - tvP
			    
			    if p < cvP then
			        cB = p / cvP; cH = cH * cB
					bB = 0; bH = 0
			    else
			        p = p - cvP
			        
			        bB = p / bvP; bH = bH * bB
			    end
			end
	    end
	else
	    if reversed then
	        if p < rhP then
	            p = p / rhP
	            rL = 1 - p; rW = rW * p
				cL = 0; cW = 0
				lL = 0; lW = 0
	        else
	            p = p - rhP

	            if p < chP then
					p = p / chP
	                cL = 1 - p; cW = cW * p
					lL = 0; lW = 0
	            else
	                p = (p - chP) / lhP
	                
	                lL = 1 - p; lW = lW * p
	            end
	        end
	    else
	        if p < lhP then
	            lR = p / lhP; lW = lW * lR
				cR = 0; cW = 0
				rR = 0; rW = 0
			else
			    p = p - lhP

			    if p < chP then
			        cR = p / chP; cW = cW * cR
					rR = 0; rW = 0
			    else
			        p = p - chP
			        
			        rR = p / rhP; rW = rW * rR
			    end
			end
	    end
	end
	
	local tx
	
	if lW > 0 and tH > 0 then
	    tx = textures[1]
	    
	    tx:SetWidth( lW ); tx:SetHeight( tH )
		tx:SetTexCoord( lL, lR, tT, tB )
		tx:Show()
	else
		textures[1]:Hide()
	end
	
	if rW > 0 and tH > 0 then
	    tx = textures[2]
	    
	    tx:SetWidth( rW ); tx:SetHeight( tH )
		tx:SetTexCoord( rL, rR, tT, tB )
		tx:Show()
	else
	    textures[2]:Hide()
	end
	
	if lW > 0 and bH > 0 then
	    tx = textures[3]
	    
	    tx:SetWidth( lW ); tx:SetHeight( bH )
		tx:SetTexCoord( lL, lR, bT, bB )
		tx:Show()
	else
	    textures[3]:Hide()
	end
	
	if rW > 0 and bH > 0 then
	    tx = textures[4]
	    
	    tx:SetWidth( rW ); tx:SetHeight( bH )
		tx:SetTexCoord( rL, rR, bT, bB )
		tx:Show()
	else
	    textures[4]:Hide()
	end
	
	if cW > 0 and tH > 0 then
	    tx = textures[5]
	    
	    tx:SetWidth( cW ); tx:SetHeight( tH )
		tx:SetTexCoord( cL, cR, tT, tB )
		tx:Show()
	else
	    textures[5]:Hide()
	end
	
	if cW > 0 and bH > 0 then
	    tx = textures[6]
	    
	    tx:SetWidth( cW ); tx:SetHeight( bH )
		tx:SetTexCoord( cL, cR, bT, bB )
		tx:Show()
	else
	    textures[6]:Hide()
	end
	
	if lW > 0 and cH > 0 then
	    tx = textures[7]
	    
	    tx:SetWidth( lW ); tx:SetHeight( cH )
		tx:SetTexCoord( lL, lR, cT, cB )
		tx:Show()
	else
	    textures[7]:Hide();
	end
	
	if rW > 0 and cH > 0 then
	    tx = textures[8]
	    
	    tx:SetWidth( rW ); tx:SetHeight( cH )
		tx:SetTexCoord( rL, rR, cT, cB )
		tx:Show()
	else
	    textures[8]:Hide()
	end
	
	if textures[9] then
	    if cW > 0 and cH > 0 then
	        tx = textures[9]
	        
	        tx:SetWidth( cW ); tx:SetHeight( cH )
			tx:SetTexCoord( cL, cR, cT, cB )
			tx:Show()
		else
		    textures[9]:Hide()
		end
	end
end

function UHB:CreateTexture( parent, descriptor, settings )
	local proto = nil
	
	if descriptor.type == "simple" then proto = SimpleTexture
	elseif descriptor.type == "compound" then proto = CompoundTexture
	end
	
	if not proto then return end
	
	return proto:Create( parent, descriptor, settings )
end
