local UH = UnderHood
local UHB = UnderHood_Bars
local L = LibStub( "AceLocale-3.0" ):GetLocale( "UnderHood" )
local LSM = LibStub( "LibSharedMedia-3.0" )

if LSM then
	LSM:Register( "statusbar", "UnderHood Vertical", [[Interface\Addons\UnderHood\Media\UHVB]] )
end

local metaStyles = {
	UnderHood = {
		title = L["UnderHood"],
	    config = {
	        width = 22,
	        height = 256,
		},
		styleSettings = {
			showBackground = true,
			bgColor = { 0.3, 0.3, 0.3, 0.75 },
			sbVertical = true,
			sbReversed = true,
			cbVertical = true,
			cbReversed = true,
			bgTexture = "UnderHood Vertical",
			sbTexture = "UnderHood Vertical",
		},
		zones = {
		    {
		        title = L["Main bar"],
			    texture = "statusbar",
		        defaultValue = 1,
			    vertical = "sbVertical",
			    reversed = "sbReversed",
			},
			{
				title = L["Outline bar"],
			    texture = "castingbar",
				defaultValue = 0,
			    vertical = "cbVertical",
			    reversed = "cbReversed",
			},
		},
		textures = {
		    background = {
		        name = L["Background"],
		        type = "simple",
		        lsmName = "bgTexture",
		        path = "Interface\\Addons\\UnderHood\\Media\\UHVB",
		        blendMode = "BLEND",
			    points = {
			        { "TOPLEFT", "TOPLEFT", 5, -5 },
			        { "BOTTOMRIGHT", "BOTTOMRIGHT", -5, 5 },
				},
				strata = "MEDIUM",
			    color = "bgColor",
			    visibility = "showBackground",
			},
			border = {
			    type = "compound",
			    path = "Interface\\Addons\\UnderHood\\Media",
			    blendMode = "BLEND",
			    strata = "LOW",
			    points = true,
			    parts = {
			        { "UHE-TL", 8 },
			        { "UHE-TR", 8 },
			        { "UHE-BL", 8 },
			        { "UHE-BR", 8 },
			        { "UHE-T", 8 },
			        { "UHE-B", 8 },
			        { "UHE-L", 8 },
			        { "UHE-R", 8 },
				},
			},
		    statusbar = {
		        name = L["Main bar"],
		        type = "simple",
		        lsmName = "sbTexture",
		        path = "Interface\\Addons\\UnderHood\\Media\\UHVB",
		        blendMode = "BLEND",
			    points = {
			        { "TOPLEFT", "TOPLEFT", 5, -5 },
			        { "BOTTOMRIGHT", "BOTTOMRIGHT", -5, 5 },
				},
				strata = "MEDIUM",
			},
		    castingbar = {
		        type = "compound",
		        path = "Interface\\Addons\\UnderHood\\Media",
		        blendMode = "BLEND",
			    points = true,
				strata = "HIGH",
			    parts = {
			        { "UHC-TL", 8, 2 },
			        { "UHC-TR", 8, 2 },
			        { "UHC-BL", 8, 2 },
			        { "UHC-BR", 8, 2 },
			        { "UHC-T", 8, 2 },
			        { "UHC-B", 8, 2 },
			        { "UHC-L", 8, 2 },
			        { "UHC-R", 8, 2 },
				},
			},
		},
	},
}

local Factory
local Style
do
	Style = {}
	Style.__index = Style

	function Style:AdjustBar( zone, p )
	    local zd = self.zones[zone]
	    local tx = zd.texture
	    
	    tx:SetStatusBarValue( p )
	end
	
	function Style:GetZones()
	    return self.meta.zones or {}
	end

	function Style:Create( bar, meta )
		local self = {}
		setmetatable( self, Style )

		self.bar = bar
		self.meta = meta
		self.textures = {}
		
		for tName, tData in pairs( meta.textures ) do
		    self.textures[tName] = UHB:CreateTexture( self.bar.frame, tData, bar.config.styleSettings )
		end

		self.zones = {}

		for z, zm in ipairs( meta.zones ) do
			local zd = {}

			zd.config = zm
			zd.texture = self.textures[zm.texture]
			zd.value = zm.defaultValue or 0
			zd.color = { r = 0, g = 0, b = 0, a = 0 }

			self.zones[z] = zd

			if type( zm.vertical ) == "boolean" then
			    zd.vertical = zm.vertical
			elseif type( zm.vertical ) == "string" then
			    zd.vertical = bar.config.styleSettings[zm.vertical]
			else
				zd.vertical = false
			end

			if type( zm.reversed ) == "boolean" then
			    zd.reversed = zm.reversed
			elseif type( zm.reversed ) == "string" then
			    zd.reversed = bar.config.styleSettings[zm.reversed]
			else
				zd.reversed = false
			end
			
			zd.texture:SetOrientation( zd.vertical, zd.reversed )
			self:AdjustBar( z, zd.value )
		end

		return self
	end

	function Style:Destroy( bar )
	    if self.textures then
		    for k, v in pairs( self.textures ) do
		        v:Destroy()
		    end
	    end
	    
	    self.textures = nil
	    self.zones = nil
	end

	function Style:SetColor( zone, r, g, b, a )
	    local zd = self.zones[zone]

	    if not zd then return end

	    zd.color.r = r
	    zd.color.g = g
	    zd.color.b = b
	    zd.color.a = a

		zd.texture:SetColor( r, g, b, a )
	end

	function Style:SetValue( zone, value )
	    local zd = self.zones[zone]

	    if not zd then return end

	    if value < 0 then value = 0 end
	    if value > 1 then value = 1 end

		if zd.value ~= value then
	    	zd.value = value

	    	self:AdjustBar( zone, self.bar:InConfigMode() and zd.config.defaultValue or value )
		end
	end

	function Style:Update()
	    for z, zd in ipairs( self.zones ) do
		    self:AdjustBar( z, self.bar:InConfigMode() and zd.config.defaultValue or zd.value )
	    end
	end

	local function HaveOptions( data )
	    return	type( data.color ) == "string" or
	            type( data.visibility ) == "string" or
	            (type( data.lsmName ) == "string" and LSM)
	end
	
	function Style:SetupOptions( options )
	    local order = 100
	    local optCache = {}
	    
	    for i, zm in ipairs( self.meta.zones ) do
	        if type( zm.vertical ) == "string" or type( zm.reversed ) == "string" then
	            local group = {
	                name = zm.title,
	                type = "group",
	                inline = true,
	                order = order,
	                args = {},
				}
				
				optCache[group.name] = group
				
				order = order + 1

				if type( zm.vertical ) == "string" then
				    local vopt = {
				        name = L["Vertical"],
				        type = "toggle",
				        order = 1,
				        get = function() return self.bar.config.styleSettings[zm.vertical] end,
				        set = function( info, value )
				            self.bar.config.styleSettings[zm.vertical] = value

							local zd = self.zones[i]
							
							zd.vertical = value

							zd.texture:SetOrientation( zd.vertical, zd.reversed )
							self:AdjustBar( i, zd.value )
				        end,
					}

					group.args.vertical = vopt
				end

				if type( zm.reversed ) == "string" then
				    local ropt = {
				        name = L["Reversed"],
				        type = "toggle",
				        order = 2,
				        get = function() return self.bar.config.styleSettings[zm.reversed] end,
				        set = function( info, value )
				            self.bar.config.styleSettings[zm.reversed] = value

							local zd = self.zones[i]

							zd.reversed = value

							zd.texture:SetOrientation( zd.vertical, zd.reversed )
							self:AdjustBar( i, zd.value )
				        end,
					}

					group.args.reversed = ropt
				end
				
				options["zone"..i] = group
			end
	    end
	    
	    for eName, eData in pairs( self.meta.textures ) do
	        if HaveOptions( eData ) then
	            local group = optCache[eData.name or eName]

				if not group then
					group = {
		                name = eData.name or eName,
		                type = "group",
		                inline = true,
		                order = order,
		                args = {},
					}

					order = order + 1

					options[eName] = group

					optCache[group.name] = group
				end
				
				if type( eData.visibility ) == "string" then
				    local vopt = {
				        name = L["Visible"],
				        type = "toggle",
				        order = 1,
				        get = function() return self.bar.config.styleSettings[eData.visibility] end,
				        set = function( info, value )
				            self.bar.config.styleSettings[eData.visibility] = value
				            
				            if value then
				                self.textures[eName]:Show()
							else
							    self.textures[eName]:Hide()
							end
				        end,
					}
					
					group.args.visible = vopt
				end
				
				if type( eData.color ) == "string" then
				    local copt = {
				        name = L["Color"],
				        type = "color",
				        order = 2,
				        hasAlpha = true,
				        get = function() return unpack( self.bar.config.styleSettings[eData.color] ) end,
				        set = function( info, r, g, b, a )
							self.bar.config.styleSettings[eData.color] = { r, g, b, a }
							
						    self.textures[eName]:SetColor( r, g, b, a )
						end,
					}
					
					group.args.color = copt
				end

				if type( eData.lsmName ) == "string" and LSM then
				    local t1opt = {
				        name = L["Texture1"],
				        type = "select",
				        order = 3,
				        get = function() return self.bar.config.styleSettings[eData.lsmName] end,
				        set = function( info, value )
				            self.bar.config.styleSettings[eData.lsmName] = value

							self.textures[eName]:SetTexture( LSM:Fetch( LSM.MediaType.STATUSBAR, value ) )
				        end,
						values = function()
							local rlst = { }

							for i,v in ipairs( LSM:List( LSM.MediaType.STATUSBAR ) ) do
								if i <= 30 then rlst[v] = v end
							end

							return rlst
						end,
					}

					group.args.texture1 = t1opt

				    local t2opt = {
				        name = L["Texture30"],
				        type = "select",
				        order = 4,
				        get = function() return self.bar.config.styleSettings[eData.lsmName] end,
				        set = function( info, value )
				            self.bar.config.styleSettings[eData.lsmName] = value

							self.textures[eName]:SetTexture( LSM:Fetch( LSM.MediaType.STATUSBAR, value ) )
				        end,
						values = function()
							local rlst = { }

							for i,v in ipairs( LSM:List( LSM.MediaType.STATUSBAR ) ) do
								if i > 30 then rlst[v] = v end
							end

							return rlst
						end,
					}

					group.args.texture30 = t2opt
				end
	        end
	    end
	end

	Factory = {}
	Factory.__index = Factory

	function Factory:New( metaName, metaStyle )
	    local self = {}
	    setmetatable( self, Factory )

		self.metaName = metaName
		self.metaStyle = metaStyle or metaStyles[metaName]
		self.name = metaName
		self.title = metaStyle.title or metaName

		return self
	end

	local function copyTable( src, dest )
		if type( dest ) ~= "table" then dest = {} end
		if type( src ) == "table" then
			for k, v in pairs( src ) do
				if type( v ) == "table" then
					-- try to index the key first so that the metatable creates the defaults, if set, and use that table
					v = copyTable( v, dest[k] )
				end

				dest[k] = v
			end
		end

		return dest
	end

	function Factory:DefaultSettings( config )
	    local metaConfig = self.metaStyle.config

	    if metaConfig then
	        if metaConfig.width then config.width = metaConfig.width end
	        if metaConfig.height then config.height = metaConfig.height end
		end

		if type( self.metaStyle.styleSettings ) == "table" then
		    config.styleSettings = copyTable( self.metaStyle.styleSettings )
		end
	end

	function Factory:Create( bar )
	    return Style:Create( bar, self.metaStyle )
	end

	for k, v in pairs( metaStyles ) do
	    local factory = Factory:New( k, v )

	    UHB:RegisterStyle( factory )
	end
end
