local UH = UnderHood
local UHB = UnderHood_Bars
local L = LibStub( "AceLocale-3.0" ):GetLocale( "UnderHood" )

local metaStyles = {
	VerticalLarge = {
		title = L["Vertical (Large)"],
	    config = {
	        width = 30,
	        height = 256,
		},
		styleSettings = {
			showBackground = true,
			bgColor = { 0.3, 0.3, 0.3, 0.75 },
		},
		zones = {
		    {
		        title = L["Main bar"],
		        element = "statusbar",
		        defaultValue = 1,
			},
			{
				title = L["Outline bar"],
				element = "castingbar",
				defaultValue = 0,
			},
		},
	    elements = {
			background = {
			    texture = "Interface\\Addons\\UnderHood\\Media\\Bar-Large",
			    layer = "BACKGROUND",
			    blend = "BLEND",
			    points = true,
			    color = { 0.3, 0.3, 0.3, 0.75 },
			},
			border = {
			    texture = "Interface\\Addons\\UnderHood\\Media\\Border-Large",
			    points = true,
			    blend = "BLEND",
				layer = "BORDER",
			},
			statusbar = {
			    texture = "Interface\\Addons\\UnderHood\\Media\\Bar-Large",
			    points = {
					{ "BOTTOM", "BOTTOM", 0, 3 },
					{ "LEFT", "LEFT", 0, 0 },
					{ "RIGHT", "RIGHT", 0, 0 },
				},
			    blend = "BLEND",
				gaps = { 3, 3, 0, 0 },
				layer = "ARTWORK",
			},
			castingbar = {
			    texture = "Interface\\Addons\\UnderHood\\Media\\CastingBar-Large",
			    points = {
					{ "BOTTOM", "BOTTOM", 0, 1 },
					{ "LEFT", "LEFT", 0, 0 },
					{ "RIGHT", "RIGHT", 0, 0 },
				},
			    blend = "BLEND",
				gaps = { 1, 1, 0, 0 },
				layer = "ARTWORK",
			},
		},
	},
}

local Factory
local Style
do
	Style = {}
	Style.__index = Style
	
	function Style:IsCastingSupported()
	    return self.bar.frames.castingbar
	end
	
	function Style:AdjustBar( zone, p )
	    local zd = self.zones[zone]
	    local tx = zd.texture
	    local data = self.meta.elements[zd.element]
		local gapTop, gapBottom, gapLeft, gapRight = 0, 0, 0, 0
		local bc = self.bar.config
		local bw, bh = bc.width, bc.height
		
		if data.gaps then
		    gapTop, gapBottom, gapLeft, gapRight = unpack( data.gaps )
		end
		
		local gapTopP = gapTop / bh
		local gapBottomP = gapBottom / bh
		local h = (bh - gapTop - gapBottom) * p
		local top = 1 - (p - gapTopP)
		local bottom = 1 - gapBottomP
		
		top = top - (gapTopP + gapBottomP) * (1 - p)
		
		tx:SetHeight( h )
		tx:SetTexCoord( 0, 1, top, bottom )
		
		if math.floor( p * 100 ) == 0 then
			tx:Hide()
		else
			tx:Show()
		end
	end
	
	function Style:CreateMetaElement( element )
	    local data = self.meta.elements[element]
	    local texture = UH:CreateFrame( "Texture", self.bar.frame, data.layer )
	    
	    texture:SetTexture( data.texture )
	    
	    if data.color then
	        local r, g, b, a = unpack( data.color )
	        
	        texture:SetVertexColor( r, g, b, a )
	    end
	    
	    if data.blend then
	        texture:SetBlendMode( data.blend )
		end
		
	    if data.points then
	        if type( data.points ) == "table" then
	            for _, point in ipairs( data.points ) do
	                local pt, rpt, x, y = unpack( point )
	                
	                texture:SetPoint( pt, self.bar.frame, rpt, x, y )
	            end
	        else
	            texture:SetAllPoints( self.bar.frame )
	        end
	    end
	    
	    if not( data.hidden ) then
	        texture:Show()
		end
		
	    self.bar.frames[element] = texture
	    
	    return texture
	end
	
	function Style:GetZones()
	    return self.meta.zones or {}
	end
	
	function Style:Create( bar, meta )
		local self = {}
		setmetatable( self, Style )
		
		self.bar = bar
		self.meta = meta
		
		if not bar.frames then bar.frames = {} end

		for element, _ in pairs( meta.elements ) do
		    self:CreateMetaElement( element )
		end
		
		if bar.frames.background then
			if bar.config.styleSettings.showBackground then
				bar.frames.background:Show()
			else
				bar.frames.background:Hide()
			end
		end
		
		self.zones = {}
		
		for z, zm in ipairs( meta.zones ) do
			local zd = {}
			
			zd.config = zm
			zd.element = zm.element
			zd.texture = bar.frames[zm.element]
			zd.value = zm.defaultValue or 0
			zd.color = { r = 0, g = 0, b = 0, a = 0 }
			
			self.zones[z] = zd
			
			self:AdjustBar( z, zd.value )
		end
		
		return self
	end
	
	function Style:Destroy( bar )
	    self.zones = nil
	end
	
	function Style:SetColor( zone, r, g, b, a )
	    local zd = self.zones[zone]
	    
	    if not zd then return end
	    
	    zd.color.r = r
	    zd.color.g = g
	    zd.color.b = b
	    zd.color.a = a
	    
	    local tx = zd.texture
	    
	    if tx then
	        tx:SetVertexColor( r, g, b, a )
		end
	end
	
	function Style:SetValue( zone, value )
	    local zd = self.zones[zone]

	    if not zd then return end
	    
	    if value < 0 then value = 0 end
	    if value > 1 then value = 1 end
	    
	    zd.value = value
	    
	    self:AdjustBar( zone, self.bar:InConfigMode() and zd.config.defaultValue or value )
	end
	
	function Style:Update()
	    for z, zd in ipairs( self.zones ) do
		    self:AdjustBar( z, self.bar:InConfigMode() and zd.config.defaultValue or zd.value )
	    end
	end
	
	function Style:SetupOptions( options )
		options.showBackground = {
			name = L["Show background"],
			type = "toggle",
			get = function() return self.bar.config.styleSettings.showBackground end,
			set = function( info, value )
				self.bar.config.styleSettings.showBackground = value
				
				if self.bar.frames and self.bar.frames.background then
					if value then
						self.bar.frames.background:Show()
					else
						self.bar.frames.background:Hide()
					end
				end
			end,
		}
	end

	Factory = {}
	Factory.__index = Factory

	function Factory:New( metaName, metaStyle )
	    local self = {}
	    setmetatable( self, Factory )
	    
		self.metaName = metaName
		self.metaStyle = metaStyle or metaStyles[metaName]
		self.name = metaName
		self.title = metaStyle.title or metaName
		
		return self
	end

	local function copyTable( src, dest )
		if type( dest ) ~= "table" then dest = {} end
		if type( src ) == "table" then
			for k, v in pairs( src ) do
				if type( v ) == "table" then
					-- try to index the key first so that the metatable creates the defaults, if set, and use that table
					v = copyTable( v, dest[k] )
				end
				
				dest[k] = v
			end
		end
		
		return dest
	end

	function Factory:DefaultSettings( config )
	    local metaConfig = self.metaStyle.config
	    
	    if metaConfig then
	        if metaConfig.width then config.width = metaConfig.width end
	        if metaConfig.height then config.height = metaConfig.height end
		end

		if type( self.metaStyle.styleSettings ) == "table" then
		    config.styleSettings = copyTable( self.metaStyle.styleSettings )
		end
	end

	function Factory:Create( bar )
	    return Style:Create( bar, self.metaStyle )
	end
	
	for k, v in pairs( metaStyles ) do
	    local factory = Factory:New( k, v )
	    
	    UHB:RegisterStyle( factory )
	end
end
