local L = LibStub( "AceLocale-3.0" ):GetLocale( "UnderHood" )
local UH = UnderHood
local LibDruidMana = LibStub( "LibDruidMana-1.0", true )
local math_abs, math_cos, math_pi, math_min, math_max = math.abs, math.cos, math.pi, math.min, math.max
local UnitMana, UnitManaMax = UnitMana, UnitManaMax
local UnitIsDeadOrGhost = UnitIsDeadOrGhost
local UnitPowerType = UnitPowerType
local UnitCanAttack = UnitCanAttack
local GetTime = GetTime

local unitClass = select( 2, UnitClass( "player" ) )
local energyTickLength, fiveSecondLength = 2, 5

local instances = nil


if LibDruidMana then
	instances = setmetatable( {}, { __mode='k' } )

	local function notifyProviders( current, max )
		for provider, _ in pairs( instances ) do
		    provider:UpdateDruidMana( current, max )
		end
	end
	
	LibDruidMana:AddListener( notifyProviders )
end

local Provider = {}
Provider.__index = Provider
Provider.name = "Ticker"
Provider.title = L["Energy/Mana Ticker"]
Provider.supportedUnits = { "player" }
Provider.realtime = true

function Provider:Create( bar, zone, config )
	local self = {}

	setmetatable( self, Provider )
	LibStub( "AceEvent-3.0" ):Embed( self )

	self.bar = bar
	self.zone = zone
	self.config = config
	
	return self
end

function Provider:UnitSupported( unit )
	return unit == "player"
end

--
-- Called from OnUpdate of bar
--
-- arguments: currentTime = time when OnUpdate started
-- return values: value, r, g, b, a
--
function Provider:GetRealtimeValueAndColor( currentTime )
	local powerType = self.powerType
	local value

	if not self.active or (powerType ~= 3 and powerType ~= 0) then return end
	
	if self.hideRequest and self.hideRequest > 1 then
		self.hideRequest = nil
		self.active = nil
		
		return 0
	end
	
	local timeDiff

	if powerType == 0 then
		timeDiff = currentTime - self.spellcastFinishTime
		
		if timeDiff > fiveSecondLength then
			self.hideRequest = 2
		else
		    value = timeDiff / fiveSecondLength
		end
	else
	    timeDiff = currentTime - self.energyLastTick
	    
	    if timeDiff > energyTickLength then
	        self.energyLastTick = self.energyLastTick + energyTickLength
	    end

	    value = timeDiff / energyTickLength
	    
	    if self.hideRequest and value >= 0.99 then
	        self.hideRequest = 2 -- Will be hidden next time OnUpdate is called
		end
	end

	return value
end

function Provider:Hide()
	if self.active and not self.hideRequest then
		self.hideRequest = 1
	end
end

function Provider:Show()
	self.active = true
	self.hideRequest = nil
end

function Provider:Connect()
	self.inCombat = InCombatLockdown()
	self.powerType = UnitPowerType( "player" )

	if unitClass == "ROGUE" or unitClass == "DRUID" then
		self.currentEnergy = 0
		
		self:RegisterEvent( "UNIT_ENERGY" )
		self:RegisterEvent( "UNIT_FACTION" )
		self:RegisterEvent( "PLAYER_TARGET_CHANGED" )
		self:RegisterEvent( "PLAYER_DEAD", "PLAYER_TARGET_CHANGED" )
		self:RegisterEvent( "PLAYER_ALIVE", "PLAYER_TARGET_CHANGED" )
		
		if unitClass == "DRUID" then
		    self:RegisterEvent( "UNIT_DISPLAYPOWER")
		    
		    if instances then
				instances[self] = true
			end
		end
	end
	
	if unitClass ~= "ROGUE" then
	    self.currentMana = UnitMana( "player" )
	    
		self:RegisterEvent( "UNIT_SPELLCAST_SUCCEEDED" )
		
		if not instances then
		    self:RegisterEvent( "UNIT_MANA" )
		end
	end

	self:UpdateColor()
	
	self.energyLastTick = 0
	self.spellcastFinishTime = 0
	self.lastSpellcast = 0
end

function Provider:Disconnect()
	self:UnregisterAllEvents()
	
	if instances then
		instances[self] = nil
	end
end

function Provider:UpdateColor()
	local r, g, b, a
	
	if self.powerType == 3 then
	    r, g, b = unpack( UH.db.profile.colors.power.energy )
	    a = 1
	elseif self.powerType == 0 then
	    r, g, b = unpack( UH.db.profile.colors.power.mana )
	    a = 1
	else
	    r = 0
	    g = 0
	    b = 0
	    a = 0
	end
	
	self.bar:SetColor( self.zone, r, g, b, a )
end

function Provider:OnEnterCombat()
	self.inCombat = true
	
	if self.powerType == 3 then
	    self:Show()
	end
end

function Provider:OnLeaveCombat()
	self.inCombat = false
	
	if self.powerType == 3 and UnitIsDeadOrGhost( "player" ) or (UnitMana( "player" ) == UnitManaMax( "player" ) and (not UnitCanAttack( "player", "target" ) or UnitIsDeadOrGhost( "target" ))) then
	    self:Hide()
	end
end

function Provider:PLAYER_TARGET_CHANGED()
	if self.powerType ~= 3 then return end
	
	if UnitIsDeadOrGhost( "player" ) or (UnitMana( "player" ) == UnitManaMax( "player" ) and (not UnitCanAttack( "player", "target" ) or UnitIsDeadOrGhost( "target" )) and not self.inCombat) then
	    self:Hide()
	else
	    self:Show()
	end
end

function Provider:UNIT_FACTION( event, unit )
	if (unit ~= "player" and unit ~= "target") or self.powerType ~= 3 then
	    return
	end
	
	self:PLAYER_TARGET_CHANGED()
end

function Provider:UNIT_ENERGY( event, unit )
	if unit ~= "player" or self.powerType ~= 3 then
	    return
	end
	
	local energy = UnitMana( "player" )
	
	if (self.currentEnergy + 20) == energy then
	    self.energyLastTick = GetTime()
	end

	if energy > self.currentEnergy then
	    
		if UnitIsDeadOrGhost( "player" ) or (energy == UnitManaMax( "player" ) and (not UnitCanAttack( "player", "target" ) or UnitIsDeadOrGhost( "target" )) and not self.inCombat) then
		    self:Hide()
		else
		    self:Show()
		end
	end
	
	self.currentEnergy = energy
end

function Provider:UNIT_MANA( event, unit )
	if unit ~= "player" or self.powerType ~= 0 then
	    return
	end
	
	local mana = UnitMana( "player" )
	
	if mana == UnitManaMax( "player" ) then
	    self:Hide()
	else
	    self:Show()
	    self.spellcastFinishTime = self.lastSpellcast
	end
	
	self.currentMana = mana
end

function Provider:UNIT_SPELLCAST_SUCCEEDED( event, unit )
	if unit == "player" then
	    self.lastSpellcast = GetTime()
	end
end

function Provider:UpdateDruidMana( current, maximum )
	if current == maximum then
	    if self.powerType == 0 then
		    self:Hide()
		end
	elseif current < self.currentMana and GetTime() - self.lastSpellcast < 0.25 then
	    if self.powerType == 0 then
		    self:Show()
		end
		
	    self.spellcastFinishTime = self.lastSpellcast
	end
	
	self.currentMana = current
end

function Provider:UNIT_DISPLAYPOWER( event, unit )
	if unit ~= "player" then return end
	
	self.powerType = UnitPowerType( "player" )

	self:UpdateColor()

	if self.powerType == 3 then
	    self.energyLastTick = GetTime()
	    self.currentEnergy = UnitMana( "player" )
	    self:Show()
	elseif self.powerType == 0 then
	    self.currentMana = UnitMana( "player" )
	    self:Show()
	else
	    self:Hide()
	end
end

UnderHood_Bars:RegisterProvider( Provider )
