local L = LibStub( "AceLocale-3.0" ):GetLocale( "UnderHood" )
local UH = UnderHood
local AceEvent = LibStub( "AceEvent-3.0" )
local AceTimer = LibStub( "AceTimer-3.0" )
local Threat = LibStub( "Threat-2.0" )
local _G = _G
local UnitExists = UnitExists
local UnitIsPlayer = UnitIsPlayer
local UnitPlayerControlled = UnitPlayerControlled
local UnitCanAttack = UnitCanAttack
local UnitName = UnitName
local UnitCastingInfo = UnitCastingInfo
local UnitGUID = UnitGUID

local Provider = {}
Provider.__index = Provider
Provider.name = "Threat"
Provider.title = L["Threat"]
Provider.manageVisibility = true

function Provider:Create( bar, zone, config )
	local self = {}

	setmetatable( self, Provider )
	AceEvent:Embed( self )
	AceTimer:Embed( self )

	self.bar = bar
	self.zone = zone
	self.config = config

	return self
end

function Provider:UnitSupported( unit )
	return UH.UnitSendEvents[unit]
end

function Provider:IsBarVisible()
	return self.unitGUID and self.hostileGUID and self.inCombat or false
end

function Provider:OnEnterCombat()
	self.inCombat = true
	
	self.bar:Show( self.zone )
	
	if self.unitGUID and self.hostileUnit then
		self:SetValue()
	else
	    self.bar:SetValue( self.zone, 0 )
	end
end

function Provider:OnLeaveCombat()
	self.inCombat = nil
	self.bar:SetValue( self.zone, 0 )
	self.bar:Hide( self.zone )
end

function Provider:Connect()
	local unit = self.bar.config.unit
	
	self.unit = unit
	self.unitGUID = UnitGUID( unit )
	self.lastTargetSwitchTime = 0
	self.selfThreat = 0
	self.isPet = nil

	local po1, po2 = unit:match( "^(.*)pet(%d*)$" )
	
	if po1 then
	    self.isPet = true
	    
	    if po1 == "" and po2 == "" then
	        self.petOwner = "player"
		elseif po1 == "party" then
		    self.petOwner = po1..po2
		end
	end
	
	Threat.RegisterCallback( self, "ThreatUpdated", "ThreatUpdated" )
	
    self:RegisterEvent( "UNIT_TARGET" )
    
    if unit == "target" then
		self:RegisterEvent( "PLAYER_TARGET_CHANGED", "UpdateUnitGUID" )
	elseif unit == "focus" then
		self:RegisterEvent( "PLAYER_FOCUS_CHANGED", "UpdateUnitGUID" )
	elseif self.isPet then
	    self:RegisterEvent( "UNIT_PET" )
	else
	    UH:RegisterUpdateTarget( self )
	end

	if self.unitGUID then
		self:UNIT_TARGET( nil, unit )
	end
end

function Provider:Disconnect()
	UH:UnregisterUpdateTarget( self )
	self:UnregisterAllEvents()
	Threat.UnregisterCallback( self, "ThreatUpdated" );
end

function Provider:UNIT_PET( event, unit )
	if unit == self.petOwner then
		self:UpdateUnitGUID( event, unit )
	end
end

function Provider:UpdateUnitGUID( event )
	self.unitGUID = UnitGUID( self.unit )
	
	if self.unitGUID then
		self:UNIT_TARGET( event, self.unit )
	else
	    self.bar:SetValue( self.zone, 0 )
	end
end

function Provider:ProcessUpdate()
	local unitGUID = UnitGUID( self.unit )
	
	if unitGUID ~= self.unitGUID then
	    self.unitGUID = unitGUID
	    
		if unitGUID then
	    	self:UNIT_TARGET( nil, self.unit )
		else
		    self.bar:SetValue( self.zone, 0 )
		end
--[[
	elseif self.aggroUnit then
		local aggroGUID = UnitGUID( self.aggroUnit )
		
		if aggroGUID ~= self.aggroGUID then
		    self.aggroGUID = aggroGUID
			self:SetValue()
		end
--]]
	end
end

function Provider:ThreatUpdated( event, srcGUID, dstGUID, threat )
	if not( self.hostileUnit and self.unitGUID ) then return end
	
	if self.hostileGUID and dstGUID == self.hostileGUID then
		self.aggroThreat = self.aggroGUID and ((srcGUID == self.aggroGUID) and threat or Threat:GetThreat( self.aggroGUID, dstGUID )) or Threat:GetMaxThreatOnTarget( dstGUID ) or 1
		self.selfThreat = (srcGUID == self.unitGUID) and threat or Threat:GetThreat( self.unitGUID, dstGUID ) or 0

		self:SetValue()
	end
end

local function isValidThreatTarget( unit )
	if not UnitExists( unit ) or UnitIsPlayer( unit ) or UnitPlayerControlled( unit ) then
		return false
	end
	return true
end

function Provider:TargetTargetCheck()
	if not self.timerTargetSwitch and UnitGUID( self.aggroUnit ) ~= self.aggroGUID then
		--UH:Debug( "Target's target changed" )
		self:UNIT_TARGET( nil, self.hostileUnit )
	end
end

function Provider:SwitchAggroTarget()
	local n = self.aggroUnit and UnitName( self.aggroUnit )
	
	if n then
		self.aggroGUID = UnitGUID( self.aggroUnit )
		--[[
		UH:Debug( "SwitchAggroTarget: H: %q / A: %q", self.hostileGUID, self.aggroGUID or "nil" )
		if self.aggroGUID == self.unitGUID then UH:Debug( "YOU HAVE AGGRO" )
		else UH:Debug( "YOU LOST AGGRO" ) end
		--]]
		
		self.lastTargetSwitchTime = GetTime()
	end
	
	self.timerTargetSwitch = nil
end

--[[
	[Unit (unit)] --> [Enemy unit (hostile unit)] --> [Friendly Unit (aggro unit)]
	[Unit (unit)] --> [Friendly unit] --> [Enemy Unit (hostile unit)] --> [Friendly Unit (aggro unit)]
--]]
function Provider:UNIT_TARGET( event, unit )
	local target = self.unit == "player" and "target" or self.unit.."target"
	-- We are only intereseted in unit's target

	--UH:Debug( "UNIT_TARGET: %q", unit )
	
	if unit == self.unit then
	    -- Watching unit has chagned target
	    
		self.hostileUnit = nil
		self.hostileGUID = nil
		self.aggroUnit = nil
		self.aggroGUID = nil

		if self.timerTargetTargetCheck then
		    self:CancelTimer( self.timerTargetTargetCheck, true )
		    self.timerTargetTargetCheck = nil
		end
		
		if isValidThreatTarget( target ) then
		    self.hostileUnit = target
		    self.aggroUnit = target.."target"
		elseif isValidThreatTarget( target.."target" ) then
		    self.hostileUnit = target.."target"
		    self.aggroUnit = target.."targettarget"
		end

		if not self.hostileUnit then
		    self.bar:SetValue( self.zone, 0 )
		    return
		end

		if not UH.UnitSendEvents[self.hostileUnit] then
		    self.timerTargetTargetCheck = self:ScheduleRepeatingTimer( "TargetTargetCheck", 0.5 )
		end

		self.hostileGUID = UnitGUID( self.hostileUnit )
		self.aggroGUID = UnitGUID( self.aggroUnit )
		--UH:Debug( "UNIT_TARGET: H: %q / A: %q", self.hostileGUID, self.aggroGUID or "nil" )
		
		self.aggroThreat = Threat:GetThreat( self.aggroGUID, self.hostileGUID ) or 1
		self.selfThreat = Threat:GetThreat( self.unitGUID, self.hostileGUID ) or 0
		
		self:SetValue()
		
		unit = target -- to pass next check
	end

	if unit == target then
		if	(not self.hostileUnit and isValidThreatTarget( target.."target")) or
			(self.hostileUnit == target.."target" and self.hostileGUID ~= UnitGUID( self.hostileUnit )) then
			self:UNIT_TARGET( event, self.unit ) -- Recurse back?
			return
		end
		
		if not self.hostileUnit then return end
		
		if self.timerTargetSwitch then
		    self:CancelTimer( self.timerTargetSwitch, true )
		    self.timerTargetSwitch = nil
		end
		
		if not UnitCastingInfo( self.hostileUnit ) or not self.aggroGUID then
			--UH:Debug( "Starting check aggro timer" )
		    self.timerTargetSwitch = self:ScheduleTimer( "SwitchAggroTarget", 0.5 )
		end
	end
end

function Provider:SetValue()
	local ps = self.config.providerSettings
	local adaptive = ps and ps.adaptive
	local wt = ps and ps.warningThreshold or 0.9
	local colorMode = ps and ps.colorMode or "static"
	local tankMode = ps and ps.tankMode
    local inMelee = Threat:UnitInMeleeRange( self.hostileUnit )
    local pullThreatAt = inMelee and 1.1 or 1.3
    local aggroGUID = self.aggroUnit and UnitGUID( self.aggroUnit )
    local unitHasAggro = self.unitGUID == aggroGUID
    local max = (adaptive and not tankMode) and pullThreatAt or 1.0

    if self.testRange ~= max then
    	self.testRange = max
    	
    	UH:Debug( "Threat scale is "..self.testRange )
    end
    
	--if self.aggroGUID ~= aggroGUID then UH:Debug( "AGGRO UNIT DETECTION ERROR: "..self.unit ) end
	
	local threat, maxThreat = self.selfThreat, tankMode and Threat:GetMaxThreatOnTarget( self.hostileGUID ) or (self.aggroThreat or 0)
	
	if maxThreat == 0 then threat = 0
	else threat = threat/maxThreat end

	local nThreat = threat/max
	local ri
   	
	if colorMode ~= "static" then
	    if tankMode then
	        if not unitHasAggro then
	            ri = "pull"
	        else
			    if threat < wt then ri = "warning"
			    else ri = "normal" end
	        end
	    else
	        if unitHasAggro then
	            ri = "pull"
			else
			    if threat < wt then ri = "normal"
			    elseif threat < pullThreatAt then ri = "warning"
			    else ri = "pull" end
			end
		end
	else
	    ri = "pull"
	end
	
	local r, g, b = unpack( UH.db.profile.colors.threat[ri] )

	self.bar:SetColor( self.zone, r, g, b, 1 )
	self.bar:SetValue( self.zone, nThreat )
end

--[[
local maximums = {
	[1.0] = L["Normal (100%)"],
	[1.1] = L["Melee (110%)"],
	[1.3] = L["Ranged (130%)"],
	[1.5] = L["Tank (150%)"],
}
--]]

local colors = {
	static = L["Static threat color"],
	amount = L["Color by threat amount"],
	--deficit = L["Color by threat deficit"],
}

function Provider:SetupOptions( options )
	options.adaptive = {
	    name = L["Adaptive scaling"],
	    desc = L["Scale bar according to range to the target"],
	    order = 100,
	    type = "toggle",
	    get = function() return self.config.providerSettings and self.config.providerSettings.adaptive end,
	    set = function( info, value )
	        if not self.config.providerSettings then self.config.providerSettings = {} end
	        
	        self.config.providerSettings.adaptive = value
	        self:SetValue()
		end,
	}

	options.tankMode = {
	    name = L["Tank mode"],
	    desc = L["Scale threat to maximum threat on target"],
	    order = 101,
	    type = "toggle",
	    get = function() return self.config.providerSettings and self.config.providerSettings.tankMode end,
	    set = function( info, value )
	        if not self.config.providerSettings then self.config.providerSettings = {} end

	        self.config.providerSettings.tankMode = value
	        self:SetValue()
		end,
	}

	options.warningThreshold = {
		name = L["Warning threshold"],
		type = "range",
		order = 102,
		min = 60,
		max = 130,
		step = 1,
		bigStep = 5,
		get = function() return (self.config.providerSettings and self.config.providerSettings.warningThreshold or 0.9)*100 end,
		set = function( info, value )
	        if not self.config.providerSettings then self.config.providerSettings = {} end

	        self.config.providerSettings.warningThreshold = value/100
	        self:SetValue()
		end,
	}

	options.colorMode = {
		name = L["Color"],
		type = "select",
		order = 103,
		values = colors,
		get = function() return self.config.providerSettings and self.config.providerSettings.colorMode or "static" end,
		set = function( info, value )
			if not self.config.providerSettings then self.config.providerSettings = {} end

			self.config.providerSettings.colorMode = value
			self:SetValue()
		end,
	}
end

UnderHood_Bars:RegisterProvider( Provider )
