local UH = UnderHood
local _G = _G
local UnitHealth = UnitHealth
local UnitHealthMax = UnitHealthMax
local UnitClass = UnitClass
local UnitExists = UnitExists
local UnitIsPlayer = UnitIsPlayer
local UnitPlayerControlled = UnitPlayerControlled
local UnitCanAttack = UnitCanAttack
local UnitIsPVP = UnitIsPVP
local UnitIsTapped = UnitIsTapped
local UnitIsTappedByPlayer = UnitIsTappedByPlayer
local UnitIsDead = UnitIsDead
local UnitReaction = UnitReaction
local UnitGUID = UnitGUID

local L = LibStub( "AceLocale-3.0" ):GetLocale( "UnderHood" )

local Provider = {}
Provider.__index = Provider
Provider.name = "Health"
Provider.title = L["Health"]
Provider.supportedUnits = { "player", "target", "focus", "pet", "targettarget" }

function Provider:Create( bar, zone, config )
	local self = {}

	setmetatable( self, Provider )
	LibStub( "AceEvent-3.0" ):Embed( self )

	self.bar = bar
	self.zone = zone
	self.config = config

	return self
end

function Provider:UnitSupported( unit )
	return true
end

function Provider:Connect()
	self.unit = self.bar.config.unit

	local unit = self.unit

	self:Update()

	if UH.UnitSendEvents[unit] then
		self:RegisterEvent( "UNIT_HEALTH" )
		self:RegisterEvent( "UNIT_MAXHEALTH" )
		self:RegisterEvent( "UNIT_FACTION" )
	else
		UH:RegisterUpdateTarget( self, "UpdateUneventedUnit" )
	end

	if unit == "player" then
		self:RegisterEvent( "PLAYER_ENTERING_WORLD", "Update" )
	elseif unit == "target" then
		self:RegisterEvent( "PLAYER_TARGET_CHANGED", "Update" )
	elseif unit == "focus" then
		self:RegisterEvent( "PLAYER_FOCUS_CHANGED", "Update" )
	elseif unit == "pet" or unit:find( "^partypet" ) or unit:find( "^raidpet" ) then
		self:RegisterEvent( "UNIT_PET", "UpdateIfUnit" )
		self:RegisterEvent( "UNIT_HAPPINESS", "UpdateIfUnit" )
	end
end

function Provider:Disconnect()
	self:UnregisterAllEvents()
	UH:UnregisterUpdateTarget( self )
end

function Provider:UpdateUneventedUnit()
	local unit = self.unit
	local totGuid = UnitGUID( unit )

	if self.totGuid ~= totGuid then
		-- Current ToT unit is not the same as last one = updating
		self.totGuid = totGuid
		self.totHealth = nil
		self:Update()
	elseif self.totGuid then
		local current, max = UnitHealth( unit ), UnitHealthMax( unit )

		current = max > 0 and current / max or 0

		if self.totHealth ~= current then
			self.totHealth = current
			self.bar:SetValue( self.zone, current )
		end
	end
end

function Provider:UNIT_HEALTH( event, unit )
	if unit == self.unit then
		self:SetValue()
	end
end

function Provider:UNIT_MAXHEALTH( event, unit )
	if unit == self.unit then
		self:SetValue()
	end
end

function Provider:UNIT_FACTION( event, unit )
	if unit == self.unit then
		self:Update()
	end
end

local happinessIndex = { "angry", "neutral", "happy" }

function Provider:UpdateIfUnit( event, unit )
	if unit == self.unit then
		self:Update()
	end
end

function Provider:Update()
	local r, g, b = unpack( UH.db.profile.colors.health.max )
	local unit = self.unit
	local _, class = UnitClass( unit )
	local rc = _G.RAID_CLASS_COLORS[class]
	local ps = self.config.providerSettings

	if not UnitExists( unit ) then return end

	if ps and ps.colorByHostility then
		if UnitIsPlayer( unit ) or UnitPlayerControlled( unit ) then
			if UnitCanAttack( unit, "player") then
				-- they can attack me
				if UnitCanAttack( "player", unit ) then
					-- and I can attack them
					r, g, b = unpack( UH.db.profile.colors.unit.hostile )
				else
					-- but I can't attack them
					r, g, b = unpack( UH.db.profile.colors.unit.civilian )
				end
			elseif UnitCanAttack( "player", unit ) then
				-- they can't attack me, but I can attack them
				r, g, b = unpack( UH.db.profile.colors.unit.neutral )
			elseif UnitIsPVP( unit ) then
				-- on my team
				if ps.colorByClass and rc then
					r, g, b = rc.r, rc.g, rc.b
				else
					r, g, b = unpack( UH.db.profile.colors.unit.friendly )
				end
			else
				-- either enemy or friend, no violence
				r, g, b = unpack( UH.db.profile.colors.unit.civilian )
			end
		elseif (UnitIsTapped( unit ) and not UnitIsTappedByPlayer( unit )) or UnitIsDead( unit ) then
			r, g, b = unpack( UH.db.profile.colors.unit.tapped )
		else
			local reaction = UnitReaction( unit, "player" )
			if reaction then
				if reaction >= 5 then
					if ps.colorByClass and rc then
						r, g, b = rc.r, rc.g, rc.b
					else
						r, g, b = unpack( UH.db.profile.colors.unit.friendly )
					end
				elseif reaction == 4 then
					r, g, b = unpack( UH.db.profile.colors.unit.neutral )
				else
					r, g, b = unpack( UH.db.profile.colors.unit.hostile )
				end
			else
				r, g, b = unpack( UH.db.profile.colors.unit.unknown )
			end
		end
	elseif ps and ps.colorByClass then
		if rc then
			r, g, b = rc.r, rc.g, rc.b
		else
			r, g, b = unpack( UH.db.profile.colors.unit.unknown )
		end
	elseif ps and ps.colorByHappiness then
		happiness = GetPetHappiness()

		if happiness then
			r, g, b = unpack( UH.db.profile.colors.pet[happinessIndex[happiness]] )
		else
			r, g, b = unpack( UH.db.profile.colors.unit.unknown )
		end
	end

	self.red = r
	self.green = g
	self.blue = b

	--self.bar:SetColor( self.zone, r, g, b, 1 )

	self:SetValue()
end

function Provider:SetValue()
	local unit = self.unit
	local current, max = UnitHealth( unit ), UnitHealthMax( unit )

	if max > 0 then
		current = current / max
	else
		current = 0
	end

	self.bar:SetValue( self.zone, current, self.red, self.green, self.blue, 1 )
end

function Provider:DefaultSettings( config )
	config.providerSettings = {}
end

function Provider:SetupOptions( options )
	options.colorByHostility = {
		name = L["Color by hostility"],
		type = "toggle",
		get = function() return self.config.providerSettings and self.config.providerSettings.colorByHostility end,
		set = function( info, value )
			if not self.config.providerSettings then
				self.config.providerSettings = {}
			end

			self.config.providerSettings.colorByHostility = value
			self:Update()
		end,
	}

	options.colorByClass = {
		name = L["Color by class"],
		type = "toggle",
		get = function() return self.config.providerSettings and self.config.providerSettings.colorByClass end,
		set = function( info, value )
			if not self.config.providerSettings then
				self.config.providerSettings = {}
			end

			self.config.providerSettings.colorByClass = value
			self:Update()
		end,
	}

	options.colorByHappiness = {
		name = L["Color by happiness"],
		type = "toggle",
		get = function() return self.config.providerSettings and self.config.providerSettings.colorByHappiness end,
		set = function( info, value )
			if not self.config.providerSettings then
				self.config.providerSettings = {}
			end

			self.config.providerSettings.colorByHappiness = value
			self:Update()
		end,
	}
end

UnderHood_Bars:RegisterProvider( Provider )
