local L = LibStub( "AceLocale-3.0" ):GetLocale( "UnderHood" )
local UH = UnderHood
local UnitCastingInfo = UnitCastingInfo
local UnitChannelInfo = UnitChannelInfo
local UnitGUID = UnitGUID
local math_abs, math_cos, math_pi, math_min, math_max = math.abs, math.cos, math.pi, math.min, math.max

local Provider = {}
Provider.__index = Provider
Provider.name = "Casting"
Provider.title = L["Casting"]
Provider.supportedUnits = { "player", "target", "focus", "pet" }
Provider.realtime = true

function Provider:Create( bar, zone, config )
	local self = {}

	setmetatable( self, Provider )
	LibStub( "AceEvent-3.0" ):Embed( self )

	self.bar = bar
	self.zone = zone
	self.config = config
	return self
end

function Provider:UnitSupported( unit )
	for k, v in ipairs( self.supportedUnits ) do
	    if unit == v then return true end
	end
end

--
-- Called from OnUpdate of bar
--
-- arguments: currentTime = time when OnUpdate started
-- return values: value, r, g, b, a
--
function Provider:GetRealtimeValueAndColor( currentTime )
	local value, r, g, b, a
	
	if self.casting then
	    if currentTime > self.castEndTime then
	        self.casting = nil
	        self.fadeOut = true
	        self.stopTime = currentTime
		end

		local showTime = math_min( currentTime, self.castEndTime )

		value = (showTime - self.castStartTime) / (self.castEndTime - self.castStartTime)
	elseif self.channeling then
	    if currentTime > self.castEndTime then
	        self.channeling = nil
	        self.fadeOut = true
	        self.stopTime = currentTime
		end

		local remainingTime = self.castEndTime - currentTime

		value = remainingTime / (self.castEndTime - self.castStartTime)
	elseif self.fadeOut then
	    if self.stopTime then
	        a = self.stopTime - currentTime + 1
	    else
			a = 0
	    end

	    if a >= 1 then
	        a = 1
		end

		if a <= 0 then
		    self.fadeOut = nil
		    self.stopTime = nil
		    value = 0
		    r = 0
		    g = 0
		    b = 0
		else
		    r, g, b = unpack( self.castingColor )
		    value = 1
		end
	else
	    value = 0
	end
	
	return value, r, g, b, a
end

function Provider:Connect()
	self.unit = self.bar.config.unit
	local unit = self.unit

	self.casting = nil
	self.channeling = nil
	self.fadeOut = nil
	
    self:RegisterEvent( "UNIT_SPELLCAST_START" )
    self:RegisterEvent( "UNIT_SPELLCAST_STOP" )
    self:RegisterEvent( "UNIT_SPELLCAST_FAILED" )
    self:RegisterEvent( "UNIT_SPELLCAST_DELAYED" )
    self:RegisterEvent( "UNIT_SPELLCAST_INTERRUPTED" )
    self:RegisterEvent( "UNIT_SPELLCAST_CHANNEL_START" )
    self:RegisterEvent( "UNIT_SPELLCAST_CHANNEL_UPDATE" )
    self:RegisterEvent( "UNIT_SPELLCAST_CHANNEL_STOP" )
	self:RegisterEvent( "UNIT_SPELLCAST_CHANNEL_INTERRUPTED", "UNIT_SPELLCAST_INTERRUPTED" )

    if unit == "target" then
        self:RegisterEvent( "PLAYER_TARGET_CHANGED", "CheckAndStartSpellcast" )
	elseif self.config.unit == "focus" then
	    self:RegisterEvent( "PLAYER_FOCUS_CHANGED", "CheckAndStartSpellcast" )
	end
end

function Provider:Disconnect()
	self:UnregisterAllEvents()
end

function Provider:UNIT_SPELLCAST_START( event, unit )
	if unit ~= self.unit then return end

	local spell, rank, displayName, icon, startTime, endTime = UnitCastingInfo( unit )

	if not startTime then
	    self.bar:SetValue( self.zone, 0 )
		return
	end

	self.castStartTime = startTime / 1000
	self.castEndTime = endTime / 1000
	self.castDelay = 0
	self.casting = true
	self.channeling = nil
	self.fadeOut = nil
	self.castingColor = UH.db.profile.colors.cast.casting

	local r, g, b = unpack( self.castingColor )

	self.bar:SetValue( self.zone, 0 )
	self.bar:SetColor( self.zone, r, g, b, 1 )
end

function Provider:UNIT_SPELLCAST_CHANNEL_START( event, unit )
	if unit ~= self.unit then return end

	local spell, rank, displayName, icon, startTime, endTime = UnitChannelInfo( unit )

	if not startTime then
	    self.bar:SetValue( self.zone, 0 )
		return
	end

	self.castStartTime = startTime / 1000
	self.castEndTime = endTime / 1000
	self.castDelay = 0
	self.casting = nil
	self.channeling = true
	self.fadeOut = nil
	self.castingColor = UH.db.profile.colors.cast.channeling

	local r, g, b = unpack( self.castingColor )

	self.bar:SetValue( self.zone, 0 )
	self.bar:SetColor( self.zone, r, g, b, 1 )
end

function Provider:UNIT_SPELLCAST_STOP( event, unit )
	if unit ~= self.unit then return end

	if self.casting then
		self.casting = nil
		self.fadeOut = true
		self.stopTime = GetTime()

		self.bar:SetValue( self.zone, 1 )
	end
end

function Provider:UNIT_SPELLCAST_CHANNEL_STOP( event, unit )
	if unit ~= self.unit then return end

	if self.channeling then
		self.channeling = nil
		self.fadeOut = true
		self.stopTime = GetTime()

		self.bar:SetValue( self.zone, 1 )
	end
end

function Provider:UNIT_SPELLCAST_FAILED( event, unit )
	if unit ~= self.unit or self.channeling then return end

	self.casting = nil
	self.channeling = nil
	self.fadeOut = true

	if not( self.stopTime ) then
	    self.stopTime = GetTime()
	end

	self.castingColor = UH.db.profile.colors.cast.failed

	local r, g, b = unpack( self.castingColor )

	self.bar:SetValue( self.zone, 1 )
	self.bar:SetColor( self.zone, r, g, b, 1 )
end

function Provider:UNIT_SPELLCAST_DELAYED( event, unit )
	if unit ~= self.unit then return end

	local spell, rank, displayName, icon, startTime, endTime = UnitCastingInfo( unit )

	if startTime then
		local oldStartTime = self.castStartTime

	    self.castStartTime = startTime / 1000
	    self.castEndTime = endTime / 1000

	    self.castDelay = (self.castDelay or 0) + (startTime - (oldStartTime or startTime))
	end
end

function Provider:UNIT_SPELLCAST_INTERRUPTED( event, unit )
	if unit ~= self.unit then return end

	self.casting = nil
	self.channeling = nil
	self.fadeOut = true

	if not( self.stopTime ) then
	    self.stopTime = GetTime()
	end

	self.castingColor = UH.db.profile.colors.cast.interrupted

	local r, g, b = unpack( self.castingColor )

	self.bar:SetValue( self.zone, 1 )
	self.bar:SetColor( self.zone, r, g, b, 1 )
end

function Provider:UNIT_SPELLCAST_CHANNEL_UPDATE( event, unit )
	if unit ~= self.unit then return end

	local spell, rank, displayName, icon, startTime, endTime = UnitChannelInfo( unit )

	if startTime then
		local oldStartTime = self.castStartTime

	    self.castStartTime = startTime / 1000
	    self.castEndTime = endTime / 1000

	    self.castDelay = (self.castDelay or 0) + (startTime - (oldStartTime or startTime))
	end
end

function Provider:CheckAndStartSpellcast()
	self.casting = nil
	self.channeling = nil
	self.fadeOut = nil

    self.bar:SetValue( self.zone, 0 )

	if not UnitExists( self.unit ) then return end

	local spell, rank, displayName, icon, startTime, endTime = UnitCastingInfo( self.unit )

	if spell then
	    self.castStartTime = startTime / 1000
	    self.castEndTime = endTime / 1000
	    self.castDelay = 0
	    self.casting = true
	    self.channeling = nil
	    self.fadeOut = nil
	    self.castingColor = UH.db.profile.colors.cast.casting
	else
	    spell, rank, displayName, icon, startTime, endTime = UnitChannelInfo( self.unit )

	    if spell then
		    self.castStartTime = startTime / 1000
		    self.castEndTime = endTime / 1000
		    self.castDelay = 0
		    self.casting = nil
		    self.channeling = true
		    self.fadeOut = nil
		    self.castingColor = UH.db.profile.colors.cast.channeling
	    end
	end

	if self.casting or self.channeling then
		local r, g, b = unpack( self.castingColor )

		self.bar:SetValue( self.zone, 0 )
		self.bar:SetColor( self.zone, r, g, b, 1 )
	end
end

UnderHood_Bars:RegisterProvider( Provider )
