local UH = UnderHood
local UHB = UnderHood_Bars
local SML = LibStub( "LibSharedMedia-3.0" )
local L = LibStub( "AceLocale-3.0" ):GetLocale( "UnderHood" )

local UnitExists = UnitExists

local Bar = {}
Bar.__index = Bar

UHB.FramePrototype = Bar

local DURATION = 0.5
local math_abs, math_cos, math_pi, math_min, math_max = math.abs, math.cos, math.pi, math.min, math.max

local function CosineInterpolate( y1, y2, mu )
	local mu2 = (1 - math_cos( mu * math_pi )) / 2
	return y1 * (1 - mu2) + y2 * mu2
end

function Bar:Create( module, config )
	local self = {}
	setmetatable( self, Bar )

	LibStub( "AceEvent-3.0" ):Embed( self )

	self.module = module
	self.config = config

	self.frame = UH:CreateFrame( "Button", nil, "SecureUnitButtonTemplate" ) -- parented to UnderHoodParentFrame
	self.frame:SetWidth( self.config.width )
	self.frame:SetHeight( self.config.height )
	self.frame:SetScale( self.config.scale )
	self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.config.x, self.config.y )

	self.style = self.module:GetStyle( self.config.style ):Create( self )

	local styleZones = self.style:GetZones()
	local count = math_max( #styleZones, #config.zones )

	self.zones = {}

	for z = 1, count do
		local sz, zm

		if z > #styleZones then
			config.zones[z] = nil
		else
			sz = styleZones[z]

			if z > #config.zones then
				zm = { provider = "" }
				config.zones[z] = zm
			else
				zm = config.zones[z]
			end

			local zd = {}

			zd.config = zm
			zd.currentValue = 0
			zd.targetValue = 0
			zd.color = { r = 0, g = 0, b = 0, a = 0 }

			self.zones[#self.zones+1] = zd

			if #zm.provider > 0 then
				zd.provider = self.module:GetProvider( zm.provider ):Create( self, z, zm )
			end
		end
	end

	self:SetupOptions()

	if self.config.enabled then
		self:Enable()
	else
		self.frame:Hide()
	end

	return self
end

function Bar:Destroy()
	if self.enabled then self:Disable() end

	for z, zd in ipairs( self.zones ) do
		zd.provider = nil
		self.zones[z] = nil
	end

	self.style:Destroy()

	if self.frames then
		for k, v in pairs( self.frames ) do
			UH:ReleaseFrame( v )
		end

		self.frames = nil
	end

	if self.frame then
		self.frame:SetAttribute( "unit", nil )

		self.frame:SetScript( "OnUpdate", nil )

		UnregisterUnitWatch( self.frame )

		UH:ReleaseFrame( self.frame )
	end

	self.frame = nil
end

function Bar:Enable()
	if self.enabled then return end

	self:RegisterEvent( "PLAYER_REGEN_DISABLED" )
	self:RegisterEvent( "PLAYER_REGEN_ENABLED" )

	for z, zd in ipairs( self.zones ) do
		if zd.provider then
			zd.provider:Connect()
		end
	end

	self:WhoManageVisibility()
	self:SetupWatch()
	self:UpdateInteractivity()

	self.frame:SetScript( "OnUpdate", function( this ) if this:IsVisible() then self:OnUpdate() end end )

	self.enabled = true
end

function Bar:Disable()
	if not self.enabled then return end

	self:UnregisterAllEvents()

	for z, zd in ipairs( self.zones ) do
		if zd.provider then
			zd.provider:Disconnect()
		end
	end

	UnregisterUnitWatch( self.frame )
	self:UpdateInteractivity( true )

	self.frame:SetScript( "OnUpdate", nil )
	self.frame:Hide()

	self.enabled = nil
end

function Bar:PLAYER_REGEN_DISABLED()
	for z, zd in ipairs( self.zones ) do
		if zd.provider and type( zd.provider.OnEnterCombat ) == "function" then
			zd.provider:OnEnterCombat()
		end
	end

	self.inCombat = true
end

function Bar:PLAYER_REGEN_ENABLED()
	self.inCombat = nil

	if self.needShow then self.frame:Show(); self.needShow = nil end
	if self.needHide then self.frame:Hide(); self.needHide = nil end

	for z, zd in ipairs( self.zones ) do
		if zd.provider and type( zd.provider.OnLeaveCombat ) == "function" then
			zd.provider:OnLeaveCombat()
		end
	end
end

function Bar:Hide( zone )
	if not zone or self.vMode ~= "manual" or zone ~= self.vZone then return end

	if not self:InConfigMode() then
		if self.inCombat then
			self.needHide = true
		else
			self.frame:Hide()
		end
	end
end

function Bar:Show( zone )
	if not zone or self.vMode ~= "manual" or zone ~= self.vZone then return end

	if self.inCombat then
		self.needShow = true
	else
		self.frame:Show()
	end
end

function Bar:SetValue( zone, value, r, g, b, a )
	local zd = self.zones[zone]

	if r then
		zd.color.r = r
		zd.color.g = g
		zd.color.b = b
		zd.color.a = a

		self.style:SetColor( zone, r, g, b, a )
	end

	if zd.targetValue == value then return end

	if not zd.config.animate or not self.frame:IsVisible() then
		zd.currentValue = value
		zd.targetValue = value

		self.style:SetValue( zone, value )
	else
		zd.targetValue = value

		if zd.currentValue ~= zd.targetValue then
			zd.animationEndTime = GetTime() + DURATION
			zd.animating = true
		end
	end
end

function Bar:SetColor( zone, r, g, b, a )
	local zd = self.zones[zone]

	zd.color.r = r
	zd.color.g = g
	zd.color.b = b
	zd.color.a = a

	self.style:SetColor( zone, r, g, b, a )
end

function Bar:WhoManageVisibility()
	self.vMode = nil
	self.vZone = nil
	self.vProvider = nil

	for z, zd in ipairs( self.zones ) do
		local zdp = zd.provider

		if zdp then
			if zdp.manageVisibility then
				-- first provider that claims visibility management is chosen over
				-- automatic and later providers.

				self.vMode = "manual"
				self.vZone = z
				self.vProvider = zdp

				return
			else
				if not self.vMode then
					self.vMode = "auto"
					self.vZone = z
					self.vProvider = zdp
				end
			end
		end
	end
end

--[[
	The main question is: what unit to monitor and how? Since there is no single provider
	anymore we rely on primary zone provider.
--]]
function Bar:SetupWatch()
	UnregisterUnitWatch( self.frame )

	if self.vMode == "manual" then
		self.frame:SetAttribute( "unit", nil )

		if self.vProvider:IsBarVisible() or self:InConfigMode() then
			self.frame:Show()
		else
			self.frame:Hide()
		end
	elseif self.vMode == "auto" then
		local watchUnit = self.config.unit

		if type( self.vProvider.GetWatchUnit ) == "function" then
			watchUnit = self.vProvider:GetWatchUnit()
		end

		self.frame:SetAttribute( "unit", watchUnit )
		self.frame:Show()

		if not self:InConfigMode() then
			RegisterUnitWatch( self.frame )
		end
	else
		-- No one? Hide it
		-- Nooo, just watch unit
		--self.frame:SetAttribute( "unit", nil )
		--self.frame:Hide()

		self.frame:SetAttribute( "unit", self.config.unit )
		self.frame:Show()

		if not self:InConfigMode() then
			RegisterUnitWatch( self.frame )
		end
	end
end

function Bar:UpdateInteractivity( turnOff )
	if self.config.interactive and not turnOff then
		self.frame:SetAttribute( "*type1", "target" )
		self.frame:SetAttribute( "*type2", "menu" )
		self.frame:EnableMouse( true )
		self.frame:RegisterForClicks( "AnyUp" )
	else
		self.frame:SetAttribute( "*type1", nil )
		self.frame:SetAttribute( "*type2", nil )
		self.frame:RegisterForClicks()
		self.frame:EnableMouse( false )
	end

	local unit = self.config.unit
	local num

	if self.config.interactive then
		if unit == "player" then
			self.frame.menu = function( self ) ToggleDropDownMenu( 1, nil, PlayerFrameDropDown, "cursor", 0, 0 ) end
		elseif unit == "target" then
			self.frame.menu = function( self ) ToggleDropDownMenu( 1, nil, TargetFrameDropDown, "cursor", 0, 0 ) end
		elseif unit == "focus" then
			self.frame.menu = function( self ) ToggleDropDownMenu( 1, nil, FocusFrameDropDown, "cursor", 0, 0 ) end
		elseif unit == "pet" then
			self.frame.menu = function( self ) ToggleDropDownMenu( 1, nil, PetFrameDropDown, "cursor", 0, 0 ) end
		else
			num = unit:match( "^party(%d)$" )

			if num then
				self.frame.menu = function( self ) ToggleDropDownMenu( 1, nil, _G["PartyMemberFrame"..num.."DropDown"], "cursor", 0, 0 ) end
			else
				self.frame.menu = nil
			end
		end
	else
		self.frame.menu = nil
	end

	ClickCastFrames = ClickCastFrames or {}

	if self.config.interactive then
		ClickCastFrames[self.frame] = true
	else
		ClickCastFrames[self.frame] = nil
	end
end

function Bar:InConfigMode()
	return UH:InConfigMode()
end

function Bar:SetConfigMode( mode )
	if mode then
		UnregisterUnitWatch( self.frame )
		self:UpdateInteractivity( true )
		self.frame:Show()
	else
		self:SetupWatch()
	end

	self.style:Update()
end

function Bar:OnUpdate()
	local currentTime = GetTime()

	for z, zd in ipairs( self.zones ) do
		if zd.provider then
			if zd.provider.realtime then
				local value, r, g, b, a = zd.provider:GetRealtimeValueAndColor( currentTime )

				if value then self.style:SetValue( z, value ) end
				if r and g and b and a then self.style:SetColor( z, r, g, b, a ) end
			elseif zd.animating and zd.currentValue and zd.targetValue then
				local value = CosineInterpolate( zd.currentValue, zd.targetValue, 1 - ((zd.animationEndTime - currentTime) / DURATION) )

				zd.currentValue = value
				self.style:SetValue( z, value )

				if zd.currentValue == zd.targetValue then
					zd.animating = nil
				end
			end
		end
	end
end

function Bar:ChangeStyle( newStyleName )
	self.style:Destroy()

	if self.frames then
		for k, v in pairs( self.frames ) do
			UH:ReleaseFrame( v )
		end

		self.frames = nil
	end

	self.config.style = newStyleName
	self.style = self.module:GetStyle( self.config.style )

	if type( self.style.DefaultSettings ) == "function" then
		self.style:DefaultSettings( self.config )
	else
		self.config.styleSettings = nil
	end

	-- Adjusting frame size in case style has changed it

	if self.frame:GetWidth() ~= self.config.width then self.frame:SetWidth( self.config.width ) end
	if self.frame:GetHeight() ~= self.config.height then self.frame:SetHeight( self.config.height ) end

	self.style = self.style:Create( self )

	if type( self.style.SetupOptions ) == "function" then
		self.options.args.appearance.plugins.style = {}
		self.style:SetupOptions( self.options.args.appearance.plugins.style )
	else
		self.options.args.appearance.plugins.style = nil
	end

	-- Validating mappings
	local styleZones = self.style:GetZones()
	local count = math_max( #styleZones, #self.config.zones )

	for z = 1, count do
		local sz, zm, zd

		if z > #styleZones then
			-- That zone doesn't exists
			zd = self.zones[z]

			if zd.provider then
				zd.provider:Disconnect()
				zd.provider = nil
			end

			self.zones[z] = nil
			self.config.zones[z] = nil
		else
			sz = styleZones[z]

			if z > #self.config.zones then
				zm = { provider = "" }
				self.config.zones[z] = zm
			else
				zm = self.config.zones[z]
				zd = self.zones[z]
			end

			if not zd then
				zd = {}
				zd.config = zm
				zd.currentValue = 0
				zd.targetValue = 0
				zd.color = { r = 0, g = 0, b = 0, a = 0 }

				self.zones[z] = zd
			else
				zd.currentValue = zd.targetValue
				zd.animationEndTime = nil
				zd.animating = nil

				self.style:SetColor( z, zd.color.r, zd.color.g, zd.color.b, zd.color.a )
				self.style:SetValue( z, zd.targetValue )
			end
		end

		self:SetupZoneOptions( z )
	end

	self:WhoManageVisibility()
	self:SetupWatch()
	self:UpdateInteractivity()

	UH:NotifyOptionsChanged()
end

function Bar:ChangeUnit( unit )
	self.config.unit = unit

	for z, zd in ipairs( self.zones ) do
		if zd.provider then
			zd.provider:Disconnect()
			zd.animationEndTime = nil
			zd.animating = nil
			zd.currentValue = 0
			zd.targetValue = 0
			zd.color = { r = 0, g = 0, b = 0, a = 0 }
		end

		local provider = self.module:GetProvider( zd.config.provider )

		if not (provider and provider:UnitSupported( unit )) then
			zd.config.provider = ""
			zd.config.providerSettings = nil
			zd.config.animate = nil
			zd.provider = nil
		else
			zd.provider:Connect()
		end

		self:SetupZoneOptions( z )
	end

	self:WhoManageVisibility()
	self:SetupWatch()
	self:UpdateInteractivity()

	UH:NotifyOptionsChanged()
end

function Bar:ChangeProvider( z, provider )
	local zd = self.zones[z]

	if not zd then return end

	if zd.provider then
		zd.provider:Disconnect()
		zd.provider = nil
		zd.animationEndTime = nil
		zd.animating = nil
		zd.currentValue = 0
		zd.targetValue = 0
		zd.color = { r = 0, g = 0, b = 0, a = 0 }
	end

	if #provider > 0 then
		zd.config.provider = provider
		provider = self.module:GetProvider( provider )

		if type( provider.DefaultSettings ) == "function" then
			provider:DefaultSettings( zd.config )
		else
			zd.config.providerSettings = nil
		end

		zd.provider = provider:Create( self, z, zd.config )
		zd.provider:Connect()
	else
		zd.config.provider = ""
		zd.config.providerSettings = nil
	end

	self:SetupZoneOptions( z )

	self:WhoManageVisibility()
	self:SetupWatch()
	self:UpdateInteractivity()

	UH:NotifyOptionsChanged()
end

function Bar:SetupOptions()
	self.options = {
		name = function() return self.config.name end,
		type = "group",
		childGroups = "tab",
		args = {
			general = {
				name = L["General"],
				type = "group",
				order = 100,
				args = {
					unit = {
						name = L["Unit"],
						type = "select",
						order = 1,
						values = function() return UH.OptionsHelper.Unit end,
						get = function() return self.config.unit end,
						set = function( info, value ) self:ChangeUnit( value ) end,
					},
					style = {
						name = L["Style"],
						type = "select",
						order = 2,
						values = function() return self.module:GetStylesNameTable() end,
						get = function() return self.config.style end,
						set = function( info, value ) self:ChangeStyle( value ) end,
					},
				},
				plugins = {
					zones = {},
				},
			},
			appearance = {
				name = L["Appearance"],
				type = "group",
				order = 101,
				args = {
					interactive = {
						name = L["Interactive"],
						desc = L["Click to target unit, Right-click for popup menu"],
						type = "toggle",
						disabled = function( info ) return self.vMode ~= "auto" end,
						order = 1,
						get = function() return self.config.interactive end,
						set = function( info, value ) self.config.interactive = value; self:UpdateInteractivity() end,
					},
				},
				plugins = {},
			},
			positionAndSize = {
				name = L["Position and Size"],
				type = "group",
				order = 102,
				args = {
					x = {
						name = L["X"],
						type = "range",
						order = 1,
						min = -800,
						max = 800,
						step = 1,
						bigStep = 1,
						get = function() return self.config.x end,
						set = function( info, value )
							self.config.x = value
							self.frame:ClearAllPoints()
							self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.config.x, self.config.y )
						end,
					},
					y = {
						name = L["Y"],
						type = "range",
						order = 2,
						min = -800,
						max = 800,
						step = 1,
						bigStep = 1,
						get = function() return self.config.y end,
						set = function( info, value )
							self.config.y = value
							self.frame:ClearAllPoints()
							self.frame:SetPoint( "CENTER", UIParent, "CENTER", self.config.x, self.config.y )
						end,
					},
					width = {
						name = L["Width"],
						type = "range",
						order = 3,
						min = 1,
						max = 500,
						step = 1,
						bigStep = 1,
						get = function() return self.config.width end,
						set = function( info, value )
							self.config.width = value
							self.frame:SetWidth( value )
							self.style:Update()
						end,
					},
					height = {
						name = L["Height"],
						type = "range",
						order = 4,
						min = 1,
						max = 500,
						step = 1,
						bigStep = 1,
						get = function() return self.config.height end,
						set = function( info, value )
							self.config.height = value
							self.frame:SetHeight( value )
							self.style:Update()
						end,
					},
					scale = {
						name = L["Scale"],
						type = "range",
						order = 5,
						min = 0.01,
						max = 5,
						step = 0.01,
						bigStep = 0.01,
						get = function() return self.config.scale end,
						set = function( info, value )
							self.config.scale = value
							self.frame:SetScale( value )
							self.style:Update()
						end,
					},
				},
			},
			zOrder = {
				name = L["Z-Order"],
				type = "group",
				order = 103,
				args = {
					strata = {
						name = L["Strata"],
						type = "select",
						order = 1,
						values = UH.OptionsHelper.Strata,
						get = function() return self.config.strata end,
						set = function( info, value ) self.config.strata = value; self.frame:SetFrameStrata( value ) end,
					},
					level = {
						name = L["Level"],
						type = "range",
						order = 2,
						min = 1,
						max = 10,
						step = 1,
						get = function() return self.config.level end,
						set = function( info, value ) self.config.level = value; self.frame:SetFrameLevel( value ) end,
					},
				},
			},
		},
		plugins = {},
	}

	for z, zd in ipairs( self.zones ) do
		self:SetupZoneOptions( z )
	end

	if self.style and type( self.style.SetupOptions ) == "function" then
		self.options.args.appearance.plugins.style = {}
		self.style:SetupOptions( self.options.args.appearance.plugins.style )
	end
end

function Bar:SetupZoneOptions( z )
	local zd = self.zones[z]
	local zo

	if zd then
		local styleZones = self.style:GetZones()
		zo = {
			name = L["Zone:"].." "..styleZones[z].title,
			type = "group",
			inline = true,
			order = 100+z,
			args = {
				provider = {
					name = L["Provider"],
					type = "select",
					order = 1,
					values = function() return self.module:GetProvidersNameTableWithNone() end,
					get = function( info ) return zd.config.provider or false end,
					set = function( info, value ) self:ChangeProvider( z, value ) end,
					validate = function( info, value )
						if #value == 0 then return true end

						local provider = self.module:GetProvider( value )

						if not provider:UnitSupported( self.config.unit ) then
							return (L["Sorry, but provider '%s' doesn't support unit '%s'."]):format(
								self.module:GetProvidersNameTableWithNone()[value],
								UH.OptionsHelper.Unit[self.config.unit] )
						end

						return true
					end,
				},
				animate = {
					name = L["Animate"],
					type = "toggle",
					order = 2,
					get = function( info ) return zd.provider and zd.config.animate end,
					set = function( info, value ) zd.config.animate = value end,
					disabled = function( info ) return not zd.provider or zd.provider.realtime end,
				},
			},
			plugins = {},
		}

		if zd.provider and type( zd.provider.SetupOptions ) == "function" then
			zo.plugins.provider = {}
			zd.provider:SetupOptions( zo.plugins.provider )
		end
	end

	self.options.args.general.plugins.zones["Zone"..z] = zo
end
