--[[
Prototype for the module, that is based on single frame.

Module MUST define the following keys:

	configDefaults - table with default values for the module's database
	configTemplate - table with values to be filled into the new or reset profile
	CreateFrame - method that will create frame
	DestroyFrame - method that will destroy frame
	SetupFrameOptions - method that will provide frame options
-]]

local L = LibStub( "AceLocale-3.0" ):GetLocale( "UnderHood" )
local Module = {}
Module.__index = {}

UnderHood.SingleFrameModulePrototype = Module

--
-- OnInitialize()
--
-- * Register database namespace with the name same as module and initializes it with
--   defaults from "self.configDefaults".
-- * Create Frames table
-- * Call SetupOptions
--
function Module:OnInitialize()
	self.db = UnderHood.db:RegisterNamespace( self.moduleName, self.configDefaults )

	self.db.RegisterCallback( self, "OnNewProfile", "FillProfileFromTemplate" )
	self.db.RegisterCallback( self, "OnProfileReset", "FillProfileFromTemplate" )
	self.db.RegisterCallback( self, "OnProfileChanged", "HandleProfileChanges" )
	self.db.RegisterCallback( self, "OnProfileCopied", "HandleProfileChanges" )

	self:SetupOptions()
end

local function copyTable(src, dest)
	if type(dest) ~= "table" then dest = {} end
	if type(src) == "table" then
		for k,v in pairs(src) do
			if type(v) == "table" then
				-- try to index the key first so that the metatable creates the defaults, if set, and use that table
				v = copyTable(v, dest[k])
			end
			dest[k] = v
		end
	end
	return dest
end

function Module:FillProfileFromTemplate( event, db, key )
	if self.configTemplate then
	    copyTable( self.configTemplate, self.db.profile )
	end

	if event == "OnProfileReset" and UnderHood.playerLoggedIn then
	    self:RefreshProfile()
	end
end

function Module:HandleProfileChanges()
	if UnderHood.playerLoggedIn then
	    self:RefreshProfile()
	end
end

--
-- OnEnable()
--
-- * Call RefreshProfile if PLAYER_LOGIN event has been fired or deffer it until
--   UnderHood_PlayerLogin message will be fired
--
function Module:OnEnable()
	if UnderHood.playerLoggedIn then
	    self:RefreshProfile()
	else
		self:RegisterMessage( "UnderHood_PlayerLogin",
			function()
				self:RefreshProfile();
				self:UnregisterMessage( "UnderHood_PlayerLogin" )
			end )
	end
end

function Module:OnDisable()
	if self.Frame and type( self.DestroyFrame ) == "function" then
	    self:DestroyFrame()
	    self.Frame = nil
	end
end

--
-- RefreshProfile()
--
-- * Destroy existing frame, if any
-- * Create new frame based on configuration from database
-- * Update frame's options table

function Module:RefreshProfile()
	if not self.db or not self.db.profile then return end

	if self.Frame and type( self.DestroyFrame ) == "function" then
	    self:DestroyFrame()
	    self.Frame = nil
	end
	
	for k, v in pairs( self.Options.plugins ) do
	    self.Options.plugins[k] = nil
	end
	
	if type( self.CreateFrame ) == "function" then
	    self:CreateFrame()
	end
	
	if type( self.SetupFrameOptions ) == "function" then
	    self:SetupFrameOptions()
	end

	UnderHood:NotifyOptionsChanged()
end

--
-- SetupOptions()
--
-- * Create module's options table

function Module:SetupOptions()
	if self.Options then return end

	self.Options = {
		name = self.moduleName,
		type = "group",
		args = {
		},
		plugins = {},
	}
end

--
-- GetOptions()
--
-- * Return module's options table

function Module:GetOptions()
	if not self.Options then
		self:SetupOptions()
	end

	return self.Options
end
