--[[
Prototype for the module, that is based on multiple frame instances.

Module MUST define the following keys:

	configDefaults - table with default values for the module's database
	configTemplate - table with values to be filled into the new or reset profile
	configTableName - key in profile table where configurations for instances are stored
	configEntryPrefix - prefix for the entries in config table, e.g. "bar", "text", "buff"
	FramePrototype - table that will serve as prototype for frame instances
	    FramePrototype MUST declare method Create( module, config ) that will create a new
	    instance of frame. That instance MUST have method Destroy() that safely destroys
	    frame releasing all UI elements.

Module MAY define the following methods:

	ExtendModuleOptions - called right after the options are created
-]]

local UH = UnderHood
local L = LibStub( "AceLocale-3.0" ):GetLocale( "UnderHood" )
local Module = {}
Module.__index = {}

UnderHood.MultiFrameModulePrototype = Module

--
-- OnInitialize()
--
-- * Register database namespace with the name same as module and initializes it with
--   defaults from "self.configDefaults".
-- * Create Frames table
-- * Call SetupOptions
--
function Module:OnInitialize()
	self.db = UnderHood.db:RegisterNamespace( self.moduleName, self.configDefaults )

	self.db.RegisterCallback( self, "OnNewProfile", "FillProfileFromTemplate" )
	self.db.RegisterCallback( self, "OnProfileReset", "FillProfileFromTemplate" )
	self.db.RegisterCallback( self, "OnProfileChanged", "HandleProfileChanges" )
	self.db.RegisterCallback( self, "OnProfileCopied", "HandleProfileChanges" )

	self.Frames = {}
	
	self:SetupOptions()
end

local function copyTable(src, dest)
	if type(dest) ~= "table" then dest = {} end
	if type(src) == "table" then
		for k,v in pairs(src) do
			if type(v) == "table" then
				-- try to index the key first so that the metatable creates the defaults, if set, and use that table
				v = copyTable(v, dest[k])
			end
			dest[k] = v
		end
	end
	
	return dest
end

function Module:FillProfileFromTemplate( event, db, key )
	if self.configTemplate then
	    copyTable( self.configTemplate, self.db.profile )
	end

	if event == "OnProfileReset" and UnderHood.playerLoggedIn then
	    self:RefreshProfile()
	end
end

function Module:HandleProfileChanges()
	if UnderHood.playerLoggedIn then
	    self:RefreshProfile()
	end
end
--
-- OnEnable()
--
-- * Call RefreshProfile if PLAYER_LOGIN event has been fired or deffer it until
--   UnderHood_PlayerLogin message will be fired
--
function Module:OnEnable()
	if UnderHood.playerLoggedIn then
	    self:RefreshProfile()
	else
		self:RegisterMessage( "UnderHood_PlayerLogin",
			function()
				self:RefreshProfile();
				self:UnregisterMessage( "UnderHood_PlayerLogin" )
			end )
	end
end

function Module:OnDisable()
	for _, frame in pairs( self.Frames ) do
		frame:Destroy()
	end

	self.Frames = {}
	self.Options.plugins.frames = {}
	self.Options.plugins.frameState = {}

	UnderHood:NotifyOptionsChanged()
end

--
-- SetConfigMode()
--
-- * Call SetConfigMode for every frame instance
--
function Module:SetConfigMode( mode )
	for _, frame in pairs( self.Frames ) do
	    if type( frame.SetConfigMode ) == "function" then
	        frame:SetConfigMode( mode )
		end
	end
end

--
-- FixFirstUnusedNumber()
--
-- * Find first "safe" number for naming items
--
function Module:FixFirstUnusedNumber()
	local pattern = self.configEntryPrefix.."(%d+)"
	local seed = 1
	local _, num

	for k, v in pairs( self.db.profile[self.configTableName] ) do
		_, _, num = string.find( k, pattern )

		if num and tonumber( num ) >= seed then
			seed = tonumber( num ) + 1
		end
	end

	self.FirstUnusedNumber = seed
end

--
-- RefreshProfile()
--
-- * Destroy all active frames
-- * Create new frames based on configuration from database
-- * Update module's options table

function Module:RefreshProfile()
	if not self.db or not self.db.profile then return end

	UH:Debug( self.moduleName..": RefreshProfile()")
	
	for _, frame in pairs( self.Frames ) do
		frame:Destroy()
	end

	self.Frames = {}
	self.Options.plugins.frames = {}
	self.Options.plugins.frameState = {}

	for name, config in pairs( self.db.profile[self.configTableName] ) do
		if type( self.UpgradeFrameConfig ) == "function" then
		    self:UpgradeFrameConfig( name, config )
		end
		
		local frame = self.FramePrototype:Create( self, config )
		self.Frames[name] = frame
		
		self.Options.plugins.frames[name] = self:ExtendFrameOptions( frame )
		
		if type( frame.Enable ) == "function" and type( frame.Disable ) == "function" then
		    local enable = {
		        name = function() return frame.config.name end,
		        type = "toggle",
		        get = function( info ) return frame.config.enabled end,
		        set = function( info, value )
		            frame.config.enabled = value
		            
		            if value then frame:Enable() else frame:Disable() end
		        end,
			}
			
			self.Options.plugins.frameState["ed"..name] = enable
		end
	end

	self:FixFirstUnusedNumber()

	UnderHood:NotifyOptionsChanged()
end

--
-- SetupOptions()
--
-- * Create module's options table

function Module:SetupOptions()
	if self.Options then return end

	self.Options = {
		name = self.moduleName,
		type = "group",
		args = {
		    createHeader = {
				type = "description",
				order = 1,
				name = L["If you wish to create a new frame you need to type a name for that frame in the input below and press an Enter key."].."\n",
			},
		    create = {
		        name = L["Type a name for the new item:"],
		        type = "input",
		        order = 2,
		        width = "full",
				validate = function( info, value )
					if string.find( value, "[ \(\[\]\)\.\:]" ) then
						return L["Invalid character in name."]
					end

					for k, v in pairs( self.db.profile[self.configTableName] ) do
						if k == value then return L["This name is already in use."] end
					end

					return true
				end,
		        get = function() return self.configEntryPrefix..self.FirstUnusedNumber end,
		        set = function( info, value ) self:CreateFrame( value ) end,
			},
			enableHeader = {
			    --name = L["Enable or disable individual frames"],
			    name = "\n"..L["You can enable or disable (without deleting) individual frames by checking or unchecking corresponding boxes below:"].."\n",
			    type = "description",
			    order = 3,
			    hidden = function() return type( self.FramePrototype.Enable ) ~= "function" or type( self.FramePrototype.Disable ) ~= "function" end
			},
		},
		plugins = {},
	}
	
	if type( self.ExtendModuleOptions ) == "function" then
	    self:ExtendModuleOptions()
	end
end

--
-- GetOptions()
--
-- * Return module's options table

function Module:GetOptions()
	if not self.Options then
		self:SetupOptions()
	end

	return self.Options
end

--
-- ExtendFrameOptions()
--
-- * Add "Name" and "Delete" entries to frame options table

function Module:ExtendFrameOptions( frame )
	local options = frame.options

	if not options.plugins then options.plugins = {} end
	if not options.plugins.module then options.plugins.module = {} end

	options.plugins.module.name = {
		name = L["Name"],
		type = "input",
		order = 1,
		validate = function( info, value )
			if string.find( value, "[ \(\[\]\)\.\:]" ) then
				return L["Invalid character in name."]
			end

			for k, v in pairs( self.db.profile[self.configTableName] ) do
				if k == value and v ~= frame.config then return L["This name is already in use."] end
			end

			return true
		end,
		get = function( info ) return frame.config.name end,
		set = function( info, value ) self:RenameFrame( frame, value ) end,
	}

	options.plugins.module.delete = {
		name = L["Delete"],
		type = "execute",
		order = 2,
		confirm = true,
		confirmText = L["Do you really want to delete this item?"],
		func = function() self:DeleteFrame( frame ) end,
	}

	return options
end

--
-- CreateFrame()
--
-- * Create new configuration entry
-- * Call OnInitializeFrameConfiguration method if any
-- * Create new frame instance

function Module:CreateFrame( name )
	local config = {}

	if not name then
		name = self.configEntryPrefix..self.FirstUnusedNumber
		self.FirstUnusedNumber = self.FirstUnusedNumber + 1
	end
	
	config.name = name
	config.enabled = true
	config.x = 0
    config.y = 0
	config.scale = 1
	config.strata = "LOW"
	config.level = 6

	local callback = self["OnInitializeFrameConfiguration"]
	
	if type( callback ) == "function" then
	    callback( self, config )
	end
	
	self.db.profile[self.configTableName][config.name] = config

	local frame = self.FramePrototype:Create( self, config )
	
	self.Frames[config.name] = frame

	self.Options.plugins.frames[config.name] = self:ExtendFrameOptions( frame )

	if type( frame.Enable ) == "function" and type( frame.Disable ) == "function" then
	    local enable = {
	        name = function() return frame.config.name end,
	        type = "toggle",
	        get = function( info ) return frame.config.enabled end,
	        set = function( info, value )
	            frame.config.enabled = value

	            if value then frame:Enable() else frame:Disable() end
	        end,
		}

		self.Options.plugins.frameState["ed"..config.name] = enable
	end

	if name:match( "^"..self.configEntryPrefix.."%d+$" ) then
	    self:FixFirstUnusedNumber()
	end
	
	UnderHood:NotifyOptionsChanged()
end

--
-- DeleteFrame()
--
-- * Destroy frame and cleanup database and options

function Module:DeleteFrame( frame )
	local name = frame.config.name

	self.Options.plugins.frames[name] = nil
	self.Options.plugins.frameState["ed"..name] = nil

	frame:Destroy()

	self.Frames[name] = nil
	self.db.profile[self.configTableName][name] = nil

	self:FixFirstUnusedNumber()

	UnderHood:NotifyOptionsChanged()
end

--
-- RenameFrame()
--
-- * Change configuration name and all references and options

function Module:RenameFrame( frame, newName )
	local oldName = frame.config.name

	frame.config.name = newName

	self.db.profile[self.configTableName][oldName] = nil
	self.db.profile[self.configTableName][newName] = frame.config

	self.Frames[oldName] = nil
	self.Frames[newName] = frame

	self.Options.plugins.frames[newName] = self.Options.plugins.frames[oldName]
	self.Options.plugins.frames[oldName] = nil

	self.Options.plugins.frameState["ed"..newName] = self.Options.plugins.frameState["ed"..oldName]
	self.Options.plugins.frameState["ed"..oldName] = nil

	self:FixFirstUnusedNumber()

	UnderHood:NotifyOptionsChanged();
end
