﻿--[[
--
-- TriviaBot Version 
-- Compatible with WoW 2.4.3
--
--]]

-- Global Variables

-- Declared colour codes for console messages
local RED     = "|cffff0000";
local WHITE   = "|cffffffff";
local TRIVIABOT_COLOR = "|cff6699ff";

TRIVIABOT_VERSION = "2.4.4" -- Version number

-- Run-time variables
TRIVIABOT_RUNNING = false;
TRIVIABOT_ACCEPT_ANSWERS = false;
TRIVIABOT_LOADED = false;
TRIVIABOT_ACTIVE_QUESTION = 0; -- The currently active question
TRIVIABOT_REPORT_COUNTER = 0;
TRIVIABOT_ROUND_COUNTER = 0; -- Count for which question we're on in a round
TRIVIABOT_QUESTION_ORDER = {}; -- An array to store in which order the questions will be asked
TRIVIABOT_QUESTION_STARTTIME = 0; -- When the question was started
TRIVIABOT_SCHEDULE = {}; -- The array used for scheduling events

-- Configuration Variables
TRIVIABOT_CONFIG = 
{
["settings"] = {}, -- Settings
["alltimescores"] = {},
["alltimequickest"] = {};
["gamescores"] = {},
["gamequickest"] = {};
};

TRIVIABOT_Questions = {};

----------------------------------------------------------------------------
-- Load event
----------------------------------------------------------------------------
function TriviaBot_OnLoad()
	 --Register Slash Command
    SLASH_TRIVIABOT1 = "/trivia";
	SLASH_TRIVIABOT2 = "/triviabot";
    SlashCmdList["TRIVIABOT"] = TriviaBot_Command;
    
    -- Register Events
	this:RegisterEvent("CHAT_MSG_CHANNEL");    
	this:RegisterEvent("CHAT_MSG_RAID");
	this:RegisterEvent("CHAT_MSG_RAID_LEADER");	
	this:RegisterEvent("CHAT_MSG_SAY");		
	this:RegisterEvent("CHAT_MSG_YELL");		
	this:RegisterEvent("CHAT_MSG_PARTY");		
	this:RegisterEvent("CHAT_MSG_GUILD");
	this:RegisterEvent("CHAT_MSG_SYSTEM");
	this:RegisterEvent("ADDON_LOADED");
end

----------------------------------------------------------------------------
-- Abort current question
----------------------------------------------------------------------------
function TriviaBot_AbortQuestion()
	if (TRIVIABOT_RUNNING) then
		TriviaBot_UnSchedule("all");
		TRIVIABOT_ACCEPT_ANSWERS = false;
	else
		TriviaBot_PrintError("No game running!");
	end
end


----------------------------------------------------------------------------
-- Print information about TriviaBot
----------------------------------------------------------------------------
function TriviaBot_About()
	TriviaBot_Print("TriviaBot Version: " .. TRIVIABOT_VERSION);
	TriviaBot_Print("Please read readme.txt for more information on the add-on and support.");
end

----------------------------------------------------------------------------
-- Ask a question
----------------------------------------------------------------------------
function TriviaBot_AskQuestion()
	-- Variables
	local currentQuestion = "";
	local questionNumber = 1;
	local c = 0;

	-- Enable Skip Button
	TriviaBotGUI_SkipButton:Enable();

	TRIVIABOT_ACTIVE_QUESTION = TRIVIABOT_ACTIVE_QUESTION + 1;

	-- Check there are questions left
	if (TRIVIABOT_ACTIVE_QUESTION > # TRIVIABOT_Questions[TRIVIABOT_CONFIG["settings"]["qlist"]]['QUESTIONS']
		or TRIVIABOT_QUESTION_ORDER[TRIVIABOT_ACTIVE_QUESTION] > # TRIVIABOT_Questions[TRIVIABOT_CONFIG["settings"]["qlist"]]['QUESTIONS']) then
		-- Reshuffle the order
		TriviaBot_Randomize();
		TRIVIABOT_ACTIVE_QUESTION = 1;
		TriviaBot_Send("Reshuffling questions...");
		TriviaBot_Print("Reshuffling questions...");
	end
	
	if (TRIVIABOT_CONFIG["settings"]["round_size"] + 1 ~= 1) then
		questionNumber = TRIVIABOT_ROUND_COUNTER + 1;
	end
	
	-- Grab the current question and put it in a reasonable variable.
	currentQuestion = TRIVIABOT_Questions[TRIVIABOT_CONFIG["settings"]["qlist"]]['QUESTIONS'][TRIVIABOT_QUESTION_ORDER[TRIVIABOT_ACTIVE_QUESTION]];
	
	-- Make sure current question is valid
	if (currentQuestion == nil) then
		TriviaBot_PrintError("currentQuestion is nil!");
		TriviaBot_PrintError("Please send this to TriviaBot tech support:");
		TriviaBot_PrintError("QList: " .. TRIVIABOT_CONFIG["settings"]["qlist"]);
		TriviaBot_PrintError("Question #: " .. TRIVIABOT_QUESTION_ORDER[TRIVIABOT_ACTIVE_QUESTION]);
		TriviaBot_PrintError("TriviaBot haulting due to fatal error in question database!");
		TriviaBot_Stop();
		return;
	end
	
	-- Find the whole "Quote:" thing
	c = string.find(currentQuestion, "Quote:");
    
    if (c == nil) then
	    TriviaBot_Send("Q" .. questionNumber .. ": " .. currentQuestion);
	else
        TriviaBot_Send(currentQuestion);	
	end
    
	TRIVIABOT_QUESTION_STARTTIME = GetTime();
	TRIVIABOT_ACCEPT_ANSWERS = true;
    TriviaBot_Schedule("QUESTION_TIMEOUT", TRIVIABOT_CONFIG["settings"]["question_timeout"]);
	TriviaBot_Schedule("QUESTION_WARN", TRIVIABOT_CONFIG["settings"]["question_timeout"] - TRIVIABOT_CONFIG["settings"]["question_timewarn"]);
end

----------------------------------------------------------------------------
-- Changes setting to value
----------------------------------------------------------------------------
function TriviaBot_ChangeSetting(setting, value)
	-- If setting is nil display the settings
	if(setting == nil) then
		TriviaBot_Print("TriviaBot settings");
		TriviaBot_Print("Warning! Be extremely careful when using this, unexpected input may cause MASSIVE errors, be prepared to use /trivia resetconfig.");
		TriviaBot_Print("--Printing All Settings--");
		
		for setting, value in pairs(TRIVIABOT_CONFIG["settings"]) do
			if(type(value) == "boolean") then
				if(value) then
					TriviaBot_Print(setting .. " = " .. "(bool)True");
				else
					TriviaBot_Print(setting .. " = " .. "(bool)False");
				end
			else
				TriviaBot_Print(setting .. " = " .. value);
			end
		end	
    elseif(TRIVIABOT_CONFIG["settings"][setting] ~= nil) then
		if(type(TRIVIABOT_CONFIG["settings"][setting]) == "boolean") then
			if(string.lower(value) == "true") then
				TRIVIABOT_CONFIG["settings"][setting] = true;
			elseif(string.lower(value) == "false") then
				TRIVIABOT_CONFIG["settings"][setting] = false;
			else
				 TriviaBot_PrintError("Error in setting, setting is a boolean value, setting must be true/false.");
				 return;
			end
		elseif(type(TRIVIABOT_CONFIG["settings"][setting]) == "number") then
			TRIVIABOT_CONFIG["settings"][setting] = tonumber(value);
		else
			TRIVIABOT_CONFIG["settings"][setting] = value;
		end
		
		TriviaBot_Print("Changed setting: " .. setting .. " to " .. value);
    else
        TriviaBot_Print("Setting not found");
    end
end

----------------------------------------------------------------------------
-- Compares user's answer to question
----------------------------------------------------------------------------
function TriviaBot_CheckAnswer(player, msg)
    -- For every answer in the list of answers
    for k, i in ipairs(TRIVIABOT_Questions[TRIVIABOT_CONFIG["settings"]["qlist"]]['ANSWERS'][TRIVIABOT_QUESTION_ORDER[TRIVIABOT_ACTIVE_QUESTION]]) do
        -- Check if answer is correct
        if ((string.lower(msg) == string.lower(i))) then
            -- Unschedule warnings and timeout
            TriviaBot_UnSchedule("all");
            
            -- Time the answer
            local timeTaken = GetTime() - TRIVIABOT_QUESTION_STARTTIME;
            timeTaken = math.floor(timeTaken  * 10^2 + 0.5) / 10^2;
            
            -- Tell player they don't suck as badly as they think they do.
            TriviaBot_Send("'".. msg .. "' is the correct answer, " .. player .. " in " .. timeTaken .. " seconds.");
            
            -- Announce if it was the all-time quickest
            if (TRIVIABOT_CONFIG["alltimequickest"]["time"] == nil or timeTaken < TRIVIABOT_CONFIG["alltimequickest"]["time"]) then
                TriviaBot_Send("---NEW ALL-TIME RECORD---");
                TRIVIABOT_CONFIG["alltimequickest"]["holder"] = player;
                TRIVIABOT_CONFIG["alltimequickest"]["time"] = timeTaken;
            end
            
            -- Announce if it was the game's quickest
            if (TRIVIABOT_CONFIG["gamequickest"]["time"] == nil or timeTaken < TRIVIABOT_CONFIG["gamequickest"]["time"]) then
                TriviaBot_Send("---NEW GAME RECORD---");
                TRIVIABOT_CONFIG["gamequickest"]["holder"] = player;
                TRIVIABOT_CONFIG["gamequickest"]["time"] = timeTaken;
            end
            
            -- Update their current game score
            local score = TRIVIABOT_CONFIG["gamescores"][player];
            if (score) then score = score + 1; else score = 1; end
            TRIVIABOT_CONFIG["gamescores"][player] = score;
           
            -- Update their overall score
            score = TRIVIABOT_CONFIG["alltimescores"][player];
            if (score) then score = score + 1; else score = 1; end
            TRIVIABOT_CONFIG["alltimescores"][player] = score;

            TriviaBot_EndQuestion(false);
        end
    end
end

----------------------------------------------------------------------------
-- Command handler
----------------------------------------------------------------------------
function TriviaBot_Command(cmd)
    -- Create variables
	local msgArgs = {};
	local numArgs = 0;
    
	-- Convert to lower case
	cmd = string.lower(cmd);
	
	-- Seperate our args
	for value in string.gmatch(cmd, "[^ ]+") do
		msgArgs[numArgs] = value;
		numArgs = numArgs + 1;
	end
	
	if (numArgs == 0) then
		-- Show the GUI
		if (not TriviaBotGUI:IsShown()) then
			TriviaBotGUI:Show();
		else
			TriviaBotGUI:Hide();
		end
	elseif (msgArgs[0] == "about") then
		TriviaBot_About();
	elseif (msgArgs[0] == "help") then
        TriviaBot_Print("---TriviaBot Help---");
        TriviaBot_Print("/trivia about - Information on TriviaBot and support");
        TriviaBot_Print("/trivia clear - Clear current game data");
        TriviaBot_Print("/trivia clearall - Clear current game and all time score data");
        TriviaBot_Print("/trivia help - Help on TriviaBot commands");
        TriviaBot_Print("/trivia load [ID#] - Load trivia via ID #");
        TriviaBot_Print("/trivia resetconfig - Reset configuration to defaults");
        TriviaBot_Print("/trivia reshuffle - Reshuffles the trivia questions");
        TriviaBot_Print("/trivia scores [option] - Option = AllTime or Game");
        TriviaBot_Print("/trivia setting [name] [value] - Change setting 'name' to become 'value'");
        TriviaBot_Print("/trivia skip - Skip the current question");
        TriviaBot_Print("/trivia start - Start trivia");
        TriviaBot_Print("/trivia stop - Stop trivia");
    elseif(msgArgs[0] == "clear") then
        -- Clear current game scores
        TRIVIABOT_CONFIG["gamescores"] = {};
        TRIVIABOT_CONFIG["gamequickest"] = {};
        TriviaBot_Print("Game scores reset.");
    elseif(msgArgs[0] == "clearall") then
        -- Clear all scores
        TRIVIABOT_CONFIG["alltimescores"] = {};
        TRIVIABOT_CONFIG["alltimequickest"] = {};
        TRIVIABOT_CONFIG["gamescores"] = {};
        TRIVIABOT_CONFIG["gamequickest"] = {};
        TriviaBot_Print("All scores reset.");
    elseif(msgArgs[0] == "load") then
        if(msgArgs[1]) then
            TriviaBot_LoadTrivia(msgArgs[1]);
        else
            TriviaBot_PrintError("Trivia ID # must be supplied!");
        end
    elseif(msgArgs[0] == "resetconfig") then
        TriviaBot_NewConfig(true);
    elseif(msgArgs[0] == "reshuffle") then
		TriviaBot_Randomize();
		TriviaBot_Print("Reshuffling...");
    elseif(msgArgs[0] == "scores") then
		if(msgArgs[1] == "" or msgArgs[1] == nil) then
			TriviaBot_Print("Invalid score command, please read /trivia help");
			return;
		end
		local scoreArg = string.lower(msgArgs[1]);
		if(scoreArg == "alltime") then
			TriviaBot_PrintAllTimeScores();
		elseif(scoreArg == "game") then
			TriviaBot_PrintGameScores();
		else
			TriviaBot_Print("Invalid score command");
		end
    elseif(msgArgs[0] == "setting") then
        TriviaBot_ChangeSetting(msgArgs[1], msgArgs[2]);
    elseif(msgArgs[0] == "skip") then
		TriviaBot_SkipQuestion();
    elseif(msgArgs[0] == "start") then
        TriviaBot_Start();
    elseif(msgArgs[0] == "stop") then
        TriviaBot_Stop();
    elseif(msgArgs[0] == "test") then
		TriviaBot_Send("Testing channel...");
    else
        TriviaBot_PrintError("Invalid Command - Use '/trivia help' for more information on how to use TriviaBot.");
    end 
end

----------------------------------------------------------------------------
-- Do Scheduled events
----------------------------------------------------------------------------
function TriviaBot_DoSchedule()
	if (TRIVIABOT_SCHEDULE ~= nil) then
		for id, events in pairs(TRIVIABOT_SCHEDULE) do
			-- Get the time of each event
			-- If it should be run (i.e. equal or less than current time)
			if (events["time"] <= GetTime()) then
				TriviaBot_OnEvent(events["name"]);
				TriviaBot_UnSchedule(id);
			end
		end
	end
end


----------------------------------------------------------------------------
-- Called when a question is finished
----------------------------------------------------------------------------
function TriviaBot_EndQuestion(showAnswer)
    -- Prevent further answers
	TRIVIABOT_ACCEPT_ANSWERS = false;
	--TriviaBotGUISkipButton:Disable();

	-- Increment the counters
	TRIVIABOT_ROUND_COUNTER = TRIVIABOT_ROUND_COUNTER  + 1;
	
	local wait = 0;
	if (showAnswer) then
		wait = wait + 4;
		TriviaBot_Schedule("SHOW_ANSWER", wait);
	end
	
	-- See if we've reached the end of the round
	if (TRIVIABOT_ROUND_COUNTER + 1 == TRIVIABOT_CONFIG["settings"]["round_size"] + 1) then
		TriviaBot_Schedule("END_REPORT", wait + 4);
		TriviaBot_Schedule("STOP_GAME", wait + 8);
	else
		-- Count how long it's been since a question report
		if (TRIVIABOT_CONFIG["settings"]["show_reports"]) then
			TRIVIABOT_REPORT_COUNTER = TRIVIABOT_REPORT_COUNTER + 1;
			if (TRIVIABOT_REPORT_COUNTER == TRIVIABOT_CONFIG["settings"]["report_interval"]) then
				wait = wait + 4;
				TriviaBot_Schedule("REPORT_SCORERS", wait);
				TRIVIABOT_REPORT_COUNTER = 0;
			end
		end
		
		TriviaBot_Schedule("NEXT_QUESTION", TRIVIABOT_CONFIG["settings"]["question_interval"] + wait);
	end
	
end

----------------------------------------------------------------------------
-- Load trivia into trivia memory
----------------------------------------------------------------------------
function TriviaBot_LoadTrivia(id)
    -- If we're running we need to skip the current question before hopping databases.
    if(TRIVIABOT_RUNNING)then
		TriviaBot_AbortQuestion();
    end

    id = tonumber(id);
    
    -- Make sure ID conversion isn't nil
    if(id == nil) then
		TriviaBot_Print("Invalid ID, ID must be a number");
		return;
    end
    
    --See if question array exists
    if(TRIVIABOT_Questions[id] and id > 0) then
        TRIVIABOT_CONFIG["settings"]['qlist'] = id;
        if(TRIVIABOT_Questions[TRIVIABOT_CONFIG["settings"]['qlist']].QUESTIONS[1]) then
            TriviaBot_Print("Questions loaded sucessfully. Question count: " .. #TRIVIABOT_Questions[TRIVIABOT_CONFIG["settings"]["qlist"]]['QUESTIONS']);
            TriviaBot_Print("Database Name: " .. TRIVIABOT_Questions[TRIVIABOT_CONFIG["settings"]['qlist']]['DESCRIPTION'] .. ". Author: " .. TRIVIABOT_Questions[TRIVIABOT_CONFIG["settings"]["qlist"]]["AUTHOR"]);
        else
			-- Fail, there is no question 1, database is improperly formatted
			TriviaBot_PrintError("Database error, question 1 does not exist or is improperly formatted! Please choose another database");
			return;
        end
                
		TriviaBot_Randomize();
        if(TRIVIABOT_RUNNING)then
            --If we're switching databases mid game, we should alert our players that the mode has changed
            TriviaBot_Send("Switching trivia database to: " .. TRIVIABOT_Questions[TRIVIABOT_CONFIG["settings"]['qlist']]['DESCRIPTION'] .. ". Author: " .. TRIVIABOT_Questions[TRIVIABOT_CONFIG["settings"]["qlist"]]["AUTHOR"]);
            TriviaBot_Schedule("NEXT_QUESTION", 6);
        end
    else
        TriviaBot_Print("Question List ID: " .. id .. " does not exist.");
        TriviaBot_Print("Available Libraries of Trivia:");
        for k,i in ipairs(TRIVIABOT_Questions) do
            TriviaBot_Print("ID: " .. k .. " - " .. i['DESCRIPTION']);
        end 
    end
end

----------------------------------------------------------------------------
-- Create new configuation
----------------------------------------------------------------------------
function TriviaBot_NewConfig(reset)
    if(reset) then
        TRIVIABOT_CONFIG["settings"] = {};
        TriviaBot_Print("Resetting configuration to default values");
    end
    
    if(not TRIVIABOT_CONFIG["alltimescores"]) then TRIVIABOT_CONFIG["alltimescores"] = {};end;
    if(not TRIVIABOT_CONFIG["alltimequickest"]) then TRIVIABOT_CONFIG["alltimequickest"] = {};end;
    if(not TRIVIABOT_CONFIG["gamescores"]) then TRIVIABOT_CONFIG["gamescores"] = {};end;
    if(not TRIVIABOT_CONFIG["gamequickest"]) then TRIVIABOT_CONFIG["gamequickest"] = {};end;
        
    -- The default channel type (guild)
    if (not TRIVIABOT_CONFIG["settings"]["channel_type"]) then TRIVIABOT_CONFIG["settings"]["channel_type"] = "GUILD";end
        
    -- The default private channel
	if (not TRIVIABOT_CONFIG["settings"]["channel"]) then TRIVIABOT_CONFIG["settings"]["channel"] = "";end
    
    -- Defaults to unlimited questions per round
	if (not TRIVIABOT_CONFIG["settings"]["round_size"]) then TRIVIABOT_CONFIG["settings"]["round_size"] = 0;end
    
	-- Default question list
	if (not TRIVIABOT_CONFIG["settings"]["qlist"]) then TRIVIABOT_CONFIG["settings"]["qlist"] = 1;end
    
	-- Default Update interval. Tweaking may increase performance.
	if (not TRIVIABOT_CONFIG["settings"]["update_interval"]) then TRIVIABOT_CONFIG["settings"]["update_interval"] = 0.5; end
    
	 -- Number of seconds between questions
	if (not TRIVIABOT_CONFIG["settings"]["question_interval"]) then TRIVIABOT_CONFIG["settings"]["question_interval"] = 30;end
    
	 -- Defaults to reporting scores every 5 answers
	if (not TRIVIABOT_CONFIG["settings"]["report_interval"]) then TRIVIABOT_CONFIG["settings"]["report_interval"] = 5;end
    
	 -- Show reports
	if (not TRIVIABOT_CONFIG["settings"]["show_reports"]) then TRIVIABOT_CONFIG["settings"]["show_reports"] = true;end
    
	 -- Show Answers
	if (not TRIVIABOT_CONFIG["settings"]["show_answers"]) then TRIVIABOT_CONFIG["settings"]["show_answers"] = true;end
    
	 -- Default question timeout
	if (not TRIVIABOT_CONFIG["settings"]["question_timeout"]) then TRIVIABOT_CONFIG["settings"]["question_timeout"] = 45;end
    
	 -- Default warning at 20 seconds.
	if (not TRIVIABOT_CONFIG["settings"]["question_timewarn"]) then TRIVIABOT_CONFIG["settings"]["question_timewarn"] = 20;end
	
	 -- Set Score Count to 5
	if (not TRIVIABOT_CONFIG["settings"]["score_count"]) then TRIVIABOT_CONFIG["settings"]["score_count"] = 5;end
        
	-- Store the version
	TRIVIABOT_CONFIG.CONFIG_VERSION = TRIVIABOT_VERSION;
end

----------------------------------------------------------------------------
-- Event handler
----------------------------------------------------------------------------
function TriviaBot_OnEvent(event)
	if (event == "ADDON_LOADED") then
		if (not TRIVIABOT_LOADED) then
			-- Load the saved variables
			if (not TRIVIABOT_CONFIG.CONFIG_VERSION) then
                TriviaBot_Print("Creating new configuation.");
				TriviaBot_NewConfig();
			end
			if (TRIVIABOT_CONFIG.CONFIG_VERSION ~= TRIVIABOT_VERSION) then
				TriviaBot_Print("Old version detected, upgrading to new version.");
				TriviaBot_Print("Old: " .. TRIVIABOT_CONFIG.CONFIG_VERSION .. " New: " .. TRIVIABOT_VERSION);
				TriviaBot_NewConfig();
			end
			
			--Start in the 'off' state
			TRIVIABOT_ACCEPT_ANSWERS = false;
			
			-- Send a message
			TriviaBot_Print("Version " .. TRIVIABOT_VERSION .. " loaded.");
			
			-- Update the GUI with the current configuration
			TriviaBotGUI_Initialize();
			TriviaBotGUI_Update();
		
			-- Load the questions
			TriviaBot_LoadTrivia(TRIVIABOT_CONFIG["settings"]["qlist"]);
            			
			-- Generate the question order.
			TriviaBot_Randomize();
			
			-- Set loaded state
			TRIVIABOT_LOADED = true;
		end
		
	elseif (event == "CHAT_MSG_CHANNEL") then
		local msg = arg1;
		local player = arg2;
        local channel = string.lower(arg9);

        if( (msg and msg ~= nil) and (player and player ~= nil) and (channel ~= nil) ) then
            if( channel == string.lower(TRIVIABOT_CONFIG["settings"]["channel"]) and TRIVIABOT_ACCEPT_ANSWERS ) then
				TriviaBot_CheckAnswer(player, msg);
            end
		end
    elseif(event == "CHAT_MSG_WHISPER") then
        local msg = arg1;
        local player = arg2;
        
	    if(string.lower(msg) == "!score")then
            TriviaBot_ReturnUserScore(player);
        end
	elseif ((event == "CHAT_MSG_SAY" or event == "CHAT_MSG_GUILD" or event == "CHAT_MSG_YELL" or event == "CHAT_MSG_RAID_LEADER"
		 or event == "CHAT_MSG_RAID" or event == "CHAT_MSG_PARTY") and TRIVIABOT_ACCEPT_ANSWERS) then
		
		-- Something was said, and the bot is on
		local msg = arg1;
		local player = arg2;

        if( (msg and msg ~= nil) and (player and player ~= nil)) then
			TriviaBot_CheckAnswer(player, msg);
		end
	elseif(event == "CHAT_MSG_SYSTEM" and (arg1 == ERR_TOO_MANY_CHAT_CHANNELS)) then
		TriviaBot_UnSchedule("all");
		TriviaBot_Print("Leave another channel before setting a new Private Channel");
	elseif (event == "RETRY_CHANNEL_CHANGE") then
		TriviaBot_ChangePrivateChannel();
	elseif (event == "NEXT_QUESTION") then
		TRIVIABOT_ACCEPT_ANSWERS = true;
		--TriviaBotGUISkipButton:Enable();
		TriviaBot_AskQuestion();
	elseif (event == "QUESTION_TIMEOUT") then TriviaBot_QuestionTimeout();
	elseif (event == "QUESTION_WARN") then TriviaBot_Send(TRIVIABOT_CONFIG["settings"]["question_timewarn"] .. " seconds left!");
	elseif (event == "REPORT_SCORERS") then TriviaBot_Report("gamereport");
	elseif (event == "END_REPORT") then TriviaBot_Report("endreport");
	elseif (event == "START_GAME") then TriviaBot_Start();
	elseif (event == "STOP_GAME") then TriviaBot_Stop();
	elseif (event == "SHOW_ANSWER") then
        local count = 0;
        local answers = "";
        for i = 1, #TRIVIABOT_Questions[TRIVIABOT_CONFIG["settings"]["qlist"]]['ANSWERS'][TRIVIABOT_QUESTION_ORDER[TRIVIABOT_ACTIVE_QUESTION]]do
            count = count + 1;
            if(TRIVIABOT_Questions[TRIVIABOT_CONFIG["settings"]["qlist"]]['ANSWERS'][TRIVIABOT_QUESTION_ORDER[TRIVIABOT_ACTIVE_QUESTION]][i + 1] ~= nil) then
                answers = answers .. TRIVIABOT_Questions[TRIVIABOT_CONFIG["settings"]["qlist"]]['ANSWERS'][TRIVIABOT_QUESTION_ORDER[TRIVIABOT_ACTIVE_QUESTION]][i] .. " & ";
            else
                answers = answers .. TRIVIABOT_Questions[TRIVIABOT_CONFIG["settings"]["qlist"]]['ANSWERS'][TRIVIABOT_QUESTION_ORDER[TRIVIABOT_ACTIVE_QUESTION]][i];
            end
        end
        if(count == 1) then
    	    TriviaBot_Send("The correct answer was: " .. answers);
        else
            TriviaBot_Send("The correct answers were: " .. answers);
        end
	end
end

----------------------------------------------------------------------------
-- OnUpdate event is run in cycles
----------------------------------------------------------------------------
function TriviaBot_OnUpdate(elapsed)
	-- OnUpdate
	this.TimeSinceLastUpdate = this.TimeSinceLastUpdate + elapsed; 	

	if (this.TimeSinceLastUpdate > TRIVIABOT_CONFIG["settings"]["update_interval"]) then
		TriviaBot_DoSchedule();
		this.TimeSinceLastUpdate = 0;
	end
end

----------------------------------------------------------------------------
-- Print information to chat box for player
----------------------------------------------------------------------------
function TriviaBot_Print(msg)
	-- Check the default frame exists
	if (DEFAULT_CHAT_FRAME) then
		-- Format the message
		msg = TRIVIABOT_COLOR .. "[TriviaBot]: " .. WHITE .. msg;
		DEFAULT_CHAT_FRAME:AddMessage(msg);
	end
end

----------------------------------------------------------------------------
-- Print error information in chat box
----------------------------------------------------------------------------
function TriviaBot_PrintError(msg)
	-- Check the default frame exists
	if (DEFAULT_CHAT_FRAME) then
		-- Format the message
		msg = RED .. "--ERROR-- [TriviaBot]: " .. WHITE .. msg;
		DEFAULT_CHAT_FRAME:AddMessage(msg);
	end
end

----------------------------------------------------------------------------
-- Print current game stats
----------------------------------------------------------------------------
function TriviaBot_PrintGameScores()
	local results = false;
	local printed = 0;
	local sortTable = {};
	
   	TriviaBot_Send("----Current game fastest----");
	if(not TRIVIABOT_CONFIG["gamequickest"]["holder"]) then
		TriviaBot_Send("No quickest time recorded.");
	else
		TriviaBot_Send(TRIVIABOT_CONFIG["gamequickest"]["holder"] .. " in " .. TRIVIABOT_CONFIG["gamequickest"]["time"] .. " sec");
	end

	-- Print game scores
	TriviaBot_Send("----Current game scores----");
	
	-- Copy players to a sorted table
	for player, score in pairs(TRIVIABOT_CONFIG["gamescores"]) do
		results = true;
		table.insert(sortTable, {["player"] = player, ["score"] = score});
	end 
	
	-- Sort that sort table!
	table.sort(sortTable, function(v1, v2)
		return v1["score"] > v2["score"];
		end);
		
	-- Loop through and print and count
	for id, record in pairs(sortTable) do
		if(printed < TRIVIABOT_CONFIG["settings"]["score_count"]) then
			TriviaBot_Send("[" .. id .. "]: " .. record["player"] .. ": " .. record["score"]);    
		end
		printed = printed + 1;
	end
		
	if(printed == 0)then
		TriviaBot_Send("No current game player data found");
		results = false;
	end
end

----------------------------------------------------------------------------
-- Print all time stats
----------------------------------------------------------------------------
function TriviaBot_PrintAllTimeScores()
    local results = false;
	local printed = 0;
	local sortTable = {};
	
   	-- Print fastest times
	TriviaBot_Send("----All time fastest----");
	if(not TRIVIABOT_CONFIG["alltimequickest"]["holder"]) then
		TriviaBot_Send("No quickest time recorded.");
	else
		TriviaBot_Send(TRIVIABOT_CONFIG["alltimequickest"]["holder"] .. " in " .. TRIVIABOT_CONFIG["alltimequickest"]["time"] .. " sec");
	end
	-- Print game scores
	TriviaBot_Send("----All time scores----");
	
	-- Copy players to a sorted table
	for player, score in pairs(TRIVIABOT_CONFIG["alltimescores"]) do
		results = true;
		table.insert(sortTable, {["player"] = player, ["score"] = score});
	end 
	
	-- Sort that sort table!
	table.sort(sortTable, function(v1, v2)
		return v1["score"] > v2["score"];
		end);
		
	-- Loop through and print and count
	for id, record in pairs(sortTable) do
		if(printed < TRIVIABOT_CONFIG["settings"]["score_count"]) then
			TriviaBot_Send("[" .. id .. "]: " .. record["player"] .. ": " .. record["score"]);    
		end
		printed = printed + 1;
	end
		
	if(printed == 0)then
		TriviaBot_Send("No current game player data found");
		results = false;
	end	
end

----------------------------------------------------------------------------
-- Answers question and prepares next one
----------------------------------------------------------------------------
function TriviaBot_QuestionTimeout()
    TriviaBot_Send("Time is up! No correct answer was given.");
	
	TriviaBot_EndQuestion(TRIVIABOT_CONFIG["settings"]["show_answers"]);	
end

----------------------------------------------------------------------------
-- Randomize the trivia questions
----------------------------------------------------------------------------
function TriviaBot_Randomize()
    -- Randomise the order of the questions
	TRIVIABOT_QUESTION_ORDER = {};
	
  	-- Initialise the table
	local noOfQuestions = table.getn(TRIVIABOT_Questions[TRIVIABOT_CONFIG["settings"]["qlist"]].QUESTIONS);
	local n = 1;
	
	while (n <= noOfQuestions) do
		TRIVIABOT_QUESTION_ORDER[n] = n;
		n = n + 1;
	end
	
	local tmp, random;
	local i;
	local j = 5;
	
	while (j > 0) do
		i = 1;
		-- Swap each element with a random element
		while (i <= noOfQuestions) do
			random = math.random(noOfQuestions);
			tmp = TRIVIABOT_QUESTION_ORDER[i];
			TRIVIABOT_QUESTION_ORDER[i] = TRIVIABOT_QUESTION_ORDER[random]
			TRIVIABOT_QUESTION_ORDER[random] = tmp;
			i = i + 1;
		end
		
		-- Decrement J
		j = j - 1
	end
end


----------------------------------------------------------------------------
-- Print report to the screen
----------------------------------------------------------------------------
function TriviaBot_Report(type)
		if (type == "gamereport") then
		-- Sort the table
		local TRIVIABOT_SORTED = {};
		
		local exists;
		for player, score in pairs(TRIVIABOT_CONFIG['gamescores']) do
			-- Add them to the sorting table
			exists = true;
			table.insert(TRIVIABOT_SORTED, {["player"] = player, ["score"] = score});
		end
		
		table.sort(TRIVIABOT_SORTED, function(v1, v2)
			return v1["score"] > v2["score"];
			end);
		
		if (exists) then
			TriviaBot_Send("Standing so far:");
		else
			TriviaBot_Send("Standing: No points earnt yet!");
		end
		
		-- Report the top 5 scorers
		for id, record in pairs(TRIVIABOT_SORTED) do
			if (id <= 5) then
				-- Report the top 3
				
				-- Ensure correct grammar.
				local ess = "s";
				if (record["score"] == 1) then
					ess = "";
				end
	
				TriviaBot_Send(id .. "]: " .. record["player"] .. " (" .. record["score"] .." point" .. ess .. ")");
				
			end
		end
		
		-- Speed record holder
		if (TRIVIABOT_CONFIG["gamequickest"]["holder"] ~= "noone" and TRIVIABOT_CONFIG["gamequickest"]["holder"] ~= nil) then
			TriviaBot_Send("Speed Record: " .. TRIVIABOT_CONFIG['gamequickest']["holder"] .. " in " .. TRIVIABOT_CONFIG['gamequickest']["time"] .. " sec");
		end
	elseif(type == "endreport") then
		-- Show the winner
		-- Sort the table
		local TRIVIABOT_SORTED = {};
		
		local exists;
		for player, score in pairs(TRIVIABOT_CONFIG['gamescores']) do
			-- Add them to the sorting table
			exists = true;
			table.insert(TRIVIABOT_SORTED, {["player"] = player, ["score"] = score});
		end
		
		if (exists) then
			TriviaBot_Send("GAME OVER! Final standings:");
		else
			TriviaBot_Send("GAME OVER! Nobody scored!");
		end
		
		table.sort(TRIVIABOT_SORTED, function(v1, v2)
			return v1["score"] > v2["score"];
			end);
	
		-- Report the top 3 scorers
		for id, record in pairs(TRIVIABOT_SORTED) do
			if (id <= 2) then
				-- Report the winner and runner up
				
				-- Ensure correct grammar.
				local ess = "s";
				if (record["score"] == 1) then
					ess = "";
				end
				
				-- Announce standing
				local standing;
				if (id == 1) then
					standing = "WINNER";
				elseif (id == 2) then
					standing = "RUNNER UP"
				end
	
				TriviaBot_Send(standing .. ": " .. record["player"] .. " (" .. record["score"] .." point" .. ess .. ")");
				
			end
		end
		
		-- Speed record holder
		if (TRIVIABOT_CONFIG['gamequickest']["holder"] ~= "noone") then
			TriviaBot_Send("QUICKEST FINGERS: " .. TRIVIABOT_CONFIG['gamequickest']["holder"] .. " in " .. TRIVIABOT_CONFIG['gamequickest']["time"] .. " sec");
		end
	end
end

----------------------------------------------------------------------------
-- Gives user their point information.
----------------------------------------------------------------------------
function TriviaBot_ReturnUserScore(user)
    --TRIVIABOT_CONFIG['alltimescores'][player]

end

----------------------------------------------------------------------------
-- Schedule an event
----------------------------------------------------------------------------
function TriviaBot_Schedule(name, time)
	thisEvent = {["name"] = name, ["time"] = GetTime() + time};
	table.insert(TRIVIABOT_SCHEDULE, thisEvent);
end

----------------------------------------------------------------------------
-- Print trivia in primary box
----------------------------------------------------------------------------
function TriviaBot_Send(msg)
	-- Send a message to the trivia channel
	
	-- Prepend the trivia tag to each message 
	local triviaMsg;
	triviaMsg = "[TriviaBot]: " .. msg;
	
	-- Get index of current channel
	local channelName = TRIVIABOT_CONFIG["settings"]["channel"];
	local channelType = TRIVIABOT_CONFIG["settings"]["channel_type"] ;
	local channelIndex = 0;
		
	-- If sending it to a channel get the name ID
	if(channelType == "CHANNEL") then
		if(channelName == "" or channelName == nil) then
			TriviaBot_PrintError("You must have a channel name set!");
			TriviaBot_Stop(true);
			return;
		end
		channelIndex = GetChannelName(channelName);
		
		-- If you're not in the channel, join it to play.
		if(channelIndex == 0) then
			TriviaBot_Print("You are not in channel: '" .. channelName .. "'. Auto-joining channel...");
			JoinChannelByName(TRIVIABOT_CONFIG["settings"]["channel"], nil, DEFAULT_CHAT_FRAME);
			ChatFrame_AddChannel(DEFAULT_CHAT_FRAME, channelName);
			channelIndex = GetChannelName(channelName);
			
			TriviaBot_Stop(true);
			TriviaBot_Print("TriviaBot must be restarted. Restarting in 5 seconds...");
			TriviaBot_Schedule("START_GAME", 5);
		end
		
		if(channelIndex == 0) then
			TriviaBot_PrintError("No valid channel set!");
			TriviaBot_Stop(true);
		else
			SendChatMessage(triviaMsg, channelType, nil, channelIndex);
		end
	elseif(channelType == "WHISPER") then
		-- Whisper to person (channelName = person)
        SendChatMessage(triviaMsg, channelType, nil, channelName);
	else
		SendChatMessage(triviaMsg, channelType, nil, channelIndex);
	end
end

----------------------------------------------------------------------------
-- Skip current question
----------------------------------------------------------------------------
function TriviaBot_SkipQuestion()
	-- Skip a question
	if (TRIVIABOT_RUNNING) then
		TriviaBot_Send("Question was skipped.");
		TriviaBot_UnSchedule("all");
		TriviaBotGUI_SkipButton:Disable();
		TRIVIABOT_ACCEPT_ANSWERS = false;
			
		-- Show the answer anyway (for those that wanted to know)
		if (TRIVIABOT_CONFIG["settings"]["show_answers"]) then
			TriviaBot_Schedule("SHOW_ANSWER", 2);
		end
			
		-- Schedule the next question
		TriviaBot_Schedule("NEXT_QUESTION", 6);
		
		TriviaBot_Print("Question skipped");
	else
		TriviaBot_PrintError("No game running!");
	end
end


----------------------------------------------------------------------------
-- Start trivia session
----------------------------------------------------------------------------
function TriviaBot_Start()
    -- Set Running
	TRIVIABOT_RUNNING = true;
	
	-- Announce the start
    TriviaBot_Send("--World of Warcraft Trivia powered by TriviaBot--");
    if(not TRIVIABOT_RUNNING) then
		-- TriviaBot bailed on us due to a channel issue, stop our stuff.
		return;
    end
    TriviaBot_Send("Using Trivia from database: " .. TRIVIABOT_Questions[TRIVIABOT_CONFIG["settings"]["qlist"]]['DESCRIPTION'] .. " Author(s): " .. TRIVIABOT_Questions[TRIVIABOT_CONFIG["settings"]["qlist"]]["AUTHOR"]);
    		
	TriviaBot_Print("First question coming up!");
	
	-- Schedule start
	TriviaBot_Schedule("NEXT_QUESTION", 7);
	
	-- Clear scores
    TRIVIABOT_CONFIG["gamescores"] = {};
    TRIVIABOT_CONFIG["gamequickest"] = {};
	TRIVIABOT_REPORT_COUNTER = 0;
	
	-- Reset Round and Report Counters
	TRIVIABOT_REPORT_COUNTER = 0;
	TRIVIABOT_ROUND_COUNTER = 0;
	
	-- GUI Functions:
	TriviaBotGUI_StartStopButton:SetText("Stop TriviaBot");
end

----------------------------------------------------------------------------
-- Stop trivia session
----------------------------------------------------------------------------
function TriviaBot_Stop(supressMessage)
	-- Clear all scheduled events
    TriviaBot_UnSchedule("all");
    TRIVIABOT_ACCEPT_ANSWERS = false;
	TRIVIABOT_RUNNING = false;
	
	if (not supressMessage) then
		TriviaBot_Send("Trivia stopped.");
	end
	
	TriviaBot_Print("Trivia stopped.")
	
	-- GUI Updates
	TriviaBotGUI_StartStopButton:SetText("Start TriviaBot");
	TriviaBotGUI_SkipButton:Disable();
end

----------------------------------------------------------------------------
-- Removes an event from the Scheduler
----------------------------------------------------------------------------
function TriviaBot_UnSchedule(id)
	-- Unschedule an event
	if (id == "all") then
		TRIVIABOT_SCHEDULE = {};
	else
		table.remove(TRIVIABOT_SCHEDULE, id);
	end
end

----------------------------------------------------------------------------
--
-- GUI OBJECTS
--
----------------------------------------------------------------------------



----------------------------------------------------------------------------
--
-- GUI FUNCTIONS
--
----------------------------------------------------------------------------

----------------------------------------------------------------------------
-- All Time Scores Button OnClick Event
----------------------------------------------------------------------------
function TriviaBotGUI_AllTimeScoresButton_OnClick()
	TriviaBot_PrintAllTimeScores();
end

----------------------------------------------------------------------------
-- Channel Enter Press (lose focus)
----------------------------------------------------------------------------
function TriviaBotGUI_Channel_OnEnterPressed()
	this:ClearFocus();
end

----------------------------------------------------------------------------
-- Channel Text Change
----------------------------------------------------------------------------
function TriviaBotGUI_Channel_OnTextChanged()
	TRIVIABOT_CONFIG["settings"]["channel"] = this:GetText();
end

----------------------------------------------------------------------------
-- Initialize the ChannelType drop down
----------------------------------------------------------------------------
function TriviaBotGUI_ChannelType_Initialize()
	local info;
	
	-- Say channel
	info = UIDropDownMenu_CreateInfo();
	info.value = "SAY";
	info.text = "Say";
	info.func = function() TriviaBotGUI_ChannelType_OnClick() end;
	UIDropDownMenu_AddButton(info);
	
	-- Yell channel
	info = UIDropDownMenu_CreateInfo();
	info.value = "YELL";
	info.text = "Yell";
	info.func = function() TriviaBotGUI_ChannelType_OnClick() end;
	UIDropDownMenu_AddButton(info);
	
	-- Party channel
	info = UIDropDownMenu_CreateInfo();
	info.value = "PARTY";
	info.text = "Party";
	info.func = function() TriviaBotGUI_ChannelType_OnClick() end;
	UIDropDownMenu_AddButton(info);
	
	-- Raid channel
	info = UIDropDownMenu_CreateInfo();
	info.value = "RAID";
	info.text = "Raid";
	info.func = function() TriviaBotGUI_ChannelType_OnClick() end;
	UIDropDownMenu_AddButton(info);
	
	-- Raid Warning channel
	info = UIDropDownMenu_CreateInfo();
	info.value = "RAID_WARNING";
	info.text = "Raid Warning";
	info.func = function() TriviaBotGUI_ChannelType_OnClick() end;
	UIDropDownMenu_AddButton(info);
	
	-- BattleGround channel
	info = UIDropDownMenu_CreateInfo();
	info.value = "BATTLEGROUND";
	info.text = "Battleground";
	info.func = function() TriviaBotGUI_ChannelType_OnClick() end;
	UIDropDownMenu_AddButton(info);
	
	-- Guild channel
	info = UIDropDownMenu_CreateInfo();
	info.value = "GUILD";
	info.text = "Guild";
	info.func = function() TriviaBotGUI_ChannelType_OnClick() end;
	UIDropDownMenu_AddButton(info);
	
	-- Whisper channel
	info = UIDropDownMenu_CreateInfo();
	info.value = "WHISPER";
	info.text = "Whisper";
	info.func = function() TriviaBotGUI_ChannelType_OnClick() end;
	UIDropDownMenu_AddButton(info);
	
	-- Custom channel
	info = UIDropDownMenu_CreateInfo();
	info.value = "CHANNEL";
	info.text = "Custom";
	info.func = function() TriviaBotGUI_ChannelType_OnClick() end;
	UIDropDownMenu_AddButton(info);
	
	
	selected = TRIVIABOT_CONFIG["settings"]["channel_type"];
    
    UIDropDownMenu_SetText(selected, TriviaBotGUI_ChannelType);
end

----------------------------------------------------------------------------
-- ChannelType OnClick Event
----------------------------------------------------------------------------
function TriviaBotGUI_ChannelType_OnClick()
	-- Change the setting
	TRIVIABOT_CONFIG["settings"]["channel_type"] = this.value;
	
	-- Update the text on the drop down menu
	UIDropDownMenu_SetText(TRIVIABOT_CONFIG["settings"]["channel_type"], TriviaBotGUI_ChannelType);
end

----------------------------------------------------------------------------
-- Game Scores Button OnClick Event
----------------------------------------------------------------------------
function TriviaBotGUI_GameScoresButton_OnClick()
	TriviaBot_PrintGameScores();
end

----------------------------------------------------------------------------
-- Initialize all objects for GUI
----------------------------------------------------------------------------
function TriviaBotGUI_Initialize()
	-- XML is a bloody nightmare for this, give me good old logical code	
	
	-- Set the main GUI screen
	TriviaBotGUI:ClearAllPoints();
	TriviaBotGUI:SetWidth(350);
	TriviaBotGUI:SetHeight(445);
	TriviaBotGUI:SetPoint("CENTER");
	TriviaBotGUI:SetBackdrop(
		{
			bgFile = "Interface/Tooltips/UI-Tooltip-Background", 
			edgeFile = "Interface/DialogFrame/UI-DialogBox-Border", 
			tile = true, tileSize = 32, edgeSize = 16, 
			insets = { left = 5, right = 5, top = 5, bottom = 5}
		});
	TriviaBotGUI:SetBackdropColor(0,0,0,1);
	
	
	-- Load Header
	TriviaBotGUI_HeaderTexture:ClearAllPoints();
	TriviaBotGUI_HeaderTexture:SetWidth(256);
	TriviaBotGUI_HeaderTexture:SetHeight(64);
	TriviaBotGUI_HeaderTexture:SetPoint("TOP", TriviaBotGUI, "TOP", 0, 12);
	TriviaBotGUI_HeaderLabel:ClearAllPoints();
	TriviaBotGUI_HeaderLabel:SetPoint("TOP", TriviaBotGUI_HeaderTexture, "TOP", 0, -14);
	
	-- Close Button
	TriviaBotGUI_CloseButton:ClearAllPoints();
	TriviaBotGUI_CloseButton:SetPoint("TOPRIGHT", TriviaBotGUI, "TOPRIGHT", 0, 0);
	TriviaBotGUI_CloseButton:SetScript("OnClick", function() TriviaBotGUI:Hide(); end);
	
	-- Question List
	TriviaBotGUI_QuestionList:ClearAllPoints();
	UIDropDownMenu_SetWidth(150, TriviaBotGUI_QuestionList);
	UIDropDownMenu_SetButtonWidth(20, TriviaBotGUI_QuestionList);
	TriviaBotGUI_QuestionList:SetPoint("TOPLEFT", TriviaBotGUI, "TOPLEFT", 160, -30);
	UIDropDownMenu_Initialize(TriviaBotGUI_QuestionList, TriviaBotGUI_QuestionList_Initialize);
	
	TriviaBotGUI_QuestionListLabel:ClearAllPoints();
	TriviaBotGUI_QuestionListLabel:SetPoint("TOPRIGHT", TriviaBotGUI_QuestionList, "TOPLEFT", -5, -10);
	
	-- Channel Type List
	TriviaBotGUI_ChannelType:ClearAllPoints();
	UIDropDownMenu_SetWidth(150, TriviaBotGUI_ChannelType);
	UIDropDownMenu_SetButtonWidth(20, TriviaBotGUI_ChannelType);
	TriviaBotGUI_ChannelType:SetPoint("TOPLEFT", TriviaBotGUI_QuestionList, "TOPLEFT", 0, -30);
	UIDropDownMenu_Initialize(TriviaBotGUI_ChannelType, TriviaBotGUI_ChannelType_Initialize);
	
	TriviaBotGUI_ChannelTypeLabel:ClearAllPoints();
	TriviaBotGUI_ChannelTypeLabel:SetPoint("TOPRIGHT", TriviaBotGUI_ChannelType, "TOPLEFT", -5, -10);
	
	-- Channel TextBox
	TriviaBotGUI_Channel:ClearAllPoints();
	TriviaBotGUI_Channel:SetWidth(100);
	TriviaBotGUI_Channel:SetHeight(32);
	TriviaBotGUI_Channel:SetPoint("TOPLEFT", TriviaBotGUI_ChannelType, "TOPLEFT", 25, -30);
	TriviaBotGUI_Channel:SetScript("OnTextChanged", TriviaBotGUI_Channel_OnTextChanged);
	TriviaBotGUI_Channel:SetScript("OnEnterPressed", TriviaBotGUI_Channel_OnEnterPressed);
	
	TriviaBotGUI_ChannelLabel:ClearAllPoints();
	TriviaBotGUI_ChannelLabel:SetPoint("TOPRIGHT", TriviaBotGUI_Channel, "TOPLEFT", -25, -10);
	
	-- Channel TextBox Textures
	TriviaBotGUI_ChannelTexture1:ClearAllPoints();
	TriviaBotGUI_ChannelTexture1:SetWidth(75);
	TriviaBotGUI_ChannelTexture1:SetHeight(32);
	TriviaBotGUI_ChannelTexture1:SetPoint("LEFT", -15, 0);
	TriviaBotGUI_ChannelTexture1:SetTexCoord(0, 0.2, 0, 1.0);
	TriviaBotGUI_ChannelTexture2:ClearAllPoints();
	TriviaBotGUI_ChannelTexture2:SetWidth(75);
	TriviaBotGUI_ChannelTexture2:SetHeight(32);
	TriviaBotGUI_ChannelTexture2:SetPoint("RIGHT", 10, 0);
	TriviaBotGUI_ChannelTexture2:SetTexCoord(0.7, 1.0, 0, 1.0);
	
	-- Question Interval
	TriviaBotGUI_QuestionInterval:ClearAllPoints();
	TriviaBotGUI_QuestionInterval:SetWidth(100);
	TriviaBotGUI_QuestionInterval:SetHeight(32);
	TriviaBotGUI_QuestionInterval:SetPoint("TOPLEFT", TriviaBotGUI_Channel, "BOTTOMLEFT", 0, 0);
	TriviaBotGUI_QuestionInterval:SetScript("OnTextChanged", TriviaBotGUI_QuestionInterval_OnTextChanged);
	TriviaBotGUI_QuestionInterval:SetScript("OnEnterPressed", TriviaBotGUI_Channel_OnEnterPressed);
	TriviaBotGUI_QuestionIntervalLabel:ClearAllPoints();
	TriviaBotGUI_QuestionIntervalLabel:SetPoint("TOPRIGHT", TriviaBotGUI_QuestionInterval, "TOPLEFT", -25, -10);
	
	-- Question Interval Textures
	TriviaBotGUI_QuestionIntervalTexture1:ClearAllPoints();
	TriviaBotGUI_QuestionIntervalTexture1:SetWidth(75);
	TriviaBotGUI_QuestionIntervalTexture1:SetHeight(32);
	TriviaBotGUI_QuestionIntervalTexture1:SetPoint("LEFT", -15, 0);
	TriviaBotGUI_QuestionIntervalTexture1:SetTexCoord(0, 0.2, 0, 1.0);
	TriviaBotGUI_QuestionIntervalTexture2:ClearAllPoints();
	TriviaBotGUI_QuestionIntervalTexture2:SetWidth(75);
	TriviaBotGUI_QuestionIntervalTexture2:SetHeight(32);
	TriviaBotGUI_QuestionIntervalTexture2:SetPoint("RIGHT", 10, 0);
	TriviaBotGUI_QuestionIntervalTexture2:SetTexCoord(0.7, 1.0, 0, 1.0);
	
	-- Question Timeout
	TriviaBotGUI_QuestionTimeOut:ClearAllPoints();
	TriviaBotGUI_QuestionTimeOut:SetWidth(100);
	TriviaBotGUI_QuestionTimeOut:SetHeight(32);
	TriviaBotGUI_QuestionTimeOut:SetPoint("TOPLEFT", TriviaBotGUI_QuestionInterval, "BOTTOMLEFT", 0, 0);
	TriviaBotGUI_QuestionTimeOut:SetScript("OnTextChanged", TriviaBotGUI_QuestionTimeOut_OnTextChanged);
	TriviaBotGUI_QuestionTimeOut:SetScript("OnEnterPressed", TriviaBotGUI_Channel_OnEnterPressed);
	TriviaBotGUI_QuestionTimeOutLabel:ClearAllPoints();
	TriviaBotGUI_QuestionTimeOutLabel:SetPoint("TOPRIGHT", TriviaBotGUI_QuestionTimeOut, "TOPLEFT", -25, -10);
	
	-- Question Timeout Textures
	TriviaBotGUI_QuestionTimeOutTexture1:ClearAllPoints();
	TriviaBotGUI_QuestionTimeOutTexture1:SetWidth(75);
	TriviaBotGUI_QuestionTimeOutTexture1:SetHeight(32);
	TriviaBotGUI_QuestionTimeOutTexture1:SetPoint("LEFT", -15, 0);
	TriviaBotGUI_QuestionTimeOutTexture1:SetTexCoord(0, 0.2, 0, 1.0);
	TriviaBotGUI_QuestionTimeOutTexture2:ClearAllPoints();
	TriviaBotGUI_QuestionTimeOutTexture2:SetWidth(75);
	TriviaBotGUI_QuestionTimeOutTexture2:SetHeight(32);
	TriviaBotGUI_QuestionTimeOutTexture2:SetPoint("RIGHT", 10, 0);
	TriviaBotGUI_QuestionTimeOutTexture2:SetTexCoord(0.7, 1.0, 0, 1.0);
	
	-- Question Warning
	TriviaBotGUI_QuestionWarning:ClearAllPoints();
	TriviaBotGUI_QuestionWarning:SetWidth(100);
	TriviaBotGUI_QuestionWarning:SetHeight(32);
	TriviaBotGUI_QuestionWarning:SetPoint("TOPLEFT", TriviaBotGUI_QuestionTimeOut, "BOTTOMLEFT", 0, 0);
	TriviaBotGUI_QuestionWarning:SetScript("OnTextChanged", TriviaBotGUI_QuestionWarning_OnTextChanged);
	TriviaBotGUI_QuestionWarning:SetScript("OnEnterPressed", TriviaBotGUI_Channel_OnEnterPressed);
	TriviaBotGUI_QuestionWarningLabel:ClearAllPoints();
	TriviaBotGUI_QuestionWarningLabel:SetPoint("TOPRIGHT", TriviaBotGUI_QuestionWarning, "TOPLEFT", -25, -10);
	
	-- Question Warning Textures
	TriviaBotGUI_QuestionWarningTexture1:ClearAllPoints();
	TriviaBotGUI_QuestionWarningTexture1:SetWidth(75);
	TriviaBotGUI_QuestionWarningTexture1:SetHeight(32);
	TriviaBotGUI_QuestionWarningTexture1:SetPoint("LEFT", -15, 0);
	TriviaBotGUI_QuestionWarningTexture1:SetTexCoord(0, 0.2, 0, 1.0);
	TriviaBotGUI_QuestionWarningTexture2:ClearAllPoints();
	TriviaBotGUI_QuestionWarningTexture2:SetWidth(75);
	TriviaBotGUI_QuestionWarningTexture2:SetHeight(32);
	TriviaBotGUI_QuestionWarningTexture2:SetPoint("RIGHT", 10, 0);
	TriviaBotGUI_QuestionWarningTexture2:SetTexCoord(0.7, 1.0, 0, 1.0);
	
	-- Score Count TextBox
	TriviaBotGUI_ScoreCount:ClearAllPoints();
	TriviaBotGUI_ScoreCount:SetWidth(100);
	TriviaBotGUI_ScoreCount:SetHeight(32);
	TriviaBotGUI_ScoreCount:SetPoint("TOPLEFT", TriviaBotGUI_QuestionWarning, "BOTTOMLEFT", 0, 0);
	TriviaBotGUI_ScoreCount:SetScript("OnTextChanged", TriviaBotGUI_ScoreCount_OnTextChanged);
	TriviaBotGUI_ScoreCount:SetScript("OnEnterPressed", TriviaBotGUI_Channel_OnEnterPressed);
	TriviaBotGUI_ScoreCountLabel:ClearAllPoints();
	TriviaBotGUI_ScoreCountLabel:SetPoint("TOPRIGHT", TriviaBotGUI_ScoreCount, "TOPLEFT", -25, -10);
	
	-- Score Count Textures
	TriviaBotGUI_ScoreCountTexture1:ClearAllPoints();
	TriviaBotGUI_ScoreCountTexture1:SetWidth(75);
	TriviaBotGUI_ScoreCountTexture1:SetHeight(32);
	TriviaBotGUI_ScoreCountTexture1:SetPoint("LEFT", -15, 0);
	TriviaBotGUI_ScoreCountTexture1:SetTexCoord(0, 0.2, 0, 1.0);
	TriviaBotGUI_ScoreCountTexture2:ClearAllPoints();
	TriviaBotGUI_ScoreCountTexture2:SetWidth(75);
	TriviaBotGUI_ScoreCountTexture2:SetHeight(32);
	TriviaBotGUI_ScoreCountTexture2:SetPoint("RIGHT", 10, 0);
	TriviaBotGUI_ScoreCountTexture2:SetTexCoord(0.7, 1.0, 0, 1.0);
	
	-- Score Frequency TextBox
	TriviaBotGUI_ScoreFrequency:ClearAllPoints();
	TriviaBotGUI_ScoreFrequency:SetWidth(100);
	TriviaBotGUI_ScoreFrequency:SetHeight(32);
	TriviaBotGUI_ScoreFrequency:SetPoint("TOPLEFT", TriviaBotGUI_ScoreCount, "BOTTOMLEFT", 0, 0);
	TriviaBotGUI_ScoreFrequency:SetScript("OnTextChanged", TriviaBotGUI_ScoreFrequency_OnTextChanged);
	TriviaBotGUI_ScoreFrequency:SetScript("OnEnterPressed", TriviaBotGUI_Channel_OnEnterPressed);
	TriviaBotGUI_ScoreFrequencyLabel:ClearAllPoints();
	TriviaBotGUI_ScoreFrequencyLabel:SetPoint("TOPRIGHT", TriviaBotGUI_ScoreFrequency, "TOPLEFT", -25, -10);
	
	-- Score Frequency Textures
	TriviaBotGUI_ScoreFrequencyTexture1:ClearAllPoints();
	TriviaBotGUI_ScoreFrequencyTexture1:SetWidth(75);
	TriviaBotGUI_ScoreFrequencyTexture1:SetHeight(32);
	TriviaBotGUI_ScoreFrequencyTexture1:SetPoint("LEFT", -15, 0);
	TriviaBotGUI_ScoreFrequencyTexture1:SetTexCoord(0, 0.2, 0, 1.0);
	TriviaBotGUI_ScoreFrequencyTexture2:ClearAllPoints();
	TriviaBotGUI_ScoreFrequencyTexture2:SetWidth(75);
	TriviaBotGUI_ScoreFrequencyTexture2:SetHeight(32);
	TriviaBotGUI_ScoreFrequencyTexture2:SetPoint("RIGHT", 10, 0);
	TriviaBotGUI_ScoreFrequencyTexture2:SetTexCoord(0.7, 1.0, 0, 1.0);
	
	-- ShowAnswer Checkbox
	TriviaBotGUI_ShowAnswerCheckBox:ClearAllPoints();
	TriviaBotGUI_ShowAnswerCheckBox:SetPoint("TOPLEFT", TriviaBotGUI_ScoreFrequency, "BOTTOMLEFT", -80, -5);
	TriviaBotGUI_ShowAnswerCheckBox:SetScript("OnClick", TriviaBotGUI_ShowAnswerCheckBox_OnClick);
	TriviaBotGUI_ShowAnswerCheckBoxLabel:ClearAllPoints();
	TriviaBotGUI_ShowAnswerCheckBoxLabel:SetPoint("TOPLEFT", TriviaBotGUI_ShowAnswerCheckBox, "TOPRIGHT", 5, -10);
	
	-- Report Checkbox
	TriviaBotGUI_ReportCheckBox:ClearAllPoints();
	TriviaBotGUI_ReportCheckBox:SetPoint("TOPLEFT", TriviaBotGUI_ShowAnswerCheckBox, "BOTTOMLEFT", 0, 0);
	TriviaBotGUI_ReportCheckBox:SetScript("OnClick", TriviaBotGUI_ReportCheckBox_OnClick);
	TriviaBotGUI_ReportCheckBoxLabel:ClearAllPoints();
	TriviaBotGUI_ReportCheckBoxLabel:SetPoint("TOPLEFT", TriviaBotGUI_ReportCheckBox, "TOPRIGHT", 5, -10);
	
	-- Start/Stop button
	TriviaBotGUI_StartStopButton:ClearAllPoints();
	TriviaBotGUI_StartStopButton:SetWidth(145);
	TriviaBotGUI_StartStopButton:SetPoint("TOPLEFT", TriviaBotGUI_ReportCheckBox, "BOTTOMLEFT", 0, 0);
	TriviaBotGUI_StartStopButton:SetScript("OnClick", TriviaBotGUI_StartStopButton_OnClick);
	
	-- Skip button
	TriviaBotGUI_SkipButton:ClearAllPoints();
	TriviaBotGUI_SkipButton:SetWidth(145);
	TriviaBotGUI_SkipButton:SetPoint("TOPLEFT", TriviaBotGUI_StartStopButton, "BOTTOMLEFT", 0, 0);
	TriviaBotGUI_SkipButton:SetScript("OnClick", TriviaBotGUI_SkipButton_OnClick);
	TriviaBotGUI_SkipButton:Disable();
	
	TriviaBotGUI_GameScoresButton:ClearAllPoints();
	TriviaBotGUI_GameScoresButton:SetWidth(145);
	TriviaBotGUI_GameScoresButton:SetPoint("TOPLEFT", TriviaBotGUI_SkipButton, "BOTTOMLEFT", 0, 0);
	TriviaBotGUI_GameScoresButton:SetScript("OnClick", TriviaBotGUI_GameScoresButton_OnClick);
	
	TriviaBotGUI_AllTimeScoresButton:ClearAllPoints();
	TriviaBotGUI_AllTimeScoresButton:SetWidth(145);
	TriviaBotGUI_AllTimeScoresButton:SetPoint("TOPLEFT", TriviaBotGUI_GameScoresButton, "BOTTOMLEFT", 0, 0);
	TriviaBotGUI_AllTimeScoresButton:SetScript("OnClick", TriviaBotGUI_AllTimeScoresButton_OnClick);
	
	
	TriviaBot_Print("GUI Initialized");
end


----------------------------------------------------------------------------
-- Question Interval OnTextChanged Event
----------------------------------------------------------------------------
function TriviaBotGUI_QuestionInterval_OnTextChanged()
	if(this:GetText() == "") then
		-- If the user is blanking it out, they're going to put a new value (hopefully), so we don't save and just exit.
		return;
	end
	local num = tonumber(this:GetText());
	if(num == nil) then
		this:SetText(TRIVIABOT_CONFIG["settings"]["question_interval"]);
	else
		TRIVIABOT_CONFIG["settings"]["question_interval"] = num 
	end
end

----------------------------------------------------------------------------
-- Question Timeout TextBox OnTextChanged Event
----------------------------------------------------------------------------
function TriviaBotGUI_QuestionTimeOut_OnTextChanged()
	if(this:GetText() == "") then
		-- If the user is blanking it out, they're going to put a new value (hopefully), so we don't save and just exit.
		return;
	end
	local num = tonumber(this:GetText());
	if(num == nil) then
		this:SetText(TRIVIABOT_CONFIG["settings"]["question_timeout"]);
	else
		TRIVIABOT_CONFIG["settings"]["question_timeout"] = num 
	end
end

----------------------------------------------------------------------------
-- Question Warning TextBox OnTextChanged Event
----------------------------------------------------------------------------
function TriviaBotGUI_QuestionWarning_OnTextChanged()
	if(this:GetText() == "") then
		-- If the user is blanking it out, they're going to put a new value (hopefully), so we don't save and just exit.
		return;
	end
	local num = tonumber(this:GetText());
	if(num == nil) then
		this:SetText(TRIVIABOT_CONFIG["settings"]["question_timewarn"]);
	else
		TRIVIABOT_CONFIG["settings"]["question_timewarn"] = num 
	end
end

----------------------------------------------------------------------------
-- Initialize the QuestionList drop down
----------------------------------------------------------------------------
function TriviaBotGUI_QuestionList_Initialize()
	local info;
	local selected = "Error!";
	
	for k,i in ipairs(TRIVIABOT_Questions) do
		-- Create and add button
		info = UIDropDownMenu_CreateInfo();
		info.value = k
		info.text = i["DESCRIPTION"];
		info.func = function() TriviaBotGUI_QuestionList_OnClick() end;
		UIDropDownMenu_AddButton(info);
		
		-- Check if this is selected button
		if (k == TRIVIABOT_CONFIG["settings"]["qlist"]) then
			selected = i["DESCRIPTION"];
		end
    end 
    
    UIDropDownMenu_SetText(selected, TriviaBotGUI_QuestionList);  
end
		
----------------------------------------------------------------------------
-- QuestionList OnClick event
----------------------------------------------------------------------------
function TriviaBotGUI_QuestionList_OnClick()
	-- Load the new trivia
	TriviaBot_LoadTrivia(this.value);
	
	-- Set the pulldown text to the new value
	for k,i in ipairs(TRIVIABOT_Questions) do
		if(k == this.value) then
			UIDropDownMenu_SetText(i["DESCRIPTION"], TriviaBotGUI_QuestionList);
		end
	end
end

----------------------------------------------------------------------------
-- Report Scores CheckBox OnClick Event
----------------------------------------------------------------------------
function TriviaBotGUI_ReportCheckBox_OnClick()
	-- Flip the flag
	TRIVIABOT_CONFIG["settings"]["show_reports"] = not TRIVIABOT_CONFIG["settings"]["show_reports"];
end

----------------------------------------------------------------------------
-- Score Count OnTextChanged Event
----------------------------------------------------------------------------
function TriviaBotGUI_ScoreCount_OnTextChanged()
	if(this:GetText() == "") then
		-- If the user is blanking it out, they're going to put a new value (hopefully), so we don't save and just exit.
		return;
	end
	local num = tonumber(this:GetText());
	if(num == nil) then
		this:SetText(TRIVIABOT_CONFIG["settings"]["score_count"]);
	else
		TRIVIABOT_CONFIG["settings"]["score_count"] = num 
	end
end

----------------------------------------------------------------------------
-- Score Frequency OnTextChanged Event
----------------------------------------------------------------------------
function TriviaBotGUI_ScoreFrequency_OnTextChanged()
	if(this:GetText() == "") then
		-- If the user is blanking it out, they're going to put a new value (hopefully), so we don't save and just exit.
		return;
	end
	local num = tonumber(this:GetText());
	if(num == nil) then
		this:SetText(TRIVIABOT_CONFIG["settings"]["report_interval"]);
	else
		TRIVIABOT_CONFIG["settings"]["report_interval"] = num 
	end
end

----------------------------------------------------------------------------
-- Show Answer CheckBox OnClick Event
----------------------------------------------------------------------------
function TriviaBotGUI_ShowAnswerCheckBox_OnClick()
	-- Flip the flag
	TRIVIABOT_CONFIG["settings"]["show_answers"] = not TRIVIABOT_CONFIG["settings"]["show_answers"];
end

----------------------------------------------------------------------------
-- Skip Click
----------------------------------------------------------------------------
function TriviaBotGUI_SkipButton_OnClick()
	TriviaBot_SkipQuestion();
end

----------------------------------------------------------------------------
-- Start/Stop OnClick
----------------------------------------------------------------------------
function TriviaBotGUI_StartStopButton_OnClick()
	if(TRIVIABOT_RUNNING) then
		TriviaBot_Stop();
	else
		TriviaBot_Start();
	end
end

----------------------------------------------------------------------------
-- Update GUI
----------------------------------------------------------------------------
function TriviaBotGUI_Update()
	-- Display Version
	TriviaBotGUI_HeaderLabel:SetText("TriviaBot " .. TRIVIABOT_VERSION);

	-- Load Question List
	TriviaBotGUI_QuestionList_Initialize();
	
	-- Load Channel Type List
	TriviaBotGUI_ChannelType_Initialize();

	-- Load Channel
	TriviaBotGUI_Channel:SetText(TRIVIABOT_CONFIG["settings"]["channel"]);
	
	TriviaBotGUI_QuestionInterval:SetText(TRIVIABOT_CONFIG["settings"]["question_interval"]);
	TriviaBotGUI_QuestionTimeOut:SetText(TRIVIABOT_CONFIG["settings"]["question_timeout"]);
	TriviaBotGUI_QuestionWarning:SetText(TRIVIABOT_CONFIG["settings"]["question_timewarn"]);
	TriviaBotGUI_ScoreCount:SetText(TRIVIABOT_CONFIG["settings"]["score_count"]);
	TriviaBotGUI_ScoreFrequency:SetText(TRIVIABOT_CONFIG["settings"]["report_interval"]);
	
	TriviaBotGUI_ShowAnswerCheckBox:SetChecked(TRIVIABOT_CONFIG["settings"]["show_answers"]);
	TriviaBotGUI_ReportCheckBox:SetChecked(TRIVIABOT_CONFIG["settings"]["show_reports"])

	-- Update start/stop button
	if(TRIVIABOT_RUNNING ~= nil and TRIVIABOT_RUNNING ~= false) then
		TriviaBotGUI_StartStopButton:SetText("Stop TriviaBot");
	else
		TriviaBotGUI_StartStopButton:SetText("Start TriviaBot");
	end
end