-- TooltipItemIcon

-- Previous Authors: Alason, Freddy, Amavana

-- See end of file for global exports
-- Documentation can be found in commands.txt and developer.txt

local version = 1.5
-- beta

--------------------------------------------------------------------------------
-- VARIABLES
--------------------------------------------------------------------------------

local IconDataTable = {}
local LocationTable = {
	lefttop = {"TOPRIGHT", "TOPLEFT", 0, -3},
	righttop = {"TOPLEFT", "TOPRIGHT", 0, -3},
	topleft = {"BOTTOMLEFT", "TOPLEFT", 3, 0},
	topright = {"BOTTOMRIGHT", "TOPRIGHT", -3, 0},
	topcenter = {"BOTTOM", "TOP", 0, 0},
	bottomleft = {"TOPLEFT", "BOTTOMLEFT", 3, 0},
	bottomright = {"TOPRIGHT", "BOTTOMRIGHT", -3, 0},
	bottomcenter = {"TOP", "BOTTOM", 0, 0},
}
local DisplayIcon_Table = {}
local HideIcon_Table = {}
local SetFunctionTable = {}

-- locals to hold subtables or settings, for access speed
local saved
local Oinsidesize

-- Set to an empty function, which will be used if another AddOn tries to call
-- before we have processed our VARIABLES_LOADED event
-- function will be discarded for garbage collection after
local DisplayIcon_Current = function () end
local HideIcon_Current = DisplayIcon_Current

--------------------------------------------------------------------------------
-- UTILITY FUNCTIONS
--------------------------------------------------------------------------------

local function TTIIPrint (msg, ...)
	DEFAULT_CHAT_FRAME:AddMessage ("TTII: "..format(msg, ...), .9, .9, .9)
end

--------------------------------------------------------------------------------
-- MODE-SPECIFIC FUNCTIONS
--------------------------------------------------------------------------------

HideIcon_Table.frame = function (parent, data)
	if data.frame then
		data.frame:Hide()
	end
end

HideIcon_Table.inside = function (parent, data)
	if data.insideicon and data.insideicon:IsShown() then
		data.insideicon:Hide()
	end
end

HideIcon_Table.background = function (parent, data)
	if data.backgroundicon and data.backgroundicon:IsShown () then
		data.backgroundicon:Hide()
	end
end

DisplayIcon_Table.frame = function (parent, data, iconpath)
	local iconframe = data.frame
	-- check for existing frame
	if not iconframe then
		-- create new icon frame
		iconframe = CreateFrame ("Frame", nil, parent)
		iconframe:ClearAllPoints()
		iconframe:SetWidth (saved.frame.size)
		iconframe:SetHeight (saved.frame.size)
		data.frame = iconframe
		-- create texture and attach to frame
		data.frameicon = iconframe:CreateTexture (nil, "ARTWORK")
		data.frameicon:SetAllPoints (iconframe)
		data.frameicon:SetAlpha (saved.frame.alpha)
		-- set anchor position
		local changepos = data.location or LocationTable.lefttop
		iconframe:SetPoint (changepos[1], parent, changepos[2], changepos[3], changepos[4])
		data.location = nil
	end
	-- show the icon
	data.frameicon:SetTexture (iconpath)
	iconframe:Show()
end

DisplayIcon_Table.background = function (parent, data, iconpath)
	local icon = data.backgroundicon
	-- check the background texture
	if not icon then
		icon = getglobal (parent:GetName().."Texture1")
		icon:SetAlpha (saved.background.alpha)
		icon:SetAllPoints (parent)
		icon:SetDrawLayer ("BACKGROUND")
		data.backgroundicon = icon
	end
	-- show the icon
	icon:SetTexture (iconpath)
	icon:Show()
end

DisplayIcon_Table.inside = function (parent, data, iconpath)
	local icon = data.insideicon
	-- check the text field
	if not icon then
		icon = getglobal (parent:GetName().."TextRight1")
		data.insideicon = icon
	end
	-- show the icon
	icon:SetFormattedText ("|T%s:%d|t", iconpath, Oinsidesize)
	icon:Show()
	parent:Show() -- required to reformat layout correctly
end

--------------------------------------------------------------------------------
-- FRAME MODE LOCATION FUNCTIONS
--------------------------------------------------------------------------------

local function GetSavedTable (frame)
	-- called once when a new frame is seen for the first time
	local name, savetable
	name = frame:GetName()
	savetable = saved.exact[name]
	if savetable then
		-- code to clean out empty tables
		if type(next(savetable)) == "nil" then
			savetable = nil
			saved.exact[name] = nil
		end
	else
		name = strmatch (name, "^(%D+)")
		savetable = saved.template[name]
		if savetable and type(next(savetable)) == "nil" then
			savetable = nil
			saved.template[name] = nil
		end
	end
	return savetable
end

local function SetSavedValue (frame, key, value)
	-- called multiple times in a loop over known frames
	-- key and/or value may be nil
	local exactname, templatename
	exactname = frame:GetName()
	templatename = strmatch (exactname, "^(%D+)")
	if exactname == templatename or not templatename then -- decide where to store this frame
		if not key then
			saved.exact[exactname] = nil
			return
		end
		if not saved.exact[exactname] then
			saved.exact[exactname] = {}
		end
		saved.exact[exactname][key] = value
	else
		if not key then
			saved.template[templatename] = nil
			return
		end
		if not saved.template[templatename] then
			saved.template[templatename] = {}
		end
		saved.template[templatename][key] = value
	end
end

--------------------------------------------------------------------------------
-- HANDLER FUNCTIONS
--------------------------------------------------------------------------------

local function DisplayIcon_Core (parent, link, location, compare)
	-- get data for this parent frame
	local data = IconDataTable[parent]
	if not data then -- new frame: do all first-time processing
		data = {}
		IconDataTable[parent] = data
		if compare then
			data.compare = true
			if not saved.options.compare then
				data.disable = true
			end
		end
		-- user-saved location has priority over addon-default location
		--data.location = GetSavedLocation (parent) or LocationTable[location]
		data.location = LocationTable[location]
		local savedtable = GetSavedTable (parent)
		if savedtable then
			data.location = LocationTable[savedtable.location] or data.location
			if savedtable.disable then
				data.disable = true
			end
		end
	end
	-- obtain icon texture from link
	local iconpath
	if link then
		iconpath = GetItemIcon (link)
		--todo: ability to accept spell links or even raw texture paths
--[ switchable test code
		if not iconpath then
			local extract, _
			extract = strmatch (link, "spell:(%-?%d+)")
			extract = extract and tonumber (extract)
			if extract then
				--iconpath = GetSpellTexture (extract)
				_,_,iconpath = GetSpellInfo (extract)
			end
			--TTIIPrint ("Debug: extract=%s, iconpath=%s", extract or "nil", iconpath or "nil")
		end
-- end disabled code ]]
	end
	-- show or hide icon
	--if iconpath and (Ocompare or not data.compare) then
	if iconpath and not data.disable then
		DisplayIcon_Current (parent, data, iconpath)
		data.shown = true
	elseif data.shown then
		HideIcon_Current (parent, data)
		data.shown = false
	end
end

local function HookItem (frame)
	local _,link = frame:GetItem()
	DisplayIcon_Core (frame, link) -- if link is nil, icon will be hidden
end

local function HookHide (frame)
	DisplayIcon_Core (frame)
end

local function HookLink (frame, link)
	DisplayIcon_Core (frame, link)
end

local function HookLWComp (frame, link)
	DisplayIcon_Core (frame, link, nil, true)
end

--------------------------------------------------------------------------------
-- COMMAND FUNCTIONS
--------------------------------------------------------------------------------

local function ResetIcons ()
	-- hides everything
	for frame, data in pairs (IconDataTable) do
		if data.frame then
			data.frame:Hide()
		end
		if data.backgroundicon then
			data.backgroundicon:Hide()
		end
		if data.insideicon then
			data.insideicon:Hide()
		end
		data.shown = false
	end
end

local function DefaultSavedVariables ()
	return {
		version = version,
		mode = "frame",
		options = {
			compare = true,
		},
		frame = {
			size = 39,
			alpha = 1,
		},
		inside = {size = 25},
		background = {alpha = .45},
		exact = {},
		template = {},
	}
end

local function SetLocalVariables ()
	saved = TooltipItemIcon_Saved
	DisplayIcon_Current = DisplayIcon_Table[saved.mode]
	HideIcon_Current = HideIcon_Table[saved.mode]
	Oinsidesize = saved.inside.size
end

SetFunctionTable.on = function (frame, data)
	data.disable = data.compare and not saved.options.compare
	SetSavedValue (frame, "disable", nil)
end

SetFunctionTable.off = function (frame, data)
	data.disable = true
	SetSavedValue (frame, "disable", true)
	if data.shown then
		HideIcon_Current (frame, data)
		data.shown = false
	end
end

SetFunctionTable.location = function (frame, data, newpos, poscode)
	data.location = newpos
	if newpos and data.frame then
		data.frame:ClearAllPoints()
		data.frame:SetPoint (newpos[1], frame, newpos[2], newpos[3], newpos[4])
	end
	SetSavedValue (frame, "location", poscode)
end

SetFunctionTable.default = function (frame, data)
	-- clears saved location so that default will be used on reloadui
	-- does not move the current icon frame (as we don't know the default location)
	data.location = nil
	SetSavedValue (frame, "location", nil)
end

SetFunctionTable.reset = function (frame, data)
	-- clears all saved values for this frame
	data.location = nil
	data.disable = nil
	SetSavedValue (frame, nil)
end

local function SearchForMatchingFrame (namestring, setfunc, value1, value2)
	-- look for a frame whose name matches the user-input string
	-- note the string will be in lower case
	local err = true

	for frame, data in pairs (IconDataTable) do
		local exactname, templatename
		exactname = strlower (frame:GetName())
		templatename = strmatch (exactname, "^(%D+)")
		if namestring == exactname or namestring == templatename then
			setfunc (frame, data, value1, value2)
			err = nil
		end
	end

	-- todo: search saved tables for frame name, in case we have seen it in a previous session
	return err
end

local function SlashTooltipActions (name, action)
	-- Settings or other actions perfomed on one or more tooltips
	-- name = code representing multiple tooltips OR name of tooltip (possibly preceded by : )

	-- decode action first:
	local value1, value2, setfunc
	value1 = LocationTable[action]
	if value1 then
		if saved.mode ~= "frame" then
			TTIIPrint "Error: command not valid in this mode"
			return
		end
		value2 = action
		action = "location"
	end
	setfunc = SetFunctionTable[action]
	if not setfunc then
		return true -- error
	end

	-- decode name
	if name == "all" then
		TTIIPrint ("Applying %s action to %s", action, "all tooltips")
		if action == "reset" then
			-- special case: clean out saved tables for all tooltips
			saved.exact = {}
			saved.template = {}
			return
		end
		for key, value in pairs (IconDataTable) do
			setfunc (key, value, value1, value2)
		end
		return
	elseif name == "screen" then
		TTIIPrint ("Applying %s action to %s", action, "visible tooltips")
		for key, value in pairs (IconDataTable) do
			if key:IsVisible() then
				setfunc (key, value, value1, value2)
			end
		end
		return
	elseif name == "compare" then
		-- works differently to the others
		-- 'on' and 'off' should not be saved for individual tooltips
		-- instead they set the global 'compare' option
		if action == "on" then
			saved.options.compare = true
			setfunc = nil
			TTIIPrint ("Compare tooltips %s", "enabled")
		elseif action == "off" then
			saved.options.compare = false
			setfunc = nil
			TTIIPrint ("Compare tooltips %s", "disabled")
		else
			TTIIPrint ("Applying %s action to %s", action, "comparison tooltips")
		end

		for key, value in pairs (IconDataTable) do
			if value.compare then
				-- actions other than 'on' or 'off'
				if setfunc then
					setfunc (key, value, value1, value2)
				end
				-- decide whether icon should now be disabled
				local savedtable = GetSavedTable (key)
				if savedtable and savedtable.disable then
					-- this tooltip explicitly disabled in saved table
					value.disable = true
				else
					-- this tooltip enabled/disabled depending on global compare option
					value.disable = not saved.options.compare
				end
				-- hide icon if now disabled
				if value.disable and value.shown then
					HideIcon_Current (key, value)
					value.shown = false
				end
			end
		end
		return
	end

	-- explicit tooltip name marked by a starting ':' symbol; strip it out
	if strsub (name, 1, 1) == ":" then
		name = strsub (name, 2) or ""
	end
	if name == "" then
		return true
	end
	TTIIPrint ("Applying %s action to %s", action, name)
	return SearchForMatchingFrame (name, setfunc, value1, value2)
end

local function SlashCommand (cmd)
	-- extract data from command string
	cmd = strlower (cmd)
	local _, pos, command, extra, extranumber
	_,pos,command = strfind (cmd, "(%S+)")
	if pos then
		_,_,extra = strfind (cmd, "%s+(.+)", pos+1)
	end
	-- make everything a valid (possibly empty) string or number
	--command = command or ""
	extra = extra or ""
	extranumber = tonumber (extra) or -1
	-- default -1 chosen to be invalid for everything but still comparable with < & >

	if not command then
		TTIIPrint ("TooltipItemIcon version %g. Mode %s", version, saved.mode)
	elseif command == "reset" then
		TooltipItemIcon_Saved = DefaultSavedVariables ()
		ResetIcons ()
		TTIIPrint "Saved variables reset to default values"
	elseif command == "size" then
		if extranumber < 1 or extranumber > 999 then
		TTIIPrint "Error: number is out of range"
		elseif saved.mode == "frame" then
			saved.frame.size = extranumber
			for _,data in pairs (IconDataTable) do
				if data.frame then
					data.frame:SetWidth (extranumber)
					data.frame:SetHeight (extranumber)
				end
			end
			TTIIPrint ("Icon size set to %d", extranumber)
		elseif saved.mode == "inside" then
			saved.inside.size = extranumber
			TTIIPrint ("Icon size set to %d", extranumber)
		else
			TTIIPrint "Error: command not valid in this mode"
		end
	elseif command == "alpha" then
		if extranumber < 0 or extranumber > 1 then
		TTIIPrint ("Error: number is out of range")
		elseif saved.mode == "background" then
			saved.background.alpha = extranumber
			for _,data in pairs (IconDataTable) do
				if data.backgroundicon then
					data.backgroundicon:SetAlpha (extranumber)
				end
			end
			TTIIPrint ("Background alpha set to %g", extranumber)
		elseif saved.mode == "frame" then
			saved.frame.alpha = extranumber
			for _,data in pairs (IconDataTable) do
				if data.frameicon then
					data.frameicon:SetAlpha (extranumber)
				end
			end
			TTIIPrint ("Background alpha set to %g", extranumber)
		else
			TTIIPrint "Error: command not valid in this mode"
		end
	elseif DisplayIcon_Table[command] then
		saved.mode = command
		ResetIcons ()
		TTIIPrint ("Changing to %s mode", command)
	elseif SlashTooltipActions (command, extra) then
		TTIIPrint "Error: unrecognised TooltipItemIcon command"
	end
	-- refresh/update for any changes
	SetLocalVariables ()
end

--------------------------------------------------------------------------------
-- SETUP FUNCTIONS
--------------------------------------------------------------------------------

function TooltipItemIcon_CheckSavedVariables ()
	-- function called at startup, then deleted
	local oldtable = TooltipItemIcon_Saved
	if type (oldtable) ~= "table" then
		TooltipItemIcon_Saved = DefaultSavedVariables ()
		return
	end
	if oldtable.version == version then
		-- only check once when a new version is installed
		return
	end
	TTIIPrint "New version detected; checking saved variables"
	local newtable = DefaultSavedVariables ()
	-- options: always boolean
	if type(oldtable.options) == "table" then
		for key, _ in pairs (newtable.options) do
			if type(oldtable.options[key]) == "boolean" then
				newtable.options[key] = oldtable.options[key]
			end
		end
	end
	-- mode
	if oldtable.mode == "frame" or oldtable.mode == "inside" or oldtable.mode == "background" then
		newtable.mode = oldtable.mode
	end
	-- inside settings
	if type(oldtable.inside) == "table" and type(oldtable.inside.size) == "number" then
		newtable.inside.size = oldtable.inside.size
	end
	-- background settings
	if type (oldtable.background) == "table" and type (oldtable.background.alpha) == "number" then
		newtable.background.alpha = oldtable.background.alpha
	end
	-- frame settings
	if type (oldtable.frame) == "table" then
		if type (oldtable.frame.size) == "number" then
			newtable.frame.size = oldtable.frame.size
		end
		if type (oldtable.frame.alpha) == "number" then
			newtable.frame.alpha = oldtable.frame.alpha
		end
	end
	-- settings for individual tooltips
	-- exact name matches
	if type (oldtable.exact) == "table" then
		for key, value in pairs (oldtable.exact) do
			if type (value) == "table" then
				local flag
				local tooltiptable = {}
				if value.disable == true then
					tooltiptable.disable = true
					flag = true
				end
				if LocationTable[value.location] then
					tooltiptable.location = value.location
					flag = true
				end
				if flag then
					newtable.exact[key] = tooltiptable
				end
			end
		end
	end
	-- template name matches
	if type (oldtable.template) == "table" then
		for key, value in pairs (oldtable.template) do
			if type (value) == "table" then
				local flag
				local tooltiptable = {}
				if value.disable == true then
					tooltiptable.disable = true
					flag = true
				end
				if LocationTable[value.location] then
					tooltiptable.location = value.location
					flag = true
				end
				if flag then
					newtable.template[key] = tooltiptable
				end
			end
		end
	end

	TooltipItemIcon_Saved = newtable
end

function TooltipItemIcon_OnEvent(frame) -- only event is VARIABLES_LOADED
	-- saved Variables
	TooltipItemIcon_CheckSavedVariables ()
	SetLocalVariables ()
	-- Blizzard tooltips
	TooltipItemIcon_HookFrame (ItemRefTooltip, EquipCompare_Enabled and "topleft" or "left")
	TooltipItemIcon_HookFrame (GameTooltip, "topleft")
	TooltipItemIcon_HookFrame (ShoppingTooltip1, "topleft", true)
	TooltipItemIcon_HookFrame (ShoppingTooltip2, "topleft", true)
	-- LinkWrangler hooks
	if LinkWrangler then
		LinkWrangler.RegisterCallback ("TooltipItemIcon", HookLink, "refresh")
		LinkWrangler.RegisterCallback ("TooltipItemIcon", HookHide, "minimize", "hide")
		LinkWrangler.RegisterCallback ("TooltipItemIcon", HookLWComp, "refreshcomp")
	end
	-- EquipCompare hooks
	if ComparisonTooltip1 then
		TooltipItemIcon_HookFrame (ComparisonTooltip1, "topleft", true)
	end
	if ComparisonTooltip2 then
		TooltipItemIcon_HookFrame (ComparisonTooltip2, "topleft", true)
	end
	-- slash commands
	SLASH_TOOLTIPITEMICON1 = "/tooltipitemicon"
	SLASH_TOOLTIPITEMICON2 = "/ttii"
	SlashCmdList ["TOOLTIPITEMICON"] = SlashCommand
	-- delete unneeded functions to free up memory
	TooltipItemIcon_CheckSavedVariables = nil
	frame:UnregisterEvent ("VARIABLES_LOADED")
	frame:SetScript ("OnEvent", nil)
	TooltipItemIcon_OnEvent = nil
	TooltipItemIcon_EventFrame = nil -- Note: may leave some non-lua data in memory
end

TooltipItemIcon_EventFrame = CreateFrame ("Frame")
TooltipItemIcon_EventFrame:SetScript ("OnEvent", TooltipItemIcon_OnEvent)
TooltipItemIcon_EventFrame:RegisterEvent("VARIABLES_LOADED")

--------------------------------------------------------------------------------
-- EXPORTS
--------------------------------------------------------------------------------

-- Export: takes parameters (ParentFrame, [Link], [Location], [Compare])
TooltipItemIcon_DisplayIcon = DisplayIcon_Core
-- Multi-purpose function used to explicitly attach an icon to your frame
-- Will work in more cases than TooltipItemIcon_HookFrame,
-- but requires the other AddOn to do more work


-- New alternative method of using TooltipItemIcon:
-- call the following function to automatically hook TTII to your frame
-- Refer to developer.txt for documentation
function TooltipItemIcon_HookFrame (frame, location, compare)
	if frame:GetScript ("OnTooltipSetItem") then
		frame:HookScript ("OnTooltipSetItem", HookItem)
	else
		frame:SetScript ("OnTooltipSetItem", HookItem)
	end
	if frame:GetScript ("OnTooltipCleared") then
		frame:HookScript ("OnTooltipCleared", HookHide)
	else
		frame:SetScript ("OnTooltipCleared", HookHide)
	end
	DisplayIcon_Core (frame, nil, location, compare) -- save location/compare settings
end

