--[[ ToDo
     Remember which quests you were watching.
]]--

ToDo = {}

ToDo.DebugLevel = 0
ToDo.Version = "0.23"
ToDo.Quests = {}
ToDo.QuestLevels = {}
ToDo.ObjectCounts = {}
ToDo.OldCounts = {}
ToDo.WHITE = { ["r"] = 1, ["g"] = 1, ["b"] = 1 }
ToDo.BLUE = { ["r"] = 0.4, ["g"] = 0.4, ["b"] = 1 }
ToDo.GOLD = { ["r"] = 1.0, ["g"] = 0.9, ["b"] = 0 }
ToDo.SILVER = { ["r"] = 1, ["g"] = 1, ["b"] = 1 }
ToDo.COPPER = { ["r"] = 0.7, ["g"] = 0.4, ["b"] = 0 }
ToDo.T = nil
ToDo.WasComplete = {}
ToDo.Live = 0
ToDo.Log = {}
ToDo.OldSkills = {}
ToDo.Skills = {}

--[[ Data structure:
  -- OLD STRUCTURE:  Converted at addonloaded
  ToDo_Options = {
    ["Version"] = 1,
    ["User@Realm"] = {
      [1] = "Quest:title",
      [2] = "Header:text",
      [3] = "Object:number (link|name)",
      [4] = "Mob:x/y name",
  }
  -- NEW STRUCTURE:
  ToDo_Options = {
    ["Version"] = 1,
    ["User@Realm"] = {
      [1] = "Quest:title",
      [2] = "Goal:gtitle",
      ["gtitle"] = {
        [1] = "Mob:x/y name",
        [2] = "Object:number (link|name)",
        [3] = "Cash:number",
        [4] = "Text:message",
      }
    }
  }
]]--

function ToDo.Debug(level, text, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9)
  if (level <= ToDo.DebugLevel) then
    local message = format(text or 'nil', arg1, arg2, arg3, arg4, arg5, arg6,
    arg7, arg8, arg9)
    ChatFrame1:AddMessage("ToDo: " .. message, 0, 1, 0.4)
    -- table.insert(ToDo.Log, message)
  end
end

-- Event stuff:

function ToDo.AQW(id, suppress)
  if (ToDo.Sync) then
    ToDo.AQWHook(id)
  end
  ToDo.Debug(1, "AQW: %d", id)

  local q = ToDo.Quests[id]

  if (q == nil) then
    ToDo.Debug(0, "Internal error: Unknown quest %d", id)
    return
  end
  local line = "Quest:" .. q

  for i, v in ipairs(ToDo.T) do
    if (v == line) then
      ToDo.Debug(1, "Hmm, re-adding quest %s?", q)
      return
    end
  end
  table.insert(ToDo.T, line)
  if (not suppress) then
    ToDo.OnShow()
  end
end

function ToDo.RQW(id)
  ToDo.RQWHook(id)
  ToDo.Debug(1, "RQW: %d", id)

  local q = ToDo.Quests[id]

  if (q == nil) then
    ToDo.Debug(0, "Internal error: Unknown quest %d", id)
    return
  end
  local line = "Quest:" .. q

  for i, v in ipairs(ToDo.T) do
    if (v == line) then
      ToDo.Debug(1, "Removing quest %s.", q)
      table.remove(ToDo.T, i)
      ToDo.OnShow()
      return
    end
  end
  ToDo.Debug(1, "No quest %s to remove?", q)
  ToDo.OnShow()
end

function ToDo.Objectives(name)
  name = ToDo.StripName(name)
  if (ToDo.T[name] == nil) then
    ToDo.T[name] = {}
  end
  return ToDo.T[name]
end

function ToDo.AddonLoaded()
  -- dummy table if none was provided
  if (ToDo.T == nil) then
    ToDo.Debug(0, "nuking entries")
    ToDo.T = { ["Version"] = 2 }
  end
  if (ToDo_Options["Version"] == nil) then
    for n, v in pairs(ToDo_Options) do
      local t = v
      local newt = {}
      for i, v2 in pairs(t) do
      	t[i] = nil
	table.insert(newt, "Quest:" .. i)
      end
      ToDo_Options[n] = newt
    end
    ToDo_Options["Version"] = 1
  end
  if (ToDo_Options["Version"] == 1) then
    local dropped = nil
    for n, v in pairs(ToDo_Options) do
      local t = v
      local newt = {}
      if (type(t) == "table") then
	for i, v2 in ipairs(t) do
	  if (string.find(v2, "^Quest")) then
	    table.insert(newt, v2)
	  else
	    dropped = true
	  end
	end
      end
    end
    if (dropped) then
      ToDo.Debug(0, "Sorry!  Converting to the new database format wiped out old custom quests.")
    end
    ToDo_Options["Version"] = 2
  end
  if (ToDo_Options[ToDo.SelfName] == nil) then
    ToDo_Options[ToDo.SelfName] = {}
  end
  ToDo.T = ToDo_Options[ToDo.SelfName]
  if (ToDo_Options["Sync"] == nil) then
    ToDo_Options["Sync"] = 1
  end
  if (ToDo_Options["Sync"] == 1) then
    ToDo.AQWHook = AddQuestWatch
    ToDo.RQWHook = RemoveQuestWatch
    AddQuestWatch = ToDo.AQW
    RemoveQuestWatch = ToDo.RQW
    ToDo.Sync = true
  else
    ToDo.Sync = false
  end
  if (ToDo_Options["Auto"] == 1) then
    ToDo.Auto = true
  else
    ToDo.Auto = false
  end
end

function ToDo.EnteringWorld()
  -- Don't bother updating quests, they aren't loaded yet.
  -- Get skill table updated
  ToDo.UpdateSkills()
  -- mark starting ranks
  for i,v in pairs(ToDo.Skills) do
    ToDo.OldSkills[ToDo.StripName(i)] = v
  end
  if (ToDo_Options["Show"]) then
    ToDo.OnSlash("show")
  else
    ToDo.OnSlash("hide")
    QuestWatch_Update()
  end
end

function ToDo.UpdateSkills()
  for i = 1,GetNumSkillLines() do
    local skillName, isHeader, isExpanded, skillRank, numTempPoints, skillModifier, skillMaxRank, isAbandonable, stepCost, rankCost, minLevel, skillCostType, skillDescription = GetSkillLineInfo(i)
    if (not isHeader) then
      ToDo.Skills[ToDo.StripName(skillName)] = skillRank
    end
  end
end

function ToDo.UpdateQuests()
  local q, _ = GetNumQuestLogEntries()
  local added = false
  -- don't do anything if we happen to get called with no quests
  ToDo.Debug(2, "quest log entries: %d", q or -1)
  if (q ~= nil and q > 1) then
    if (ToDo.Live == 0) then
      ToDo.Debug(1, "going live, %d log entries", q)
    end
  else
    return false
  end
  ToDo.Quests = {}
  ToDo.QuestLevels = {}
  for i = 1, q do
    local questTitle, level, questTag, suggestedGroup, isHeader, isCollapsed, isComplete = GetQuestLogTitle(i)
    questTitle = string.gsub(questTitle, "^%[.*%] *", "")
    questTitle = string.gsub(questTitle, "(Complete)", "(*)")
    if (not isHeader) then
      ToDo.Quests[questTitle] = i
      ToDo.Debug(3, "'%s' -> '%s'", questTitle, ToDo.StripName(questTitle))
      ToDo.Quests[ToDo.StripName(questTitle)] = i
      ToDo.QuestLevels[ToDo.StripName(questTitle)] = level
      ToDo.Quests[i] = questTitle
      if (not added and ToDo.Auto and ToDo.OldQuests ~= nil and ToDo.OldQuests[questTitle] == nil) then
        -- it's a new quest, so add it!
	added = true
	ToDo.Debug(2, "Adding quest '%s'", questTitle)
	ToDo.AQW(i, true)
      end
    end
  end
  ToDo.OldQuests = ToDo.Quests
  ToDo.Debug(2, "updated quest list.")
  for i = 1,table.getn(ToDo.T) do
    local v = ToDo.T[i] or ""
    local q
    ToDo.Debug(2, "todo list %d: %s", i, v or "nil")
    q = string.match(v, "Quest:(.*)")
    if (q) then
      if (ToDo.Quests[ToDo.StripName(q)] == nil) then
        ToDo.Debug(0, "Crossed off %s", q)
	table.remove(ToDo.T, i)
	break
      end
    end
  end
  return true
end

function ToDo.WatchQuests()
  for i, v in ipairs(ToDo.T) do
    ToDo.Debug(2, "todo %d: %s", i, v)
    q = string.match(v, "Quest:(.*)")
    if (q ~= nil) then
      if (ToDo.Quests[ToDo.StripName(q)]) then
	ToDo.Debug(2, "Adding quest '%s' (%d) to quest watch.", q, ToDo.Quests[ToDo.StripName(q)])
	ToDo.AQWHook(ToDo.Quests[ToDo.StripName(q)])
      else
	ToDo.Debug(0, "Can't find quest '%s'.", q)
	ToDo.T[i] = nil
      end
    end
  end
end

function ToDo.OnEvent(event, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9)
  local event_type
  local event_name
  local event_entry
  
  ToDo.Debug(1, "Event: %s (%s, %s, %s, %s)", event,
    (arg1 or "(nil)"),
    (arg2 or "(nil)"),
    (arg3 or "(nil)"),
    (arg4 or "(nil)"))

  if (event == "ADDON_LOADED") then
    if (arg1 == "ToDo") then
      ToDo.AddonLoaded()
      ToDo_Frame:UnregisterEvent("ADDON_LOADED")
    end
    return
  end

  if (event == "PLAYER_ENTERING_WORLD") then
    ToDo.Debug(3, "EnteringWorld")
    ToDo.EnteringWorld()
    ToDo_Frame:UnregisterEvent("PLAYER_ENTERING_WORLD")
    ToDo_Frame:RegisterEvent("QUEST_LOG_UPDATE")
    ToDo_Frame:RegisterEvent("BAG_UPDATE")
    ToDo_Frame:RegisterEvent("CHAT_MSG_SKILL")
    ToDo_Frame:RegisterEvent("PLAYER_MONEY")
    ToDo_Frame:RegisterEvent("COMBAT_LOG_UNFILTERED")
    ToDo.OnShow()
    return
  end

  if (event == "COMBAT_LOG_UNFILTERED") then
    local mob = nil
    if (arg2 == "UNIT_DIED") then
      slayer = arg4
      mob = arg7
      if (slayer ~= UnitName("player")) then
        local found = false
        for i = 1, 4 do
	  if (slayer == UnitName("party" .. i)) then
	    found = true
	  end
        end
        if (not found) then
	  mob = nil
        end
      end
    end
    if (mob) then
      mob = ToDo.StripName(mob)
      for i, v in pairs(ToDo.T) do
        if (type(v) == "table") then
	  for j, w in pairs(v) do
	    if (ToDo.TypeOf(w) == "Mob") then
	      local count, goal, name = string.match(ToDo.NameOf(w), "(%d+)/(%d+) (.*)")
	      count = tonumber(count)
	      if (name and string.find(mob, ToDo.StripName(name))) then
		count = count + 1
		v[j] = format("Mob:%d/%s %s", count, goal, name)
		goal = tonumber(goal)
		if (count <= goal) then
		  local complete
		  if (count == goal) then
		    complete = " -- complete!"
		  else
		    complete = ""
		  end
		  ToDo.Progress("%s slain: %d/%s%s", name, count, goal, complete)
		end
		ToDo.OnShow()
	      end
	    end
	  end
	end
      end
    end
  end

  if (event == "QUEST_LOG_UPDATE") then
    if (ToDo.UpdateQuests()) then
      ToDo.Live = 1
    end
    ToDo.OnShow()
  end

  if (event == "BAG_UPDATE") then
    ToDo.UpdateObjects()
    ToDo.OnShow()
  end

  if (event == "CHAT_MSG_SKILL") then
    ToDo.UpdateSkills()
    ToDo.OnShow()
  end

  -- Money is so cheap to track that we just do it in the OnShow().
  if (event == "PLAYER_MONEY") then
    ToDo.OnShow()
  end
end

-- returns eg. "monster", "Young Nightsaber slain", 1, 7, nil
function ToDo.GetLeaderBoardDetails(boardIndex, questIndex)
  local leaderboardTxt, itemType, isDone = GetQuestLogLeaderBoard(boardIndex,questIndex)
  local itemName, numItems, numNeeded = string.match(leaderboardTxt, "(.*):%s*([^%s]+)%s*/%s*([^%s]+)")
  if (itemName == nil) then
    return itemType, leaderboardTxt, isDone, 1, isDone
  else
    return itemType, itemName, numItems, numNeeded, isDone
  end
end

function ToDo.NameOf(obj)
  if (obj) then
    local type, name = string.match(obj, "([^:]-):(.*)")
    return name or ""
  else
    return ""
  end
end

function ToDo.TypeOf(obj)
  if (obj) then
    local type, name = string.match(obj, "([^:]-):(.*)")
    return type or ""
  else
    return ""
  end
end

function ToDo.AddLine(text, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9)
  local str = format(text, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9)
  local textitem
  ToDo.Debug(4, "AddLine: %s", str)
  -- reset height so first line starts out at the top
  if (ToDo.CurrentLine == 1) then
    ToDo_Frame:SetHeight(0)
  end
  if (ToDo.Children[ToDo.CurrentLine]) then

    textitem = ToDo.Children[ToDo.CurrentLine]
    textitem:SetText(str)
    textitem:Show()
  else
    textitem = ToDo_Frame:CreateFontString("ToDo_Line" ..  ToDo.CurrentLine, OVERLAY, "GameFontNormal")
    textitem:SetText(str)
    ToDo.Children[ToDo.CurrentLine] = textitem
  end

  textitem:SetPoint("TOPLEFT", ToDo_Frame, "TOPLEFT", 0, -1 * ToDo_Frame:GetHeight())
  ToDo_Frame:SetHeight(ToDo_Frame:GetHeight() + textitem:GetHeight() + 1)
  ToDo.CurrentLine = ToDo.CurrentLine + 1

  if (textitem:GetStringWidth() > ToDo_Frame:GetWidth()) then
    ToDo_Frame:SetWidth(textitem:GetStringWidth())
  end
end

function ToDo.ResetLines()
  if (ToDo.CurrentLine) then
    for i = 2, ToDo.CurrentLine do
      if (ToDo.Children[i]) then
        ToDo.Children[i]:Hide()
      end
    end
  end
  local textitem
  if (ToDo.Children[1] == nil) then
    textitem = ToDo_Frame:CreateFontString("ToDo_Line" ..  ToDo.CurrentLine, OVERLAY, "GameFontNormal")
    ToDo.Children[1] = textitem
  else
    textitem = ToDo.Children[1]
  end
  textitem:SetText(ToDo.DimColor(ToDo.BLUE, true) .. "ToDo List|r")
  textitem:SetPoint("TOPLEFT", ToDo_Frame, "TOPLEFT", 0, 0)
  ToDo.CurrentLine = 1
  ToDo_Frame:SetHeight(textitem:GetHeight())
  ToDo_Frame:SetWidth(100)
end

function ToDo.ObjectName(item)
  local name
  if (item == nil) then
    return nil
  end
  name = string.match(item, ".*%[(.*)%].*")
  if (name == nil) then
    name = item
  end
  return name
end

function ToDo.UpdateObjects()
  ToDo.Debug(2, "UpdateObjects")
  ToDo.OldCounts = ToDo.ObjectCounts
  ToDo.ObjectCounts = {}
  for bag = 0, 4 do
    for slot = 1, GetContainerNumSlots(bag) do
      local link = GetContainerItemLink(bag, slot)
      local name = ToDo.ObjectName(link)
      if (name) then
        name = ToDo.StripName(name)
        local texture, itemCount, locked, quality, readable = GetContainerItemInfo(bag, slot)
	if (ToDo.ObjectCounts[name]) then
          ToDo.ObjectCounts[name] = ToDo.ObjectCounts[name] + itemCount
        else
	  ToDo.ObjectCounts[name] = itemCount
	end
      end
    end
  end
end

function ToDo.DimColor(color, bright)
  local factor
  if (bright) then
    factor = 255
  else
    factor = 200
  end
  local formatted = string.format("|cff%02x%02x%02x",
  		color.r * factor,
		color.g * factor,
		color.b * factor)
  return formatted
end

function ToDo.PrettyPrint(amount, bright)
  local gold, silver, copper
  local money = ""

  gold = math.floor(amount / 10000)
  silver = math.floor(amount / 100) % 100
  copper = amount % 100

  if (gold > 0) then
    money = money .. ToDo.DimColor(ToDo.GOLD, bright) .. gold .. "g"
  end
  if (silver > 0 or gold > 0) then
    money = money .. ToDo.DimColor(ToDo.SILVER, bright) .. silver .. "s"
  end
  money = money .. ToDo.DimColor(ToDo.COPPER, bright) ..  copper .. "c|r"
  return money
end

function ToDo.OnShow()
  -- nothing to do until the system seems to be "up"
  if (ToDo.Live == 0) then
    return
  end
  ToDo.ResetLines()
  if (ToDo_Frame:IsShown()) then
    QuestWatchFrame:Hide()
  else
    QuestWatchFrame:Show()
  end
  if (ToDo.T ~= nil) then
    for i,v in ipairs(ToDo.T) do
      local complete = ToDo.IsComplete(i)
      local type, name = string.match(v, "([^:]*):(.*)")
      ToDo.Debug(2, "Item %d: %s.", i, v)
      if (type == "Quest") then
	local id = ToDo.Quests[ToDo.StripName(name)]
	if (id == nil) then
	  ToDo.Debug(0, "Unknown quest: '%s'", name)
	else
	  SelectQuestLogEntry(id)
	  local questTitle, level, questTag, suggestedGroup, isHeader, isCollapsed, isComplete = GetQuestLogTitle(id)
	  local objectives = GetNumQuestLeaderBoards()
	  -- a quest with no objectives is "complete", in that
	  -- it is ready to be handed in.
	  if (objectives == nil or objectives == 0) then
	    isComplete = 1
	  end
	  local tag
	  if (questTag) then
	    tag = string.sub(questTag, 1, 1)
	  else
	    tag = ""
	  end
	  local comp
	  local color = ToDo.DimColor(GetDifficultyColor(level), isComplete)

	  if (isComplete) then
	    if (isComplete < 0) then
	      comp = " (Fail)"
	    else
	      comp = " (*)"
	    end
	  else
	    comp = ""
	  end
	  ToDo.AddLine("%s[%d%s] %s%s|r", color, level, tag, name, comp)
	  if (objectives == nil or objectives == 0) then
	    local questDescription, questObjectives = GetQuestLogQuestText()
	    questObjectives = string.gsub(questObjectives, ".*wants you to ", "")
	    questObjectives = string.gsub(questObjectives, ".*wishes you to ", "")
	    questObjectives = string.gsub(questObjectives, ".*has asked that you ", "")
	    if (string.len(questObjectives) > 35) then
	      questObjectives = string.sub(questObjectives, 1, 33) .. "..."
	    end
	    ToDo.AddLine("%s -- %s", ToDo.DimColor(ToDo.WHITE, true), questObjectives)
	  else
	    for item = 1, objectives do
	      local itemType, itemName, numItems, numNeeded, isDone = ToDo.GetLeaderBoardDetails(item)
	      ToDo.Debug(3, "item type: %s (%s)", itemType or "Unknown type",
	      		itemName or "Unknown name")
	      if (string.len(itemName) > 35) then
	        itemName = string.gsub(itemName, ".* %- ", "")
		if (string.len(itemName) > 35) then
	          itemName = string.sub(itemName, 1, 33) .. "..."
	        end
	      end
	      ToDo.AddLine("%s -- %s: %s/%s", ToDo.DimColor(ToDo.WHITE, isDone),
	      	itemName or "Unknown item", numItems or "0", numNeeded or "0")
	    end
	  end
	end
      elseif (type == "Goal") then
	-- first print a header.
	local color = ToDo.DimColor(ToDo.BLUE, not complete)
	if (complete) then
	  ToDo.AddLine("%s%s (*)|r", color, name)
        else
	  ToDo.AddLine("%s%s|r", color, name)
        end
	for slot, objective in ipairs(ToDo.Objectives(name)) do
	  local type, name = ToDo.TypeOf(objective), ToDo.NameOf(objective)
	  local complete = ToDo.ObjComplete(objective)
	  local color = ToDo.DimColor(ToDo.WHITE, complete)
	  if (type == "Object") then
	    local goal, item = string.match(name, "(%d+) (.*)")
	    ToDo.Debug(2, "Checking for %s '%s'.", goal, item)
	    if (goal == nil) then
	      ToDo.AddLine("Invalid object reference: %s", name)
	    else
	      -- safe, because we got it by matching "%d+"
	      goal = tonumber(goal)
	      local name = ToDo.ObjectName(item)
	      local lname = ToDo.StripName(name)
	      local new = ToDo.ObjectCounts[lname] or 0
	      local old = ToDo.OldCounts[lname] or 0
	      ToDo.AddLine("%s -- %s%s: %d/%d", color, item, color, new, goal)
	      if (old ~= new and ((new < goal) or (old < goal))) then
		local complete_text
		if ((old < goal) and (new >= goal)) then
		  complete_text = " -- complete!"
		else
		  complete_text = ""
		end
		ToDo.Progress("%s: %d/%d%s", name, new, goal, complete_text)
		ToDo.OldCounts[lname] = new
	      end
	    end
	  elseif (type == "Mob") then
	    local count, goal, name = string.match(name, "(%d+)/(%d+) (.*)")
	    ToDo.AddLine("%s -- %s slain: %s/%s", color, name, count, goal)
	  elseif (type == "Skill") then
	    local goal, name = string.match(name, "(%d+) (.*)")
	    local current = ToDo.Skills[ToDo.StripName(name)] or 0
	    local old = ToDo.OldSkills[ToDo.StripName(name)] or 0
	    goal = tonumber(goal)
	    if (current > old) then
	      local complete_text
	      if ((old < goal) and (current >= goal)) then
		complete_text = " -- complete!"
	      else
		complete_text = ""
	      end
	      ToDo.Progress("Skill up: %s to %d/%d%s", name, current, goal, complete_text)
	      ToDo.OldSkills[ToDo.StripName(name)] = ToDo.Skills[ToDo.StripName(name)]
	    end
	    ToDo.AddLine("%s -- %s %d/%d", color, name, current, goal)
	  elseif (type == "Text") then
	    ToDo.AddLine("%s -- %s", color, name)
	  elseif (type == "Cash") then
	    local goal = tonumber(name)
	    local current = GetMoney()
	    ToDo.AddLine("%s -- Cash: %s%s/%s",
	    	color, ToDo.PrettyPrint(current, complete),
	    	color, ToDo.PrettyPrint(goal, complete))
	  else
	    ToDo.Debug(0, "Unknown objective type '%s'.", type)
	  end
	end
      else
        ToDo.AddLine("WTF: %s", name)
      end
    end
  else
    ToDo.AddLine("No todo items for %s.", ToDo.SelfName)
  end
end

function ToDo.Progress(text, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9)
  UIErrorsFrame:AddMessage(format(text or 'nil', arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9), 1,1,0.3)
end

function ToDo.ObjComplete(objective)
  local type, text = ToDo.TypeOf(objective), ToDo.NameOf(objective)
  if (type == nil or text == nil) then
    ToDo.Debug(0, "Help!  Invalid objective '%s'", objective or "nil")
    return true
  end
  if (type == "Mob") then
    local count, goal, mobname = string.match(text, "(%d+)/(%d+) (.*)")
    if (tonumber(count) >= tonumber(goal)) then
      return true
    else
      return nil
    end
  elseif (type == "Object") then
    local goal, itemname = string.match(text, "(%d+) (.*)")
    local x = ToDo.ObjectCounts[ToDo.ObjectName(itemname):lower()] or 0
    return (x >= tonumber(goal))
  elseif (type == "Cash") then
    local goal = tonumber(text)
    local current = GetMoney()
    return (current >= goal)
  elseif (type == "Text") then
    return true
  elseif (type == "Skill") then
    local goal, name = string.match(text, "(%d+) (.*)")
    return (ToDo.Skills[ToDo.StripName(name)] and ToDo.Skills[ToDo.StripName(name)] >= tonumber(goal))
  else
    ToDo.Debug(0, "Unknown objective type '%s'", type)
    return true
  end
end

function ToDo.RemoveObjective(quest, objective)
  objective = ToDo.ObjectName(ToDo.StripName(objective))
  for i, v in ipairs(quest) do
    local type = ToDo.TypeOf(v)
    local name = ToDo.NameOf(v)
    name = string.match(name, "[%d/]+ (.*)")
    name = ToDo.ObjectName(ToDo.StripName(name))
    if ((name == objective) or (type == "Cash" and objective == "cash")) then
      table.remove(quest, i)
      return
    end
  end
  ToDo.Debug(0, "No objective '%s' in quest.", objective)
end

function ToDo.AddMob(quest, objective)
  local object, count = string.match(objective, "(.*) x(%d+)")
  if (object == nil or count == nil) then
    object = objective
    count = 1
  else
    count = tonumber(count)
  end
  table.insert(quest, format("Mob:0/%d %s", count, object))
end

function ToDo.AddObject(quest, objective)
  local object, count = string.match(objective, "(.*) x(%d+)")
  if (object == nil or count == nil) then
    object = objective
    count = 1
  else
    count = tonumber(count)
  end
  table.insert(quest, format("Object:%d %s", count, object))
end

function ToDo.IsComplete(slot)
  local quest = ToDo.T[slot]
  local type, name = ToDo.TypeOf(quest), ToDo.NameOf(quest)
  if (type == "Goal") then
    local obj = ToDo.Objectives(name)
    for i,v in ipairs(obj) do
      if (not ToDo.ObjComplete(v)) then
        ToDo.Debug(3, "Incomplete objective '%s'", v)
	ToDo.WasComplete[name] = nil
        return nil
      else
        ToDo.Debug(3, "Complete objective '%s'", v)
      end
    end
    if (not ToDo.WasComplete[name]) then
      ToDo.Progress("You have completed %s!", name)
    end
    ToDo.WasComplete[name] = true
    return true
  elseif (type == "Quest") then
    local id = ToDo.Quests[name]
    ToDo.Debug(3, "interpreting quest '%s'... %d", name, (id or -3))
    if (id) then
      local questTitle, level, questTag, suggestedGroup, isHeader, isCollapsed, isComplete = GetQuestLogTitle(id)
      return isComplete
    else
      return false
    end
  else
    ToDo.Debug(0, "Unknown type '%s'", type or "nil")
    return nil
  end
end

function ToDo.StripName(name)
  return string.gsub(name:lower(), "%A", "")
end

function ToDo.MatchName(a, b)
  return ToDo.StripName(a) == ToDo.StripName(b)
end

function ToDo.OnSlash(command)
  local cmd, args = string.match(command, "([^ ]*) (.*)")
  if (args == nil) then
    cmd = command
    args = ""
  end
  if (cmd == nil or cmd == "") then
    cmd = "help"
  end

  ToDo.Debug(2, "ToDo (version %s) slashcmd: %s", ToDo.Version, cmd)
  ToDo.Debug(3, "cmd: '%s' args '%s'", (cmd or "nil"), (args or "nil"))
  if (cmd == "help" or cmd == "usage") then
    ToDo.Debug(0, "ToDo:")
    ToDo.Debug(0, "/todo list -- list items")
    ToDo.Debug(0, "/todo add <name> -- add quest to list")
    ToDo.Debug(0, "/todo new <name> -- add user quest to list")
    ToDo.Debug(0, "/todo get <name>, <object> [xN] -- add object(s) to user quest")
    ToDo.Debug(0, "/todo kill <name>, <mob> [xN] -- add mob(s) to user quest")
    ToDo.Debug(0, "/todo skill <name>, skill N -- add skill goal to user quest")
    ToDo.Debug(0, "/todo cash <name>, <amount> -- add money to user quest")
    ToDo.Debug(0, "/todo text <name>, <message> -- add generic text to user quest")
    ToDo.Debug(0, "/todo drop <name>, <objective> -- drop objective from user quest")
    ToDo.Debug(0, "/todo remove <slot> -- remove slot from list")
    ToDo.Debug(0, "/todo remove <name> -- remove quest (in-game or user) from list")
  elseif (cmd == "debug") then
    if (ToDo.DebugLevel == 0) then
      ToDo.DebugLevel = 2
    else
      ToDo.DebugLevel = 0
    end
  elseif (cmd == "show") then
    ToDo_Frame:Show()
    QuestWatchFrame:Hide()
    QuestWatchFrame.Show = QuestWatchFrame.Hide
    ToDo_Options["Show"] = true
  elseif (cmd == "hide") then
    ToDo_Frame:Hide()
    QuestWatchFrame.Show = ToDo.QWShowHook
    QuestWatchFrame:Show()
    ToDo_Options["Show"] = nil
  elseif (cmd == "reset") then
    ToDo_Frame:SetPoint("TOPLEFT", "UIParent", "TOPLEFT", 400, -300)
  elseif (cmd == "add") then
    local quest
    args = string.gsub(args, "%[.*%] ?", "")
    quest = ".*" .. ToDo.StripName(args) .. ".*"
    for k, v in pairs(ToDo.Quests) do
      if (string.match(k, quest)) then
        args = ToDo.Quests[v]
        ToDo.Debug(3, "Calling AQW(%d) for '%s'", ToDo.Quests[args], args)
        ToDo.AQW(ToDo.Quests[args])
        QuestWatch_Update()
	return
      end
    end
    ToDo.Debug(0, "Can't find a quest named '%s' to add.", args)
  elseif (cmd == "new") then
    table.insert(ToDo.T, "Goal:" .. args)
    ToDo.T[ToDo.StripName(args)] = {}
    ToDo.OnShow()
  elseif (cmd == "get") then
    local name, args = string.match(args, "([^,]*), ?(.*)")
    ToDo.Debug(2, "get(%s): (%s)", name or "nil", args or "nil")
    if (name == nil) then
      ToDo.Debug(0, "usage: /todo get name, objects")
      return
    end
    if (ToDo.Objectives(name)) then
      while (string.find(args, ",")) do
        local objective
	objective, args = string.match(args, "([^,]*), ?(.*)")
	ToDo.AddObject(ToDo.Objectives(name), objective)
      end
      ToDo.AddObject(ToDo.Objectives(name), args)
    else
      ToDo.Debug(0, "No objective table for name '%s'.", name)
    end
    ToDo.OnShow()
  elseif (cmd == "skill") then
    local name, args, goal = string.match(args, "([^,]*), ?(.*) (%d+)")
    args = string.sub(args, 1, 1):upper() .. string.sub(args, 2, -1):lower()
    ToDo.Debug(2, "skill(%s): (%s) to (%d)", name or "nil", args or "nil", goal or "nil")
    if (name == nil) then
      ToDo.Debug(0, "usage: /todo skill name, skill number")
      return
    end
    if (ToDo.Objectives(name)) then
      table.insert(ToDo.Objectives(name), format("Skill:%s %s", goal, ToDo.StripName(args)))
    else
      ToDo.Debug(0, "No objective table for name '%s'.", name)
    end
    ToDo.OnShow()
  elseif (cmd == "kill") then
    local name, args = string.match(args, "([^,]*), ?(.*)")
    ToDo.Debug(2, "kill(%s): (%s)", name, args)
    if (name == nil) then
      ToDo.Debug(0, "usage: /todo kill name, targets")
      return
    end
    if (ToDo.Objectives(name)) then
      while (string.find(args, ",")) do
        local objective
	objective, args = string.match(args, "([^,]*), ?(.*)")
	ToDo.AddMob(ToDo.Objectives(name), objective)
      end
      ToDo.AddMob(ToDo.Objectives(name), args)
    else
      ToDo.Debug(0, "No objective table for name '%s'.", name)
    end
    ToDo.OnShow()
  elseif (cmd == "text") then
    local name, args = string.match(args, "([^,]*), ?(.*)")
    if (name == nil) then
      ToDo.Debug(0, "usage: /todo text name, text")
      return
    end
    if (ToDo.Objectives(name)) then
      table.insert(ToDo.Objectives(name), format("Text:%s", args))
      ToDo.OnShow()
    else
      ToDo.Debug(0, "No objective table for name '%s'.", name)
    end
  elseif (cmd == "cash") then
    local name, args = string.match(args, "([^,]*), ?(.*)")
    ToDo.Debug(2, "cash(%s): (%s)", name, args)
    if (name == nil) then
      ToDo.Debug(0, "usage: /todo cash name, amount (e.g., 1g50s)")
      return
    end
    if (ToDo.Objectives(name)) then
      -- amount must be of form NgNsNc, although g can be omitted.
      local g, s, c
      local type, count, remaining
      repeat
	count, type, remaining = string.match(args, "(%d+)([gsc])(.*)")
	if (count == nil or type == nil) then
	  ToDo.Debug(0, "Couldn't understand cash '%s'", args)
	  return
	end
	if (type == "c" and c == nil) then
	  c = tonumber(count)
	elseif (type == "s" and s == nil) then
	  s = tonumber(count)
	elseif (type == "g" and g == nil) then
	  g = tonumber(count)
	else
	  ToDo.Debug(0, "Couldn't figure out what to make of '%s%s'.",
	  	count, type)
	  return
	end
	args = remaining
      until (args == "")
      table.insert(ToDo.Objectives(name), format("Cash:%d", (g or 0) * 10000 + (s or 0) * 100 + (c or 0)))
    else
      ToDo.Debug(0, "No objective table for name '%s'.", name)
    end
    ToDo.OnShow()
  elseif (cmd == "load") then
    ToDo.EnteringWorld()
  elseif (cmd == "auto") then
    if (args == "") then
      if (ToDo.Auto) then
	args = "off"
      else
	args = "on"
      end
    end
    if (args == "on") then
      ToDo.Auto = true
      ToDo_Options["Auto"] = 1
      ToDo.Debug(0, "Auto-adding new quests.")
    elseif (args == "off") then
      ToDo.Auto = false
      ToDo_Options["Auto"] = 0
      ToDo.Debug(0, "Not auto-adding quests.")
    else
      ToDo.Debug(0, "Usage: /todo auto (on|off)")
    end
  elseif (cmd == "sync") then
    if (args == "") then
      if (ToDo.Sync) then
	args = "off"
      else
	args = "on"
      end
    end
    if (args == "on") then
      ToDo.Sync = true
      ToDo_Options["Sync"] = 1
      AddQuestWatch = ToDo.AQW
      RemoveQuestWatch = ToDo.RQW
      ToDo.Debug(0, "Synchronizing with quest watch in future.")
    elseif (args == "off") then
      ToDo.Sync = false
      ToDo_Options["Sync"] = 0
      AddQuestWatch = ToDo.AQWHook
      RemoveQuestWatch = ToDo.RQWHook
      ToDo.Debug(0, "No longer trying to sync with quest watch.")
    else
      ToDo.Debug(0, "Usage: /todo sync (on|off)")
    end
  elseif (cmd == "list") then
    ToDo.OnShow()
    for i, v in ipairs(ToDo.Children) do
      if (v:IsShown()) then
        local t = v:GetText()
        ToDo.Debug(0, "%s", t)
      end
    end
  elseif (cmd == "move") then
    if (ToDo_Frame:IsMouseEnabled()) then
      ToDo_Frame:EnableMouse(false)
      ToDo_Frame:SetBackdropColor(0.05, 0.05, 0.05, 0.0)
    else
      local title = ToDo_Frame:GetTitleRegion()
      if (title == nil) then
        ToDo.Debug(1, "Creating title region to drag.")
        title = ToDo_Frame:CreateTitleRegion()
      end
      title:SetAllPoints()
      ToDo_Frame:EnableMouse(true)
      ToDo_Frame:SetBackdropColor(0.05, 0.05, 0.05, 0.5)
    end
  elseif (cmd == "drop") then
    local name, args = string.match(args, "([^,]*), ?(.*)")
    if (name == nil or args == nil) then
      ToDo.Debug(0, "Usage: drop name, objective")
      return
    end
    ToDo.Debug(2, "drop(%s): (%s)", name, args)
    if (ToDo.Objectives(name)) then
      while (string.find(args, ",")) do
        local objective
	objective, args = string.match(args, "([^,]*), ?(.*)")
	ToDo.RemoveObjective(ToDo.Objectives(name), objective)
      end
      ToDo.RemoveObjective(ToDo.Objectives(name), args)
    else
      ToDo.Debug(0, "No objective table for name '%s'.", name)
    end
    ToDo.OnShow()
  elseif (cmd == "remove") then
    local slot = tonumber(args)
    if (slot ~= nil and slot > 0) then
      local objective = ToDo.T[slot]
      local type = ToDo.TypeOf(objective)
      local name = ToDo.NameOf(objective)
      if (objective) then
	ToDo.Debug(0, "Removing %s %s from ToDo list.", type, ToDo.NameOf(objective))
	if (type == "Quest") then
	  if (ToDo.Sync and ToDo.Quests[ToDo.StripName(name)]) then
	    -- Remove from regular quest watch too, for consistency.
	    ToDo.RQWHook(ToDo.Quests[ToDo.StripName(name)])
            QuestWatch_Update()
	  end
          table.remove(ToDo.T, slot)
	elseif (type == "Goal") then
	  -- remove the associated objectives, too.
	  ToDo.T[name] = nil
	  table.remove(ToDo.T, slot)
        else
          table.remove(ToDo.T, slot)
	end
      else
        ToDo.Debug(0, "Nothing in slot %d", slot)
      end
    elseif (args == "all") then
      for i, v in ipairs(ToDo.T) do
	local type, name = ToDo.TypeOf(v), ToDo.NameOf(v)
	if (ToDo.TypeOf(v) == "Quest") then
	  if (ToDo.Sync and ToDo.Quests[ToDo.StripName(args)]) then
	    -- Remove from regular quest watch too, for consistency.
	    ToDo.RQWHook(ToDo.Quests[ToDo.StripName(args)])
	    if (not ToDo_Frame:IsShown()) then
	      QuestWatch_Update()
	    end
	  end
	else
	  ToDo.T[ToDo.StripName(name)] = nil
	end
      end
      ToDo_Options[ToDo.SelfName] = {}
      ToDo.T = ToDo_Options[ToDo.SelfName]
      ToDo.OnShow()
      return
    else
      local quest = ".*" .. ToDo.StripName(args) .. ".*"
      for i, v in ipairs(ToDo.T) do
        if (string.match(ToDo.StripName(ToDo.NameOf(v)), quest)) then
	  if (ToDo.TypeOf(v) == "Goal") then
	    ToDo.Debug(0, "Removing goal '%s' from ToDo list.", ToDo.NameOf(v))
	    ToDo.T[ToDo.StripName(args)] = nil
	  else
	    ToDo.Debug(0, "Removing quest '%s' from ToDo list.", ToDo.NameOf(v))
	    if (ToDo.Sync and ToDo.Quests[ToDo.StripName(args)]) then
	      -- Remove from regular quest watch too, for consistency.
	      ToDo.RQWHook(ToDo.Quests[ToDo.StripName(args)])
	    end
	  end
	  table.remove(ToDo.T, i)
          ToDo.OnShow()
	  if (not ToDo_Frame:IsShown()) then
            QuestWatch_Update()
	  end
	  return
	end
      end
    end
    ToDo.Debug(0, "Can't find an item named '%s' to remove.", args)
    ToDo.OnShow()
  elseif (cmd == "log") then
    local x = ToDo.Log
    ToDo.Log = {}
    for i,v in ipairs(x) do
      ToDo.Debug(0, "%s", v)
    end
  elseif (cmd == "status") then
    ToDo.Debug(0, "%d items, window is %f x %f",
      table.getn(ToDo.T),
      ToDo_Frame:GetWidth(),
      ToDo_Frame:GetHeight())
  elseif (cmd == "sort") then
    table.sort(ToDo.T, ToDo.ByLevel);
    ToDo.OnShow()
  else
    ToDo.Debug(0, "Unknown command '%s'.", cmd)
  end
end

function ToDo.ByLevel(a, b)
  local a1, b1
  a1 = ToDo.QuestLevels[ToDo.StripName(ToDo.NameOf(a))] or 99
  b1 = ToDo.QuestLevels[ToDo.StripName(ToDo.NameOf(b))] or 99
  return a1 < b1
end

function ToDo.OnLoad()
  ToDo.SelfName = GetUnitName("player") .. "@" .. GetRealmName()
  -- start out with empty set, to be corrected when variables are loaded
  ToDo_Options = { ["Version"] = 1 }
  ToDo.Children = {}
  ToDo_Frame:RegisterEvent("ADDON_LOADED")
  ToDo_Frame:RegisterEvent("PLAYER_ENTERING_WORLD")
  SlashCmdList["TODO"] = ToDo.OnSlash
  ToDo.QWShowHook = QuestWatchFrame.Show
  ToDo.CurrentLine = 1
  ToDo_Frame:SetBackdropColor(0.05, 0.05, 0.05, 0.0)
  ToDo.ResetLines()
  ToDo.OldQuests = nil
end

SLASH_TODO1 = "/todo"
