-- ******************************** Constants *******************************
TITAN_VECTOR_ID = "Vector"
TITAN_VECTOR_VERSION = "1.00";
TITAN_VECTOR_BUTTON_ICON = "Interface\\Icons\\Ability_Rogue_Sprint.blp";
TITAN_VECTOR_FREQUENCY = 0.5;

-- *********************************  Strings *******************************
TITAN_VECTOR_BASE_WORD = "Vector";
TITAN_VECTOR_TOOLTIP_TEXT = "Your flying angle and speed".."\n".."Right click for options.";
TITAN_VECTOR_SET_NEW_RATE = "Set/Reset current rate (while running at 100%)";
TITAN_VECTOR_RESET_ALL = "Reset all currently stored rates!";
TITAN_VECTOR_RESET_ALL_QUERY = "Are you sure you want to reset all the speed rates?";
TITAN_VECTOR_SET_DATA = "Set Data";
TITAN_VECTOR_UP_INDICATOR = "U";
TITAN_VECTOR_DOWN_INDICATOR = "D";
TITAN_VECTOR_HORIZONTAL_INDICATOR = "H";
TITAN_VECTOR_SHOW_LABEL_TEXT = "Show label text";

-- ******************************** Variables *******************************
local SpeedData = {};
local TimeSlice, VectorSpeed = 0, 0;
local XPosition, YPosition, LastX, LastY = 0, 0, 0, 0;
local ResetRate, RoundedPitch;

-- **************************************************************************
-- NAME : TitanPanelVectorButton_OnLoad()
-- DESC : Registers the mod when it loads
-- **************************************************************************
function TitanPanelVectorButton_OnLoad()
	this.registry = {
		id = TITAN_VECTOR_ID,
		version = TITAN_VECTOR_VERSION,
		menuText = TITAN_VECTOR_ID,
		tooltipTitle = TITAN_VECTOR_ID.." "..TITAN_VECTOR_VERSION,
		tooltipTextFunction = "TitanPanelVectorButton_GetToolTipText",
		buttonTextFunction = "TitanPanelVectorButton_GetButtonText",
		category = "Information",		
		icon = TITAN_VECTOR_BUTTON_ICON,
		iconWidth = 16,
		savedVariables = {
			ShowIcon = 1,
			ShowColoredText = 1,
			ShowLabelText = 1
		},
	}
	this:RegisterEvent("ZONE_CHANGED_NEW_AREA");
	this:RegisterEvent("VARIABLES_LOADED");
	this:RegisterEvent("PLAYER_ENTERING_WORLD");
	WorldMapFrame:HookScript("OnHide", function() MapHidden = 1; end)
	if (not VectorSavedArray) then 
	 	VectorSavedArray={};
		VectorSavedArray["SpeedData"] = {};
	end	
end

-- **************************************************************************
-- NAME : TitanPanelVectorButton_OnEvent()
-- DESC : This section will grab the events registered to the add on and act on them
-- **************************************************************************
function TitanPanelVectorButton_OnEvent()
	if ( event == "ZONE_CHANGED_NEW_AREA" ) then
		SetMapToCurrentZone();
	elseif ( event == "VARIABLES_LOADED" ) then
		SpeedData = VectorSavedArray["SpeedData"];
	elseif ( event == "PLAYER_ENTERING_WORLD" ) then
		LastX, LastY = GetPlayerMapPosition("player");
	end
end

-- *******************************************************************************************
-- Name: TitanPanelVectorButton_GetToolTipText()
-- Desc: Gets our tool-tip text, what appears when we hover over our item on the Titan bar.
-- *******************************************************************************************
function TitanPanelVectorButton_GetToolTipText()
	return TITAN_VECTOR_TOOLTIP_TEXT;
end

-- *******************************************************************************************
-- Name: TitanPanelVectorButton_GetButtonText()
-- Desc: Gets our button text, what appears on our button on the Titan bar.
-- *******************************************************************************************
function TitanPanelVectorButton_GetButtonText()
	if VectorSpeed == -2 then
		TITAN_VECTOR_BUTTON_TEXT = TITAN_VECTOR_SET_DATA;
	elseif VectorSpeed == -1 then
		TITAN_VECTOR_BUTTON_TEXT =  "??%";
	else
		TITAN_VECTOR_BUTTON_TEXT = floor(VectorSpeed + 0.5).."%";
	end
	if ( IsFlying() ) then
		SaveView(5);
		RoundedPitch = tonumber(string.format("%.2f", tonumber(GetCVar("cameraPitchD"))));
		if ( RoundedPitch == 0.00 ) then
			TITAN_VECTOR_BUTTON_TEXT = TITAN_VECTOR_HORIZONTAL_INDICATOR..format("%.2f", GetCVar("cameraPitchD") * -1).."\194\176".." - "..TITAN_VECTOR_BUTTON_TEXT;		
		elseif ( RoundedPitch > 0.00 ) then
			TITAN_VECTOR_BUTTON_TEXT = TITAN_VECTOR_DOWN_INDICATOR..format("%.2f", GetCVar("cameraPitchD")).."\194\176".." - "..TITAN_VECTOR_BUTTON_TEXT;
	  else
	  	TITAN_VECTOR_BUTTON_TEXT = TITAN_VECTOR_UP_INDICATOR..format("%.2f", GetCVar("cameraPitchD") * -1).."\194\176".." - "..TITAN_VECTOR_BUTTON_TEXT;
	  end
	end
	-- Set the TITAN_VECTOR_BUTTON_TEXT based on the ShowColoredText settings
	if (TitanGetVar(TITAN_VECTOR_ID, "ShowColoredText")) then
		TITAN_VECTOR_BUTTON_TEXT = TitanUtils_GetGreenText(TITAN_VECTOR_BUTTON_TEXT);
  end
  if (TitanGetVar(TITAN_VECTOR_ID, "ShowLabelText")) then
  	return TITAN_VECTOR_BASE_WORD..": "..TITAN_VECTOR_BUTTON_TEXT;
  else
  	return TITAN_VECTOR_BUTTON_TEXT;
  end
end

-- *******************************************************************************************
-- Name: TitanPanelRightClickMenu_PrepareVectorMenu
-- Desc: Builds the right click config menu
-- *******************************************************************************************
function TitanPanelRightClickMenu_PrepareVectorMenu()
	if ( UIDROPDOWNMENU_MENU_LEVEL == 1 ) then
		TitanPanelRightClickMenu_AddTitle(TitanPlugins[TITAN_VECTOR_ID].menuText);
		TitanPanelRightClickMenu_AddSpacer();
		LocalUIDropDownMenu_AddButton(TITAN_VECTOR_SET_NEW_RATE, TITAN_VECTOR_SET_NEW_RATE, SetNewRate, false, false, false, UIDROPDOWNMENU_MENU_LEVEL);
		LocalUIDropDownMenu_AddButton(TITAN_VECTOR_RESET_ALL, TITAN_VECTOR_RESET_ALL, QueryResetAll, false, false, false, UIDROPDOWNMENU_MENU_LEVEL);
		TitanPanelRightClickMenu_AddSpacer();
		TitanPanelRightClickMenu_AddToggleIcon(TITAN_VECTOR_ID);
		TitanPanelRightClickMenu_AddToggleColoredText(TITAN_VECTOR_ID);
		TitanPanelRightClickMenu_AddToggleVar(TITAN_VECTOR_SHOW_LABEL_TEXT, TITAN_VECTOR_ID, "ShowLabelText")
		TitanPanelRightClickMenu_AddSpacer();
		TitanPanelRightClickMenu_AddCommand(TITAN_PANEL_MENU_HIDE, TITAN_VECTOR_ID, TITAN_PANEL_MENU_FUNC_HIDE);
	end
end

-- *******************************************************************************************
-- Name: TitanPanelVectorButton_OnShow()
-- Desc: Function to create a timer object when the button is shown
-- *******************************************************************************************
function TitanPanelVectorButton_OnShow()
 -- first we check if a timer has been defined for our plugin
  local hasTimer = TitanPanel:HasTimer("TitanPanel"..TITAN_VECTOR_ID);
   if ( not TitanPanel:HasTimer("TitanPanel"..TITAN_VECTOR_ID) ) then
   	-- if no timer is present we need to define one like this
     TitanPanel:AddRepeatingTimer("TitanPanel"..TITAN_VECTOR_ID, TITAN_VECTOR_FREQUENCY, TitanPanelPluginHandle_OnUpdate, TITAN_VECTOR_ID, TITAN_PANEL_UPDATE_BUTTON);
   end
end
-- *******************************************************************************************
-- Name: TitanPanelVectorButton_OnHide()
-- Desc: Function to create a timer object when the button is hidden
-- *******************************************************************************************
function TitanPanelVectorButton_OnHide()
  -- since our plugin is hidden, it's most likely disabled
  -- so we check again if a repeating timer exists/is defined
  if ( TitanPanel:HasTimer("TitanPanel"..TITAN_VECTOR_ID) ) then
  	-- if a timer is found then we remove it
    TitanPanel:RemoveTimer("TitanPanel"..TITAN_VECTOR_ID);
  end
end

-- *******************************************************************************************
-- Name: TitanPanelVectorButton_OnUpdate(self, elapsed)
-- Desc: Function to react to the Global OnUpdate call for each frame and calculate the movement
-- *******************************************************************************************
function TitanPanelVectorButton_OnUpdate(self, elapsed)
	TimeSlice = TimeSlice + elapsed;
	-- If the TITAN_VECTOR_FREQUENCY timer hasn't fired yet then return
	if ( TimeSlice < TITAN_VECTOR_FREQUENCY ) then 
		return;
	else		
		-- Get the X,Y position of the player
		XPosition, YPosition = GetPlayerMapPosition("player");
	  if ( XPosition == 0 and YPosition == 0 ) then
			VectorSpeed = -1;
		elseif ( XPosition == LastX and YPosition == LastY ) then
			VectorSpeed = 0;			
		else						
			local MoveX, MoveY;
			MoveX = LastX - XPosition;
			MoveY = LastY - YPosition;
			-- The 2.25 is a fudge factor as the world is not square
			local Distance = sqrt((math.pow(MoveX, 2) * 2.25) + math.pow(MoveY, 2));		  
		  local Speed = Distance / TimeSlice;
		  
			-- Check to see if we have this data yet				
			if ( SpeedData[GetZoneText()] == nil ) then
				VectorSpeed = -2;
			else
				VectorSpeed = (Speed / SpeedData[GetZoneText()]) * 100;
			end
			
			-- If we are inserting or resetting the rate write it to the SpeedData array
			if ( ResetRate ) then
				if ( SpeedData[GetZoneText()] == nil ) then
					SpeedData[GetZoneText()] = Speed;
				else
					SpeedData[GetZoneText()] = Speed;
				end
				ResetRate = false;
				VectorSpeed = (Speed / SpeedData[GetZoneText()]) * 100;
			end				
		end
		LastX, LastY = XPosition, YPosition;
	end
	TimeSlice = 0
	TitanPanelButton_UpdateButton(TITAN_VECTOR_ID);
end

-- *******************************************************************************************
-- Name: SetNewRate()
-- Desc: Toggle the ResetRate boolean for next TitanPanelVectorButton_OnUpdate
-- *******************************************************************************************
function SetNewRate()
	ResetRate = true;
end

-- *******************************************************************************************
-- Name: ResetAll()
-- Desc: Reset all the stored SpeedData
-- *******************************************************************************************
function ResetAll()	
	SpeedData = {};
	VectorSavedArray["SpeedData"] = SpeedData;
end

-- *******************************************************************************************
-- Name: ResetAll
-- Desc: Query if the user really wants to reset all the stored data 
-- *******************************************************************************************
function QueryResetAll()
	StaticPopupDialogs.TITAN_VECTOR_RESET_ALL = StaticPopupDialogs.TITAN_VECTOR_RESET_ALL or {
		text = TEXT(TITAN_VECTOR_RESET_ALL_QUERY),
		button1 = TEXT(OKAY),
		button2 = TEXT(CANCEL),
		OnAccept = ResetAll(),
		timeout = 0,
		exclusive = 1
	}
	StaticPopup_Show("TITAN_VECTOR_RESET_ALL")
end

-- *******************************************************************************************
-- Name: LocalUIDropDownMenu_AddButton
-- Desc: Performs a default TitanPanel UIDropDownMenu_AddButton with parameters
-- *******************************************************************************************
function LocalUIDropDownMenu_AddButton(text, value, funcname, checked, keepshown, hasarrow, level)
	MenuItem = {};
  MenuItem.text = text;
  MenuItem.value = value;
  MenuItem.func = funcname;
  MenuItem.checked = checked;
  MenuItem.keepShownOnClick = keepshown;
  MenuItem.hasArrow = hasarrow;
	UIDropDownMenu_AddButton(MenuItem, level);
end