-- **************************************************************************
-- * TitanXP.lua
-- *
-- * By: TitanMod, Dark Imakuni, Adsertor and the Titan Development Team
-- *     (HonorGoG, jaketodd422, joejanko, Lothayer, Tristanian)
-- **************************************************************************

-- ******************************** Constants *******************************
TITAN_XP_ID = "XP";
TITAN_XP_FREQUENCY = 1;

-- ******************************** Variables *******************************

-- ******************************** Functions *******************************

-- **************************************************************************
-- NAME : TitanPanelXP_OnLoad()
-- DESC : Registers the plugin upon it loading
-- **************************************************************************
function TitanPanelXPButton_OnLoad()
     this.registry = { 
          id = TITAN_XP_ID,
          builtIn = 1,
          version = TITAN_VERSION,
          menuText = TITAN_XP_MENU_TEXT, 
          buttonTextFunction = "TitanPanelXPButton_GetButtonText",
          tooltipTitle = TITAN_XP_TOOLTIP, 
          tooltipTextFunction = "TitanPanelXPButton_GetTooltipText",
          iconWidth = 16,
          savedVariables = {
               ShowXPPerHourSession = 1,
               ShowIcon = 1,
               ShowLabelText = 1,
          }
     };

     this:RegisterEvent("PLAYER_ENTERING_WORLD");
     this:RegisterEvent("TIME_PLAYED_MSG");
     this:RegisterEvent("PLAYER_XP_UPDATE");
     this:RegisterEvent("PLAYER_LEVEL_UP");
end

-- **************************************************************************
-- NAME : TitanPanelXPButton_OnShow()
-- DESC : Display the icon in the bar
-- NOTE : For a lack of better check at the moment TitanPanel_ButtonAdded
--        is a global variable set to true only when a button has just been
--        added to the panel
-- **************************************************************************
function TitanPanelXPButton_OnShow()
     TitanPanelXPButton_SetIcon();
     if (TitanPanel_ButtonAdded) then
          RequestTimePlayed();
     end
end


-- **************************************************************************
-- NAME : TitanPanelXPButton_OnEvent(arg1, arg2)
-- DESC : Parse events registered to addon and act on them
-- VARS : arg1 = <research> , arg2 = <research>
-- **************************************************************************
function TitanPanelXPButton_OnEvent(arg1, arg2)
     if (event == "PLAYER_ENTERING_WORLD") then
          if (not this.sessionTime) then
               this.sessionTime = time();
               --RequestTimePlayed();
          end
          if (not this.initXP) then
               this.initXP = UnitXP("player");
               this.accumXP = 0;
               this.sessionXP = 0;
               this.startSessionTime = time();
          end
     elseif (event == "TIME_PLAYED_MSG") then
          -- Remember play time
          this.totalTime = arg1;
          this.levelTime = arg2;
     elseif (event == "PLAYER_XP_UPDATE") then
          if (not this.initXP) then
               this.initXP = UnitXP("player");
               this.accumXP = 0;
               this.sessionXP = 0;
               this.startSessionTime = time();
          end
          this.sessionXP = UnitXP("player") - this.initXP + this.accumXP;
     elseif (event == "PLAYER_LEVEL_UP") then
          this.levelTime = 0;
          this.accumXP = this.accumXP + UnitXPMax("player") - this.initXP;
          this.initXP = 0;
     end
end

-- **************************************************************************
-- NAME : TitanPanelXPButton_OnUpdate(elapsed)
-- DESC : Update button data
-- VARS : elapsed = <research>
-- **************************************************************************
function TitanPanelXPButton_OnUpdate(elapsed)
     TITAN_XP_FREQUENCY = TITAN_XP_FREQUENCY - elapsed;
     if (TITAN_XP_FREQUENCY <=0) then
          TITAN_XP_FREQUENCY = 1;
          TitanPanelButton_UpdateButton(TITAN_XP_ID);
          TitanPanelButton_UpdateTooltip();
     end
     if (this.totalTime) then
          this.totalTime = this.totalTime + elapsed;
          --this.sessionTime = this.sessionTime + elapsed;
          this.levelTime = this.levelTime + elapsed;
     end
end


-- **************************************************************************
-- NAME : TitanPanelXPButton_GetButtonText(id)
-- DESC : Calculate time based logic for button text
-- VARS : id = button ID
-- NOTE : Because the panel gets loaded before XP we need to check whether
--        the variables have been initialized and take action if they haven't
-- **************************************************************************
function TitanPanelXPButton_GetButtonText(id)
     if (TitanPanelXPButton.startSessionTime == nil) then
          return;
     end

     local button, id = TitanUtils_GetButton(id, true);
     local totalXP = UnitXPMax("player");
     local currentXP = UnitXP("player");
     local toLevelXP = totalXP - currentXP;     
     local sessionXP = button.sessionXP;
     local xpPerHour, xpPerHourText, timeToLevel, timeToLevelText;
     local sessionTime = time() - TitanPanelXPButton.startSessionTime;
     local levelTime = TitanPanelXPButton.levelTime;
     
     if (levelTime) then
          if (TitanGetVar(TITAN_XP_ID, "ShowXPPerHourSession")) then          
               xpPerHour = sessionXP / sessionTime * 3600;
               timeToLevel = TitanUtils_Ternary((sessionXP == 0), -1, toLevelXP / sessionXP * sessionTime);
          
               xpPerHourText = format(TITAN_XP_FORMAT, xpPerHour);
               timeToLevelText = TitanUtils_GetEstTimeText(timeToLevel)
          
               return TITAN_XP_BUTTON_LABEL_XPHR_SESSION, TitanUtils_GetHighlightText(xpPerHourText),
                    TITAN_XP_BUTTON_LABEL_TOLEVEL_TIME_SESSION, TitanUtils_GetHighlightText(timeToLevelText);
          else
               xpPerHour = currentXP / levelTime * 3600;
               timeToLevel = TitanUtils_Ternary((currentXP == 0), -1, toLevelXP / currentXP * levelTime);
          
               xpPerHourText = format(TITAN_XP_FORMAT, xpPerHour);
               timeToLevelText = TitanUtils_GetEstTimeText(timeToLevel);
          
               return TITAN_XP_BUTTON_LABEL_XPHR_LEVEL, TitanUtils_GetHighlightText(xpPerHourText),
                    TITAN_XP_BUTTON_LABEL_TOLEVEL_TIME_LEVEL, TitanUtils_GetHighlightText(timeToLevelText);
          end
     end
end

-- **************************************************************************
-- NAME : TitanPanelXPButton_GetTooltipText()
-- DESC : Display tooltip text
-- **************************************************************************
function TitanPanelXPButton_GetTooltipText()
     local totalTime = TitanPanelXPButton.totalTime;
     local sessionTime = time() - TitanPanelXPButton.startSessionTime;
     local levelTime = TitanPanelXPButton.levelTime;
     local totalXP = UnitXPMax("player");
     local currentXP = UnitXP("player");
     local toLevelXP = totalXP - currentXP;
     local currentXPPercent = currentXP / totalXP * 100;
     local toLevelXPPercent = toLevelXP / totalXP * 100;
     local xpPerHourThisLevel = currentXP / levelTime * 3600;
     local xpPerHourThisSession = this.sessionXP / sessionTime * 3600;
     local estTimeToLevelThisLevel = TitanUtils_Ternary((currentXP == 0), -1, toLevelXP / currentXP * levelTime);
     local estTimeToLevelThisSession = TitanUtils_Ternary((this.sessionXP == 0), -1, toLevelXP / this.sessionXP * sessionTime);
     
     return ""..
          TITAN_XP_TOOLTIP_TOTAL_TIME.."\t"..TitanUtils_GetHighlightText(TitanUtils_GetAbbrTimeText(totalTime)).."\n"..
          TITAN_XP_TOOLTIP_LEVEL_TIME.."\t"..TitanUtils_GetHighlightText(TitanUtils_GetAbbrTimeText(levelTime)).."\n"..
          TITAN_XP_TOOLTIP_SESSION_TIME.."\t"..TitanUtils_GetHighlightText(TitanUtils_GetAbbrTimeText(sessionTime)).."\n"..
          "\n"..          
          TITAN_XP_TOOLTIP_TOTAL_XP.."\t"..TitanUtils_GetHighlightText(totalXP).."\n".. 
          TITAN_XP_TOOLTIP_LEVEL_XP.."\t"..TitanUtils_GetHighlightText(format(TITAN_XP_PERCENT_FORMAT, currentXP, currentXPPercent)).."\n".. 
          TITAN_XP_TOOLTIP_TOLEVEL_XP.."\t"..TitanUtils_GetHighlightText(format(TITAN_XP_PERCENT_FORMAT, toLevelXP, toLevelXPPercent)).."\n"..
          TITAN_XP_TOOLTIP_SESSION_XP.."\t"..TitanUtils_GetHighlightText(this.sessionXP).."\n"..
          "\n"..
          TITAN_XP_TOOLTIP_XPHR_LEVEL.."\t"..TitanUtils_GetHighlightText(format(TITAN_XP_FORMAT, xpPerHourThisLevel)).."\n"..
          TITAN_XP_TOOLTIP_XPHR_SESSION.."\t"..TitanUtils_GetHighlightText(format(TITAN_XP_FORMAT, xpPerHourThisSession)).."\n"..
          TITAN_XP_TOOLTIP_TOLEVEL_LEVEL.."\t"..TitanUtils_GetHighlightText(TitanUtils_GetAbbrTimeText(estTimeToLevelThisLevel)).."\n"..
          TITAN_XP_TOOLTIP_TOLEVEL_SESSION.."\t"..TitanUtils_GetHighlightText(TitanUtils_GetAbbrTimeText(estTimeToLevelThisSession));
end

-- **************************************************************************
-- NAME : TitanPanelXPButton_SetIcon()
-- DESC : Define icon based on faction
-- **************************************************************************
function TitanPanelXPButton_SetIcon()
     local icon = TitanPanelXPButtonIcon;
     local factionGroup, factionName = UnitFactionGroup("player");     

     if (factionGroup == "Alliance") then
          icon:SetTexture("Interface\\TargetingFrame\\UI-PVP-Alliance");
          icon:SetTexCoord(0.046875, 0.609375, 0.03125, 0.59375);
     elseif (factionGroup == "Horde") then
          icon:SetTexture("Interface\\TargetingFrame\\UI-PVP-Horde");
          icon:SetTexCoord(0.046875, 0.609375, 0.015625, 0.578125);
     else
          icon:SetTexture("Interface\\TargetingFrame\\UI-PVP-FFA");
          icon:SetTexCoord(0.046875, 0.609375, 0.03125, 0.59375);
     end
end

-- **************************************************************************
-- NAME : TitanPanelRightClickMenu_PrepareXPMenu()
-- DESC : Display rightclick menu options
-- **************************************************************************
function TitanPanelRightClickMenu_PrepareXPMenu()     
     TitanPanelRightClickMenu_AddTitle(TitanPlugins[TITAN_XP_ID].menuText);

     local info = {};
     info.text = TITAN_XP_MENU_SHOW_XPHR_THIS_SESSION;
     info.func = TitanPanelXPButton_ShowXPPerHourSession;
     info.checked = TitanGetVar(TITAN_XP_ID, "ShowXPPerHourSession");
     UIDropDownMenu_AddButton(info);

     info = {};
     info.text = TITAN_XP_MENU_SHOW_XPHR_THIS_LEVEL;
     info.func = TitanPanelXPButton_ShowXPPerHourLevel
     info.checked = TitanUtils_Toggle(TitanGetVar(TITAN_XP_ID, "ShowXPPerHourSession"));
     UIDropDownMenu_AddButton(info);

     TitanPanelRightClickMenu_AddSpacer();
     TitanPanelRightClickMenu_AddCommand(TITAN_XP_MENU_RESET_SESSION, TITAN_XP_ID, "TitanPanelXPButton_ResetSession");
     TitanPanelRightClickMenu_AddCommand(TITAN_XP_MENU_REFRESH_PLAYED, TITAN_XP_ID, "TitanPanelXPButton_RefreshPlayed");

     TitanPanelRightClickMenu_AddSpacer();
     TitanPanelRightClickMenu_AddToggleIcon(TITAN_XP_ID);
     TitanPanelRightClickMenu_AddToggleLabelText(TITAN_XP_ID);

     TitanPanelRightClickMenu_AddSpacer();
     TitanPanelRightClickMenu_AddCommand(TITAN_PANEL_MENU_HIDE, TITAN_XP_ID, TITAN_PANEL_MENU_FUNC_HIDE);
end

-- **************************************************************************
-- NAME : TitanPanelXPButton_ShowXPPerHourSession()
-- DESC : Display per hour in session data in bar if set
-- **************************************************************************
function TitanPanelXPButton_ShowXPPerHourSession()
     TitanSetVar(TITAN_XP_ID, "ShowXPPerHourSession", 1);
     TitanPanelButton_UpdateButton(TITAN_XP_ID);
end

-- **************************************************************************
-- NAME : TitanPanelXPButton_ShowXPPerHourLevel()
-- DESC : Display per hour to level data in bar if set
-- **************************************************************************
function TitanPanelXPButton_ShowXPPerHourLevel()
     TitanSetVar(TITAN_XP_ID, "ShowXPPerHourSession", nil);
     TitanPanelButton_UpdateButton(TITAN_XP_ID);
end

-- **************************************************************************
-- NAME : TitanPanelXPButton_ResetSession()
-- DESC : Reset session and accumulated variables
-- **************************************************************************
function TitanPanelXPButton_ResetSession()
     TitanPanelXPButton.initXP = UnitXP("player");
     TitanPanelXPButton.accumXP = 0;
     TitanPanelXPButton.sessionXP = 0;
     TitanPanelXPButton.startSessionTime = time();
end

-- **************************************************************************
-- NAME : TitanPanelXPButton_RefreshPlayed()
-- DESC : Get total time played
-- **************************************************************************
function TitanPanelXPButton_RefreshPlayed()
     RequestTimePlayed();
end