-- TextSpy by Recluse@Norgannon


--				[[ ***** LOCAL SETTINGS ***** ]]
--[[----------------------------------------------------------------------------------
Notes:
* Setup our locals
 ------------------------------------------------------------------------------------]]
local Tablet = AceLibrary("Tablet-2.0")
local L = AceLibrary("AceLocale-2.2"):new("TextSpy")
local waterfall = AceLibrary:HasInstance("Waterfall-1.0") and AceLibrary("Waterfall-1.0")
local _G = getfenv(0)
--[[----------------------------------------------------------------------------------
Notes:
* The framework for a default trigger
 ------------------------------------------------------------------------------------]]
local defaultTrigger = {
	name = L["triggername"],
	text = L["trigger text"],
	enabled = true,
	casesensitive = false,
}
--[[----------------------------------------------------------------------------------
Notes:
* Allows TextSpy to copy one table into another.
Arguments:
table - the table to copy
Returns:
* table - the copy to be passed back
 ------------------------------------------------------------------------------------]]
local function TblCopy(t, lookup_table)
	local copy = {}
	if type(t) ~= "table" then return t end
	for i,v in pairs(t) do
		if type(v) ~= "table" then
			copy[i] = v
		else
			lookup_table = lookup_table or {}
			lookup_table[t] = copy
			if lookup_table[v] then
				copy[i] = lookup_table[v] -- we already copied this table. reuse the copy.
			else
				copy[i] = TblCopy(v,lookup_table) -- not yet copied. copy it.
			end
		end
	end
	return copy
end
--[[----------------------------------------------------------------------------------
Notes:
* Creates and removes key/vals in the current table, resulting in the current table
  having an identical table layout as the new table.
* All values for keys which are not to be removed from curT will be preserved.
* newT should contain default values, as they will be inserted into curT if not present.
Arguments:
* curT - The current table
* newT - The new table
 ------------------------------------------------------------------------------------]]
local function MigrateTable(curT, newT)
	--[[First, we need to add keys to the current table which need to be added.
		If a new key is added, we assign the default value provided by the new table passed in.
		After checking, determine if we are looking at a table, and if so, recurse into it.]]
	for newK, newV in pairs(newT) do
		if curT[newK] == nil then
			curT[newK] = newT[newK]
		end
		if (type(newT[newK]) == "table") then
			MigrateTable(curT[newK], newT[newK])
		end
	end
	--[[Second, we need to remove keys and their values which do not exist in the new table.
		Remember to set the key to nil to remove it, rather than calling table.remove()!
		After checking, determine if we are looking at a table, and if so, recurse into it.]]
	for curK, curV in pairs(curT) do
		if newT[curK] == nil then
			curT[curK] = nil
		end
		if (type(curT[curK]) == "table") then
			MigrateTable(curT[curK], newT[curK])
		end
	end
end
local t = {}
local eventMap = {
    CHAT_MSG_CHANNEL = true,
    CHAT_MSG_SAY = true,
    CHAT_MSG_GUILD = true,
    CHAT_MSG_WHISPER = true,
    CHAT_MSG_WHISPER_INFORM = true,
    CHAT_MSG_YELL = true,
    CHAT_MSG_PARTY = true,
    CHAT_MSG_OFFICER = true,
    CHAT_MSG_RAID = true,
    CHAT_MSG_RAID_LEADER = true,
    CHAT_MSG_RAID_WARNING = true,
    CHAT_MSG_BATTLEGROUND = true,
    CHAT_MSG_BATTLEGROUND_LEADER = true,
    CHAT_MSG_SYSTEM = true,
    CHAT_MSG_DND = true,
    CHAT_MSG_AFK = true,
}
local function safestr(s) return s or "" end
local NULL_INFO = {r = 1.0, g = 1.0, b = 1.0, id = 0 }

--				[[ ***** TEXTSPY SETUP ***** ]]
TextSpy = AceLibrary("AceAddon-2.0"):new("FuBarPlugin-2.0", "AceDB-2.0", "AceConsole-2.0", "AceEvent-2.0", "AceHook-2.1")
--[[----------------------------------------------------------------------------------
Notes:
* Set up our basic options.
 ------------------------------------------------------------------------------------]]
TextSpy.name = "TextSpy"
TextSpy.revision = tonumber(string.sub("$Revision: 49801 $", 12, -3))
TextSpy.version = "r"..TextSpy.revision
TextSpy.date = string.sub("$Date: 2007-09-26 02:08:45 -0400 (Wed, 26 Sep 2007) $", 8, 17)
TextSpy.hasIcon = "Interface\\AddOns\\TextSpy\\media\\images\\textspy"
TextSpy.cannotDetachTooltip = true
TextSpy.hideWithoutStandby = true
TextSpy:RegisterDB("TextSpyDB")
--[[----------------------------------------------------------------------------------
Notes:
* The basic TextSpy menu
 ------------------------------------------------------------------------------------]]
TextSpy.options = {
	type = 'group',
	args = {
		monitor = {
			type = 'toggle',
			name = L["Monitor Chat"],
			desc = L["Toggles whether or not chat is watched for trigger text. (Global)"],
			order = 5,
			get = "IsMonitoring",
			set = "ToggleMonitoring",
		},
		shortlabel = {
			type = 'toggle',
			name = L["Short Label"],
			desc = L["If checked, TextSpy will  show [ON] rather than TextSpy [ON]"],
			hidden = false,
			order = 10,
			get = "IsShortLabel",
			set = "ToggleShortLabel",
		},
		spacerOne = {
			name = " ",
			type = "header",
			order = 15,
		},
		alltriggers = {
			type = 'group',
			name = L["Triggers"],
			desc = L["Trigger Settings"],
			hidden = function() return not (#TextSpy.db.profile.triggers > 0) end,
			order = 20,
			args = {},
		},
		newTrigger = {
			type = 'execute',
			name = L["Create Trigger"],
			desc = L["Create a new trigger"],
			order = 25,
			func = function(v) TextSpy:CreateTrigger(nil) end,
		},
		globalSettings = {
			type = 'group',
			name = L["New Settings"],
			desc = L["Sets default options for new triggers"],
			hidden = false,
			order = 30,
			args = {}
		},
		spacerTwo = {
			name = " ",
			type = 'header',
			order = 35,
		},
		components = {
			type = 'group',
			name = L["Components"],
			desc = L["Components add functionality to triggers."],
			hidden = false,
			order = 40,
			args = {},
		},
		spacerThree = {
			name = " ",
			type = 'header',
			order = 45,
		},
	},
}
TextSpy.OnMenuRequest = TextSpy.options
--[[----------------------------------------------------------------------------------
* A list of all components which should be loaded.  Component scripts add themselves
  to this table.
* These get checked out and loaded by :InitializeComponents()
 ------------------------------------------------------------------------------------]]
TextSpy.componentList = {}
--[[----------------------------------------------------------------------------------
* eventID.current is incremented each CHAT_MSG event fired, eventID.lastProcessed is
  just a marker so we know if the current event has already been fired.  This prevents
  our triggers from firing twice off of the same CHAT_MSG event.
 ------------------------------------------------------------------------------------]]
TextSpy.eventID = {
	current = 0,
	lastProcessed = 0,
}
TextSpy.eventData = {}
TextSpy.Events = {
    PRE_ADDMESSAGE = "TextSpy_PreAddMessage",
    POST_ADDMESSAGE = "TextSpy_PostAddMessage",
}
TextSpy.CurrentMsg = nil
TextSpy.SplitMessageOut = {
    MESSAGE = "",
    TYPE = "",
    TARGET = "",
    CHANNEL = "",
    LANGUAGE = "",
}
TextSpy.SplitMessageOrg = {}
TextSpy.INFO = {r = 1.0, g = 1.0, b = 1.0, id = 0 }
TextSpy.SplitMessageSrc = {
    PRE = "",

    cC = "",
        CHANNELNUM = "",
        CC = "",
        CHANNEL = "",
        zZ = "",
            ZONE = "",
        Zz = "",
    Cc = "",

    TYPEPREFIX = "",

    fF = "",
        FLAG = "",
    Ff = "",

    pP = "",
        lL = "",  -- link start
        PLAYERLINK= "",
        PLAYERLINKDATA = "",
        LL = "", --  link text start

        PREPLAYER = "",
        PLAYER = "",

        sS = "",
            SERVER = "",
        Ss = "",
        POSTPLAYER = "",

        Ll = "",  -- link text end
    Pp = "",

    TYPEPOSTFIX = "",

    mM = "",
        gG = "",
        LANGUAGE = "",
        Gg = "",
        MESSAGE = "",
    Mm = "",

    POST = "",
}
TextSpy.Separators = {
    ZONE = "zz",
    SERVER = "ss",
    MESSAGE = "mm",
    LANGUAGE = "gg",
    PLAYERLINK = "ll",
    FLAG = "ff",
    CHANNEL = "cc",
    PLAYER = "pp",
}
TextSpy.SplitMessageIdx = {
    "PRE",
    "cC",
        "CHANNELNUM",
        "CC",
        "CHANNEL",
    "Cc",
    "TYPEPREFIX",
    "fF",
        "FLAG",
    "Ff",
    "pP",
        "lL",
            "PLAYERLINK",
            "PLAYERLINKDATA",
        "LL",
        "PREPLAYER",
        "PLAYER",
        "sS",
            "SERVER",
        "Ss",
        "POSTPLAYER",
        "Ll",
    "Pp",
    "TYPEPOSTFIX",
    "mM",
        "gG",
        "LANGUAGE",
        "Gg",
        "MESSAGE",
    "Mm",
    "POST",
}
TextSpy.SplitMessage = {}
TextSpy.fireData = {}
TextSpy.testData = {}
--[[----------------------------------------------------------------------------------
Notes:
* Registers TextSpy with Waterfall
 ------------------------------------------------------------------------------------]]
if waterfall then
	waterfall:Register("TextSpy",
						"aceOptions", TextSpy.options,
						"title", L["TextSpy Configuration"],
						"treeLevels", 5);
end


--				[[ ***** EVENT FUNCTIONS ***** ]]
--[[----------------------------------------------------------------------------------
Notes:
 * Checks to see if monitoring is turned on and updates the tooltip accordingly
 ------------------------------------------------------------------------------------]]
function TextSpy:OnTextUpdate()
	-- Check monitoring status
	if (self.db.profile.monitor) then
		-- Set monitoring to "on"
		monitorstatus = "|cff00ff00"..L["ON"].."|r"
	else
		-- Set monitoring to "off"
		monitorstatus = "|cffff0000"..L["OFF"].."|r"
	end
	-- Update the FuBar text with the monitoing status
	if (not(self.db.profile.shortlabel)) then
		thelabel = "TextSpy ["..monitorstatus.."]"
	else
		thelabel = "["..monitorstatus.."]"
	end
	self:SetText(thelabel)
end
--[[----------------------------------------------------------------------------------
Notes:
 * Sets the text for the tooltip when mousing over this addon
 ------------------------------------------------------------------------------------]]
function TextSpy:OnTooltipUpdate()
	-- Set our tooltip
	if waterfall then
		Tablet:SetHint(L["Left click for GUI configuration. Right click for menu configuration."])
	else
		Tablet:SetHint(L["Right click for options"])
	end
	TextSpy:CreateMenus()
end
--[[----------------------------------------------------------------------------------
Notes:
* Called when enabling a profile
* Calls our trigger initialization function
 ------------------------------------------------------------------------------------]]
function TextSpy:OnProfileEnable()
    self:InitTriggers()
end
--[[----------------------------------------------------------------------------------
Notes:
* Called when enabling our addon
* Calls our trigger initialization function
 ------------------------------------------------------------------------------------]]
function TextSpy:OnEnable()
	self:InitTriggers()
end
--[[----------------------------------------------------------------------------------
Notes:
* Checks if Waterfall is running when the user left-clicks on TextSpy.  If so, it opens the Waterfall GUI
 ------------------------------------------------------------------------------------]]
function TextSpy:OnClick()
	if waterfall then
		waterfall:Open("TextSpy")
	else
		self:Print(L["Waterfall-1.0 is required to access the GUI."])
	end
end
--[[----------------------------------------------------------------------------------
Notes:
* Initializes TextSpy
 ------------------------------------------------------------------------------------]]
function TextSpy:OnInitialize()
	self:RegisterAllEvents("OnEvent")
	self:RegisterEvent(TextSpy.Events.POST_ADDMESSAGE)
	self:RegisterEvent(TextSpy.Events.PRE_ADDMESSAGE)
	    -- Inbound Hooking
    self:Hook("ChatFrame_MessageEventHandler", true)
	-- Call the chat frame hook function so we can start listening to channels.
	self:HookChatFrames()
	-- This ensures that our defaultTrigger table is complete and updated in regards to
	-- what components are installed.
	for k,v in pairs(TextSpy.componentList) do
		defaultTrigger[k] = TextSpy.componentList[k].triggerDefaults
	end
	-- Default profile options (if they are not set, this is what they'll get set to
	self:RegisterDefaults('profile', {
		monitor = true,
		triggers = {},
		squelches = {},
		shortlabel = false,
		global = {},
		components = {},
	})
	-- This 'upgrades' our triggers if options were added/removed since we last ran WoW.
	for i in pairs(TextSpy.db.profile.triggers) do
		MigrateTable(TextSpy.db.profile.triggers[i], defaultTrigger)
	end
	MigrateTable(TextSpy.db.profile.global, defaultTrigger)
	
	-- Set up our slash command(s)
	self:RegisterChatCommand( L["AceConsole-Commands"], TextSpy.OnMenuRequest )
	-- Compatibility for no FuBar, though it shouldn't happen XD
	if not FuBar then
		self.OnMenuRequest.args.hide.guiName = L["HideIcon"]
		self.OnMenuRequest.args.hide.desc = L["HideIcon"]
	end
end
function TextSpy:OnDisable()
	self:UnregisterAllEvents()
end
function TextSpy:TextSpy_PreAddMessage(message, selfpointer, event, chatmessage, r, g, b, id)
	TextSpy.fireData.fired = false
	-- If monitoring is turned on...
	if TextSpy.db.profile.monitor then
		-- Check each trigger
		for k, v in pairs(TextSpy.db.profile.triggers) do
			-- If the trigger is enabled...
			if (v.enabled) then
				if (not(TextSpy.eventID.current == TextSpy.eventID.lastProcessed)) then
					-- Call to process the text to see if our string was found
					TextSpy:ProcessText(chatmessage, k)
				end
			end
		end
	end
end
function TextSpy:TextSpy_PostAddMessage(m, selfpointer, event, mOUTPUT, r, g, b, id)
	if TextSpy.db.profile.monitor then -- Check that TextSpy is supposed to check stuff
		if TextSpy.fireData.fired then -- This is set if the trigger search was found
			-- Now we just call the postFire() function for our components provided they and their triggers are all enabled
			if TextSpy.db.profile.triggers[TextSpy.fireData.index].enabled then
				if (not(TextSpy.eventID.current == TextSpy.eventID.lastProcessed)) then
					for k,v in pairs(TextSpy.db.profile.components) do
						if TextSpy.db.profile.components[k].enabled then
							if TextSpy.db.profile.triggers[TextSpy.fireData.index][k].enabled then
								TextSpy.componentList[k].functions.postFire(TextSpy.fireData)
							end
						end
					end
				end
			end
		end
	end
	TextSpy.fireData = {} -- Clear the data we saved about this fired trigger
	TextSpy.eventID.lastProcessed = TextSpy.eventID.current -- Mark event as fired
end

--				[[ ***** EVENT OVERRIDE FUNCTIONS ***** ]]

function TextSpy:AddMessage(frame, text, r, g, b, id)
    local s = TextSpy.SplitMessage
    if s.OUTPUT == nil and s.CAPTUREOUTPUT then
        self.INFO.r, self.INFO.g, self.INFO.b, self.INFO.id = r, g, b, id
        s.OUTPUT = text
        s.INFO = TextSpy.INFO
    else
        TextSpy.hooks[frame].AddMessage(frame, text, r, g, b, id)
    end
end
function TextSpy:ChatFrame_MessageEventHandler(event)
    local event = event
    local message, info
    local process = eventMap[event]
    local CMEResult
    message, info = self:SplitChatMessage(this, event)
    if not info then
       return self.hooks["ChatFrame_MessageEventHandler"](event)
    else
        local m = TextSpy.SplitMessage
        self.CurrentMsg = m
        m.OUTPUT  = nil
        m.DONOTPROCESS = nil
        m.CAPTUREOUTPUT = true
        CMEResult = self.hooks["ChatFrame_MessageEventHandler"](event)
        m.CAPTUREOUTPUT = false
        if type(m.OUTPUT) == "string" and not m.DONOTPROCESS then
            local r,g,b,id = self.INFO.r, self.INFO.g, self.INFO.b, self.INFO.id
            self:TriggerEvent(TextSpy.Events.PRE_ADDMESSAGE, message, this, event, self:BuildChatText(message), r,g,b,id )
            if not m.DONOTPROCESS then 
                if process then
                    m.OUTPUT  = self:BuildChatText(message)
                    this:AddMessage(m.OUTPUT, r,g,b,id);
                else
                    if type(m.OUTPUT) == "string" then
                        m.OUTPUT  = (m.PRE or "")..m.OUTPUT..(m.POST or "")
                        this:AddMessage(m.OUTPUT, r,g,b,id);
                    end
                end
                self:TriggerEvent(TextSpy.Events.POST_ADDMESSAGE,  m, this, event, m.OUTPUT, r,g,b,id)
            end
        end
        m.CAPTUREOUTPUT = nil
        m.OUTPUT  = nil
        m.INFO = nil
        self.CurrentMessage = nil
    end
    return CMEResult
end
--[[----------------------------------------------------------------------------------
Notes:
* Used to simply assign an eventID to the current event. Thanks, Sylvanaar.
 ------------------------------------------------------------------------------------]]
function TextSpy:OnEvent(...)
    local event = AceLibrary("AceEvent-2.0").currentEvent
	-- Save event data for passing to components.
	TextSpy.eventData = {}
	if arg2 then
		TextSpy.eventData.author = arg2
	end
	if arg8 then
		TextSpy.eventData.channelNumber = arg8
	end
	if arg9 then
		TextSpy.eventData.channelName = arg9
	end
    if type(event) == "string" and event ~= "CHAT_MSG_ADDON" and strsub(event, 1, 8) == "CHAT_MSG" then
		TextSpy.eventData.type = event
		-- Assign a 'serial number' to this event
		TextSpy.eventID.current = TextSpy.eventID.current + 1
    end
end
--[[----------------------------------------------------------------------------------
Notes:
* Uses AceHook to hook all of our chat windows, filtering the Chat_Frame:AddMessage() function through the TextSpy:AddMessage() function
* This allows TextSpy to scan the incoming text for any trigger matches.
 ------------------------------------------------------------------------------------]]
function TextSpy:HookChatFrames()
	-- Loop uses NUM_CHAT_WINDOWS rather than a static number in case Blizzard adds more windows in the future.
	for i=1,NUM_CHAT_WINDOWS do
		-- Create a reference to the chat window based on our loop integer
		local cf = getglobal("ChatFrame"..i);
		-- Hook AddMessage() for this chat window
		self:Hook(cf, "AddMessage", true)
	end
end
--[[----------------------------------------------------------------------------------
Notes:
* Hooked ChatFrame_AddMessage()
* Checks to see if we should check the incoming message for our trigger or not
Arguments:
table - a table containing information about the frame which this message is destined for
string - the text which will be displayed
integer - red value
integer - green value
integer - blue value
string - a flag contianing the type of message this is
Returns:
The original AddMessage()
 ------------------------------------------------------------------------------------]]


--				[[ ***** TRIGGER FUNCTIONS ***** ]]

--[[----------------------------------------------------------------------------------
Notes:
 * Changes the name of a trigger
 Arguments:
 string - the new name of the trigger
 integer - the index of the trigger to set the new name on
 ------------------------------------------------------------------------------------]]
function TextSpy:ChangeName(trigname,i)
	if not trigname then trigname = "" end
	TextSpy.OnMenuRequest.args.alltriggers.args["TextSpy"..i].name = i..". "..trigname
	TextSpy.OnMenuRequest.args.alltriggers.args["TextSpy"..i].desc = i..". "..trigname
end
--[[----------------------------------------------------------------------------------
Notes:
* Removes a trigger from the database
Arguments:
integer - the index of the trigger to delete
 ------------------------------------------------------------------------------------]]
function TextSpy:RemoveTrigger(index)
	local triggers = TextSpy.db.profile.triggers
	triggers[index] = {}
	table.remove(triggers,index)
	-- Recreate our menus since a trigger was removed
	self:CreateMenus()
end
--[[----------------------------------------------------------------------------------
Notes:
 * Creates a new trigger in the TextSpy database
 Arguments:
 integer - set to nil or 0 will create a trigger at the end of the list of triggers
 ------------------------------------------------------------------------------------]]
function TextSpy:CreateTrigger(index)
	local triggers = self.db.profile.triggers

	if not index or index == 0 then
		table.insert(triggers, TblCopy(self.db.profile.global))
		index = #triggers
	end
	-- Recreate our menu since a trigger was added
	self:CreateMenus()
end
--[[----------------------------------------------------------------------------------
Notes:
 * Checks the incoming text to determine if it matches our search string.
 * Plays the sound file associated with this search string, if it is found
 * Sends a text message to SCT and/or Parrot if they are installed, enabled, and not squelched.
 Arguments:
string - the text which we are going to search inside of
integer - the index of the trigger containing the search string
------------------------------------------------------------------------------------]]
function TextSpy:ProcessText(haystack, index)
	local needle = self.db.profile.triggers[index].text
	-- Check that we have data
	if ((needle) and (haystack)) then
		-- If the text we're going to search in is smaller than what we're looking for, no need to search
		if ((string.len(haystack)) >= (string.len(needle))) then 
			-- Check if our string is found
			if ((((not(self.db.profile.triggers[index].casesensitive)) and string.find(string.lower(haystack),string.lower(needle))) or (string.find(haystack,needle)))) then
				local match = nil
				if (not(self.db.profile.triggers[index].casesensitive)) then
					_, _, match = string.find(string.lower(haystack),string.lower(needle))
				else
					_, _, match = string.find(haystack,needle)
				end
				-- create a table of useful data to pass to components
				TextSpy.fireData = {
					fired = true,
					needle = needle,
					haystack = haystack,
					match = match,
					index = index,
					event = TblCopy(TextSpy.eventData)
				}
				-- fire any components which are enabled for the trigger that fired
				for k,v in pairs(TextSpy.db.profile.components) do
					if TextSpy.db.profile.components[k].enabled then
						if TextSpy.db.profile.triggers[index][k].enabled then
							TextSpy.componentList[k].functions.fire(TextSpy.fireData)
						end
					end
				end
			end
		end
	end
end

--				[[ ***** MENU FUNCTIONS ***** ]]

--[[----------------------------------------------------------------------------------
Notes:
 * Initializes our triggers (updates the menu)
 ------------------------------------------------------------------------------------]]
function TextSpy:InitTriggers()
	-- Create global option menu
	TextSpy.OnMenuRequest.args.globalSettings.args = self:CreateTriggerMenu(0)
	-- Create triggers
	for i in pairs(TextSpy.db.profile.triggers) do
		self:CreateTrigger(i)
	end
	-- Create our full menu
	self:CreateMenus()
end
--[[----------------------------------------------------------------------------------
Notes:
 * Creates the trigger portion of our menu
 * Passing in a 0 as the argument flags it as the global trigger settings menu
 Arguments:
 integer - the index of the trigger to create a menu for.
 ------------------------------------------------------------------------------------]]
function TextSpy:CreateTriggerMenu(index)
	-- Stores the namespace for each trigger
	local triggerData
	-- If the index is 0, this is the global trigger setting menu and not a user-created one. Set the namespace appropriately.
	if index == 0 then
		triggerData = self.db.profile.global
	else
		triggerData = self.db.profile.triggers[index]
	end
	
	-- Create the table for each trigger's menu
	local option = {
		triggerenabled = {
			type='toggle',
			name = L["Trigger Enabled"],
			desc = L["Monitor chat for this trigger"],
			get = function()
				return triggerData.enabled
			end,
			set = function()
				triggerData.enabled = (not triggerData.enabled)
			end,
			order = 5,
		},
		spacerOne = {
			type = 'header',
			name = " ",
			order = 10,
		},
		triggercase = {
			type='toggle',
			name = L["Case Sensitive"],
			desc = L["Distinguish between upper/lowercase letters. If checked 'CASE' will not match 'case'.  If unchecked, 'CASE' will match 'case'"],
			get = function()
				return triggerData.casesensitive
			end,
			set = function()
				triggerData.casesensitive = (not triggerData.casesensitive)
			end,
			order = 15,
		},
		triggername = {
			type='text',
			name = L["Trigger Name"],
			desc = L["Short name for this trigger"],
			usage = "DeepBreath, PartyMbrOom",
			get = function() return triggerData.name end,
			set = function(val)
				if index ~= 0 then
					triggerData.name = val
					self:ChangeName(triggerData.name,index)
				end
			end,
			hidden = function() if index == 0 then return true else return false end end,
			order = 20,
			canCopy = true,
			copyType = "text",
		},
		triggertext = {
			type='text',
			name = L["Trigger Text"],
			desc = L["Text to search chatbox for"],
			usage = L["out of mana, begins to grow, yourcharactername, etc"],
			get = function()
				return triggerData.text
			end,
			set = function(val) 
				triggerData.text = val
			end,
			order = 25,
			canCopy = true,
			copyType = "text",
		},
		spacerTwo = {
			name = " ",
			type = 'header',
			order = 29,
		},
		spacerThree = {
			name = L["Components"],
			type = "header",
			order = 30,
		},
		spacerFour = {
			name = " ",
			type = 'header',
			order = 98,
		},
		triggerdel = {
			type='execute',
			name = L["Delete Trigger"],
			desc = L["Deletes this trigger"],
			hidden = function() if index == 0 then return true else return false end end,
			order = 99,
			func = function(v) TextSpy:RemoveTrigger(index) end,
		},
	}
	-- Add component trigger menus here
	componentOrder = 31 -- Start value, increment by 1 in loop
	for k,v in pairs(TextSpy.componentList) do
		option[k] = TextSpy.componentList[k].triggerMenu(index)
		option[k].order = componentOrder
		componentOrder = componentOrder + 1
	end
	-- Create a group with this name if we are not adding in the global settings menu
	if index ~= 0 then
		return {
			type = 'group',
			name = index..'. '..triggerData.name, 
			desc = index..'. '..triggerData.name, 
			order = index,
			args = option
		}
	else
		return option
	end
	
end
--[[----------------------------------------------------------------------------------
Notes:
 * Creates the TextSpy menu
 ------------------------------------------------------------------------------------]]
function TextSpy:CreateMenus()
	self.OnMenuRequest.args.alltriggers.args = {}
	self.OnMenuRequest.args.globalSettings.args = self:CreateTriggerMenu(0)
	for i in pairs(TextSpy.db.profile.triggers) do
		self.OnMenuRequest.args.alltriggers.args["TextSpy"..i] = self:CreateTriggerMenu(i)
	end
	self:InitializeComponents()
end


--				[[ ***** SETTINGS FUNCTIONS ***** ]]

--[[----------------------------------------------------------------------------------
Notes:
* Checks if user has the short label flag set
 ------------------------------------------------------------------------------------]]
function TextSpy:IsShortLabel()
	return self.db.profile.shortlabel
end
--[[----------------------------------------------------------------------------------
Notes:
* Turns the short label flag on or off
 ------------------------------------------------------------------------------------]]
function TextSpy:ToggleShortLabel()
	self.db.profile.shortlabel = (not(self.db.profile.shortlabel))
	self:OnTextUpdate()
end
--[[----------------------------------------------------------------------------------
Notes:
* Checks to see if the addon is actively monitoring chat (user option)
Returns:
bool - true if chat is being monitored, false if chat is not being monitored
 ------------------------------------------------------------------------------------]]
function TextSpy:IsMonitoring()
	return self.db.profile.monitor
end
--[[----------------------------------------------------------------------------------
Notes:
* Toggles the global monitoring setting.
 ------------------------------------------------------------------------------------]]
function TextSpy:ToggleMonitoring()
	self.db.profile.monitor = (not self.db.profile.monitor)
	self:OnTextUpdate()
end
--[[----------------------------------------------------------------------------------
Notes:
* Sets up any components which have registered themselves in the components table.
 ------------------------------------------------------------------------------------]]
function TextSpy:InitializeComponents()
	-- Creates and inserts default component options.
	componentSettings = {}
	for k,v in pairs(TextSpy.componentList) do
		componentSettings[k] = {
			enabled = false,
		}
	end
	MigrateTable(TextSpy.db.profile.components, componentSettings)
	
	-- Creates and inserts menu items for turning components on and off.
	componentOrder = 0
	componentMenus = {}
	for k,v in pairs(TextSpy.componentList) do
		componentOrder = componentOrder + 5
		componentMenus[k] = {
			type = 'toggle',
			name = TextSpy.componentList[k].name,
			desc = TextSpy.componentList[k].desc,
			order = componentOrder,
			get = function()
				return TextSpy.db.profile.components[k].enabled
			end,
			set = function()
				TextSpy.db.profile.components[k].enabled = not TextSpy.db.profile.components[k].enabled
			end,
		}
	end
	TextSpy.OnMenuRequest.args.components.args = componentMenus
end


--				[[ ***** TEXT PROCESSING ***** ]]

function TextSpy:ClearSplitItem(split, item)
    if item and split and split[item] then
        local seps = TextSpy.Separators[item]

        split[item] = ""
        if seps then
            for k,v in pairs(split) do
                if strlower(k) == seps then
                    v = ""
                end
            end
        end
    end
end
function TextSpy:SplitChatMessage(frame, event)
    self:ClearChatSections(self.SplitMessage)
    if ( strsub(safestr(event), 1, 8) == "CHAT_MSG" ) then
        local type = strsub(event, 10);
        local info = ChatTypeInfo[type];
        local s = self.SplitMessage
        s.CHATTYPE = type
        s.MESSAGE = safestr(arg1)
        local chatget = getglobal("CHAT_"..type.."_GET")
        if chatget then
            s.TYPEPREFIX, s.TYPEPOSTFIX = string.match(TEXT(chatget), "(.*)%%s(.*)")
        end
        s.TYPEPOSTFIX = safestr(s.TYPEPOSTFIX)
        s.TYPEPREFIX = safestr(s.TYPEPREFIX)
        arg2 = safestr(arg2)
        if strlen(arg2) > 0 then

        	if ( strsub(type, 1, 7) == "MONSTER" or type == "RAID_BOSS_EMOTE" ) then
        		-- no link
        	else
               local plr, svr = strsplit("-", arg2)


                s.pP = "["
                s.lL = "|Hplayer:"
                s.PLAYERLINK = arg2
                s.LL = "|h"
                s.PLAYER = plr

                if svr and strlen(svr) > 0 then
                    s.sS = "-"
                    s.SERVER = svr
                end

                if arg11 then
                    s.PLAYERLINKDATA = ":"..safestr(arg11)
                end

                s.Ll = "|h"
                s.Pp = "]"
            end
        end
        arg6 = safestr(arg6)
        if strlen(arg6) > 0 then
            s.fF = ""
            s.FLAG = TEXT(getglobal("CHAT_FLAG_"..arg6))
            s.Ff = ""
        end
        arg3 = safestr(arg3)
        if ( (strlen(arg3) > 0) and (arg3 ~= "Universal") and (arg3 ~= this.defaultLanguage) ) then
            s.gG = "["
            s.LANGUAGE = arg3
            s.Gg = "] "
        else
            s.LANGUAGE_NOSHOW = arg3
        end
        arg9 = safestr(arg9)
        if strlen(arg9) > 0 then
            local bracket, post_bracket = string.match(s.TYPEPREFIX, "%[(.*)%](.*)")
            bracket = safestr(bracket)
            if strlen(bracket) > 0 then
                s.cC = "["
                s.Cc = "]"
                s.CHANNEL = bracket
                s.TYPEPREFIX = safestr(post_bracket)
            end


            if strlen(safestr(arg8)) > 0 and arg8 > 0 then
                s.CHANNELNUM = tostring(arg8)
                s.CC = ". "
            end

            if arg7 > 0 then
                s.cC = "["
                s.Cc = "] "
                s.CHANNEL, s.zZ, s.ZONE = string.match(arg9, "(.*)(%s%-%s)(.*)")

                if s.CHANNEL == nil then
                    s.CHANNEL = arg9
                end

                s.CHANNEL = safestr(s.CHANNEL)
                s.zZ = safestr(s.zZ)
                s.ZONE = safestr(s.ZONE)

--                s.CHANNEL = gsub(arg9, "(.*)(%s%-%s)(.*)", "%1")
--                s.zZ = gsub(arg9, "(.*)(%s%-%s)(.*)", "%2")
--                s.ZONE = gsub(arg9, "(.*)(%s%-%s)(.*)", "%3")
                s.Zz = ""
            else
                if strlen(arg9) > 0 then
                    s.CHANNEL = arg9
                    s.cC = "["
                    s.Cc = "] "
                end
            end
        end
        self:ClearChatSections(self.SplitMessageOrg)
        for k,v in pairs(s) do
            self.SplitMessageOrg[k] = v
        end
        if type == "SYSTEM" then
            local pl, p, rest = string.match(s.MESSAGE, "|Hplayer:(.-)|h%[(.-)%]|h(.+)")
            if pl and p then
                local plr, svr = strsplit("-", pl)
                s.pP = "["
                s.lL = "|Hplayer:"
                s.PLAYERLINK = pl
                s.LL = "|h"
                s.PLAYER = plr
                s.Ll = "|h"
                s.Pp = "]"
                s.MESSAGE = rest

                if svr and strlen(svr) > 0 then
                    s.sS = "-"
                    s.SERVER = svr
                end

                if arg11 then
                    s.PLAYERLINKDATA = ":"..safestr(arg11)
                end
            end
        end
        s.ORG = self.SplitMessageOrg
        return self.SplitMessage, info
    end
end
function TextSpy:ValidateMessageTable(message, reporterror)
     -- Ensure the standard stuff isnt nil
    for k,v in pairs(self.SplitMessageSrc) do
        if message[k] then
            message[k] = message[k]
        else
            if reporterror then self:Print("ValidateMessageTable - BAD KEY: ", k) end
            message[k] = ""
        end
    end

    message.INFO = message.INFO or NULL_INFO
end
function TextSpy:BuildChatText(message, index)
    local index = index or self.SplitMessageIdx
    local s = message

    for k in pairs(t) do
        t[k] = nil
    end

    for i,v in ipairs(index) do
        local part = s[v]
        if type(part) == "string" and strlen(part) > 0 then
            table.insert(t, part)
        end
    end

    return table.concat(t, "")
end
function TextSpy:ClearChatSections(message)
    for k,v in pairs(message) do
        message[k] = self.SplitMessageSrc[k] and "" or nil
    end
end
