
TtpsFrames = {};

local L = AceLibrary("AceLocale-2.2"):new("TankadinTps");
local Graph = AceLibrary("Graph-1.0");
local Dewdrop = AceLibrary("Dewdrop-2.0");

local detailsFrame = nil;
local reportFrame = nil;

local tblDataMode = "tps" -- || "dmg"
local rowDataDetails = nil;
local rowDataPinDetails = nil;
local rowData = nil;
local rowDataPin = nil;
local dataReference = {key=nil, source=nil};
local dataReferencePin = {key=nil, source=nil};

local MAXDETAILROWS = 8;

-- report options
local reportToOptions = {
	type = "group",
	handler = TtpsFrames,
	args = {
		whisperTar = {
		   type = "execute",
		   name = L["reportToWhisperTar"],
		   desc = L["reportToDesc"],
		   func = function() local g,s = TtpsFrames:GetDataReference(); TtpsFrames:Export(g, s, "WHISPER", UnitName("target")); end,
		   order = 10,
		},
		whisper = {
			type = "text",
			name = L["reportToWhisper"],
			desc = L["reportToDesc"],
			usage = "<name>",
			get = false,
			set = function(v) local g,s = TtpsFrames:GetDataReference(); TtpsFrames:Export(g, s, "WHISPER", v); end,
			validate = function(v) return type(v) == "string" and v:trim():len() > 0 end,
			order = 15,
			--disabled = function() return false end,
		},
		raid = {
		   type = "execute",
		   name = L["reportToRaid"],
		   desc = L["reportToDesc"],
		   func = function() local g,s = TtpsFrames:GetDataReference(); TtpsFrames:Export(g, s, "RAID"); end,
		   order = 20,
		},
		guild = {
		   type = "execute",
		   name = L["reportToGuild"],
		   desc = L["reportToDesc"],
		   func = function() local g,s = TtpsFrames:GetDataReference(); TtpsFrames:Export(g, s, "GUILD"); end,
		   order = 30,
		},
	}
};

-- get an entry from the correct table (source)
local function getData(key, source)
	local data;
	if (source == "data") then
		data = TtpsData:GetDataOutOfCombat(key);
	elseif (source == "saved") then
		data = TtpsData:GetSavedData(key);
	elseif (source == "imported") then
		data = TtpsData:GetImportedData(key);
	else
		data = TtpsData:GetDataOutOfCombat(key);
		source = "data";
	end
	
	return data;
end

-- update the table scrollframe
local function updateTableScrollFrame(frame)
  local line;
  local offset; -- an index into our data calculated from the scroll offset
  
  local numLines = 0;
  
  local rData;
  if (frame == detailsFrame.Pinned) then
	rData = rowDataPin;
  else
	rData = rowData;
  end
  
  if (rData) then
	numLines = #rData;
  end
  
  FauxScrollFrame_Update(frame.ScrollFrame,numLines,MAXDETAILROWS, 11);
  
  for line=1,MAXDETAILROWS do
    offset = line + FauxScrollFrame_GetOffset(frame.ScrollFrame);
	
    if offset <= numLines then
		frame.rows[line].ColKey:SetVertexColor(rData[offset].color[1], rData[offset].color[2], rData[offset].color[3], 1.0);
		
		local n, r = GetSpellInfo(rData[offset].skillId);
		frame.rows[line].Name:SetText((n or L["auto-hits"])..((r and " ("..r..")") or ""));
		frame.rows[line].Count:SetText(rData[offset].count);
		frame.rows[line].Total:SetText(string.format("%.1f", rData[offset].sumValues));
		frame.rows[line].ValueAvg:SetText(string.format("%.1f", rData[offset].valueAvg));
		frame.rows[line].Col1:SetText(string.format("%.2f", rData[offset].col1));
		frame.rows[line].Col2:SetText(string.format("%.2f", rData[offset].col2));
		frame.rows[line]:Show();
		
		if (rData[offset].showDetails) then
			frame.rows[line].Background:Show();
		else
			frame.rows[line].Background:Hide();
		end
    else
		frame.rows[line]:Hide();
    end
  end
end

-- update the details scrollframe
local function updateDetailsScrollFrame(frame)
  local line;
  local offset; -- an index into our data calculated from the scroll offset
  
  local numLines = 0;
  
  local rData;
  if (frame == detailsFrame.Pinned) then
	rData = rowDataPinDetails.rows;
  else
	rData = rowDataDetails.rows;
  end
  
  if (rData) then
	numLines = #rData;
  end
  
  FauxScrollFrame_Update(frame.DetailsScrollFrame,numLines,MAXDETAILROWS, 11);
  
  for line=1,MAXDETAILROWS do
    offset = line + FauxScrollFrame_GetOffset(frame.DetailsScrollFrame);
	
    if offset <= numLines then
		frame.sDetailBox.rows[line].Text:SetText(rData[offset].txt);
		frame.sDetailBox.rows[line]:Show();
    else
		frame.sDetailBox.rows[line]:Hide();
    end
  end
end

-- shows tooltip
local function detailsTooltip(i, frame)

	local offset;
	local hData, rData;
	if (frame == detailsFrame.Pinned) then
		rData = rowDataPinDetails.rows;
		hData = rowDataPinDetails.head;
	else
		rData = rowDataDetails.rows;
		hData = rowDataDetails.head;
	end
	
	offset = i + FauxScrollFrame_GetOffset(frame.DetailsScrollFrame);
	
	if (rData[offset].tip) then	
		GameTooltip:SetOwner(frame.sDetailBox.rows[i], "ANCHOR_BOTTOMRIGHT");
		GameTooltip:SetText("# |cffffffff"..hData.txt.."|r", hData.col[1], hData.col[2], hData.col[3]);
		GameTooltip:AddLine(rData[offset].tip, 1.0, 1.0, 1.0);
		
		GameTooltip:Show();
	end
end

-- called by table rows
local function rowClicked(i, frame)

	local scrollFrame = frame.ScrollFrame;
	local offset = i + FauxScrollFrame_GetOffset(scrollFrame);
	
	local rData;
	if (frame == detailsFrame.Pinned) then
		rData = rowDataPin;
	else
		rData = rowData;
	end
	
	for i=1,#rData do
		if (i ~= offset) then
			rData[i].showDetails = false;
		end
	end
	
	local dBox = frame.sDetailBox;
	local d = rData[offset];
	
	if (d.showDetails) then
		d.showDetails = false;
		frame.Pie:SetAlpha(1.0);
		dBox:Hide();
	else
		d.showDetails = true;
		frame.Pie:SetAlpha(0.2);
		
		local n, r = GetSpellInfo(rData[offset].skillId);
		dBox.Heading.ColKey:SetVertexColor(d.color[1], d.color[2], d.color[3], 1.0);
		dBox.Heading.Text:SetText((n or L["auto-hits"]));
				
		-- set text
		local detailsRow = {};
		local hitRows = {};
		local mitigationRows = {};
		local missRows = {};
		
		table.insert(detailsRow, {txt=string.format(L["min"]..": %d", d.skillData.minValue), tip=nil});
		table.insert(detailsRow, {txt=string.format(L["max"]..": %d", d.skillData.maxValue), tip=nil});
		
		for k,v in pairs(d.hits) do
			if (v.count > 0) then
				if (k == "BLOCK" or k == "RESIST" or k == "ABSORB") then
					local tipTxt;
					if (tblDataMode == "dmg") then
						tipTxt = string.format("%d "..L["damage"].."\n%d "..L[k].." (%.2f%%)", (v.sum+v.mtgSum), v.mtgSum, ((v.mtgSum/(v.sum+v.mtgSum))*100));
					else
						tipTxt = string.format("%d "..L["threat"].."\n%d "..L[k].." (%.2f%%)", (v.sum+v.mtgSum), v.mtgSum, ((v.mtgSum/(v.sum+v.mtgSum))*100));
					end
					table.insert(mitigationRows, {
								txt=string.format(L[k]..": %d (%.1f%%)", v.count, (v.count/(d.count)*100)),
								tip=tipTxt
								});
				else
					table.insert(hitRows, {
								txt=string.format(L[k]..": %d (%.1f%%)", v.count, (v.count/(d.count)*100)),
								tip=nil;
								});
				end
			end
		end
		if (#hitRows > 0) then
			table.insert(detailsRow, {txt=string.format(""), tip=nil});
			table.insert(detailsRow, {txt=string.format("|cffffffff"..L["hits"]..":"), tip=nil});
			for k,v in pairs(hitRows) do
				table.insert(detailsRow, v);
			end
		end
		if (#mitigationRows > 0) then
			table.insert(detailsRow, {txt=string.format(""), tip=nil});
			table.insert(detailsRow, {txt=string.format("|cffffffff"..L["mitigated"]..":"), tip=nil});
			for k,v in pairs(mitigationRows) do
				table.insert(detailsRow, v);
			end
		end
		
		for k,v in pairs(d.misses) do
			if (v > 0) then
				table.insert(missRows, {txt=string.format(L[k]..": %d (%.1f%%)", v, (v/(d.count)*100)), tip=nil});
			end
		end
		if (#missRows > 0) then
			table.insert(detailsRow, {txt=string.format(""), tip=nil});
			table.insert(detailsRow, {txt=string.format("|cffffffff"..L["missed"]..":"), tip=nil});
			for k,v in pairs(missRows) do
				table.insert(detailsRow, v);
			end
		end

		if (frame == detailsFrame.Pinned) then
			rowDataPinDetails = {head={txt=(n or L["auto-hits"]),col=d.color}, rows=detailsRow};
		else
			rowDataDetails = {head={txt=(n or L["auto-hits"]),col=d.color}, rows=detailsRow};
		end

		updateDetailsScrollFrame(frame);
		dBox:Show();
	end
	
	updateTableScrollFrame(frame);
end

-- opens the 2nd window
local function pinClicked()
	if (detailsFrame.Pinned:IsShown()) then
		detailsFrame.Pinned:Hide();
		
		detailsFrame.PinBtn:SetNormalTexture("Interface/ChatFrame/UI-ChatIcon-ScrollUp-Up");
	else
		TtpsFrames:PinDetails(dataReference.key, dataReference.source);
		
		detailsFrame.PinBtn:SetNormalTexture("Interface/ChatFrame/UI-ChatIcon-ScrollDown-Up");
	end
end

-- returns successor of (key, source)
local function getNextData(key, source)
	local cData, fData;
	local fSource;
	if (source == "data") then
		cData = TtpsData:GetDataOutOfCombat();
		fData = TtpsData:GetImportedData();
		fSource = "imported";
	elseif (source == "imported") then
		cData = TtpsData:GetImportedData();
		fData = TtpsData:GetSavedData();
		fSource = "saved";
	elseif (source == "saved") then
		cData = TtpsData:GetSavedData();
		fData = TtpsData:GetDataOutOfCombat();
		fSource = "data";

	end
	
	if (key < #cData) then
		return (key+1), source;
	else
		if (#fData == 0) then
			return getNextData(1, fSource);
		else
			return 1, fSource;
		end
	end
end

-- returns predecessor of (key, source)
local function getPreviousData(key, source)
	local cData, fData;
	local fSource;
	if (source == "data") then
		cData = TtpsData:GetDataOutOfCombat();
		fData = TtpsData:GetSavedData();
		fSource = "saved";
	elseif (source == "imported") then
		cData = TtpsData:GetImportedData();
		fData = TtpsData:GetDataOutOfCombat();
		fSource = "data";
	elseif (source == "saved") then
		cData = TtpsData:GetSavedData();
		fData = TtpsData:GetImportedData();
		fSource = "imported";
	end
	
	if (key > 1) then
		return (key-1), source;
	else
		if (#fData == 0) then
			return getPreviousData(1, fSource)
		else
			return (#fData), fSource;
		end
	end
end


local function getRowDataThreat(tblData)
	local returnData = {};
	
	-- time in ms
	local dTime = tblData.timeEnd - tblData.timeStart - tblData.timePaused;
	local totalTime = tblData.timeEnd - tblData.timeStart;
	
	if (tblData.skills) then 
		for k,v in pairs(tblData.skills) do
			local r = {};
			local p = (v.sumValues/tblData.sumSkills)*100;
			local avg;
			if (v.count >0) then avg = (v.sumValues / v.count); else avg = 0; end

			r["skillId"] = k;
			r["sumValues"] = v.sumValues;
			r["count"] = v.count;
			r["valueAvg"] = avg;
			r["col1"] = p;
			r["col2"] = v.sumValues / dTime;
			
			r["showDetails"] = false;
			r["misses"] = v.misses;
			r["hits"] = v.hits;
			r["skillData"] = v;
			r["percent"] = p;
			
			table.insert(returnData, r);
		end
	end
	table.sort(returnData, function(a,b) return a.sumValues > b.sumValues; end);
	
	return returnData;
end

local function getRowDataDmgIn(tblData)
	local returnData = {};
	
	-- time in ms
	local dTime = tblData.timeEnd - tblData.timeStart - tblData.timePaused;
	local totalTime = tblData.timeEnd - tblData.timeStart;
	
	if (tblData.dmgIn.skills) then 
		for k,v in pairs(tblData.dmgIn.skills) do
			local r = {};
			local p = (v.sumValues/tblData.dmgIn.sumSkills)*100;
			local mitigated = 0;
			local missed = 0;
			
			local hits = 0;
			local avg;
			for k2,v2 in pairs(v.hits) do
				hits = hits + v2.count;
			end
			if (hits > 0) then avg = (v.sumValues / hits); else avg = 0; end
			
			missed = v.misses["DODGE"] + v.misses["MISS"] + v.misses["PARRY"]
					+ v.misses["ABSORB"] + v.misses["BLOCK"] + v.misses["DEFLECT"]
					+ v.misses["RESIST"] + v.misses["IMMUNE"] + v.misses["REFLECT"];
			
			if (v.sumValues > 0) then
				local s = (v.hits["RESIST"].sum + v.hits["BLOCK"].sum + v.hits["ABSORB"].sum);
				local m = (v.hits["RESIST"].mtgSum + v.hits["BLOCK"].mtgSum + v.hits["ABSORB"].mtgSum);
				mitigated = (m/(s+m))*100;
			end

			if (v.count > 0) then
				missed = (missed/v.count)*100;
			else
				missed = 0;
			end
			

			r["skillId"] = k;
			r["sumValues"] = v.sumValues;
			r["count"] = v.count;
			r["valueAvg"] = avg;
			r["col1"] = mitigated;
			r["col2"] = missed;
			
			r["showDetails"] = false;
			r["misses"] = v.misses;
			r["hits"] = v.hits;
			r["skillData"] = v;
			r["percent"] = p;
			
			table.insert(returnData, r);
		end
	end
	table.sort(returnData, function(a,b) return a.sumValues > b.sumValues; end);
	
	return returnData;
end


local function showThreatTable(frame)
	local data;
	local rData;
	tblDataMode = "tps";
	
	if (frame == detailsFrame.Pinned) then
		data = getData(dataReferencePin.key, dataReferencePin.source);
		if (not data) then return; end
		
		rowDataPin = getRowDataThreat(data);
		rData = rowDataPin;
	else
		data = getData(dataReference.key, dataReference.source);
		if (not data) then return; end
		
		rowData = getRowDataThreat(data);
		rData = rowData;
	end
	
	frame.TableHeader.Background:SetVertexColor(0.1,0.5,0.5,0.3);
	frame.PieHeader.Background:SetVertexColor(0.1,0.5,0.5,0.3);
	
	frame.TableHeader.Name:SetText(L["name"]);
	frame.TableHeader.Count:SetText(L["count"]);
	frame.TableHeader.Total:SetText(L["threat"]);
	frame.TableHeader.ValueAvg:SetText("\195\152 "..L["threat"]);
	frame.TableHeader.Col1:SetText("%");
	frame.TableHeader.Col2:SetText(L["tps"]);
	
	frame.Pie:ResetPie();
	for i=1,#rData do
		rData[i]["color"] = frame.Pie:AddPie(rData[i]["percent"]);
	end
	frame.Pie:CompletePie();
	
	frame.Pie:SetAlpha(1.0);
	frame.sDetailBox:Hide();
	
	updateTableScrollFrame(frame);
end

local function showDmgInTable(frame)
	local data;
	local rData;
	tblDataMode = "dmg";
	
	if (frame == detailsFrame.Pinned) then
		data = getData(dataReferencePin.key, dataReferencePin.source);
		if (not data) then return; end
		
		rowDataPin = getRowDataDmgIn(data);
		rData = rowDataPin;
	else
		data = getData(dataReference.key, dataReference.source);
		if (not data) then return; end
		
		rowData = getRowDataDmgIn(data);
		rData = rowData;
	end
	
	frame.TableHeader.Background:SetVertexColor(0.8,0.1,0.1,0.3);
	frame.PieHeader.Background:SetVertexColor(0.8,0.1,0.1,0.3);
	
	frame.TableHeader.Name:SetText(L["name"]);
	frame.TableHeader.Count:SetText(L["count"]);
	frame.TableHeader.Total:SetText(L["damage"]);
	frame.TableHeader.ValueAvg:SetText("\195\152 "..L["damage"]);
	frame.TableHeader.Col1:SetText(L["mitigated"].." %");
	frame.TableHeader.Col2:SetText(L["missed"].." %");
	
	frame.Pie:ResetPie();
	for i=1,#rData do
		rData[i]["color"] = frame.Pie:AddPie(rData[i]["percent"]);
	end
	frame.Pie:CompletePie();
	
	frame.Pie:SetAlpha(1.0);
	frame.sDetailBox:Hide();
	
	updateTableScrollFrame(frame);
end

function TtpsFrames:PinDetails(key, source)
	dataReferencePin.key = key;
	dataReferencePin.source = source;
		
	local data = getData(key, source);	
	if (not data) then return; end
	
	-- time in ms
	local dTime = data.timeEnd - data.timeStart - data.timePaused;
	local totalTime = data.timeEnd - data.timeStart;

	detailsFrame.Pinned.Headline:SetText(data.name);
	detailsFrame.Pinned.HeadlineMeta:SetText("["..data.playerName..", "..date("%Y-%m-%d %H:%M", data.timeStart).."]");
		
	detailsFrame.Pinned.infoRows.Time:SetText(string.format("%.2f "..L["seconds"].." (%.2f "..L["seconds"].." "..L["pause"]..")", totalTime, data.timePaused));
	detailsFrame.Pinned.infoRows.Total:SetText(string.format("%.2f", data.sumSkills));
	if (dTime >0) then
		detailsFrame.Pinned.infoRows.Tps:SetText(string.format("%.2f", (data.sumSkills / dTime)));
	else
		detailsFrame.Pinned.infoRows.Tps:SetText("--");
	end

	showThreatTable(detailsFrame.Pinned);
	
	detailsFrame.Pinned:Show();
end

function TtpsFrames:OpenDetails(key, source)
	dataReference.key = key;
	dataReference.source = source;

	if (not detailsFrame) then
		detailsFrame = TtpsFrames:CreateDetailsWindow();
		updateTableScrollFrame(detailsFrame);
	end
		
	local data = getData(key, source);	
	if (not data) then return; end
	
	-- time in ms
	local dTime = data.timeEnd - data.timeStart - data.timePaused;
	local totalTime = data.timeEnd - data.timeStart;
	
	detailsFrame.Headline:SetText(data.name);
	detailsFrame.HeadlineMeta:SetText("["..data.playerName..", "..date("%Y-%m-%d %H:%M", data.timeStart).."]");
	
	detailsFrame.infoRows.Time:SetText(string.format("%.2f "..L["seconds"].." (%.2f "..L["seconds"].." "..L["pause"]..")", totalTime, data.timePaused));
	detailsFrame.infoRows.Total:SetText(string.format("%.2f", data.sumSkills));
	if (dTime >0) then
		detailsFrame.infoRows.Tps:SetText(string.format("%.2f", (data.sumSkills / dTime)));
	else
		detailsFrame.infoRows.Tps:SetText("--");
	end

	showThreatTable(detailsFrame);
	
	-- set buttons
	detailsFrame.ReportBtn:SetScript("OnClick", function() TtpsFrames:OpenReport(key, source); end);
	detailsFrame.ExportBtn:SetScript("OnClick", function() Dewdrop:Open(detailsFrame.ExportBtn) end);
	
	if (source == "saved") then
		detailsFrame.SaveBtn:Disable();
		detailsFrame.DeleteBtn:SetScript("OnClick", function() TtpsData:DeleteSaved(key); end);
	else
		detailsFrame.SaveBtn:Enable();
		detailsFrame.SaveBtn:SetScript("OnClick", function() TtpsData:Save(key); end);
		detailsFrame.DeleteBtn:SetScript("OnClick", function() TtpsData:Delete(key); end);
	end
	
	detailsFrame:Show();
end

function TtpsFrames:CloseDetails()
	dataReference.key = nil;
	dataReference.source = nil;
end

function TtpsFrames:OpenReport(key, source)
	if (not reportFrame) then
		reportFrame = TtpsFrames:CreateReportWindow();
	end
		
	local data = getData(key, source);
	if (not data) then return; end
	
	-- time in ms
	local dTime = data.timeEnd - data.timeStart - data.timePaused;
	local cTime = data.timeEnd - data.timeStart;

	
	local box = reportFrame.Box;
	
	box:Insert(data.name.."\n");
	box:Insert(L["threat"]..":"..data.sumSkills.."\n");
	box:Insert(string.format(L["time"]..": %.3f (%.3f "..L["pause"]..")\n", cTime, data.timePaused));
	box:Insert(string.format(L["avgThreat"]..": %.3f\n", (data.sumSkills / dTime)));
	box:Insert("\n");
	box:Insert("+++++++++++++++++\n");
	box:Insert("threat\n");
	box:Insert("+++++++++++++++++\n");

	if (data.skills) then 
		for k,v in pairs(data.skills) do
			local p = (v.sumValues/data.sumSkills)*100;
			local n, r = GetSpellInfo(k);
			box:Insert((n or L["auto-hits"])..((r and " ("..r..")") or "").."\n");
			box:Insert(string.format(".."..L["threat"]..": %.2f ("..L["avgThreat"]..": %.2f)\n", v.sumValues, (v.sumValues / v.count)));
			box:Insert(".."..L["count"]..":"..v.count.."\n");
			box:Insert(".."..L["min"]..":"..v.minValue.." "..L["max"]..":"..v.maxValue.."\n");
			box:Insert(string.format(".."..L["percent"]..": %.2f%%\n", p));
			box:Insert(string.format(".."..L["tps"]..": %.2f\n", (v.sumValues / dTime)));
			box:Insert("\n");
		end
	end
	box:Insert("+++++++++++++++++\n");
	box:Insert("incoming damage\n");
	box:Insert("+++++++++++++++++\n");
	
	if (data.dmgIn.skills) then 
		for k,v in pairs(data.dmgIn.skills) do
			local p = (v.sumValues/data.dmgIn.sumSkills)*100;
			local n, r = GetSpellInfo(k);
			box:Insert((n or L["auto-hits"])..((r and " ("..r..")") or "").."\n");
			box:Insert(string.format("..".."dmg"..": %.2f (".."avg. damage"..": %.2f)\n", v.sumValues, (v.sumValues / v.count)));
			box:Insert(".."..L["count"]..":"..v.count.."\n");
			box:Insert(".."..L["min"]..":"..v.minValue.." "..L["max"]..":"..v.maxValue.."\n");
			box:Insert(string.format(".."..L["percent"]..": %.2f%%\n", p));
			box:Insert(string.format("..".."dps"..": %.2f\n", (v.sumValues / dTime)));
			box:Insert("\n");
		end
	end
	
	reportFrame:Show();
end


function TtpsFrames:Export(key, source, channel, player)
	local data = getData(key, source);
	if (not data) then return; end
		
	if (TtpsCommunication:SendData(data, channel, player)) then
		DEFAULT_CHAT_FRAME:AddMessage("sending data via "..(channel or ""));
	else
		DEFAULT_CHAT_FRAME:AddMessage("could not send data");
	end
end

local function createDetails(frame)

	frame.DmgInBtn = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate");
	frame.DmgInBtn:SetText(L["ouch!"]);
	frame.DmgInBtn:SetPoint("TOPRIGHT", frame, "TOPRIGHT", -30, -30);
	frame.DmgInBtn:SetWidth(80);
	frame.DmgInBtn:SetHeight(24);
	frame.DmgInBtn:SetScript("OnClick", function() showDmgInTable(frame); end);
	
	frame.TpsBtn = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate");
	frame.TpsBtn:SetText(L["tps"]);
	frame.TpsBtn:SetPoint("TOPRIGHT", frame.DmgInBtn, "TOPLEFT", -2, 0);
	frame.TpsBtn:SetWidth(80);
	frame.TpsBtn:SetHeight(24);
	frame.TpsBtn:SetScript("OnClick", function() showThreatTable(frame); end);
	

	local r; -- tmp var for rows

	-- headline
	frame.Headline = frame:CreateFontString(nil,"OVERLAY","GameFontNormalLarge");
	frame.Headline:SetPoint("TOPLEFT",frame,"TOPLEFT",15,-10);
	frame.Headline:SetText("Headline");
	frame.Headline:SetTextColor(1.0,1.0,1.0,1.0);
	
	frame.HeadlineMeta = frame:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
	frame.HeadlineMeta:SetPoint("LEFT",frame.Headline,"RIGHT",10,0);
	frame.HeadlineMeta:SetText("[some meta infos]");
	frame.HeadlineMeta:SetTextColor(1.0,1.0,1.0,1.0);
	
	-- thread, time, tps
	local txt;
	local tmpAnchor;
	local infoRowsKey = {L["threat"], L["time"], L["tps"]};
	local i=1;

	frame.infoRows = {};
	frame.infoRows.Total = frame:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
	frame.infoRows.Time = frame:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
	frame.infoRows.Tps = frame:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
	
	for k,v in pairs(frame.infoRows) do
		txt = frame:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
		txt:SetText(infoRowsKey[i]..":");
		
		v:SetText("12345");
		v:SetTextColor(1.0,1.0,1.0,1.0);
		
		if (not tmpAnchor) then
			txt:SetPoint("TOPLEFT",frame.Headline,"BOTTOMLEFT",0,-5);
		else
			txt:SetPoint("LEFT",tmpAnchor,"RIGHT",5,0);
		end
		
		v:SetPoint("LEFT", txt, "RIGHT", 2, 0);
		tmpAnchor = v;
		
		i = i+1;
	end
	
	-- pie  column header
	r = CreateFrame("frame",nil,frame);

	r:SetWidth(200);
	r:SetHeight(11);
	r:SetPoint("TOPLEFT",frame,"TOPLEFT",10,-55);
	
	r.Background = r:CreateTexture(nil,"BACKGROUND");
	r.Background:SetAllPoints(r);
	r.Background:SetTexture("Interface\\Buttons\\WHITE8X8.blp");
	r.Background:SetVertexColor(0.1,0.5,0.5,0.3);
	
	r.Name = r:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
	r.Name:SetPoint("LEFT",r,"LEFT",20,0);
	r.Name:SetText(L["pieColHeading"]);
	
	frame.PieHeader = r;
	
	-- pie chart
	frame.Pie = Graph:CreateGraphPieChart("ThreatPieChart",frame,"TOPLEFT","TOPLEFT",30,-65,125,125)

	frame.Pie:AddPie(35,{1.0,0.0,0.0})
	frame.Pie:AddPie(21,{0.0,1.0,0.0})	
	frame.Pie:AddPie(10,{1.0,1.0,1.0})
	frame.Pie:CompletePie({0.2,0.2,1.0})
	frame.Pie:Show();
			
	-- miss details
	frame.sDetailBox = CreateFrame("frame",nil,frame);
	frame.sDetailBox:SetPoint("TOPLEFT", frame.Pie, "TOPLEFT", -25, 0);
	frame.sDetailBox:SetPoint("BOTTOMRIGHT", frame.Pie, "BOTTOMRIGHT");
	
	r = CreateFrame("frame",nil,frame.sDetailBox);
	r:SetPoint("TOPLEFT",frame.sDetailBox,"TOPLEFT",5,-10);
	r:SetWidth(200);
	r:SetHeight(11);
	r.ColKey = r:CreateTexture(nil,"OVERLAY");
	r.ColKey:SetPoint("LEFT",r,"LEFT",0,0);
	r.ColKey:SetTexture("Interface\\Buttons\\WHITE8X8.blp");
	r.ColKey:SetWidth(12);
	r.ColKey:SetHeight(12);
	r.Text = r:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
	r.Text:SetPoint("LEFT",r,"LEFT",20,0);
	r.Text:SetText("test");
	frame.sDetailBox.Heading = r;
	
	frame.sDetailBox.rows = {};
	for i=1, MAXDETAILROWS do
		r = CreateFrame("frame",nil,frame.sDetailBox);
		r:SetPoint("TOPLEFT",frame.sDetailBox,"TOPLEFT",5,-(11*i)-16);
		r:SetWidth(175);
		r:SetHeight(11);
		r:EnableMouse(true);
		r:SetScript("OnEnter", function() detailsTooltip(i, frame); end);
		r:SetScript("OnLeave", function() GameTooltip:Hide(); end);
		r.Text = r:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
		r.Text:SetPoint("LEFT",r,"LEFT",0,0);
		r.Text:SetText("test");
		
		frame.sDetailBox.rows[i] = r;
	end
	frame.sDetailBox:Hide();
	
	-- details scroll frame
	local scrollFrame = CreateFrame("SCROLLFRAME",frame:GetName().."DetailsScrollFrame",frame.sDetailBox,"FauxScrollFrameTemplate");
	scrollFrame:EnableMouse(true);
	scrollFrame:SetScript("OnVerticalScroll", function() FauxScrollFrame_OnVerticalScroll(11, function() updateDetailsScrollFrame(frame) end ) end);
	scrollFrame:SetPoint("TOPLEFT",frame.sDetailBox.rows[1],"TOPLEFT")	
	scrollFrame:SetPoint("BOTTOMRIGHT",frame.sDetailBox.rows[MAXDETAILROWS],"BOTTOMRIGHT")
	
	local fLvl = scrollFrame:GetFrameLevel();
	for i=1, #frame.sDetailBox.rows do
		frame.sDetailBox.rows[i]:SetFrameLevel(fLvl+1);
	end
	
	frame.DetailsScrollFrame = scrollFrame;
	
	
	-- table column headers
	r = CreateFrame("frame",nil,frame);

	r:SetWidth(495+50);
	r:SetHeight(11);
	r:SetPoint("TOPRIGHT",frame,"TOPRIGHT",-35,-55);
	
	r.Background = r:CreateTexture(nil,"BACKGROUND");
	r.Background:SetAllPoints(r);
	r.Background:SetTexture("Interface\\Buttons\\WHITE8X8.blp");
	r.Background:SetVertexColor(0.1,0.5,0.5,0.3);
	
	r.Name = r:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
	r.Name:SetPoint("LEFT",r,"LEFT",20,0);
	r.Name:SetText("name");

	r.Count = r:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
	r.Count:SetPoint("RIGHT",r,"RIGHT",-280,0);
	r.Count:SetText("count");

	r.Total = r:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
	r.Total:SetPoint("RIGHT",r,"RIGHT",-210,0);
	r.Total:SetText("total");
	
	r.ValueAvg = r:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
	r.ValueAvg:SetPoint("RIGHT",r,"RIGHT",-140,0);
	r.ValueAvg:SetText("avg value");

	r.Col1 = r:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
	r.Col1:SetPoint("RIGHT",r,"RIGHT",-70,0);
	r.Col1:SetText("col1");
	
	r.Col2 = r:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
	r.Col2:SetPoint("RIGHT",r,"RIGHT",0,0);
	r.Col2:SetText("col2");
	
	frame.TableHeader = r;
	
	-- rows
	frame.rows = {};
	for i=1,MAXDETAILROWS do
		r = CreateFrame("frame",nil,frame);
		r.id = i;
		r:EnableMouse(true);
		r:SetScript("OnMouseDown", function() rowClicked(this.id, frame); end);
		
		r:SetWidth(495+50);
		r:SetHeight(11);
		r:SetPoint("TOPRIGHT",frame,"TOPRIGHT",-35,-(11+2)*i-55);

		r.Background = r:CreateTexture(nil,"BACKGROUND");
		r.Background:SetAllPoints(r);
		r.Background:SetTexture("Interface\\Buttons\\WHITE8X8.blp");
		r.Background:SetVertexColor(0.5,0.5,0.8,0.3);
		r.Background:Hide();

		r.ColKey = r:CreateTexture(nil,"OVERLAY");
		r.ColKey:SetPoint("LEFT",r,"LEFT",0,0);
		r.ColKey:SetTexture("Interface\\Buttons\\WHITE8X8.blp");
		r.ColKey:SetWidth(12);
		r.ColKey:SetHeight(12);
		
		r.Name = r:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
		r.Name:SetPoint("LEFT",r,"LEFT",20,0);
		r.Name:SetText("Test");
		r.Name:SetTextColor(1.0,1.0,1.0,1.0);

		r.Count = r:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
		r.Count:SetPoint("RIGHT",r,"RIGHT",-280,0);
		r.Count:SetText("123");
		r.Count:SetTextColor(1.0,1.0,1.0,1.0);

		r.Total = r:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
		r.Total:SetPoint("RIGHT",r,"RIGHT",-210,0);
		r.Total:SetText("12345");
		r.Total:SetTextColor(1.0,1.0,1.0,1.0);
		
		r.ValueAvg = r:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
		r.ValueAvg:SetPoint("RIGHT",r,"RIGHT",-140,0);
		r.ValueAvg:SetText("12345");
		r.ValueAvg:SetTextColor(1.0,1.0,1.0,1.0);

		r.Col1 = r:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
		r.Col1:SetPoint("RIGHT",r,"RIGHT",-70,0);
		r.Col1:SetText("10%");
		r.Col1:SetTextColor(1.0,1.0,1.0,1.0);
		
		r.Col2 = r:CreateFontString(nil,"OVERLAY","GameFontNormalSmall");
		r.Col2:SetPoint("RIGHT",r,"RIGHT",0,0);
		r.Col2:SetText("123");
		r.Col2:SetTextColor(1.0,1.0,1.0,1.0);
		
		frame.rows[i] = r;
	end
	
	-- table scroll frame
	scrollFrame = CreateFrame("SCROLLFRAME",frame:GetName().."ScrollFrame",frame,"FauxScrollFrameTemplate");
	scrollFrame:EnableMouse(true);
	scrollFrame:SetScript("OnVerticalScroll", function() FauxScrollFrame_OnVerticalScroll(11, function() updateTableScrollFrame(frame) end ) end);
	scrollFrame:SetPoint("TOPLEFT",frame.rows[1],"TOPLEFT");
	scrollFrame:SetPoint("BOTTOMRIGHT",frame.rows[MAXDETAILROWS],"BOTTOMRIGHT");
	
	fLvl = scrollFrame:GetFrameLevel();
	for i=1, #frame.rows do
		frame.rows[i]:SetFrameLevel(fLvl+1);
	end
	frame.ScrollFrame = scrollFrame;

end

function TtpsFrames:CreateDetailsWindow()
	local detailsWin = CreateFrame("Frame", "TankadinTpsDetailsFrame",UIParent);

	detailsWin:SetWidth(800);
	detailsWin:SetHeight(200);
	detailsWin:SetBackdrop({bgFile = "Interface/DialogFrame/UI-DialogBox-Background", 
                          edgeFile = "Interface/Tooltips/UI-Tooltip-Border", 
                          tile = true, tileSize = 16, edgeSize = 16, 
                          insets = { left = 4, right = 4, top = 4, bottom = 4 }
						 });

	detailsWin:SetMovable(true);
	detailsWin:EnableMouse(true);
	detailsWin:SetPoint("CENTER", UIParent, "CENTER");
	detailsWin:SetScript("OnMouseDown", function() this:StartMoving(); end);
	detailsWin:SetScript("OnMouseUp", function() this:StopMovingOrSizing(); end);
	detailsWin:SetScript("OnHide", function() TtpsFrames:CloseDetails(); end);
	detailsWin:SetFrameStrata("DIALOG");
	
	local closeBtn = CreateFrame("Button", "TankadinTpsDetailsFrameClose",detailsWin,"UIPanelCloseButton");
	closeBtn:SetPoint("TOPRIGHT", detailsWin, "TOPRIGHT", 0, 0);
	
	createDetails(detailsWin);
	
	local detailsPinnedWin = CreateFrame("Frame", "TankadinTpsDetailsPinnedFrame",detailsWin);

	detailsPinnedWin:SetWidth(800);
	detailsPinnedWin:SetHeight(200);
	detailsPinnedWin:SetBackdrop({bgFile = "Interface/DialogFrame/UI-DialogBox-Background", 
                          edgeFile = "Interface/Tooltips/UI-Tooltip-Border", 
                          tile = true, tileSize = 16, edgeSize = 16, 
                          insets = { left = 4, right = 4, top = 4, bottom = 4 }
						 });
	detailsPinnedWin:EnableMouse(true);
	detailsPinnedWin:SetScript("OnMouseDown", function() detailsWin:StartMoving(); end);
	detailsPinnedWin:SetScript("OnMouseUp", function() detailsWin:StopMovingOrSizing(); end);
	detailsPinnedWin:SetPoint("BOTTOM", detailsWin, "TOP");
	createDetails(detailsPinnedWin);
	detailsPinnedWin:Hide();
	
	detailsWin.Pinned = detailsPinnedWin;
	
	-- pin data button
	detailsWin.PinBtn = CreateFrame("Button", nil, detailsWin, "UIPanelButtonTemplate2");
	detailsWin.PinBtn:SetPoint("RIGHT", closeBtn, "LEFT", 0, 0);
	detailsWin.PinBtn:SetWidth(24);
	detailsWin.PinBtn:SetHeight(24);
	detailsWin.PinBtn:SetScript("OnClick", function() pinClicked(); end);
	detailsWin.PinBtn:SetScript("OnMouseDown", nil);
	detailsWin.PinBtn:SetScript("OnMouseUp", nil);
	
	detailsWin.PinBtn:SetNormalTexture("Interface/ChatFrame/UI-ChatIcon-ScrollUp-Up");
	
	-- next data button
	detailsWin.NextBtn = CreateFrame("Button", nil, detailsWin, "UIPanelButtonTemplate2");
	detailsWin.NextBtn:SetPoint("RIGHT", detailsWin.PinBtn, "LEFT", 0, 0);
	detailsWin.NextBtn:SetWidth(24);
	detailsWin.NextBtn:SetHeight(24);
	detailsWin.NextBtn:SetScript("OnClick", function() local k,s = getNextData(dataReference.key, dataReference.source); TtpsFrames:OpenDetails(k,s) end);
	detailsWin.NextBtn:SetScript("OnMouseDown", nil);
	detailsWin.NextBtn:SetScript("OnMouseUp", nil);
	
	detailsWin.NextBtn:SetNormalTexture("Interface/Buttons/UI-SpellbookIcon-NextPage-Up");
	
	-- previous data button
	detailsWin.PreviousBtn = CreateFrame("Button", nil, detailsWin, "UIPanelButtonTemplate2");
	detailsWin.PreviousBtn:SetPoint("RIGHT", detailsWin.NextBtn, "LEFT", 0, 0);
	detailsWin.PreviousBtn:SetWidth(24);
	detailsWin.PreviousBtn:SetHeight(24);
	detailsWin.PreviousBtn:SetScript("OnClick", function() local k,s = getPreviousData(dataReference.key, dataReference.source); TtpsFrames:OpenDetails(k,s) end);
	detailsWin.PreviousBtn:SetScript("OnMouseDown", nil);
	detailsWin.PreviousBtn:SetScript("OnMouseUp", nil);
	
	detailsWin.PreviousBtn:SetNormalTexture("Interface/Buttons/UI-SpellbookIcon-PrevPage-Up");
		
	detailsWin.SaveBtn = CreateFrame("Button", nil, detailsWin, "UIPanelButtonTemplate");
	detailsWin.SaveBtn:SetText(L["save"]);
	detailsWin.SaveBtn:SetPoint("TOPRIGHT", detailsWin, "BOTTOMRIGHT", -2, 0);
	detailsWin.SaveBtn:SetWidth(80);
	detailsWin.SaveBtn:SetHeight(24);
	
	detailsWin.DeleteBtn = CreateFrame("Button", nil, detailsWin, "UIPanelButtonTemplate");
	detailsWin.DeleteBtn:SetText(L["delete"]);
	detailsWin.DeleteBtn:SetPoint("RIGHT", detailsWin.SaveBtn, "LEFT", -2, 0);
	detailsWin.DeleteBtn:SetWidth(80);
	detailsWin.DeleteBtn:SetHeight(24);
	
	detailsWin.ReportBtn = CreateFrame("Button", nil, detailsWin, "UIPanelButtonTemplate");
	detailsWin.ReportBtn:SetText(L["report"]);
	detailsWin.ReportBtn:SetPoint("RIGHT", detailsWin.DeleteBtn, "LEFT", -2, 0);
	detailsWin.ReportBtn:SetWidth(80);
	detailsWin.ReportBtn:SetHeight(24);
	
	detailsWin.ExportBtn = CreateFrame("Button", nil, detailsWin, "UIPanelButtonTemplate");
	detailsWin.ExportBtn:SetText(L["export"]);
	detailsWin.ExportBtn:SetPoint("TOP", detailsWin.ReportBtn, "BOTTOM", 0, 2);
	detailsWin.ExportBtn:SetWidth(80);
	detailsWin.ExportBtn:SetHeight(24);
	
	Dewdrop:Register(detailsWin.ExportBtn,
    'children', function()
        Dewdrop:FeedAceOptionsTable(reportToOptions);
    end
)
	
	return detailsWin;
end

function TtpsFrames:CreateReportWindow()

	local reportWin = CreateFrame("Frame", "TankadinTpsReportFrame",detailsFrame);

	reportWin:SetWidth(300);
	reportWin:SetHeight(400);
	reportWin:SetBackdrop({bgFile = "Interface/DialogFrame/UI-DialogBox-Background", 
                          edgeFile = "Interface/Tooltips/UI-Tooltip-Border", 
                          tile = true, tileSize = 16, edgeSize = 16, 
                          insets = { left = 4, right = 4, top = 4, bottom = 4 }
						 });

	reportWin:SetMovable(true);
	reportWin:EnableMouse(true);
	reportWin:SetPoint("CENTER", UIParent, "CENTER")
	reportWin:SetScript("OnMouseDown", function() this:StartMoving(); end)
	reportWin:SetScript("OnMouseUp", function() this:StopMovingOrSizing(); end)
	reportWin:SetFrameStrata("DIALOG");
	
	local closeBtn = CreateFrame("Button", nil, TankadinTpsReportFrame, "UIPanelCloseButton");
	closeBtn:SetPoint("TOPRIGHT", TankadinTpsReportFrame, "TOPRIGHT", 0, 0);

	local scroll = CreateFrame("ScrollFrame", "TtpsReportScrollFrame", reportWin, "UIPanelScrollFrameTemplate");

	scroll:SetPoint("TOPLEFT",reportWin,"TOPLEFT",10,-30)
	scroll:SetPoint("BOTTOMRIGHT",reportWin,"BOTTOMRIGHT",-30,10)

	reportWin.Box = CreateFrame("EditBox",nil,scroll);
	reportWin.Box:SetScript("OnEscapePressed", function() this:ClearFocus(); end);
	reportWin.Box:SetScript("OnEditFocusLost", function() this:HighlightText(0, 0); end);
	reportWin.Box:SetScript("OnEditFocusGained", function() this:HighlightText(); end);
	reportWin.Box:SetFontObject("ChatFontNormal");
	reportWin.Box:SetMultiLine(true);
	reportWin.Box:SetWidth(280);
	reportWin.Box:SetHeight(380);

	reportWin.Box:SetAutoFocus(false)
	
	scroll:SetScrollChild(reportWin.Box);
	
	return reportWin;
end

function TtpsFrames:GetDataReference()
	return dataReference.key, dataReference.source;
end
