
TankadinTps_data = {};
TtpsData = {};

TtpsData.debugMode = false;

local DATA_VERSION = 1;
local MAX_ENTRIES = 10;

--[[
data.ic = {
	[tarGUID] = { 
		name
		playerName
		dataVersion
		sumSkills
		timeStart
		timeEnd
		timePaused
		pauseData = {
			lastActivity
			start
		}
		skills = {
			[skillId] = {
				sumValues
				minValue
				maxValue
				count
				misses = {
				}
				hits = {
				}
				}
			}
		} 
		dmgIn = {
			sumSkills
			skills = {
				[skillId] = {
					sumValues
					minValue
					maxValue
					count
					misses = {
					}
					hits = {
					}
					}
				}
			} 
		}
}
data.ooc = same structure as data.ic, but without a key "tarGUID" and without pauseData
--]]
local data = {};
data.ic = {};
data.ooc = {};
data.import = {};
local dataSaved = TankadinTps_data;

local pause = false;
local deadOrOOC = false;

--[[
	removes entries in data, to keep its size below MAX_ENTRIES
	you can't use a table with keys (data.ic), because #tbl will always return 0
--]]
local function trimSize(tbl)
	while (#tbl > MAX_ENTRIES) do
		table.remove(tbl, 1);
	end
end

-- returns the table entry from data.ic[tarGUID] or creates a new one
local function getOrCreateTargetData(tarGUID, tarName, timestamp)
	if (not data.ic[tarGUID]) then
		local e = {};
		e.name = tarName or "<???>";
		e.playerName = UnitName("player");
		e.dataVersion = DATA_VERSION;
		e.sumSkills = 0;
		e.timeStart = timestamp;
		e.timeEnd = 0;
		e.timePaused = 0;
		e.skills = {};
		e.dmgIn = {};
		e.dmgIn.sumSkills = 0;
		e.dmgIn.skills = {};
		
		e.pauseData = {};
		e.pauseData.start = 0;
		e.pauseData.lastActivity = 0;
		
		data.ic[tarGUID] = e;
		trimSize(data.ooc);
		
		if (TtpsData.debugMode) then TtpsDebug:Out("new data: "..(tarName or "")..", "..(tarGUID or "")); end
	end
	
	return data.ic[tarGUID];
end

-- returns the table entry from data.ic[tarGUID].skills[skillId] or creates a new one
local function getOrCreateSkillData(data, skillId)
	if (not data.skills[skillId]) then
		local e = {};
		e.sumValues = 0;
		e.minValue = 999999;
		e.maxValue = 0;
		e.count = 0;
		
		e.misses = {};
		e.misses["ABSORB"] = 0;
		e.misses["BLOCK"] = 0;
		e.misses["DEFLECT"] = 0;
		e.misses["DODGE"] = 0;
		e.misses["EVADE"] = 0;
		e.misses["IMMUNE"] = 0;
		e.misses["MISS"] = 0;
		e.misses["PARRY"] = 0;
		e.misses["REFLECT"] = 0;
		e.misses["RESIST"]  = 0;
		
		e.hits = {};
		e.hits["NORMAL"] = {count=0,sum=0,mtgSum=0};
		e.hits["CRITICAL"] = {count=0,sum=0,mtgSum=0};
		e.hits["GLANCING"] = {count=0,sum=0,mtgSum=0};
		e.hits["CRUSHING"] = {count=0,sum=0,mtgSum=0};
		e.hits["RESIST"] = {count=0,sum=0,mtgSum=0};
		e.hits["BLOCK"] = {count=0,sum=0,mtgSum=0};
		e.hits["ABSORB"] = {count=0,sum=0,mtgSum=0};

		data.skills[skillId] = e;
		
		if (TtpsData.debugMode) then TtpsDebug:Out("new skill data: "..(data.name or "").."> "..(skillId or "")); end
	end
	
	return data.skills[skillId];
end

-- ---------------------------------------------------------------------------------------------------

local function addData(tblData, name, timestamp, value, skillId)
	local skill = getOrCreateSkillData(tblData, skillId);
	
	tblData.sumSkills = tblData.sumSkills + value;
	
	skill.sumValues = skill.sumValues + value;
	skill.count = skill.count + 1;
	skill.minValue = math.min(skill.minValue, value);
	skill.maxValue = math.max(skill.maxValue, value);
	
	return tblData.sumSkills, skill;
end

local function addMiss(tblData, name, timestamp, missType, skillId)
	local skill = getOrCreateSkillData(tblData, skillId);
	skill.count = skill.count + 1;
	skill.misses[missType] = skill.misses[missType] + 1;
	
	return skill;
end

local function addHitInfo(tblData, name, timestamp, value, resist, block, absorb, critical, glancing, crushing, skillId)
	local skill = getOrCreateSkillData(tblData, skillId);
	--DEFAULT_CHAT_FRAME:AddMessage((resist or "-").." "..(block or "-").." "..(absorb or "-").." "..(critical or "-").." "..(glancing or "-").." "..(crushing or "-"));
	
	local hitType;
	local mtgVal;
	if (critical) then
		hitType = "CRITICAL";
		mtgVal = 0;
	elseif (glancing) then
		hitType = "GLANCING";
		mtgVal = 0;
	elseif (crushing) then
		hitType = "CRUSHING";
		mtgVal = 0;
	elseif (resist) then
		hitType = "RESIST";
		mtgVal = resist;
	elseif (block) then
		hitType = "BLOCK";
		mtgVal = block;
	elseif (absorb) then
		hitType = "ABSORB";
		mtgVal = absorb;
	else
		hitType = "NORMAL";
		mtgVal = 0;
	end
		
	skill.hits[hitType].count = skill.hits[hitType].count + 1;
	skill.hits[hitType].sum = skill.hits[hitType].sum + value;
	skill.hits[hitType].mtgSum = skill.hits[hitType].mtgSum + mtgVal;
	
	return skill;
end

-- ---------------------------------------------------------------------------------------------------

function TtpsData:CheckPause()
	local ts = time();
	for k,v in pairs(data.ic) do
		if (v.pauseData.start > 0) then
			if (v.pauseData.lastActivity > v.pauseData.start and not pause) then
				-- reactivate
				v.timePaused = v.timePaused + (v.pauseData.lastActivity - v.pauseData.start);
				v.pauseData.start = 0;
			end
		else
			if (ts - v.pauseData.lastActivity > 5) then
				-- deactivate
				v.pauseData.start = v.pauseData.lastActivity;
			end
		end
	end
end

function TtpsData:SetPause(status)
	pause = status;
	
	for k,v in pairs(data.ic) do
		if (pause) then
			v.pauseData.start = v.pauseData.lastActivity;
		end
	end
end

function TtpsData:GetPause()
	return pause;
end

function TtpsData:SetDead(status)
	deadOrOOC = status;
end

function TtpsData:EnterCombat(timestamp)
	deadOrOOC = false;
end

function TtpsData:LeaveCombat(timestamp)
	for k,v in pairs(data.ic) do
		v.timeEnd = timestamp;
		v.pauseData = nil;
		table.insert(data.ooc, v);
		data.ic[k] = nil;
	end
	deadOrOOC = true;
	
	if (TtpsData.debugMode) then TtpsDebug:Out("combat left, all data.ic entries copied to data.ooc"); end
end

-- --------------------------------

--[[
	adds new data to a target
	creates a new entry in data, if the guid is unknown
 --]]
function TtpsData:AddData(tarGUID, tarName, timestamp, threat, skillId)
	if (deadOrOOC) then return; end
	
	skillId = skillId or 'SWING'; -- p1 is nil on white hits
	local data = getOrCreateTargetData(tarGUID, tarName, timestamp);

	if (pause) then return; end
	
	data.pauseData.lastActivity = timestamp;
	addData(data, tarName, timestamp, threat, skillId);
		
	if (TtpsData.debugMode) then TtpsDebug:Out("data added: "..(tarName or "").."("..(tarGUID or "")..") threat:"..(threat or "").." skillId:"..(skillId or "")); end
end

--[[
	adds global threat (heals, mana, ...) to all entries with timeEnd == 0
--]]
function TtpsData:AddDataGlobal(timestamp, threat, skillId)
	if (deadOrOOC or pause) then return; end

	for k,v in pairs(data.ic) do
		if (v.pauseData.start == 0) then
			
			local skill = getOrCreateSkillData(v, skillId);
			
			v.sumSkills = v.sumSkills + threat;
						
			skill.sumValues = skill.sumValues + threat;
			skill.count = skill.count + 1;
			
			skill.minValue = math.min(skill.minValue, threat);
			skill.maxValue = math.max(skill.maxValue, threat);
		end
	end
	
	if (TtpsData.debugMode) then TtpsDebug:Out("data added (global): threat:"..(threat or "").." skillId:"..(skillId or "")); end
end

--[[
	adds misses from skills
--]]
function TtpsData:AddMiss(tarGUID, tarName, timestamp, missType, skillId)
	if (deadOrOOC) then return; end
	
	skillId = skillId or 'SWING'; -- p1 is nil on white hits
	local data = getOrCreateTargetData(tarGUID, tarName, timestamp);

	if (pause) then return; end
	
	data.pauseData.lastActivity = timestamp;
	addMiss(data, tarName, timestamp, missType, skillId);
end

--[[
	adds hits from skills
--]]
function TtpsData:AddHitInfo(tarGUID, tarName, timestamp, value, resist, block, absorb, critical, glancing, crushing, skillId)
	if (deadOrOOC) then return; end
	
	skillId = skillId or 'SWING'; -- p1 is nil on white hits
	local data = getOrCreateTargetData(tarGUID, tarName, timestamp);

	if (pause) then return; end
	
	data.pauseData.lastActivity = timestamp;
	addHitInfo(data, tarName, timestamp, value, resist, block, absorb, critical, glancing, crushing, skillId);
end

--[[
	adds incoming damage
--]]
function TtpsData:AddDmgInData(srcGUID, srcName, timestamp, dmg, skillId)
	if (deadOrOOC) then return; end
	
	skillId = skillId or 'SWING'; -- p1 is nil on white hits
	local data = getOrCreateTargetData(srcGUID, srcName, timestamp);

	if (pause) then return; end
	
	addData(data.dmgIn, srcName, timestamp, dmg, skillId);
end

--[[
	adds incoming misses
--]]
function TtpsData:AddDmgInMiss(srcGUID, srcName, timestamp, missType, skillId)
	if (deadOrOOC) then return; end
	
	skillId = skillId or 'SWING'; -- p1 is nil on white hits
	local data = getOrCreateTargetData(srcGUID, srcName, timestamp);

	if (pause) then return; end

	addMiss(data.dmgIn, srcName, timestamp, missType, skillId);
end

--[[
	adds incoming hits
--]]
function TtpsData:AddDmgInHitInfo(srcGUID, srcName, timestamp, value, resist, block, absorb, critical, glancing, crushing, skillId)
	if (deadOrOOC) then return; end
	
	skillId = skillId or 'SWING'; -- p1 is nil on white hits
	local data = getOrCreateTargetData(srcGUID, srcName, timestamp);

	if (pause) then return; end
	
	addHitInfo(data.dmgIn, srcName, timestamp, value, resist, block, absorb, critical, glancing, crushing, skillId);
end

-- --------------------------------

function TtpsData:UnitDied(tarGUID, timestamp)
	if (data.ic[tarGUID]) then
		data.ic[tarGUID].timeEnd = timestamp;
		if (data.ic[tarGUID].pauseData.start >0) then
			data.ic[tarGUID].timePaused = data.ic[tarGUID].timePaused + (timestamp - data.ic[tarGUID].pauseData.start);
		end
		data.ic[tarGUID].pauseData = nil;
		table.insert(data.ooc, data.ic[tarGUID]);
		data.ic[tarGUID] = nil;
		
		if (TtpsData.debugMode) then TtpsDebug:Out("unit died, data copied to data.ooc: "..(tarGUID or "")); end
	end
end

function TtpsData:ResetData()
	data.ic = {};
	data.ooc = {};
end

function TtpsData:Load()
	dataSaved = TankadinTps_data or {};
	TtpsData:CheckDataFormat();
end

function TtpsData:Save(key)
	if (key and data.ooc[key]) then
		table.insert(dataSaved, data.ooc[key]);
		DEFAULT_CHAT_FRAME:AddMessage("saved ("..key..")");
	else
		DEFAULT_CHAT_FRAME:AddMessage("no key");
	end
end

function TtpsData:Delete(key)
	if (key) then
		table.remove(data.ooc, key);
		DEFAULT_CHAT_FRAME:AddMessage("deleted ("..key..")");
	else
		DEFAULT_CHAT_FRAME:AddMessage("no key");
	end
end

function TtpsData:DeleteSaved(key)
	if (key) then
		table.remove(dataSaved, key);
		DEFAULT_CHAT_FRAME:AddMessage("deleted ("..key..")");
	else
		DEFAULT_CHAT_FRAME:AddMessage("no key");
	end
end

function TtpsData:GetDataInCombat(key)
	if (key) then
		return data.ic[key];
	else
		return data.ic;
	end
end

function TtpsData:GetDataOutOfCombat(key)
	if (key) then
		return data.ooc[key];
	else
		return data.ooc;
	end
end

function TtpsData:GetImportedData(key)
	if (key) then
		return data.import[key];
	else
		return data.import;
	end
end

function TtpsData:GetSavedData(key)
	if (key) then
		return dataSaved[key];
	else
		return dataSaved;
	end
end

local function checkData(v)
	if (not v.dataVersion) then
		if (not v.name) then v.name = ""; end
		if (not v.playerName) then v.playerName = myName; end
		if (not v.sumSkills) then v.sumSkills = 0; end
		if (not v.timeStart) then v.timeStart = 0; end
		if (not v.timeEnd) then v.timeEnd = 1; end
		if (not v.timePaused) then v.timePaused = 0; end
		if (not v.skills) then v.skills = {}; end
		if (not v.dmgIn) then v.dmgIn = {}; end
		for k2,v2 in pairs(v.skills) do
			if (not v2.misses) then v2.misses = {}; end
			if (not v2.hits) then v2.hits = {}; end
			if (not v2.minValue) then v2.minValue = 0; end
			if (not v2.maxValue) then v2.maxValue = 0; end
			if (not v2.sumValues) then v2.sumValues = 0; end
			if (v2.minThreat) then v2.minValue = v2.minThreat; v2.minThreat = nil; end
			if (v2.maxThreat) then v2.maxValue = v2.maxThreat; v2.maxThreat = nil; end
			if (v2.threat) then v2.sumValues = v2.threat; v2.threat = nil; end
		end
		if (v.tpsData) then v.tpsData = nil; end
		if (v.threatSum) then v.sumSkills = v.threatSum; v.threatSum = nil; end
		v.dataVersion = 1;
	end
	
	if (v.dataVersion > DATA_VERSION) then
		return nil;
	end

	--[[
	if (v.dataVersion == 1) then
		-- do something
	end
	if (v.dataVersion == 2) then
		-- do something
	end
	--]]
	return v; -- v.dataVersion == DATA_VERSION
end

function TtpsData:CheckDataFormat()
	local myName = UnitName("player");
	local v;
	local removePos = {};
	for i=1,#dataSaved do
		v = dataSaved[i];
		if (not v) then
			table.insert(removePos, i);
		else
			checkData(v)
		end
	end
	
	for i=1,#removePos do
		table.remove(dataSaved, (removePos[i] -(i-1)));
	end
end

function TtpsData:Import(newEntry)
	newEntry = checkData(newEntry);
	
	if (newEntry) then
		table.insert(data.import, newEntry);
		trimSize(data.import);
		DEFAULT_CHAT_FRAME:AddMessage("you received new data from "..(newEntry.playerName or "<???>"));
	else
		DEFAULT_CHAT_FRAME:AddMessage((newEntry.playerName or "<???>").." tried to send you data, but the import failed. Please update to the latest version of TankadinTps.");
	end
end

