--[[*****************************************************************
	StanceSets 3 + 1 v2.3 
	01.09.2008

	Author: CapnBry modified by Tageshi and Vorkosigan
	*****************************************************************
	Description:
    This add-on allow you to quickly swap weapons, shield and ammo on a single key press, and cycles between 
    two or three different weapons sets. Additionaly this add-on can have different weapons sets cycle for 
    different stances and shapes.	

	Commands:
	  /stancesets will open the stance set configuration dialog or use the minimap button
	  The panel toggle and set cycle can be bound to a key combination. Look in the interface
	  menu to define a hot-key.
	*****************************************************************
]]

local WeaponQuickSwap = {}

-- Unit variable to hold the stack of weapon swaps
local wswap                        = { c = 0, sb = {}, si = {}, db = {}, di = {} };
local WeaponSwap_IsSwapping        = nil;
local WeaponSwap_IsInSwapIteration = nil;
local WeaponSwap_SwapTime          = nil;
local WeaponSwap_HasAmmoSlot       = true;
local rarg                         = {};
local STANCESETS3_MAX_STANCES      = 8
local LastMain                     = {bag = nil, slot = nil};
local LastOff                      = {bag = nil, slot = nil};
local LastRange                    = {bag = nil, slot = nil};
local LastAmmo                     = {bag = nil, slot = nil};
local mainhandslot, offhandslot, rangedslot, ammoslot = 16, 17, 18, 0;
local StanceSets3_LastForm;

tinsert(UISpecialFrames,"StanceSets3Frame");

local defaultStanceSets3ButtonOptions = {
	["ButtonPosition"] = 356;
	["ButtonRadius"]   = 78;
	["ButtonShown"]    = true;
};

function CloneTable(t)			  -- return a copy of the table t
  local new = {};             -- create a new table
  local i, v = next(t, nil);	-- i is an index of t, v = t[i]
  while i do
    if type(v)=="table" then 
	  v=CloneTable(v);
    end 
    new[i] = v;
	  i, v = next(t, i);			  -- get next index
  end
  return new;
end

---------------------------------------------------------------------------------------
-- StanceSets3 Code
---------------------------------------------------------------------------------------

function StanceSets3_OnLoad()
  this:RegisterEvent("UPDATE_BONUS_ACTIONBAR");
  this:RegisterEvent("UPDATE_SHAPESHIFT_FORM");
  this:RegisterEvent("SPELLS_CHANGED");
  this:RegisterEvent("VARIABLES_LOADED");
  this:RegisterEvent("PLAYER_AURAS_CHANGED");   
  this:RegisterEvent("PLAYER_LOGIN");
  this:RegisterEvent("ADDON_LOADED");
  
  SlashCmdList["STANCESETS"] = StanceSets3_SlashCmd;
  SLASH_STANCESETS1 = "/stancesets";
  
  if not StanceSetsVars then 
    StanceSetsVars = { }; 
  end
  
  if ( StanceSets3ButtonOptions == nil ) then
    StanceSets3ButtonOptions = CloneTable(defaultStanceSets3ButtonOptions);
  end
  
  WeaponQuickSwap.OnLoad();

  if (select(2, UnitClass("player")) == "DRUID") 
  or (select(2, UnitClass("player")) == "PALADIN") then
    WeaponSwap_HasAmmoSlot = false;
  end
    
  StanceSets3Button_Init();
  StanceSets3Button_UpdatePosition();
 
  DEFAULT_CHAT_FRAME:AddMessage(StanceSets3Locale["Intro"],0,1,0,1); 
end

function StanceSets3_OnShow()
  local chkMiniMapButton = getglobal("StanceSets3MiniMapButton");
  -- Wenn ButtonShow TRUE zeige Hkchen im Dialog
  if (StanceSets3ButtonOptions.ButtonShown) then
    chkMiniMapButton:SetChecked(1);
  else
    chkMiniMapButton:SetChecked(nil);
  end
  return StanceSets3_UpdateAllStances(); 
end

function StanceSets3_OnEvent(event, arg1)
  if event == "UPDATE_BONUS_ACTIONBAR" then   
    StanceSets3_FormChanged();
  elseif event == "UPDATE_SHAPESHIFT_FORM" then
    StanceSets3_FormChanged();
  elseif event == "PLAYER_AURAS_CHANGED" then
    StanceSets3_FormChanged();
  elseif event == "SPELLS_CHANGED" then     
    StanceSets3_SpellsChanged();
  elseif event == "PLAYER_ENTERING_WORLD" then
    return WeaponSwap_RegisterEvents();
  elseif event == "PLAYER_LEAVING_WORLD" then
    return WeaponSwap_UnregisterEvents();
  elseif event == "ITEM_LOCK_CHANGED" then
    return WeaponQuickSwap.ExecuteSwapIteration();  	
  elseif event == "PLAYER_DEAD" then
    if WeaponSwap_IsSwapping then 
      return WeaponQuickSwap.WeaponQuickSwap_OnSwapError(StanceSets3Locale["PLAYER_DIED"]);
    end
  elseif event == "VARIABLES_LOADED" then     
    -- register for myaddons
    if(myAddOnsFrame) then
      myAddOnsList.StanceSets = {name = "StanceSets 3", description = "Stance Sets 3+1 (modified by Vorkosigan)", version = 2, category = MYADDONS_CATEGORY_COMBAT, frame = "StanceSets3Frame", optionsframe = "StanceSets3Frame"};
    end
  end
end

function StanceSets3_SlashCmd(cmd)
  if cmd == "next" then
    return StanceSets3_Next();
  elseif cmd == "next2" then
    return StanceSets3_Next(2);
  elseif cmd == "next3" then
    return StanceSets3_Next(3);
  elseif cmd == "set1" then
    StanceSets3_EquipSet(0);
  elseif cmd == "set2" then
    StanceSets3_EquipSet(1);
  elseif cmd == "set3" then
    StanceSets3_EquipSet(2);
  else
    StanceSets3_Toggle();
  end
end

function StanceSets3_Toggle()
  if StanceSets3Frame:IsVisible() then
    HideUIPanel(StanceSets3Frame);
  else
    ShowUIPanel(StanceSets3Frame);
  end
end

function StanceSets3_SpellsChanged()
  -- Spells have changed, this may mean that we've got a new shapeshift form
  if StanceSets3Frame:IsVisible() then
    return StanceSets3_UpdateAllStances();
  end
end

function StanceSets3_SetButtonTextureAndName(suffix,texture,itemName)
  local btn = getglobal("StanceSets3FrameSet"..suffix);
  if btn then
    SetItemButtonTexture(btn, texture);
    btn.itemName = itemName;

    if texture == nil and UnitHasRelicSlot("player") then
      local id = btn:GetID();
      if id == 3 or id == 7 or id == 11 then
        SetItemButtonTexture(btn, "Interface\\Paperdoll\\UI-PaperDoll-Slot-Relic.blp");
      end
    end
  end
end

function StanceSets3_FindAndSetButtonTexture(suffix, itemName)
  if itemName and itemName ~= "" then
    local bag, slot = WeaponQuickSwap.FindItem(itemName,0,true);
    if bag and slot then
      local texture;
      if bag == -1 then
        texture = GetInventoryItemTexture("player", slot);
      else
        local link = GetContainerItemLink(bag, slot);
        if link then
          local _, _, _, _, _, _, _, _, _, itemTexture = GetItemInfo(link);
          texture = itemTexture;
        else
          DEFAULT_CHAT_FRAME:AddMessage(StanceSets3Locale["Item"]..itemName..StanceSets3Locale["DEFAULT_TEXTURE"],1,0,0,1); 
          texture = "Interface\\Icons\\INV_Misc_Gift_01";
        end
      end
      StanceSets3_SetButtonTextureAndName(suffix, texture, itemName);
    else
      -- if not found, just use a generic icon
      StanceSets3_SetButtonTextureAndName(suffix, "Interface\\Icons\\INV_Misc_Gift_01", itemName);
    end
  else
    -- no item name, no icon
    StanceSets3_SetButtonTextureAndName(suffix, nil, itemName);
  end
end

function StanceSets3_UpdateAllStances()
  local allstances = StanceSets3_GetAllForms();
  
  for i = 1, STANCESETS3_MAX_STANCES do
    local stanceName  = allstances[i];
    local frameStance = getglobal("StanceSets3FrameSet"..i);
    local frameLabel  = getglobal("StanceSets3FrameSet"..i.."Title");
    
    if frameStance then
      local chkForceFirst = getglobal(frameStance:GetName().."ForceFirst");

      if stanceName then
        frameStance.StanceName = stanceName;

        if stanceName == STANCESETS3_SETGROUP2 or stanceName == STANCESETS3_SETGROUP3 then
          chkForceFirst:Hide();
        else
          chkForceFirst:Show();
        end
      
        frameStance:Show();
        if frameLabel then frameLabel:SetText(stanceName); end
        
        local setName = stanceName;
        
        if (StanceSetsVars[setName]) then
          local stanceSet = StanceSetsVars[setName];
        
          if StanceSetsVars[setName.."_ForceFirst"] then
            chkForceFirst:SetChecked(1);
          else
            chkForceFirst:SetChecked(nil);
          end

          StanceSets3_FindAndSetButtonTexture(i.."MainHand1", stanceSet[1]);
          StanceSets3_FindAndSetButtonTexture(i.."OffHand1",  stanceSet[2]);
          StanceSets3_FindAndSetButtonTexture(i.."Ranged1",   stanceSet[3]);
          StanceSets3_FindAndSetButtonTexture(i.."Munition1", stanceSet[4]);
          StanceSets3_FindAndSetButtonTexture(i.."MainHand2", stanceSet[5]);
          StanceSets3_FindAndSetButtonTexture(i.."OffHand2",  stanceSet[6]);
          StanceSets3_FindAndSetButtonTexture(i.."Ranged2",   stanceSet[7]);
          StanceSets3_FindAndSetButtonTexture(i.."Munition2", stanceSet[8]);
          StanceSets3_FindAndSetButtonTexture(i.."MainHand3", stanceSet[9]);
          StanceSets3_FindAndSetButtonTexture(i.."OffHand3",  stanceSet[10]);
          StanceSets3_FindAndSetButtonTexture(i.."Ranged3",   stanceSet[11]);
          StanceSets3_FindAndSetButtonTexture(i.."Munition3", stanceSet[12]);
        else
          StanceSets3_SetButtonTextureAndName(i.."MainHand1", nil, nil);
          StanceSets3_SetButtonTextureAndName(i.."OffHand1",  nil, nil);
          StanceSets3_SetButtonTextureAndName(i.."Ranged1",   nil, nil);
          StanceSets3_SetButtonTextureAndName(i.."Munition1", nil, nil);
          StanceSets3_SetButtonTextureAndName(i.."MainHand2", nil, nil);
          StanceSets3_SetButtonTextureAndName(i.."OffHand2",  nil, nil);
          StanceSets3_SetButtonTextureAndName(i.."Ranged2",   nil, nil);
          StanceSets3_SetButtonTextureAndName(i.."Munition2", nil, nil);
          StanceSets3_SetButtonTextureAndName(i.."MainHand3", nil, nil);
          StanceSets3_SetButtonTextureAndName(i.."OffHand3",  nil, nil);
          StanceSets3_SetButtonTextureAndName(i.."Ranged3",   nil, nil);
          StanceSets3_SetButtonTextureAndName(i.."Munition3", nil, nil);
        end
      else        
        frameStance:Hide();
      end
    end  -- if frame found
  end  -- for i 1,MAX
end

function StanceSets3_GetCurrentStanceSet(arg)
  local currentForm = StanceSets3_GetCurrentForm(arg);
  local setName     = currentForm;
  local sets        = StanceSetsVars[setName];

  if not sets or
     (not sets[1] and not sets[2] and not sets[3] and
      not sets[5] and not sets[6] and not sets[7] and 
      not sets[9] and not sets[10] and not sets[11]) then
    setName = "Default";
  end
  return StanceSetsVars[setName], setName;
end

function StanceSets3_Next(arg)
  local sets = StanceSets3_GetCurrentStanceSet(arg);
  if sets then
    return WeaponQuickSwap.WeaponSwap(sets[1], sets[2], sets[3], sets[4], sets[5], sets[6], sets[7], sets[8], sets[9], sets[10], sets[11], sets[12]);
  end
end

function StanceSets3_FormChanged()
  local curForm = StanceSets3_GetCurrentForm();
  if StanceSets3_LastForm then
    if curForm == StanceSets3_LastForm then
      return;
    end
  end
  StanceSets3_LastForm = curForm;

  local playerFormSet, setName = StanceSets3_GetCurrentStanceSet();
  if playerFormSet then   
    if StanceSetsVars[setName.."_ForceFirst"] then
      -- must have at least one thing to do a swap
      if playerFormSet[1] or playerFormSet[2] or playerFormSet[3] or playerFormSet[4] then
        return WeaponQuickSwap.WeaponSwap(playerFormSet[1], playerFormSet[2], playerFormSet[3], playerFormSet[4]);
      end
    end
  end
end

function StanceSets3_EquipSet(num) 
  local playerFormSet, setName = StanceSets3_GetCurrentStanceSet();
  if playerFormSet then   
    -- must have at least one thing to do a swap    
    if playerFormSet[1 + (num * 4)] or playerFormSet[2 + (num * 4)] or playerFormSet[3 + (num * 4)] then
      WeaponQuickSwap.WeaponSwap(playerFormSet[1 + (num * 4)], playerFormSet[2 + (num * 4)], playerFormSet[3 + (num * 4)], playerFormSet[4 + (num * 4)]);
    end
  end
end

function StanceSets3_GetCurrentForm(arg)
  if arg == 2 then
    return STANCESETS3_SETGROUP2;
  elseif arg == 3 then
    return STANCESETS3_SETGROUP3;
  else
    for i = 1, GetNumShapeshiftForms(), 1 do
      local _, name, isActive = GetShapeshiftFormInfo(i);
      if isActive then      
        return name;
      end
    end
  end
  
  return "Default";
end

function StanceSets3_GetAllForms()
  local retVal = { };
  table.insert(retVal, "Default");

  for i = 1, GetNumShapeshiftForms() do
    local _, name = GetShapeshiftFormInfo(i);
    table.insert(retVal, name);
  end
  table.insert(retVal, STANCESETS3_SETGROUP2);
  table.insert(retVal, STANCESETS3_SETGROUP3);
  
  local maxnum = GetNumShapeshiftForms() + 3;
  if maxnum > STANCESETS3_MAX_STANCES then
    maxnum = STANCESETS3_MAX_STANCES;
  end
  local height = 40 + (maxnum * 70);
  StanceSets3Frame:SetHeight(height);
  
  return retVal;
end

function StanceSets3Slot_OnEnter()
  if this.itemName and this.itemName ~= "" then
    GameTooltip:SetOwner(this, "ANCHOR_RIGHT");
    local bag, slot = WeaponQuickSwap.FindItem(this.itemName);
    if bag ~= nil and slot ~= nil then
  	  if bag and slot then
        if bag == -1 then
  	      GameTooltip:SetInventoryItem("player", slot);
  	    else
  	      GameTooltip:SetBagItem(bag, slot);
  	    end
      elseif bag == -1 and slot == 0 then
        GameTooltip:SetInventoryItem("player", slot);
      else
        GameTooltip:SetText(this.itemName, 1.0, 1.0, 1.0);
  	  end
  	  GameTooltip:Show();
    end
  end
end

function StanceSets3Slot_IDToSlotID(id)
  if id == 1 or id == 5 or id == 9 then
    return 16;
  elseif id == 2 or id == 6 or id == 10 then
    return 17;
  elseif id == 3 or id == 7 or id == 11 then
    return 18;
  else
    return 0;
  end
end

function StanceSets3Slot_OnDragStart()
  local frmParent = this:GetParent();
  local setName = frmParent.StanceName;
  
  if StanceSetsVars[setName] then
    StanceSetsVars[setName][this:GetID()] = nil;
    ResetCursor();
    StanceSets3_UpdateAllStances();
  end
end

function StanceSets3Slot_TakeItemOffCursor(srcBag, srcSlot)
  if srcBag == -1 then
    PickupInventoryItem(srcSlot);
  else
    PickupContainerItem(srcBag, srcSlot);
  end
end

function StanceSets3Slot_OnReceiveDrag()
  StanceSets3Slot_OnClick("LeftButton");
end

StanceSets3_PickedupItem = {};

function StanceSets3Slot_OnClick(arg1)
  local slotID = StanceSets3Slot_IDToSlotID(this:GetID());
  
  arg1 = arg1 or "";
  if arg1 == "LeftButton" then
    if CursorHasItem() and StanceSets3_PickedupItem.bag then
      if StanceSets3_PickedupItem.bag == -1 and StanceSets3_PickedupItem.slot == 0 then
        DEFAULT_CHAT_FRAME:AddMessage(StanceSets3Locale["USE_BAG"],1,0,0,1);
        PickupInventoryItem(0);
      else
	      local itemID = WeaponQuickSwap.GetItemID(StanceSets3_PickedupItem.bag, StanceSets3_PickedupItem.slot);
	      -- Cursor item can go in the slot or is Egan's Blaster.
	      if CursorCanGoInSlot(slotID) or itemID == 13289 then
	        local frmParent = this:GetParent();
	        local setName   = frmParent.StanceName;
	
	        if not StanceSetsVars then
	          StanceSetsVars = {};
	        end
	        if not StanceSetsVars[setName] then
	          StanceSetsVars[setName] = {};
	        end
	        StanceSetsVars[setName][this:GetID()] = WeaponQuickSwap.GetItemLink(StanceSets3_PickedupItem.bag, StanceSets3_PickedupItem.slot);
	         
	        StanceSets3_UpdateAllStances();
	      else
	        DEFAULT_CHAT_FRAME:AddMessage(itemName..StanceSets3Locale["CAN_NOT_GO"],1,0,0,1);
	      end
	    end  
      ClearCursor();
      ResetCursor();
    end  -- if has item and we know where it came from

    StanceSets3_PickedupItem.bag = nil;  
  end  -- if left button
end

local StanceSets3PickupContainerItem = function (bag, slot)
  StanceSets3_PickedupItem.bag  = bag;
  StanceSets3_PickedupItem.slot = slot;
end
hooksecurefunc("PickupContainerItem", StanceSets3PickupContainerItem);

local StanceSets3PickupInventoryItem = function (slot)
  StanceSets3_PickedupItem.bag  = -1;
  StanceSets3_PickedupItem.slot = slot;
end
hooksecurefunc("PickupInventoryItem", StanceSets3PickupInventoryItem);

function StanceSets3ForceCheck_OnClick()
  local frmParent = this:GetParent();
  local setName = frmParent.StanceName.."_ForceFirst";
  
  if (this:GetChecked()) then
    StanceSetsVars[setName] = true;
    PlaySound("igMainMenuOptionCheckBoxOff");
  else
    StanceSetsVars[setName] = nil;
    PlaySound("igMainMenuOptionCheckBoxOn");
  end
end     

---------------------------------------------------------------------------------------
-- WeaponQuickSwap Code
---------------------------------------------------------------------------------------

--[[
  WeaponQuickSwap - by CapnBry modified by Tageshi and Vorkosigan
  A script for moving weapons by name between hands, taking slot locking into account.
  
  Public Domain.  Feel free to use any of this code or ideas in your own mods
--]]

--
-- "Exported" functions for the user
--

function WeaponQuickSwap.WeaponSwap(...)
  rarg[1] = nil; rarg[2]  = nil; rarg[3]  = nil; rarg[4]  = nil;
  rarg[5] = nil; rarg[6]  = nil; rarg[7]  = nil; rarg[8]  = nil;
  rarg[9] = nil; rarg[10] = nil; rarg[11] = nil; rarg[12] = nil;
  local p = 1;
  local ignore1 = true;
  local ignore2 = true;
  local ignore3 = true;
  local ignore4 = true;
  -- Regularize argument list
  for i = 1, 9, 4 do
    if (select(i,...) or "") ~= "" or (select(i + 1,...) or "") ~= "" or (select(i + 2,...) or "") ~= "" or (select(i + 3,...) or "") ~= "" then
      if not select(i,...) then
        rarg[p] = "";
      else
        rarg[p] = select(i,...);
        ignore1 = false;
      end
      if not select(i + 1,...) then
        rarg[p + 1] = "";
      else
        rarg[p + 1] = select(i + 1,...);
        ignore2 = false;
      end
      if not select(i + 2,...) then
        rarg[p + 2] = "";
      else
        rarg[p + 2] = select(i + 2,...);
        ignore3 = false;
      end
      if not select(i + 3,...) then
        rarg[p + 3] = "";
      else
        rarg[p + 3] = select(i + 3,...);
        ignore4 = false;
      end
      p = p + 4;
    end
  end
  -- If all right hand slots are empty, right hand is ignored.
  -- If all left hand slots are empty, left hand is ignored.
  -- If all ranged slots are empty, ranged is ignored.
  -- If all munition slots are empty, munition is ignored.
  for i = 1, 9, 4 do
    if ignore1 and rarg[i] then
      rarg[i] = "*";
    end
    if ignore2 and rarg[i + 1] then
      rarg[i + 1] = "*";
    end
    if ignore3 and rarg[i + 2] then
      rarg[i + 2] = "*";
    end
    if ignore4 and rarg[i + 3] then
      rarg[i + 3] = "*";
    end
  end

  return WeaponQuickSwap.WeaponSwapCommon(rarg);
end

WeaponSwap = WeaponQuickSwap.WeaponSwap

---------------------------------------------------------------------------------------
-- Internal functions and callbacks for WeaponQuickSwap
---------------------------------------------------------------------------------------

function WeaponQuickSwap.OnLoad()
  this:RegisterEvent("ITEM_LOCK_CHANGED");
end

function WeaponSwap_RegisterEvents()
  WeaponQuickSwapFrame:RegisterEvent("ITEM_LOCK_CHANGED");
  WeaponQuickSwapFrame:RegisterEvent("PLAYER_DEAD");
end

function WeaponSwap_UnregisterEvents()
  WeaponQuickSwapFrame:UnregisterEvent("ITEM_LOCK_CHANGED");
  WeaponQuickSwapFrame:UnregisterEvent("PLAYER_DEAD");
end

local function swaplist_push(list, sb, si, db, di)
  local i = list.c + 1;
  list.c, list.sb[i], list.si[i], list.db[i], list.di[i] = i, sb, si, db, di;
end

local function swaplist_popfirst(list)
  if list.c == 0 then return; end
  list.c = list.c - 1;
end

local function ExtractItemID(link)
	local _, itemLink, _, _, _, _, _, _, _, _ = GetItemInfo(link);
	if itemLink then
    local found, _, itemString = string.find(itemLink, "^|c%x+|H(.+)|h%[.+%]");
    local _, itemId, enchantId, jewelId1, jewelId2, jewelId3, jewelId4, suffixId, uniqueId = strsplit(":", itemString);
    return itemId;
  end
  return nil;
end

local function ExtractItemLink(link)
	local _, itemLink, _, _, _, _, _, _, _, _ = GetItemInfo(link);
	if itemLink then
    return itemLink;
  end
  return nil;
end

function WeaponQuickSwap.ItemIsLocked(bag, slot)
  if bag == -1 and slot == -1 then return false; end

  if bag == -1 then
    return IsInventoryItemLocked(slot);
  else
    local _,_,retVal = GetContainerItemInfo(bag, slot);
    return retVal;
  end
end

function WeaponQuickSwap.AnyItemLocked()
  -- Checks all the bags and the 4 equip slots to see if any slot is still locked
  for i = 0, NUM_BAG_FRAMES do
    for j = 1,GetContainerNumSlots(i) do
      local _,_,retVal = GetContainerItemInfo(i, j);
      if retVal then
        return retVal;
      end
    end
  end
  return IsInventoryItemLocked(16) or IsInventoryItemLocked(17) or IsInventoryItemLocked(18) or IsInventoryItemLocked(0);
end

function WeaponQuickSwap.ExecuteSwapIteration()
  
  if WeaponSwap_IsInSwapIteration then return; end
  WeaponSwap_IsInSwapIteration = 1;

  if wswap.c == 0 then 
    if WeaponSwap_IsSwapping and not WeaponQuickSwap.AnyItemLocked() then
      WeaponSwap_IsInSwapIteration = nil;
      return WeaponQuickSwap.OnSwapComplete();
    end
    WeaponSwap_IsInSwapIteration = nil;
    return;
  end

  if WeaponQuickSwap.ItemIsLocked(wswap.sb[wswap.c], wswap.si[wswap.c]) or
     WeaponQuickSwap.ItemIsLocked(wswap.db[wswap.c], wswap.di[wswap.c]) then
    WeaponSwap_IsInSwapIteration = nil;
    return;
  end

  ClearCursor();

  -- aufnehmen der Quelle
  if wswap.sb[wswap.c] == -1 then
    PickupInventoryItem(wswap.si[wswap.c]);                    -- aus dem Inventar
  else
    PickupContainerItem(wswap.sb[wswap.c], wswap.si[wswap.c]); -- oder einem Taschenplatz
  end

  -- ablegen im Ziel
  if wswap.db[wswap.c] == -1 then
    if wswap.di[wswap.c] == -1 then
      PutItemInBackpack();                                     -- in einen Taschenplatz ablegen
    else
      PickupInventoryItem(wswap.di[wswap.c]);                  -- akt. Cursor in das Inventar legen
    end
  else
    PickupContainerItem(wswap.db[wswap.c], wswap.di[wswap.c]); -- an diese Stelle in der Tasche ablegen
  end

  swaplist_popfirst(wswap);                                    -- stack vom letzten zum ersten (lifo)
  WeaponSwap_IsInSwapIteration = nil;

  if not WeaponQuickSwap.PerformSlowerSwap then
    -- rekursiver Aufruf
    return WeaponQuickSwap.ExecuteSwapIteration();
  end

end

function WeaponQuickSwap.OnSwapComplete()
  wswap.c                      = 0;
  WeaponSwap_IsSwapping        = nil;
  WeaponSwap_IsInSwapIteration = nil;
  WeaponSwap_SwapTime          = nil;
  -- this is just here to allow people to hook the completion event
end

function WeaponQuickSwap.OnSwapError(error)
  wswap.c                      = 0;
  WeaponSwap_IsSwapping        = nil;
  WeaponSwap_IsInSwapIteration = nil;
  WeaponSwap_SwapTime          = nil;
  -- this is just here to allow people to hook the completion event
  return DEFAULT_CHAT_FRAME:AddMessage(error,1,0,0,1);
end

function WeaponQuickSwap.GetItemID(bag, slot)
  local link, itemID = nil, nil;
  if (bag == -1) then
    link = GetInventoryItemLink("player", slot);
  else
    link = GetContainerItemLink(bag, slot);
  end

  if link then
    itemID = ExtractItemID(link);
    return itemID;
  else
    return "";
  end
end

function WeaponQuickSwap.GetItemLink(bag, slot)
  local link, itemLink = nil, nil;
  if (bag == -1) then
    link = GetInventoryItemLink("player", slot);
  else
    link = GetContainerItemLink(bag, slot);
  end

  if link then
    itemLink = ExtractItemLink(link); 
    return itemLink;
  else
    return "";
  end
end

function WeaponQuickSwap.IsItem(bag, slot, ItemLink)
  local CurrentID, SearchID;
  local CurrentLink, SearchLink;
  if bag == -1 and slot == 0 then
    local link = GetInventoryItemLink("player", GetInventorySlotInfo("AmmoSlot"))
    if link then
      CurrentID = ExtractItemID(link);
      SearchID  = ExtractItemID(ItemLink);
      if CurrentID == SearchID then
        return true;
      end
    end      
  else 
    SearchLink  = ExtractItemLink(ItemLink);
    CurrentLink = WeaponQuickSwap.GetItemLink(bag, slot);
    if CurrentLink == SearchLink then
      return true;
    end
  end
  return false;
end


function WeaponQuickSwap.FindItem(ItemLink, skipcount, CompareIds)

  skipcount  = skipcount  or 0;
  CompareIds = CompareIds or false;
  
  -- First check the inventory slots 16, 17 and 18
  for i = 16, 18, 1 do
    if (WeaponQuickSwap.IsItem(-1, i, ItemLink)) then 
      if skipcount == 0 then return -1, i; end
      skipcount = skipcount - 1;
    end
  end

  -- not found check bags
  for i = 0, NUM_BAG_SLOTS do
    for j = 1, GetContainerNumSlots(i) do
      local link = GetContainerItemLink(i, j)
      if link then
        if CompareIds == true then
          local CurrentID = ExtractItemID(link);
          local SearchID  = ExtractItemID(ItemLink);
          if CurrentID == SearchID then
            return i, j
          end
        else
          local CurrentLink = ExtractItemLink(link);
          local SearchLink  = ExtractItemLink(ItemLink);
          if CurrentLink == SearchLink then
            if skipcount == 0 then return i, j; end
            skipcount = skipcount - 1;
          end
        end
      end
    end 
  end 

  -- not found return nil,nil
  return nil, nil;
end

function WeaponQuickSwap.IsNormalBag(id)
  if id == 0 then
    return true;
  end
  local link = GetInventoryItemLink("player", ContainerIDToInventoryID(id));
  if link then
    local _, _, id = string.find(link, "item:(%d+)");
    if id then
      local _, _, _, _, _, itemType, subType = GetItemInfo(id);
      if not (itemType == "Quiver" or string.find(subType, " ")) then
        return true;
      end
    end
  end
  return false;
end

function WeaponQuickSwap.SearchBagPlaceInList(bag, slot)
  local i = wswap.c;
  local NotFound = true;
  while i do
    if wswap.db[i] == bag and wswap.di[i] == slot then
      NotFound = false;
      break;
    end
    if i == 0 then
      break;
    end
    i = i - 1;
  end
  return NotFound;
end

function WeaponQuickSwap.FindLastEmptyBagSlot(bag_affinity, slot_affinity)

  -- try to put the item in the requested affinity, if possible
  if bag_affinity and slot_affinity and not GetContainerItemInfo(bag_affinity, slot_affinity) then
    if WeaponQuickSwap.SearchBagPlaceInList(bag_affinity, slot_affinity) then
      return bag_affinity, slot_affinity;
    end
  end
  
  -- if we couldn't get the bag and slot we wanted, just try the same bag
  if bag_affinity then
    for j = GetContainerNumSlots(bag_affinity), 1, -1 do
      if not GetContainerItemInfo(bag_affinity, j) then
        if WeaponQuickSwap.SearchBagPlaceInList(bag_affinity, j) then
          return bag_affinity, j;
        end
      end
    end
  end
  
  -- no affinity, chek all bags
  for i = NUM_BAG_FRAMES, 0, -1 do
    if bag_affinity ~= i then
      -- Make sure this isn't a quiver nor profession bag, those won't hold shit
      if WeaponQuickSwap.IsNormalBag(i) then
        for j = GetContainerNumSlots(i), 1, -1 do
          if not GetContainerItemInfo(i, j) then
            if WeaponQuickSwap.SearchBagPlaceInList(i, j) then
              return i, j;
            end
          end  
        end  
      end  
    end 
  end  

  return nil, nil;
end

function WeaponQuickSwap.FindCurrentSetIndex(startIndex, setsList)
  -- loop through the paramters and find what we have in our hands  
  local main, off;
  local argidx = startIndex;
  local retVal;
  while setsList[argidx] do
    main, off, ranged, ammo = setsList[argidx], setsList[argidx + 1], setsList[argidx + 2], setsList[argidx + 3];
    
    if not main then break; end
    
    -- don't need to specify the offhand if this is just a single deal
    if not off      then off      = ""; end
    if not ranged   then ranged   = ""; end
    if not ammo     then ammo     = ""; end

    if (main     == "*" or WeaponQuickSwap.IsItem(-1, mainhandslot, main))   and 
       (off      == "*" or WeaponQuickSwap.IsItem(-1, offhandslot,  off))    and
       (ranged   == "*" or WeaponQuickSwap.IsItem(-1, rangedslot,   ranged)) then 
      retVal = argidx;
      break;
    end

    argidx = argidx + 4;
  end
  
  return retVal;
end

function WeaponQuickSwap.FindLastUsedBagPlace(Inventory)
  local bag, slot;
  if Inventory == 16 then
    if LastMain.bag and LastMain.slot then
      if not GetContainerItemInfo(LastMain.bag, LastMain.slot) then
        if WeaponQuickSwap.SearchBagPlaceInList(LastMain.bag, LastMain.slot) then
          bag  = LastMain.bag;
          slot = LastMain.slot;
        end
      end
    end
    if not (bag and slot) then 
      if LastOff.bag and LastOff.slot then
        if not GetContainerItemInfo(LastOff.bag, LastOff.slot) then
          if WeaponQuickSwap.SearchBagPlaceInList(LastOff.bag, LastOff.slot) then
            bag  = LastOff.bag;
            slot = LastOff.slot;
          end
        end
      end
    end
    if not (bag and slot) then 
      bag, slot = WeaponQuickSwap.FindLastEmptyBagSlot();
    end
    return bag, slot;
  end

  if Inventory == 17 then
    if LastOff.bag and LastOff.slot then
      if not GetContainerItemInfo(LastOff.bag, LastOff.slot) then
        if WeaponQuickSwap.SearchBagPlaceInList(LastOff.bag, LastOff.slot) then
          bag  = LastOff.bag;
          slot = LastOff.slot;
        end
      end
    end
    if not (bag and slot) then 
      if LastMain.bag and LastMain.slot then
        if not GetContainerItemInfo(LastMain.bag, LastMain.slot) then
          if WeaponQuickSwap.SearchBagPlaceInList(LastMain.bag, LastMain.slot) then
            bag  = LastMain.bag;
            slot = LastMain.slot;
          end
        end
      end
    end
    if not (bag and slot) then 
      bag, slot = WeaponQuickSwap.FindLastEmptyBagSlot();
    end
    return bag, slot;
  end
  return bag, slot;
end

function WeaponQuickSwap.WeaponSwapCommon(arg)
  -- I explicitly use arg as a parameter instead of ... to prevent
  -- having to call unpack on the arg list from the caller
  if WeaponSwap_IsSwapping then 
    return;
  end

  wswap.c                           = 0;
  WeaponSwap_IsSwapping             = 1;
  WeaponQuickSwap.PerformSlowerSwap = nil;

  local bb, bi;
  local multiinst = 0;
  local main, off, ranged, ammo;
  local m_sb, m_si = -1, mainhandslot;
  local o_sb, o_si = -1, offhandslot;
  local r_sb, r_si = -1, rangedslot;
  local a_sb, a_si = -1, ammoslot;

  local matchingsetidx = WeaponQuickSwap.FindCurrentSetIndex(1, arg);
  
  -- if we found a match, and there is at least one weapon speficied in the next
  -- set, then use that set.  Else use the first 4
  if matchingsetidx and arg[matchingsetidx + 4] then
    main, off, ranged, ammo = arg[matchingsetidx + 4], arg[matchingsetidx + 5], arg[matchingsetidx + 6], arg[matchingsetidx + 7];
  else
    main, off, ranged, ammo = arg[1], arg[2], arg[3], arg[4];
  end
  
  -- prfen was schon angelegt ist.
  local m_ok = WeaponQuickSwap.IsItem(-1, mainhandslot, main);
  local o_ok = WeaponQuickSwap.IsItem(-1, offhandslot,  off);
  local r_ok = WeaponQuickSwap.IsItem(-1, rangedslot,   ranged);
  local a_ok;
  if WeaponSwap_HasAmmoSlot then
    a_ok = WeaponQuickSwap.IsItem(-1, ammoslot,     ammo);
  end

  if not off then
    off = "*";
  end
  if not ranged then
    ranged = "*";
  end
  if not ammo and WeaponSwap_HasAmmoSlot then
    ammo = "*";
  end
  
  if not m_ok then 
    if main == "*" then
      m_sb, m_si = -1, -1;-- aus dem Inventar nehmen, Platz wird nicht neu belegt              
    else
      m_sb, m_si = WeaponQuickSwap.FindItem(main);
      -- if FindItem returned the offhand weapon and it is already ok
      -- don't remove it.  Look harder
      if o_ok and m_sb == -1 and m_si == offhandslot then
        m_sb, m_si = WeaponQuickSwap.FindItem(main, 1);
      end
    end
    if not (m_sb and m_si) then
      return WeaponQuickSwap.OnSwapError(StanceSets3Locale["NO_MAIN"]..main);
    end
    -- wenn aus taschenplatz geholt merke dir die position 
    if m_sb ~= -1 then 
      LastMain.bag  = m_sb; 
      LastMain.slot = m_si;
    end
  end -- if not m_ok

  -- if you're using 2 of the same weapon FindItem needs to 
  -- know not to just return the first
  if main == off then multiinst = 1; else multiinst = 0; end
  
  if not o_ok then 
    if off == "*" then
      o_sb, o_si = -1, -1;-- aus dem Inventar nehmen, Platz wird nicht neu belegt
    else
      o_sb, o_si = WeaponQuickSwap.FindItem(off, multiinst);
      -- note that here we don't have to "look harder" because
      -- that would mean that both weapons are the same so multinst
      -- would already be set to 1
    end
    if not (o_sb and o_si) then
      return WeaponQuickSwap.OnSwapError(StanceSets3Locale["NO_OFF"]..off);
    end
    -- wenn aus taschenplatz geholt merke dir die position 
    if o_sb ~= -1 then 
      LastOff.bag  = o_sb; 
      LastOff.slot = o_si;
    end
  end  -- if not o_ok

  if not r_ok then 
    if ranged == "*" then
      r_sb, r_si = -1, -1;-- aus dem Inventar nehmen, Platz wird nicht neu belegt
    else
      r_sb, r_si = WeaponQuickSwap.FindItem(ranged);
    end
    if not (r_sb and r_si) then
      return WeaponQuickSwap.OnSwapError(StanceSets3Locale["NO_RANGE"]..ranged);
    end
    -- wenn aus taschenplatz geholt merke dir die position 
    if r_sb ~= -1 then 
      LastRange.bag  = r_sb; 
      LastRange.slot = r_si;
    end
  end  -- if not r_ok

  if not a_ok and WeaponSwap_HasAmmoSlot then 
    if ammo == "*" then
      a_sb, a_si = -1, -1;-- aus dem Inventar nehmen, Platz wird nicht neu belegt
    else
      a_sb, a_si = WeaponQuickSwap.FindItem(ammo, 0, true);
    end
    if not (a_sb and a_si) then
      return WeaponQuickSwap.OnSwapError(StanceSets3Locale["NO_AMMO"]..ammo);
    end
    -- wenn aus taschenplatz geholt merke dir die position 
    if r_sb ~= -1 then 
      LastAmmo.bag  = r_sb; 
      LastAmmo.slot = r_si;
    end
  end  -- if not a_ok
  
  -- Moving ranged equipment from a bag
  if not r_ok then 
    if r_sb ~= -1 then
      ClearCursor();
      PickupContainerItem(r_sb, r_si);
      PickupInventoryItem(rangedslot);
      r_ok = true;
    end
  end
  
  -- Moving munition from a bag (ammo always in a bag)
  if not a_ok and WeaponSwap_HasAmmoSlot then 
    if a_sb ~= -1 then
      ClearCursor();
      PickupContainerItem(a_sb, a_si);
      PickupInventoryItem(ammoslot);
      a_ok = true;
    end
  end
  -- Moving both hands from bags, that's easy
  if m_sb ~= -1 and o_sb ~= -1 then
    -- Load main first because if it is a 2h and we try to load offhand
    -- we get a "Cannot Equip that with a Two-handed weapon" error
    ClearCursor();
    PickupContainerItem(m_sb, m_si);
    PickupInventoryItem(mainhandslot);
    PickupContainerItem(o_sb, o_si);
    PickupInventoryItem(offhandslot);
    m_ok = true;
    o_ok = true;
    -- es wird keine Liste mehr aufgebaut ExecuteSwapIteration() wird zu beginn
    -- beendet da wswap.c = 0
  end

  -- Simple hand swap
  if m_sb == -1 and m_si == offhandslot and o_sb == -1 and o_si == mainhandslot then
    ClearCursor();
    PickupInventoryItem(mainhandslot);
    PickupInventoryItem(offhandslot);
    m_ok = true;
    o_ok = true;
    -- es wird keine Liste mehr aufgebaut ExecuteSwapIteration() wird zu beginn
    -- beendet da wswap.c = 0
  end

  -- Push the list.  We want to:
  -- Take the mainhand weapon out if it isn't going to offhand
  -- Move from wherever to the offhand.  If offhand is supposed to be empty, empty it.
  -- Install the main hand weapon.  No blank main hand is supported unless are going to be.
  --
  -- Do it backward, the swaplist is a stack
 
  -- Install main hand
  if not m_ok then
    -- if nothing going to the main hand
    if (m_sb == -1 and m_si == -1) then
      -- and the main is not going to the off: put it in a bag
      if not (o_sb == -1 and o_si == mainhandslot) then
        bb, bi = WeaponQuickSwap.FindLastUsedBagPlace(mainhandslot);
        if not (bb and bi) then 
          return WeaponQuickSwap.OnSwapError(StanceSets3Locale["NO_BAG_PLACE"]); 
        end
        swaplist_push(wswap, -1, mainhandslot, bb, bi);
        -- when moving A,"" -> "",B where A is a 2h, the offhand doesn't lock properly
        -- so work around it by swapping slowly (only one swap per lock notify)
        WeaponQuickSwap.PerformSlowerSwap = true;
      end
    else
      -- Aus Tasche oder Inventar (nebenhand) in die haupthand
      -- Der Fall m_sb == -1 and m_si == offhandslot wird hiermit behandelt
      swaplist_push(wswap, m_sb, m_si, -1, mainhandslot);
    end
  end
  
  -- Load offhand if not already there
  if not o_ok then
    -- if nothing going to the off hand
    if (o_sb == -1 and o_si == -1) then
      -- and the off is not going to the main: put it in a bag
      if not (m_sb == -1 and m_si == offhandslot) then
        bb, bi = WeaponQuickSwap.FindLastUsedBagPlace(offhandslot);
        if not (bb and bi) then 
          return WeaponQuickSwap.OnSwapError(StanceSets3Locale["NO_BAG_PLACE"]); 
        end
        swaplist_push(wswap, -1, offhandslot, bb, bi);
      end
    else
      -- if the main hand weapon is coming from the offhand slot
      -- we need to fix up its source to be where the offhand is 
      -- GOING to be after the bag->off swap
      if wswap.c > 0 and (m_sb == -1 and m_si == offhandslot) then
        wswap.sb[wswap.c] = o_sb;
        wswap.si[wswap.c] = o_si;
        -- don't set o_sb, o_si they're tested later
      end
      
      if WeaponQuickSwap.PerformSlowerSwap then
        --We need another workaround when moving 2H,"" -> "", 1H .
        --Do mainhand first by swapping list:
        swaplist_push(wswap, wswap.sb[wswap.c], wswap.si[wswap.c], wswap.db[wswap.c], wswap.di[wswap.c]);
        wswap.c = wswap.c - 2;
        swaplist_push(wswap, o_sb, o_si, -1, offhandslot);
        wswap.c = wswap.c + 1;
      else
        -- Aus Tasche oder Inventar (Haupthand) in die nebenhand
        -- Der Fall o_sb == -1 and o_si == mainhandslot wird hiermit behandelt
        swaplist_push(wswap, o_sb, o_si, -1, offhandslot);
      end
    end
  end
  
  -- Special Case: Moving off to main, and not main to off
  -- This is because maybe the main hand weapon is main only
  -- Wenn die Nebenhand in die Haupthand wechselt muss die jetzige Haupthand in das
  -- Inventar zurck gelegt werden. Der Listeneintrag fr den Wechsel off->main ist vorher
  -- gemacht worden. Da die Liste rckwrst abgearbeitet wird, wird diese Anweisung zuerst
  -- ausgefhrt, d.h. der Platz in der Haupthand wird gerumt spter wird die Nebenhand eingesetzt. 
  if not m_ok then
    if (m_sb == -1 and m_si == offhandslot) and not (o_sb == -1 and o_si == mainhandslot) then
      bb, bi = WeaponQuickSwap.FindLastUsedBagPlace(mainhandslot);
      if not (bb and bi) then 
        return WeaponQuickSwap.OnSwapError(StanceSets3Locale["NO_BAG_PLACE"]); 
      end
      swaplist_push(wswap, -1, mainhandslot, bb, bi);
    end
  end
  
  -- Same thing for off hand
  if not o_ok then
    if (o_sb == -1 and o_si == mainhandslot) and not (m_sb == -1 and m_si == offhandslot) then
      bb, bi = WeaponQuickSwap.FindLastUsedBagPlace(offhandslot);
      if not (bb and bi) then 
        return WeaponQuickSwap.OnSwapError(StanceSets3Locale["NO_BAG_PLACE"]); 
      end
      swaplist_push(wswap, -1, offhandslot, bb, bi);
    end
  end

  if not r_ok then
    -- if nothing going to the ranged slot
    if (r_sb == -1 and r_si == -1) then
      if LastRange.bag then
        bb, bi = WeaponQuickSwap.FindLastEmptyBagSlot(LastRange.bag, LastRange.slot);
      else
        bb, bi = WeaponQuickSwap.FindLastEmptyBagSlot();
      end
      if not (bb and bi) then 
        return WeaponQuickSwap.OnSwapError(StanceSets3Locale["NO_BAG_PLACE"]); 
      end
      swaplist_push(wswap, -1, rangedslot, bb, bi);
    else
      swaplist_push(wswap, r_sb, r_si, -1, rangedslot);
    end
  end
  
  if not a_ok and WeaponSwap_HasAmmoSlot then
    -- if nothing going to the munition slot
    if (a_sb == -1 and a_si == -1) then
      local bb, bi = WeaponQuickSwap.FindLastEmptyBagSlot();
      if not (bb and bi) then 
        return WeaponQuickSwap.OnSwapError(StanceSets3Locale["NO_BAG_PLACE"]); 
      end
      swaplist_push(wswap, -1, ammoslot, bb, bi);
    else
      swaplist_push(wswap, a_sb, a_si, -1, ammoslot);
    end
  end

  -- Start moving
  return WeaponQuickSwap.ExecuteSwapIteration();
end
