--[[
    Database of all minipets and mounts
    $Revision: 85 $
]]--

--[[
Copyright (c) 2008, LordFarlander
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation
      and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
]]--

local MAJOR_VERSION = "LibPetAndMountDatabase-1.1";
local MINOR_VERSION = tonumber(("$Revision: 85 $"):match("(%d+)")) + 90000

local LibPetAndMountDatabase, oldminor = LibStub:NewLibrary( MAJOR_VERSION, MINOR_VERSION );
if( not LibPetAndMountDatabase ) then
    return;
end--if

if( not LibPetAndMountDatabase.FoundSets ) then
    LibPetAndMountDatabase.FoundSets = {};
end--if    
if( not LibPetAndMountDatabase.SubSets ) then
    LibPetAndMountDatabase.SubSets = {};
end--if

local LibPeriodicTable = LibStub( "LibPeriodicTable-3.1", true );
if( not LibPeriodicTable ) then
    error( MAJOR_VERSION .. " requires LibPeriodicTable-3.1" );
end

--[[
    Value meanings:
    
    Mounts.*: Maximum speed increase over running in percent, where running is 100%
    MiniPets.Reagented: ItemID of item needed to summon (snowball)
    MiniPets.Quest: QuestID of quest item is used for
    MiniPets.Equipment: SlotName of slot the equipment goes in
    
    Negative number is a SPELL instead of an ITEM
]]--

function LibPetAndMountDatabase.NormalizeSetName( set )
    if( set:find( "PetAndMountDatabase." ) ~= 1 ) then
        set = "PetAndMountDatabase." .. set;
    end--if
    if( set == "PetAndMountDatabase.MiniPets.Holiday" ) then
        set = "PetAndMountDatabase.Critters.Reagented.Snowball";
    end--if
    return set;
end--LibPetAndMountDatabase.NormalizeSetName( set )

function LibPetAndMountDatabase:HasSet( set, normalize )
    local sets = LibPeriodicTable.sets;

    if( ( normalize == true ) or ( normalize == nil ) ) then
        set = self.NormalizeSetName( set );
    end--if
    if( self.FoundSets[set] ) then
        return true;
    end--if
    for setName, _ in pairs( sets ) do
        if( setName:find( set ) == 1 ) then
            self.FoundSets[set] = true;
            return true;
        end--if
    end--if
    return false;
end--LibPetAndMountDatabase:HasSet( set, normalize )

function LibPetAndMountDatabase:GetSubsets( set, normalize )
    local sets = LibPeriodicTable.sets;

    if( ( normalize == true ) or ( normalize == nil ) ) then
        set = self.NormalizeSetName( set );
    end--if
    if( self.SubSets[set] ) then
        return self.SubSets[set];
    end--if
    set = set .. ".";
    for setName, _ in pairs( sets ) do
        local start, ends = setName:find( set );

        if( ( start == 1 ) and ( ends ~= setName:len() ) ) then
            setName = setName:sub( ends + 1 );
            if( setName:find( "." ) ) then
                setName = ("%s%s"):format( set, strsplit( ".", setName ) );
            end--if
            if( self.SubSets[set] == nil ) then
                self.SubSets[set] = {};
            end--if
            self.SubSets[set][setName] = setName;
        end--if
    end--if
    return self.SubSets[set];
end--LibPetAndMountDatabase:GetSubsets( set, normalize )

function LibPetAndMountDatabase:ItemInSet( item, set, normalize )
    if( item == nil ) then
        return false;
    end--if
    if( ( normalize == true ) or ( normalize == nil ) ) then
        set = self.NormalizeSetName( set );
    end--if
    return LibPeriodicTable:ItemInSet( item, set );
end--LibPetAndMountDatabase:ItemInSet( item, set, normalize )

function LibPetAndMountDatabase:SpellInSet( spellID, set, normalize )
    if( spellID == nil ) then
        return false;
    end--if
    if( spellID > 0 ) then
        spellID = spellID * -1;
    end--if
    return self:ItemInSet( spellID, set, normalize );
end--LibPetAndMountDatabase:SpellInSet( item, set )

function LibPetAndMountDatabase:IterateSet( set, normalize )
    if( ( normalize == true ) or ( normalize == nil ) ) then
        set = self.NormalizeSetName( set, normalize );
    end--if
    return LibPeriodicTable:IterateSet( set );
end--LibPetAndMountDatabase:IterateSet( set, normalize )

function LibPetAndMountDatabase:GetMountAttributes( value )
    local speed, ridingSkill, passengers = strsplit( "|", value );

    return tonumber( speed ), tonumber( ridingSkill ), passengers and tonumber( passengers ) or 1;
end--LibPetAndMountDatabase:GetMountAttributes( value )

if( GetNumCompanions ) then
    function LibPetAndMountDatabase:GetPlayersCompanions( type )
        local returnList = {};
    
        for i = 1, GetNumCompanions( type ) do
            local creatureID, creatureName, creatureSpellID, creatureIcon, _ = GetCompanionInfo( type, i );
    
            returnList[i] = { spellID = creatureSpellID, icon = creatureIcon, modelID = creatureID, name = creatureName };
        end--for
        return returnList;
    end--LibPetAndMountDatabase:GetPlayersCompanions( type )
    
    function LibPetAndMountDatabase:IsCompanionType( companionName, type )
        for i = 1, GetNumCompanions( type ) do
            if( select( 2, GetCompanionInfo( type, i ) ) == companionName ) then
                return true;
            end--if
        end--for
        return false;
    end--LibPetAndMountDatabase:IsCompanionType( companionName, type )
end--if
