--[[
    Helper functions
    $Revision: 85 $
]]--

--[[
Copyright (c) 2008, LordFarlander
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation
      and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
]]--

local MAJOR_VERSION = "LibLordFarlander-2.0";
local MINOR_VERSION = tonumber(("$Revision: 85 $"):match("(%d+)")) + 90000

local LibLordFarlander, oldminor = LibStub:NewLibrary( MAJOR_VERSION, MINOR_VERSION );
if( not LibLordFarlander ) then
    return;
end--if

local gratuity = LibStub( "LibGratuity-3.0", true );

LibLordFarlander.OutlandZones = nil;
LibLordFarlander.NorthrendZones = nil;
LibLordFarlander.ColdWeatherFlying = 54197;

-- Notes:
-- Checks if an item is soulbound or quest
-- Arguments:
--     number - bag the item is in
--     number - slot the item is in
-- Returns:
-- * boolean - if the item given is soulbound
function LibLordFarlander.ItemIsSoulbound( bag, slot )
    if( gratuity ) then
        local strings;

        gratuity:SetBagItem( bag, slot );
        strings = gratuity:GetText( 2, 4, false, true );
        if( strings ) then
            for _, text in pairs( strings ) do
                text = text[1];
                if( ( text == ITEM_SOULBOUND ) or ( text == ITEM_BIND_QUEST ) ) then
                    return true;
                end--if
            end--for
        end--if
    end--if
    return false;
end--LibLordFarlander.ItemIsSoulbound( bag, slot )

function LibLordFarlander.PlayerItemCount( item )
    local _, itemLink, _, _, _, _, _, _, _, _ = GetItemInfo( item );

    if( itemLink ~= nil ) then
        return GetItemCount( itemLink );
    end--if
    return 0;
end--LibLordFarlander.PlayerItemCount( item )

-- Notes:
-- Gets an item string from a link
-- Arguments:
--     string - a link, can be any type of link
-- Returns:
-- * string - the item string from the link
function LibLordFarlander.GetItemStringFromLink( itemLink )
    if( type( itemLink ) == "string" ) then
        return itemLink:match( "^|c%x+|H(.+)|h%[.+%]" );
    end--if
    return nil;
end--LibLordFarlander.GetItemStringFromLink( itemLink )

function LibLordFarlander.IsFlyableZone( zone, noRequirementsCheck )
    local L = LibStub( "LibBabble-Zone-3.0", true );

    if( L and ( ( zone == L["Dalaran"] ) or ( zone == L["Wintergrasp"] ) ) ) then
        return false;
    end--if
    if( not LibLordFarlander.OutlandZones ) then
        LibLordFarlander.OutlandZones = table.concat( { GetMapZones( 3 ) }, "|" );
    end--if
    if( not LibLordFarlander.NorthrendZones ) then
        LibLordFarlander.NorthrendZones = table.concat( { GetMapZones( 4 ) }, "|" );
    end--if
    return ( LibLordFarlander.OutlandZones:find( zone ) ~= nil ) or
        --For Northrend the player has to have learned the Cold Weather Flying spell first
        ( ( LibLordFarlander.NorthrendZones:find( zone ) ~= nil ) and ( noRequirementsCheck or ( LibLordFarlander.HasPlayerSpell( LibLordFarlander.ColdWeatherFlying ) ) ) );
end--LibLordFarlander.IsFlyableZone( zone )

-- Notes:
-- Determines if the player is in a zone where flying is allowed
-- Returns:
-- * boolean - whether or not the player is in a zone where flying is allowed
function LibLordFarlander.IsInFlyableZone()
    return LibLordFarlander.IsFlyableZone( GetRealZoneText() );
end--LibLordFarlander.IsInFlyableZone()

-- Notes:
-- Gets an ID from a string
-- Arguments:
--     string - a string
-- Returns:
-- * number - the ID from the string
function LibLordFarlander.GetItemIDFromString( itemID )
    if( type( itemID ) == "string" ) then
        return tonumber( itemID:match( "(%d+)" ) );
    end--if
    return nil;
end--LibLordFarlander.GetItemIDFromString( itemID )

-- Notes:
-- Gets an ID from a link
-- Arguments:
--     string - a link, can be any type
-- Returns:
-- * number - the ID from the link
function LibLordFarlander.GetItemIDFromLink( itemLink )
    if( type( itemLink ) == "string" ) then
        return LibLordFarlander.GetItemIDFromString( LibLordFarlander.GetItemStringFromLink( itemLink ) );
    end--if
    return nil;
end--LibLordFarlander.GetItemIDFromLink( itemLink )

-- Notes:
-- Gets an item string from an ID
-- Arguments:
--     number - an item ID
-- Returns:
-- * string - the string from the ID
function LibLordFarlander.GetItemString( itemID )
    return LibLordFarlander.GetItemStringFromLink( LibLordFarlander.GetItemLink( itemID ) );
end--LibLordFarlander.GetItemString( itemID )

-- Notes:
-- Gets an item link from and ID
-- Arguments:
--     number - an item ID
-- Returns:
-- * string - the link from the ID
function LibLordFarlander.GetItemLink( itemID )
    return select( 2, GetItemInfo( itemID ) );
end--LibLordFarlander.GetItemLink( itemID )

-- Notes:
-- Checks to see if the player has a spell
-- Arguments:
--     number, string - A spell ID or spell link
-- Returns:
-- * boolean - whether or not the player has the spell
function LibLordFarlander.HasPlayerSpell( spell )
    local spellName, _, _, _, _, _, _, _, _ = GetSpellInfo( spell );

    if( spellName ) then
        return GetSpellLink( spell ) == GetSpellLink( spellName );
    end
    return false;
end--LibLordFarlander.PlayerHasSpell( spell )

-- Notes:
-- Deep copies a table to another table
-- Arguments:
--     table - Destination table (MUST already be a table)
--     table - Source table
function LibLordFarlander.CopyTable( to, from )
    for i, v in pairs( from ) do
        if( type( v ) == "table" ) then
            to[i] = {};
            LibLordFarlander.CopyTable( to[i], v );
        else
            to[i] = v;
        end--if
    end--for
end--LibLordFarlander.CopyTable( to, from )

function LibLordFarlander.AppendTable( to, from )
    for _, v in pairs( from ) do
        table.insert( to, v );
    end--for
end--LibLordFarlander.AppendTable( to, from )

-- #nodef
function LibLordFarlander.SimpleCompareTables( table1, table2 )
    if( table1 and table2 and ( type( table1 ) == "table" ) and ( type( table2 ) == "table" ) ) then
        for i, v in pairs( table1 ) do
            if( type( v ) == "table" ) then
                if( type( table2[i] ) == "table" ) then
                    if( not LibLordFarlander.SimpleCompareTables( v, table2[i] ) ) then
                        return false;
                    end--if
                else
                    return false;
                end--if
            elseif( table2[i] ~= v ) then
                return false;
            end--if
        end--for
    else
        return false;
    end--if
    return true;
end--LibLordFarlander.SimpleCompareTables( table1, table2 )

function LibLordFarlander.TableCount( t )
    local val = 0;

    for i, _ in pairs( t ) do
        val = val + 1;
    end--for
    return val;
end--LibLordFarlander.TableCount( t )

function LibLordFarlander.HasTableMembers( t )
    for i, _ in pairs( t ) do
        return true;
    end--for
    return false;
end--LibLordFarlander.HasTableMember( t )

-- Notes:
-- Compares two tables to see if they have all the same members and the values are the same
-- Arguments:
--     table - First table to compare
--     table - Second table to compare
-- Returns:
-- * boolean - whether or not the tables are exactly alike
function LibLordFarlander.CompareTables( table1, table2 )
    return LibLordFarlander.SimpleCompareTables( table1, table2 ) and LibLordFarlander.SimpleCompareTables( table2, table1 );
end--LibLordFarlander.CompareTables( table1, table2 )

-- Notes:
-- Debug message printing, suitable for including in your addon's class.
-- Set your addon's DEBUG member to true to print.
-- Arguments:
--     addon - The addon object
--     string - The message to print
function LibLordFarlander.dprint( addon, msg )
    if( addon.DEBUG ) then
        DEFAULT_CHAT_FRAME:AddMessage( msg, 1.0, 0.0, 0.0 );
    end--if
end--LibLordFarlander.dprint( addon, msg )

function LibLordFarlander.round( num, idp )
    local mult = 10 ^ ( idp or 0 );

    return math.floor( num * mult + 0.5 ) / mult;
end--LibLordFarlander.round( num, idp )

function LibLordFarlander.GetTOCVersion()
    return tonumber( select( 4, GetBuildInfo() ) );
end--LibLordFarlander.GetTOCVersion()

function LibLordFarlander.GetMapZonesWithExtra( continent )
    local L = LibStub( "LibBabble-Zone-3.0", true );
    local mapzones = { GetMapZones( continent ) };

    if( L ) then
        L = L:GetLookupTable();
        if( continent == 2 ) then -- Eastern Kingdoms
            if( tonumber( GetBuildInfo():sub( 1, 1 ) ) >= 3 ) then
                tinsert( mapzones, L["Alterac Valley"] ); tinsert( mapzones, L["Arathi Basin"] ); tinsert( mapzones, L["Armory"] ); tinsert( mapzones, L["Blackrock Depths"] ); tinsert( mapzones, L["Blackrock Spire"] ); tinsert( mapzones, L["Champions' Hall"] );
                tinsert( mapzones, L["Cathedral"] ); tinsert( mapzones, L["The Deadmines"] ); tinsert( mapzones, L["Deeprun Tram"] ); tinsert( mapzones, L["Gnomeregan"] ); tinsert( mapzones, L["Graveyard"] ); tinsert( mapzones, L["Library"] );
                tinsert( mapzones, L["Lower Blackrock Spire"] ); tinsert( mapzones, L["Magisters' Terrace"] ); tinsert( mapzones, L["Molten Core"] ); tinsert( mapzones, L["Onyxia's Lair"] ); tinsert( mapzones, L["Ruins of Lordaeron"] );
                tinsert( mapzones, L["Scholomance"] ); tinsert( mapzones, L["Shadowfang Keep"] ); tinsert( mapzones, L["The Stockade"] ); tinsert( mapzones, L["Stratholme"] ); tinsert( mapzones, L["Sunwell Plateau"] ); tinsert( mapzones, L["The Temple of Atal'Hakkar"] );
                tinsert( mapzones, L["Uldaman"] ); tinsert( mapzones, L["Upper Blackrock Spire"] ); tinsert( mapzones, L["Zul'Aman"] ); tinsert( mapzones, L["Zul'Gurub"] );
            else -- Naxxramas is now in Northrend
                tinsert( mapzones, L["Alterac Valley"] ); tinsert( mapzones, L["Arathi Basin"] ); tinsert( mapzones, L["Armory"] ); tinsert( mapzones, L["Blackrock Depths"] ); tinsert( mapzones, L["Blackrock Spire"] ); tinsert( mapzones, L["Champions' Hall"] );
                tinsert( mapzones, L["Cathedral"] ); tinsert( mapzones, L["The Deadmines"] ); tinsert( mapzones, L["Deeprun Tram"] ); tinsert( mapzones, L["Gnomeregan"] ); tinsert( mapzones, L["Graveyard"] ); tinsert( mapzones, L["Library"] );
                tinsert( mapzones, L["Lower Blackrock Spire"] ); tinsert( mapzones, L["Magisters' Terrace"] ); tinsert( mapzones, L["Molten Core"] ); tinsert( mapzones, L["Naxxramas"] ); tinsert( mapzones, L["Onyxia's Lair"] ); tinsert( mapzones, L["Ruins of Lordaeron"] );
                tinsert( mapzones, L["Scholomance"] ); tinsert( mapzones, L["Shadowfang Keep"] ); tinsert( mapzones, L["The Stockade"] ); tinsert( mapzones, L["Stratholme"] ); tinsert( mapzones, L["Sunwell Plateau"] ); tinsert( mapzones, L["The Temple of Atal'Hakkar"] );
                tinsert( mapzones, L["Uldaman"] ); tinsert( mapzones, L["Upper Blackrock Spire"] ); tinsert( mapzones, L["Zul'Aman"] ); tinsert( mapzones, L["Zul'Gurub"] );
            end--if
        elseif( continent == 1 ) then -- Kalimdor
            tinsert( mapzones, L["Blackfathom Deeps"] ); tinsert( mapzones, L["The Black Morass"] ); tinsert( mapzones, L["Dire Maul (East)"] ); tinsert( mapzones, L["Dire Maul (West)"] ); tinsert( mapzones, L["Dire Maul (North)"] );
            tinsert( mapzones, L["Hall of Legends"] ); tinsert( mapzones, L["Hyjal Summit"] ); tinsert( mapzones, L["Maraudon"] ); tinsert( mapzones, L["Old Hillsbrad Foothills"] ); tinsert( mapzones, L["Ragefire Chasm"] ); tinsert( mapzones, L["Razorfen Downs"] );
            tinsert( mapzones, L["Razorfen Kraul"] ); tinsert( mapzones, L["Ruins of Ahn'Qiraj"] ); tinsert( mapzones, L["Temple of Ahn'Qiraj"] ); tinsert( mapzones, L["Wailing Caverns"] ); tinsert( mapzones, L["Warsong Gulch"] );
        elseif( continent == 3 ) then -- Outland
            tinsert( mapzones, L["The Arcatraz"] ); tinsert( mapzones, L["Auchenai Crypts"] ); tinsert( mapzones, L["Black Temple"] ); tinsert( mapzones, L["Blade's Edge Arena"] ); tinsert( mapzones, L["The Blood Furnace"] ); tinsert( mapzones, L["The Botanica"] );
            tinsert( mapzones, L["The Eye"] ); tinsert( mapzones, L["Eye of the Storm"] ); tinsert( mapzones, L["Gruul's Lair"] ); tinsert( mapzones, L["Hellfire Ramparts"] ); tinsert( mapzones, L["Magtheridon's Lair"] ); tinsert( mapzones, L["Mana-Tombs"] );
            tinsert( mapzones, L["The Mechanar"] ); tinsert( mapzones, L["Nagrand Arena"] ); tinsert( mapzones, L["Serpentshrine Cavern"] ); tinsert( mapzones, L["Sethekk Halls"] ); tinsert( mapzones, L["Shadow Labyrinth"] ); tinsert( mapzones, L["The Shattered Halls"] );
            tinsert( mapzones, L["The Slave Pens"] ); tinsert( mapzones, L["The Steamvault"] ); tinsert( mapzones, L["Twisting Nether"] ); tinsert( mapzones, L["The Underbog"] );
        elseif( continent == 4 ) then -- Northrend
            tinsert( mapzones, L["Azjol-Nerub"] ); tinsert( mapzones, L["Chamber of Aspects"] ); tinsert( mapzones, L["Drak'Tharon Keep"] ); tinsert( mapzones, L["Gundrak"] ); tinsert( mapzones, L["Halls of Stone"] ); tinsert( mapzones, L["Halls of Lightning"] );
            tinsert( mapzones, L["The Eye of Eternity"] ); tinsert( mapzones, L["Naxxramas"] ); tinsert( mapzones, L["The Nexus"] ); tinsert( mapzones, L["The Oculus"] ); tinsert( mapzones, L["Stratholme Past"] ); tinsert( mapzones, L["Ulduar"] ); tinsert( mapzones, L["Utgarde Keep"] );
            tinsert( mapzones, L["Utgarde Pinnacle"] ); tinsert( mapzones, L["The Violet Hold"] );
        end--if
    end--if
    return mapzones;
end--LibLordFarlander.GetMapZonesWithExtra( continent )

function LibLordFarlander.NamifyString( str )
    local returnString = str:gsub( "([%c%s%p])", function( input ) return ("_%03i"):format( strbyte( input ) ) end );

    return returnString;
end--LibLordFarlander.NamifyString( str )

function LibLordFarlander.UnnamifyString( str )
    local returnString = str:gsub( "_(%d%d%d)", function( input ) return strchar( tonumber( input ) ) end );
    
    return returnString;
end--LibLordFarladner.UnnamifyString( str )
