--[[

Spellcraft: This is a World of Warcraft Mage Utility AddOn.
Copyright (C) 2007  Patrick J. Donnelly (batrick@unm.edu)

Permission is NOT granted to modify/redistribute this software.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

--]]


--=======
--GLOBALS
--=======

SC_myConfig = { };


--=====================================
--Setting up the environment and module
--=====================================

local Spellcraft = Spellcraft or { };
local _G = getfenv(0);

Spellcraft.name = "Spellcraft";
Spellcraft.version = 1.115;

_G[Spellcraft.name] = Spellcraft;

setmetatable(Spellcraft, getmetatable(Spellcraft) or {__index = _G});
setfenv(1, Spellcraft);

--======
--Locals
--======

local tclass = {
  DRUID = "Druid",
  HUNTER = "Hunter",
  MAGE = "Mage",
  PALADIN = "Paladin",
  PRIEST = "Priest",
  ROGUE = "Rogue",
  SHAMAN = "Shaman",
  WARLOCK = "Warlock",
  WARRIOR = "Warrior"
};

local modules = { };

local errors = { };

local EventsNotify = { };

local MainTabs = {num = 0, frames = { }, placement = 10};

SC = CreateFrame("Frame", "SC", UIParent);

do
  local SC_Confirmation = CreateFrame("Frame", "SC_Confirmation", SC);
  SC_Confirmation:SetWidth(500);
  SC_Confirmation:SetHeight(100);
  SC_Confirmation:SetPoint("CENTER", UIParent, "CENTER");
  SC_Confirmation:SetBackdrop({
      bgFile = "Interface/DialogFrame/UI-DialogBox-Background",
      edgeFile = "Interface/DialogFrame/UI-DialogBox-Border",
      tile = true,
      tileSize = 32,
      edgeSize = 32,
    insets = { left = 5, right = 5, top = 5, bottom = 5 }
  });
  SC_Confirmation:Hide();
  local fs = SC_Confirmation:CreateFontString(
      "SC_Confirmation_Confirm", "ARTWORK", "GameFontNormal");
  fs:SetPoint("TOP", SC_Confirmation, "TOP", 0, -10);
  fs:SetText("");
  fs:SetWidth(480);

  local button = CreateFrame("Button", "SC_Confirmation_Yes", SC_Confirmation,
      "OptionsButtonTemplate");
  button:SetPoint("BOTTOM", SC_Confirmation, "BOTTOM", -50, 20);
  button:SetText(YES);

  button = CreateFrame("Button", "SC_Confirmation_No", SC_Confirmation,
      "OptionsButtonTemplate");
  button:SetPoint("BOTTOM", SC_Confirmation, "BOTTOM", 50, 20);
  button:SetText(NO);
end

do
  local SC_Notification = CreateFrame("Frame", "SC_Notification", SC);
  SC_Notification:SetWidth(500);
  SC_Notification:SetHeight(100);
  SC_Notification:SetPoint("CENTER", UIParent, "CENTER");
  SC_Notification:SetBackdrop({
      bgFile = "Interface/DialogFrame/UI-DialogBox-Background",
      edgeFile = "Interface/DialogFrame/UI-DialogBox-Border",
      tile = true,
      tileSize = 32,
      edgeSize = 32,
    insets = { left = 5, right = 5, top = 5, bottom = 5 }
  });
  SC_Notification:Hide();
  local fs = SC_Notification:CreateFontString(
      "SC_Notification_Message", "ARTWORK", "GameFontNormalLarge");
  fs:SetPoint("TOP", SC_Notification, "TOP", 0, -10);
  fs:SetText("");
  fs:SetWidth(480);

  local button = CreateFrame("Button", "SC_Notification_Ok", SC_Notification,
      "OptionsButtonTemplate");
  button:SetPoint("BOTTOM", SC_Notification, "BOTTOM", 0, 20);
  button:SetText(OKAY);
  button:SetScript("OnClick", function()
    SC_Notification:Hide();
  end);
end


--===============
--Local Functions
--===============

local function errhandler(err)
  local trace = debugstack();
  table.insert(errors, {date(), err, trace, event});
  return error(string.format(SC_MSG_ERROR, event, err, trace));
end


--==============================
--Helper Functions and Variables
--==============================

changes = { };

player = {name = "", class = "", level = 0, GUID = ""};

function print(message, r, g, b, t)
  return DEFAULT_CHAT_FRAME:AddMessage("|cff4477ffSC:|r "..tostring(message),
      r or 0, g or 0.2, b or 0.6, t or 0.0);
end

function error(message, trace)
  return print(message, 1.0, 0, 0, 0);
end

function inform(message)
  if SC_myConfig.inform then
    return print(message, 0, 1.0, 0, 0);
  end
end

function decompose(unform, form, ...)
  local t, p, mod = {...}, 1, "";
  if GetLocale() == "enEN" or GetLocale() == "enUS" then
    mod = string.gsub(unform, "%%s", function()
      for _,v in pairs(t) do
        if v == p then
          p = p + 1;
          return "(.*)";
        end
      end
      p = p + 1;
      return ".*";
    end);
  elseif GetLocale() == "deDE" then
    mod = string.gsub(unform, "%%(%d+)%$s", function(d)
      for _,v in pairs(t) do
        if v == tonumber(d) then
          return "(.*)";
        end
      end
      return ".*";
    end);
  end
  return string.find(form, mod);
end

function CreateSecureButton(Name, Parent, SizeX, SizeY, Anchor, ax, ay, Texture)
  local x = CreateFrame("Button", Name, Parent, "SecureActionButtonTemplate");
  x:SetWidth(SizeX);
  x:SetHeight(SizeY);
  x:SetPoint("CENTER", Parent, Anchor, ax, ay);
  if Texture then x:SetNormalTexture(Texture); end
  return x;
end

function ctruth(t)
  if t and t ~= 0 then
    return true;
  else
    return false;
  end
end

function isXDebuffUp(who, debuff)
  if UnitIsDead(who) or not UnitExists(who) then
    return false
  end
  local u, b = 1;
  local z = '';
  repeat b = UnitDebuff(who, u);
   if b == nil then break end
   z = z..b;
   u = u + 1;
  until u == 18;
  if string.find(z, debuff) then return true else return false end
end

function isFocusDebuffUp(debuff)
  return isXDebuffUp("focus", debuff);
end

function numberSane(num)
  local number = tonumber(num);
  if number then
    return true, number;
  else
    return false;
  end
end

function isHealer(unitid)
  local _, class = UnitClass(unitid);
  return class and (class == "PRIEST" or class == "SHAMAN" or
         class == "PALADIN" or class == "DRUID");
end

function standardName(s)
  local string = string;
  return string.upper(string.sub(s, 1, 1))..string.sub(string.lower(s), 2);
end

function isInBattleground()
  local zone = GetRealZoneText();
  return zone == SC_ZONE_ALTERACVALLEY or
         zone == SC_ZONE_WARSONG or zone == SC_ZONE_ARATHIBASIN or
         zone == SC_ZONE_EYE
end

function tcopy(table, _lookup_table)
  _lookup_table = _lookup_table or {}
  local copy = {}
  if not _lookup_table[table] then
    _lookup_table[table] = copy;
  end
  for i,v in pairs(table) do
    if type(i) == "table" then
      if _lookup_table[i] then
        i = _lookup_table[i]
      else
        i = tcopy( i, _lookup_table )
      end
    end
    if type( v ) ~= "table" then
      copy[i] = v
    else
      if _lookup_table[v] then
        copy[i] = _lookup_table[v]
      else
        copy[i] = tcopy( v, _lookup_table )
      end
    end
  end
  return copy
end

function Confirmation(text, r, g, b, Yes, No)
  SC_Confirmation_Confirm:SetText(text);
  SC_Confirmation_Confirm:SetTextColor(r, g, b);
  SC_Confirmation_Yes:SetScript("OnClick", Yes);
  SC_Confirmation_No:SetScript("OnClick", No);
  SC_Confirmation:Show();
end

function Notification(text, r, g, b)
  SC_Notification_Message:SetText(text);
  SC_Notification_Message:SetTextColor(r or 1, g or 0, b or 0);
  SC_Notification:Show();
end


--================
--Module Functions
--================

function RegisterModule(Module)
  modules[Module] = true;
  Spellcraft[Module.name] = Module;
end

-- This is to only be used when the module chooses not to load,
-- e.g. the module is for mages only
function UnregisterModule(Module)
  Spellcraft[Module.name] = nil;
  modules[Module] = nil;
  for event in pairs(EventsNotify) do
    EventsNotify[event][Module] = nil;
  end
end

function SC_RegEvent(mod, event)
  EventsNotify[event] = EventsNotify[event] or { };
  if not next(EventsNotify[event]) then
    SC:RegisterEvent(event);
  end
  EventsNotify[event][mod] = mod;
end

function SC_UnRegEvent(mod, event)
  EventsNotify[event] = EventsNotify[event] or { };-- Why would this happen?
  EventsNotify[event][mod] = nil;
  if not next(EventsNotify[event]) then
    SC:UnregisterEvent(event);
  end
end

function LoadModules()
  local remove = { };
  for mod in pairs(modules) do
    if not SC_myConfig[mod.name] then mod.Defaults(); end
    if SC_myConfig[mod.name].load then
      mod.Load();
    else
      remove[mod] = true;
    end
  end
  for mod in pairs(remove) do
    local m = {
      name = mod.name,
      lname = mod.lname,
      info = mod.info,
      version = mod.version,
      Defaults = mod.Defaults
    };
    setfenv(m.Defaults, Spellcraft);
    modules[mod] = nil;
    modules[m] = false;
    Spellcraft[mod.name] = nil;
  end
  collectgarbage();
end

function OnEvent(frame, event, ...)
  if EventsNotify[event] then
    for mod in pairs(EventsNotify[event]) do
      --xpcall(v.Notify, errhandler);
      if mod[event] then
        mod[event](...);
      end
    end
  end
  if event == "PLAYER_ENTERING_WORLD" then
    Update();
  end
end

function Update()
  Player();
  -- Send update to all modules
end

function Player()
  local _, class = UnitClass("player");
  player.class = tclass[class];
  player.level = UnitLevel("player");
  player.name = UnitName("player");
  player.GUID = UnitGUID("player");
end

function SetDefaults()
  _G.SC_myConfig = {inform = true};
  config = _G.SC_myConfig;
  for mod in pairs(modules) do
    if mod.Defaults then
      mod.Defaults();
    end
  end
end

--===================
--Main Frame Handling
--===================

local function ShowMain()
  if InCombatLockdown() then
    error(SC_MSG_SETTINGS_COMBAT);
  else
    if not SC_Main then MakeMainFrame(); end
    SC_Main:Show();
    SC_MainTab1:Click();
  end
end

--======================
--Configuration Handling
--======================

local function MakeGeneralFrame()
  local general = CreateFrame("Frame", "SC_Main_General", SC_Main);
  general:SetHeight(425);
  general:SetWidth(360);
  general:SetPoint("TOPLEFT", SC_Main, "TOPLEFT", 20, -50);
  general:SetBackdrop({
      bgFile = "Interface/Tooltips/UI-Tooltip-Background",
      edgeFile = "Interface/Tooltips/UI-Tooltip-Border",
      tile = true,
      tileSize = 16,
      edgeSize = 12,
      insets = { left = 2, right = 2, top = 2, bottom = 2 }
    });
  general:SetBackdropColor(0.1, 0.1, 0.1);
  general:SetBackdropBorderColor(0.4, 0.1, 0.8);
  general:SetScript("OnShow", function()
      SC_Main_Current:SetText(SC_MAIN_GENERAL);
    end);

  local options = general:CreateFontString("SC_Main_General_Options",
      "ARTWORK", "GameFontNormal");
  options:SetText(SC_MAIN_GENERAL_OPTIONS);
  options:SetTextColor(.3, 0, 1);
  options:SetPoint("TOPLEFT", general, "TOPLEFT", 5, -5);

  local temp = CreateFrame("Button", "SC_Main_General_Done", SC_Main_General,
      "OptionsButtonTemplate");
  temp:SetText(DONE);
  temp:SetPoint("CENTER", SC_Main_General, "BOTTOM", -50, 20);
  temp:SetScript("OnClick", function()
    SettingsDone();
    SC_Main:Hide();
  end);

  temp = CreateFrame("Button", "SC_Main_General_Defaults", SC_Main_General,
      "OptionsButtonTemplate");
  temp:SetText(DEFAULT);
  temp:SetPoint("BOTTOM", SC_Main_General, "TOP", 0, -180);
  temp:SetScript("OnClick", function()
    Confirmation(SC_MAIN_GENERAL_DEFAULTS, 1, 0, 0, function()
      SetDefaults();
      SC_Confirmation:Hide();
      ReloadUI();
    end, function()
      SC_Confirmation:Hide();
      SC_Main:Show();
    end);
    SC_Main:Hide();
  end);

  temp = CreateFrame("Button", "SC_Main_General_Cancel", SC_Main_General,
      "OptionsButtonTemplate");
  temp:SetText(CANCEL);
  temp:SetPoint("CENTER", SC_Main_General, "BOTTOM", 50, 20);
  temp:SetScript("OnClick", function()
    SC_Main:Hide();
  end);

  RegisterMainTab(general, SC_MAIN_GENERAL);

  -- Module Frame
  local scroll = CreateFrame("ScrollFrame", "SC_Main_General_ScrollModules",
    SC_Main_General, "UIPanelScrollFrameTemplate");
  scroll:SetHeight(165);
  scroll:SetWidth(200);
  scroll:SetPoint("BOTTOMLEFT", SC_Main_General, "BOTTOMLEFT", 20, 50);
  scroll:SetBackdrop({
      bgFile = "Interface/Tooltips/UI-Tooltip-Background",
      edgeFile = "Interface/Tooltips/UI-Tooltip-Border",
      tile = true,
      tileSize = 16,
      edgeSize = 12,
      insets = { left = 2, right = 2, top = 2, bottom = 2 }
    });
  scroll:SetBackdropColor(0.1, 0.1, 0.7);
  scroll:SetBackdropBorderColor(0.2, 0.5, 0.3);

  local fs = scroll:CreateFontString("SC_Main_General_Modules", "ARTWORK",
      "GameFontNormal");
  fs:SetText(SC_MAIN_GENERAL_MODULES);
  fs:SetTextColor(0, 0, 1);
  fs:SetPoint("BOTTOM", scroll, "TOP", 0, 10);

  local mod = CreateFrame("Frame", "SC_Main_General_ScrollModulesFrame",
    scroll);
  mod:SetHeight(250);
  mod:SetWidth(300);
  mod:SetPoint("TOPLEFT", scroll, "TOPLEFT");

  --List Modules, allow them to be removed.
  mod.children = { };
  local y, i = -10, 1;
  for m in pairs(modules) do
    local button = CreateFrame("Button", "SC_Modules"..i, mod);
    mod.children[m.name] = button;
    button:SetWidth(180);
    button:SetHeight(15);
    if changes[m.name] and changes[m.name].load or SC_myConfig[m.name].load then
      button:SetText(string.format("|c0000FF00%s    %s!|r", m.lname,
          SC_ENABLED));
    else
      button:SetText(string.format("|c00FF0000%s    %s!|r", m.lname,
          SC_DISABLED));
    end
    button:SetTextFontObject("GameFontHighlight");
    (button:GetFontString()):SetAllPoints(button);
    (button:GetFontString()):SetJustifyH("LEFT");
    button:SetPoint("TOPLEFT", mod, "TOPLEFT", 10, y);
    button:SetScript("OnEnter", function()
      GameTooltip:SetOwner(this, "ANCHOR_RIGHT");
      GameTooltip:SetText(string.format("%s (v%.3f): %s", m.lname,
          m.version, m.info));
      GameTooltip:Show();
    end);
    button:SetScript("OnLeave", function()
      GameTooltip:Hide();
    end);
    button:SetScript("OnClick", function()
      changes[m.name] = changes[m.name] or {load = SC_myConfig[m.name].load};
      if changes[m.name].load then
        changes[m.name].load = false;
        button:SetText(string.format("|c00FF0000%s    %s!|r", m.lname,
            SC_DISABLED));
      else
        changes[m.name].load = true;
        button:SetText(string.format("|c0000FF00%s    %s!|r", m.lname,
            SC_ENABLED));
      end
      Reload = true;
    end);
    y = y - button:GetHeight() - 5;
    i = i + 1;
  end

  scroll:SetScrollChild(mod);
  mod:SetHeight(math.ceil(math.abs(y)));
  scroll:UpdateScrollChildRect();


  local num, place = 1, -20;
  -- Function to register an option with general!
  function RegisterGeneralOption(text, checked, funcClick)
    local fs = SC_Main_General:CreateFontString("SC_Main_General_Option"..num,
        "ARTWORK", "GameFontNormal");
    fs:SetText(text);
    fs:SetTextColor(0, 1, .4);
    fs:SetWidth(300);
    fs:SetJustifyH("LEFT");
    fs:SetPoint("TOPLEFT", SC_Main_General, "TOPLEFT", 10, place);

    local cb = CreateFrame("CheckButton", "SC_Main_General_OptionCB"..num,
        SC_Main_General, "OptionsCheckButtonTemplate");
    cb:SetPoint("LEFT", fs, "RIGHT", 15, 0)
    cb:SetChecked(checked);
    cb:SetScript("OnClick", funcClick);
    num = num + 1;
    place = place - fs:GetHeight() - 5;
  end

  RegisterGeneralOption(SC_MAIN_GENERAL_OPTION1, SC_myConfig.inform, function()
    changes.inform = ctruth(this:GetChecked());
  end);

  MakeGeneralFrame = nil;
end

function RegisterMainTab(object, text)
  MainTabs.num = MainTabs.num + 1;
  local num = MainTabs.num;
  MainTabs.frames[object] = num;

  local tab = CreateFrame("Button", "SC_MainTab"..num, SC_Main,
      "CharacterFrameTabButtonTemplate");
  tab.id = num;
  tab:SetText(text);
  PanelTemplates_TabResize(0, tab)
  tab:SetPoint("LEFT", SC_Main, "BOTTOMLEFT", MainTabs.placement, -10);

  tab:SetScript("OnClick", function()
    PanelTemplates_SetTab(SC_Main, this.id);
    object:Show();
    for f in pairs(MainTabs.frames) do
      if f ~= object then
        f:Hide();
      end
    end
  end);

  -- Setting Tabs
  PanelTemplates_SetNumTabs(SC_Main, MainTabs.num);
  MainTabs.placement = MainTabs.placement + tab:GetWidth() - 10;

  return num;
end


--=================
--Main Parent Frame
--=================

local function MakeChanges(t, c)
  for i,v in pairs(t) do
    if type(v) == "table" then
      MakeChanges(v, c[i]);
    else
      c[i] = v;
    end
  end
end

function SettingsDone()
  MakeChanges(changes, SC_myConfig);
  if Reload then
    SC_Main:Hide();
    Confirmation(SC_RELOAD, 1, 0, 0, ReloadUI, function()
      SC_Confirmation:Hide();
    end);
  end
  changes = { };
end

function MakeMainFrame()
  --Making Main Configuration Frame
  local main = CreateFrame("Frame", "SC_Main", SC);
  main:SetHeight(500);
  main:SetWidth(400);
  main:ClearAllPoints();
  main:SetPoint("CENTER", UIParent, "CENTER");
  main:SetFrameStrata("DIALOG");
  main:SetMovable(true);
  main:SetBackdrop({
      bgFile = "Interface/DialogFrame/UI-DialogBox-Background",
      edgeFile = "Interface/DialogFrame/UI-DialogBox-Border",
      tile = true,
      tileSize = 32,
      edgeSize = 32,
    insets = { left = 5, right = 5, top = 5, bottom = 5 }
  });
  main:CreateTexture("SC_Main_FrameHeader", "ARTWORK");
  SC_Main_FrameHeader:SetPoint("CENTER", SC_Main, "TOP", 0, -15);
  SC_Main_FrameHeader:SetTexture("Interface/DialogFrame/UI-DialogBox-Header");
  main:CreateFontString("SC_Main_FrameHeader_Name", "ARTWORK",
    "ChatFontNormal");
  SC_Main_FrameHeader_Name:SetPoint("CENTER", SC_Main, "TOP");
  main:CreateFontString("SC_Main_Current", "ARTWORK", "ChatFontNormal");
  SC_Main_Current:SetPoint("CENTER", SC_Main, "TOPLEFT", 50, -30);
  table.insert(UISpecialFrames, main:GetName());
  main.elapsed = 0;
  main:Hide();

  -- SC_Main Buttons
  local temp = CreateFrame("Button", "SC_Main_CloseX", SC_Main,
    "UIPanelCloseButton");
  temp:SetPoint("CENTER", SC_Main, "TOPRIGHT", -20, -20);
  temp:SetScript("OnClick", function() SC_Main:Hide(); end);
  temp:SetFrameStrata("DIALOG");
  temp:SetFrameLevel(5);

  MainFrames = { };

  MakeGeneralFrame();

  for m in pairs(modules) do
    if m.CreateSettingsFrame then m.CreateSettingsFrame(); end
  end

  -- All Tabs must be created by their respective modules
  SC_Main_FrameHeader_Name:SetText(string.format("|cff4477ff%s (v%.3f)|r",
    SC_NAME, version));

  MakeMainFrame = nil;
end



-- Initialization Sub Module (placed here to prevent accidental user deletion)
local function init()
  --=====================================
  --Setting up the environment and module
  --=====================================

  local module = { };
  local _G = getfenv(0);

  setmetatable(module, {__index = _G.Spellcraft});
  setfenv(1, module);
  -- We don't Register this Module!

  --===============
  --Local Functions
  --===============

  local function Initialize()
    if not SC_myConfig.version then
      SetDefaults();
    elseif SC_myConfig.version < 1.00 then
      print(SC_MSG_RESET, 1, 0, 0, 0);
      SetDefaults();
    else
      if SC_myConfig.version < 1.02 then
        SC_myConfig.silent = { };
        SC_myConfig.silent.trade = false;
      end
      if SC_myConfig.version < 1.09 then
        print(SC_MSG_VERSION_UPDATE, 1, 0, 0, 0);
        Merchant.SetDefaultReagents();
      end
      if SC_myConfig.version < 1.100 then
        SC_myConfig.Bandage = {load = SC_myConfig.healer};
        SC_myConfig.Evocate = {load = SC_myConfig.spirit};
        SC_myConfig.Invisibility = {load = SC_myConfig.invisible};
        -- Large Merchant Changes
        SC_myConfig.Merchant = {
          load = SC_myConfig.merchant,
          silent = SC_myConfig.silent.trade,
          reagents = { }
        };
        for i,v in pairs(SC_myConfig.myReagents) do
          SC_myConfig.Merchant.reagents[i] = { };
          for j,w in pairs(SC_myConfig.myReagents[i]) do
            SC_myConfig.Merchant.reagents[i][j] = w[2];
          end
        end
        -- End Merchant Changes
        SC_myConfig.Minimap = {load = SC_myConfig.MinimapButton.bool,
            hearth = true};
        SC_myConfig.Minimap.button = { };
        SC_myConfig.Minimap.button.position = { };
        SC_myConfig.Minimap.button.position.degree =
            SC_myConfig.MinimapButton.position.degree;
        SC_myConfig.Minimap.button.position.radius =
            SC_myConfig.MinimapButton.position.radius;
        SC_myConfig.Poly = SC_myConfig.poly;
          SC_myConfig.Poly.load = true;
          SC_myConfig.Poly.auto = SC_myConfig.poly.bool;
        SC_myConfig.Release = {load = SC_myConfig.release};
        SC_myConfig.Buttons = {load = true};
        SC_myConfig.Collect = {load = true};
        SC_myConfig.Buffs = {load = true};
        SC_myConfig.healer = nil;
        SC_myConfig.spirit = nil;
        SC_myConfig.invisible = nil;
        SC_myConfig.merchant = nil;
        SC_myConfig.myReagents = nil;
        SC_myConfig.MinimapButton = nil;
        SC_myConfig.poly.bool = nil;
        SC_myConfig.silent = nil;
        SC_myConfig.poly = nil;
        SC_myConfig.release = nil;
      end
      if SC_myConfig.version < 1.103 then
        Spellcraft.Merchant.Defaults();
        error("All Merchant Settings have been reset to defaults.\n"..
            "Please restore your settings using: /changereagent");
        SC_myConfig.inform = true;
      end
      if SC_myConfig.version < 1.108 then
        SC_myConfig.Poly.broken = false;
      end
    end
    if not SC_myConfig.Merchant.reagents then
      Spellcraft.Merchant.Defaults();
    end
    if type(SC_myConfig.Merchant.reagents.Mage[1]) == "table" then
      for i,v in pairs(SC_myConfig.Merchant.reagents) do
        SC_myConfig.Merchant.reagents[i] = { };
        for j,w in pairs(v) do
          SC_myConfig.Merchant.reagents[i][j] = w[2];
        end
      end
    end
    SC_myConfig.version = version;
    Update();
  end


  --================
  --Module Functions
  --================

  function Load()
    -- Events
    SC_RegEvent(module, "ADDON_LOADED");
  end

  function ADDON_LOADED(addon)
    if addon == "Spellcraft" then
      inform(string.format("%s v%.3f %s", SC_NAME, version, SC_LOADED));
      --print(SC_BETA);
      _G.Spellcraft.config = _G.SC_myConfig;
      Initialize();
      LoadModules();
      SC_UnRegEvent(module, "ADDON_LOADED");
      module = nil; -- Remove the module.
    end
  end

  -- Load this Module immediately:
  Load();
end



do -- Main
  SC:SetScript("OnEvent", OnEvent);
  SC:RegisterEvent("PLAYER_ENTERING_WORLD");
  _G.SLASH_SPELLCRAFT1 = "/spellcraft";
  _G.SLASH_SPELLCRAFT2 = "/sc";
  _G.SlashCmdList["SPELLCRAFT"] = ShowMain;

  init();
end
