--[[

Spellcraft: This is a World of Warcraft Mage Utility AddOn.
Copyright (C) 2007  Patrick J. Donnelly (batrick@unm.edu)

Permission is NOT granted to modify/redistribute this software.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

--]]


--=====================================
--Setting up the environment and module
--=====================================

local module = { };
local _G = getfenv(0);
setmetatable(module, {__index = _G.Spellcraft});
setfenv(1, module);

module.name = "Poly";
module.lname = SC_MODULE_POLY;
module.version = 1.115;
module.info = SC_MODULE_POLY_INFO;

RegisterModule(module);

--======
--Locals
--======

local player = player;
local tab;
local sheep_GUID;
local sheep_name = "";
local breaker;
local casting = false;
local casting_GUID;
local hasSheep = false;
local myMessageButtons = { };
local LastMessage = { };
local defaults = {
  whisper = {
    [1] = {SC_WHISPER_POLY_DEFAULT1, 100}
  },
  party = {
    [1] = {SC_PARTY_POLY_DEFAULT1, 30},
    [2] = {SC_PARTY_POLY_DEFAULT2, 30},
    [3] = {SC_PARTY_POLY_DEFAULT3, 40}
  },
  raid = {
    [1] = {SC_RAID_POLY_DEFAULT1, 100}
  },
  sheep = {
    [1] = {SC_SHEEP_POLY_DEFAULT1, 100}
  }
};

--===============
--Local Functions
--===============

local PolyGroupTarget,
      PolySendMessages,
      PolyAddMessage,
      PolyChangeMessage,
      PolyGetMessage,
      PolyRemoveMessage,
      Poly_Show,
      Poly_Set,
      Poly_AutoMsgClick,
      Poly_LoadMessages,
      Poly_Defaults,
      Poly_Hide,
      Poly_Messages_AddShow,
      Poly_Messages_AddSet,
      Poly_Messages_EditShow,
      Poly_Messages_EditSet,
      Poly_EditRemove;

PolyAddMessage = function(message, percent, grp)
  percent = math.abs(tonumber(percent) or 0);
  table.insert(config.Poly[grp].sayings, {message, tonumber(percent)});
  return true;
end

PolyChangeMessage = function(MSG, percent, grp, saying)
  percent = math.abs(tonumber(percent) or 0);
  table.insert(config.Poly[grp].sayings, saying, {MSG, tonumber(percent)});
  table.remove(config.Poly[grp].sayings, saying + 1);
  return true;
end

PolyGetMessage = function(grp)
  local x, a = 0, math.random(0, 99);
  for _,v in ipairs(config.Poly[grp].sayings) do
    x = x + v[2];
    if a < x then
      return v[1]:gsub('%%t', sheep_name);
    end
  end
  if #config.Poly[grp].sayings == 0 then
    error(SC_MSG_POLY_SAYINGS_ERROR:format(grp, grp));
  end
  return nil;
end

PolyRemoveMessage = function(grp, index)
  table.remove(config.Poly[grp].sayings, index);
end

PolyGroupTarget = function(idiot)
  if sheep_name and not hasSheep then
    return;
  elseif not isFocusDebuffUp(SC_DEBUFF_OTHER_POLY) or not hasSheep then
    sheep_name = nil;
    hasSheep = false;
    return;
  end
  local temp, name, realm = PolyGetMessage("sheep"), UnitName(idiot);
  if temp and name ~= player.name then
    local now = tonumber(date("%j%H%M%S"));
    if (LastMessage[name] or 0) + 300 < now then
      LastMessage[name] = now;
      SendChatMessage(temp, "WHISPER", nil, realm and name.."-"..realm
          or name);
    end
  end
end

PolySendMessages = function()
  local t, c;
  if GetNumRaidMembers() > 0 then
    c = "raid";
    t = GetNumRaidMembers();
  else
    c = "party";
    t = GetNumPartyMembers();
  end
  if config.Poly.whisper.bool then
    for i = 1, t do
      if not UnitIsUnit(c..i, "player") then 
        if UnitIsUnit(c..i.."target", "focus") then
          local name, realm = UnitName(c..i);
          local temp = PolyGetMessage("whisper");
          if temp then
            if not realm then
              SendChatMessage(temp, "WHISPER", nil, name);
            else
              SendChatMessage(temp, "WHISPER", nil, name.."-"..realm);
            end
          end
        end
      end
    end
  end
  if config.Poly.party.bool and (GetNumPartyMembers() > 0
      or GetNumRaidMembers() > 0 and not config.Poly.raid.bool) then
    local temp = PolyGetMessage("party");
    if temp then
      SendChatMessage(temp, "PARTY", nil);
    end
  end
  if config.Poly.raid.bool then
    local temp = PolyGetMessage("raid");
    if temp then
      if isInBattleground() then
        SendChatMessage(temp, "BATTLEGROUND", nil);
      elseif GetNumRaidMembers() > 0 then
        SendChatMessage(temp, "RAID", nil);
      end
    end
  end
end

--==================================
--Polymorph Settings Frame Functions
--==================================

Poly_Show = function()
  SC_Main_Poly_AutoMsgCheck:SetChecked(config.Poly.auto);
  if not config.Poly.auto then
    config.Poly.sheep.bool = false;
    SC_Main_Poly_Option2:SetTextColor(.5, .5, .5);
    SC_Main_Poly_AutoTarCheck:Disable();
  else
    SC_Main_Poly_Option2:SetTextColor(.2, .8, .5);
    SC_Main_Poly_AutoTarCheck:Enable();
  end
  SC_Main_Poly_AutoTarCheck:SetChecked(config.Poly.sheep.bool);
  SC_Main_Poly_WhisperCheck:SetChecked(config.Poly.whisper.bool);
  SC_Main_Poly_PartyCheck:SetChecked(config.Poly.party.bool);
  SC_Main_Poly_RaidCheck:SetChecked(config.Poly.raid.bool);
  SC_Main_Poly_BrokenCheck:SetChecked(config.Poly.broken);
  SC_Main_Poly_Slider:SetMinMaxValues(1, 4);
  SC_Main_Poly_Slider:SetValue(1);
  SC_Main_Poly_Slider:SetValueStep(1);
  SC_Main_Poly_SliderLow:SetText("");
  SC_Main_Poly_SliderHigh:SetText("");
  Poly_LoadMessages();
end

Poly_LoadMessages = function()
  local value, grp = SC_Main_Poly_Slider:GetValue();
  if value == 1 then
    grp = "whisper";
    SC_Main_Poly_SliderState:SetText(SC_MAIN_POLY_SLIDERSTATE1);
  elseif value == 2 then
    grp = "sheep";
    SC_Main_Poly_SliderState:SetText(SC_MAIN_POLY_SLIDERSTATE2);
  elseif value == 3 then
    grp = "party";
    SC_Main_Poly_SliderState:SetText(SC_MAIN_POLY_SLIDERSTATE3);
  else
    grp = "raid";
    SC_Main_Poly_SliderState:SetText(SC_MAIN_POLY_SLIDERSTATE4);
  end
  SC_Main_Poly_SliderState:SetTextColor(0, 1, 0);
  local temp = config.Poly[grp].sayings;
  local y, k = -10, 1;
  for i,v in pairs(temp) do
    k = i + 1;
    local button;
    if not myMessageButtons[i] then
      button = CreateFrame("Button", "SC_Sayings"..i,
          SC_Main_Poly_MessagesFrame);
      button:SetHeight(25);
      button:SetWidth(300);
      button:SetText("a");
      button:SetTextFontObject("GameFontHighlight");
      (button:GetFontString()):SetAllPoints(button);
      (button:GetFontString()):SetJustifyH("LEFT");
      myMessageButtons[#myMessageButtons+1] = button;
    else
      button = myMessageButtons[i];
    end
    if #temp[i][1] > 29 then
      button:SetText(("\"%.30s\"|cFFFFFF00...|r"):format(temp[i][1]));
    else
      button:SetText(("%q %d%%"):format(temp[i][1], temp[i][2]));
    end
    button.mySaying = i;
    button.group = grp;
    button:SetPoint("TOPLEFT", 10, y);
    button:SetScript("OnEnter", function(self)
      GameTooltip:SetOwner(self, "ANCHOR_RIGHT");
      GameTooltip:SetText(("%q %d%%"):format(temp[i][1], temp[i][2]));
      GameTooltip:Show();
    end);
	button:SetScript("OnLeave", function()
               GameTooltip:Hide();
      end);
    button:SetScript("OnClick", function(self)
        Poly_Messages_EditShow(self.mySaying, self.group);
      end);
    button:Show();
    y = y - button:GetTextHeight();
  end
  SC_Main_Poly_MessagesFrame:SetHeight(math.ceil(math.abs(y)));
  for i = k, #myMessageButtons do
    myMessageButtons[i]:Hide();
  end
  SC_Main_Poly_Messages:UpdateScrollChildRect();
end

Poly_Defaults = function()
  Confirmation(string.format("%s", SC_MAIN_POLY_DEFAULTS), 1, 0, 0, function()
    Defaults();
    SC_Confirmation:Hide();
    SC_Main:Show();
    SC_MainTab2:Click();
  end, function()
    SC_Confirmation:Hide();
    SC_Main:Show();
    SC_MainTab2:Click()
  end);
  SC_Main:Hide();
end

Poly_Hide = function()
  for i,v in pairs(myMessageButtons) do
    myMessageButtons[i]:Hide();
  end
end

Poly_Messages_AddShow = function()
  SC_Main:Hide();
  SC_Main_Poly_EditFrame:Show();
  SC_Main_Poly_EditFrameNote:SetText(SC_MAIN_POLY_EDITFRAMENOTE2);
  SC_Main_Poly_EditFrame_EditBox1:SetText("");
  SC_Main_Poly_EditFrame_EditBox2:SetText("10");
  SC_Main_Poly_EditFrameRemove:Hide();
  SC_Main_Poly_EditFrameDone:SetScript("OnClick", function()
    Poly_Messages_AddSet();
      end)
end

Poly_Messages_AddSet = function()
  local temp = SC_Main_Poly_Slider:GetValue();
  if temp == 1 then
    temp = "whisper";
  elseif temp == 2 then
    temp = "sheep";
  elseif temp == 3 then
    temp = "party";
  else
    temp = "raid";
  end
  local MSG, percent = SC_Main_Poly_EditFrame_EditBox1:GetText(),
                       SC_Main_Poly_EditFrame_EditBox2:GetText();
  PolyAddMessage(MSG, percent, temp);
  SC_Main_Poly_EditFrame:Hide();
  SC_Main:Show();
  SC_MainTab2:Click();
end

Poly_Messages_EditShow = function(saying, group)
  local temp1, temp2 = config.Poly[group].sayings[saying][1],
        config.Poly[group].sayings[saying][2];
  SC_Main:Hide();
  SC_Main_Poly_EditFrame:Show();
  SC_Main_Poly_EditFrameNote:SetText(SC_MAIN_POLY_EDITFRAMENOTE);
  SC_Main_Poly_EditFrame_EditBox1:SetText(temp1);
  SC_Main_Poly_EditFrame_EditBox2:SetText(temp2);
  SC_Main_Poly_EditFrameRemove:Show();
  SC_Main_Poly_EditFrame.saying = saying;
  SC_Main_Poly_EditFrame.group = group;
  SC_Main_Poly_EditFrameDone:SetScript("OnClick", function()
    Poly_Messages_EditSet();
      end)
end

Poly_Messages_EditSet = function()
  local temp1, temp2 = SC_Main_Poly_EditFrame.saying,
                       SC_Main_Poly_EditFrame.group;
  local MSG, percent = SC_Main_Poly_EditFrame_EditBox1:GetText(),
                       SC_Main_Poly_EditFrame_EditBox2:GetText();
  PolyChangeMessage(MSG, percent, temp2, temp1);
  SC_Main_Poly_EditFrame:Hide();
  SC_Main:Show();
  SC_MainTab2:Click();
end

Poly_EditRemove = function()
  local temp1, temp2 = SC_Main_Poly_EditFrame.saying,
                       SC_Main_Poly_EditFrame.group;
  local temp3 = config.Poly[temp2].sayings[temp1];
  SC_Confirmation_Confirm:SetText(string.format("%s: %q?",
        SC_MAIN_POLY_EDITFRAME_REMOVE, temp3[1]));
  SC_Confirmation_Confirm:SetTextColor(1, 0, 0);
  SC_Main_Poly_EditFrame:Hide();
  SC_Confirmation:Show();
  SC_Confirmation_Yes:SetScript("OnClick", function()
      PolyRemoveMessage(temp2, temp1)
      SC_Confirmation:Hide();
      SC_Main:Show();
      SC_MainTab2:Click();
    end);
  SC_Confirmation_No:SetScript("OnClick", function()
      SC_Confirmation:Hide();
      SC_Main:Show();
      SC_MainTab2:Click();
    end);
end


  --==================
  --Polymorph Handling
  --==================

local function Slash_Polym(msg)
  if msg:find("config") then
    if not SC_Main_Poly then
      CreateSettingsFrame();
    end
    SC_Main:Show();
    _G["SC_MainTab"..tab]:Click();
  elseif msg:find(" fo?c?u?s?%W*$") then
    sheep_name = UnitName("focus");
    PolySendMessages();
  elseif msg:find(" ta?r?g?e?t?%W*$") then
    sheep_name = UnitName("target");
    PolySendMessages();
  else
    error(SC_MSG_POLYM_ERROR);
  end
end


--================
--Module Functions
--================

function Load()
  -- Slash Commands
  _G.SLASH_POLYM1 = "/polym";
  _G.SlashCmdList["POLYM"] = Slash_Polym;

  -- Events
  SC_RegEvent(module, "UNIT_TARGET");
  SC_RegEvent(module, "UNIT_SPELLCAST_SENT");
  SC_RegEvent(module, "UNIT_SPELLCAST_FAILED");
  SC_RegEvent(module, "UNIT_SPELLCAST_INTERRUPTED");
  SC_RegEvent(module, "COMBAT_LOG_EVENT_UNFILTERED");
  SC_RegEvent(module, "PLAYER_ENTERING_WORLD");

  local SC_DEBUFF_OTHER_POLY = SC_DEBUFF_OTHER_POLY;
  local UnitGUID = UnitGUID;

  local function csbn(name, on_self, target)
    if name:find(SC_DEBUFF_OTHER_POLY) then
      casting = true;
      casting_GUID = UnitGUID(target or "target");
    end
  end;
  setfenv(csbn, _G); -- for module deletion, can't be done later
  hooksecurefunc("CastSpellByName", csbn);
  local function stu(unit)
    casting_GUID = UnitGUID(unit or "target");
  end
  setfenv(stu, _G);
  hooksecurefunc("SpellTargetUnit", stu);
  local function cs(spell_id, book)
    if GetSpellName(spell_id, "spell"):find(SC_DEBUFF_OTHER_POLY) then
      casting = true;
      casting_GUID = UnitGUID("target");
    end
  end
  setfenv(cs, _G);
  hooksecurefunc("CastSpell", cs);
end

function Defaults()
  config.Poly = {
    load = true,
    auto = true,
    broken = true,
    whisper = {
      bool = true,
      sayings = tcopy(defaults.whisper),
    },
    party = {
      bool = true,
      sayings = tcopy(defaults.party),
    },
    raid = {
      bool = false,
      sayings = tcopy(defaults.raid),
    },
    sheep = {
      bool = false,
      sayings = tcopy(defaults.sheep),
    },
  };
end


  --========================
  --Polymorph Settings Frame
  --========================

function CreateSettingsFrame()
  if not SC_Main then MakeMainFrame(); end

  local poly = CreateFrame("Frame", "SC_Main_Poly", SC_Main)
  poly:SetWidth(360);
  poly:SetHeight(425);
  poly:SetPoint("TOPLEFT", SC_Main, "TOPLEFT", 20, -50);
  poly:SetBackdrop({
      bgFile = "Interface/Tooltips/UI-Tooltip-Background",
      edgeFile = "Interface/Tooltips/UI-Tooltip-Border",
      tile = true,
      tileSize = 16,
      edgeSize = 12,
      insets = { left = 2, right = 2, top = 2, bottom = 2 }
    });
  poly:SetBackdropColor(0.1, 0.1, 0.2);
  poly:SetBackdropBorderColor(0.4, 0.1, 0.8);
  poly:SetScript("OnShow", function()
    SC_Main_Current:SetText(SC_MAIN_POLY);
    Poly_Show();
  end);
  poly:SetScript("OnHide", Poly_Hide);

  tab = RegisterMainTab(poly, SC_MAIN_POLY);

  -- Options
  local f = poly:CreateFontString("SC_Main_Poly_Option1", "ARTWORK",
    "GameFontNormal");
  f:SetText(SC_MAIN_POLY_OPTION1);
  f:SetPoint("LEFT", poly, "TOPLEFT", 10, -20);
  f:SetWidth(275);
  f:SetJustifyH("LEFT");
  f:SetTextColor(.2, .8, .5);

  f = poly:CreateFontString("SC_Main_Poly_Option2", "ARTWORK",
    "GameFontNormal");
  f:SetText(SC_MAIN_POLY_OPTION2);
  f:SetPoint("LEFT", poly, "TOPLEFT", 40, -50);
  f:SetWidth(245);
  f:SetJustifyH("LEFT");
  f:SetTextColor(.2, .8, .5);

  f = poly:CreateFontString("SC_Main_Poly_Option3", "ARTWORK",
    "GameFontNormal");
  f:SetText(SC_MAIN_POLY_OPTION3);
  f:SetPoint("LEFT", poly, "TOPLEFT", 10, -80);
  f:SetWidth(275);
  f:SetJustifyH("LEFT");
  f:SetTextColor(.2, .8, .5);

  f = poly:CreateFontString("SC_Main_Poly_Option4", "ARTWORK",
    "GameFontNormal");
  f:SetText(SC_MAIN_POLY_OPTION4);
  f:SetPoint("LEFT", poly, "TOPLEFT", 10, -110);
  f:SetWidth(275);
  f:SetJustifyH("LEFT");
  f:SetTextColor(.2, .8, .5);

  f = poly:CreateFontString("SC_Main_Poly_Option5", "ARTWORK",
    "GameFontNormal");
  f:SetText(SC_MAIN_POLY_OPTION5);
  f:SetPoint("LEFT", poly, "TOPLEFT", 10, -140);
  f:SetWidth(275);
  f:SetJustifyH("LEFT");
  f:SetTextColor(.2, .8, .5);

  f = poly:CreateFontString("SC_Main_Poly_Option6", "ARTWORK",
    "GameFontNormal");
  f:SetText(SC_MAIN_POLY_OPTION6);
  f:SetPoint("LEFT", poly, "TOPLEFT", 10, -170);
  f:SetWidth(275);
  f:SetJustifyH("LEFT");
  f:SetTextColor(.2, .8, .5);

  local cb = CreateFrame("CheckButton", "SC_Main_Poly_AutoMsgCheck", poly,
    "OptionsCheckButtonTemplate");
  cb:SetPoint("RIGHT", poly, "TOPRIGHT", -13, -20);
  cb:SetScript("OnClick", function(self)
    changes.Poly = changes.Poly or { };
    if ctruth(self:GetChecked()) then
      changes.Poly.auto = true;
      SC_Main_Poly_Option2:SetTextColor(.2, .8, .5);
      SC_Main_Poly_AutoTarCheck:Enable();
      SC_Main_Poly_AutoTarCheck:SetChecked(false);
    else
      changes.Poly.auto = false;
      SC_Main_Poly_Option2:SetTextColor(.5, .5, .5);
      SC_Main_Poly_AutoTarCheck:Disable();
      SC_Main_Poly_AutoTarCheck:SetChecked(false);
    end
  end);

  cb = CreateFrame("CheckButton", "SC_Main_Poly_AutoTarCheck", poly,
    "OptionsCheckButtonTemplate");
  cb:SetPoint("RIGHT", poly, "TOPRIGHT", -13, -50);
  cb:SetScript("OnClick", function(self)
    changes.Poly = changes.Poly or { };
    changes.Poly.sheep = changes.Poly.sheep or { };
    changes.Poly.sheep.bool = ctruth(self:GetChecked());
  end);

  cb = CreateFrame("CheckButton", "SC_Main_Poly_WhisperCheck", poly,
    "OptionsCheckButtonTemplate");
  cb:SetPoint("RIGHT", poly, "TOPRIGHT", -13, -80);
  cb:SetScript("OnClick", function(self)
    changes.Poly = changes.Poly or { };
    changes.Poly.whisper = changes.Poly.whisper or { };
    changes.Poly.whisper.bool = ctruth(self:GetChecked());
  end);

  cb = CreateFrame("CheckButton", "SC_Main_Poly_PartyCheck", poly,
    "OptionsCheckButtonTemplate");
  cb:SetPoint("RIGHT", poly, "TOPRIGHT", -13, -110);
  cb:SetScript("OnClick", function(self)
    changes.Poly = changes.Poly or { };
    changes.Poly.party = changes.Poly.party or { };
    changes.Poly.party.bool = ctruth(self:GetChecked());
  end);

  cb = CreateFrame("CheckButton", "SC_Main_Poly_RaidCheck", poly,
    "OptionsCheckButtonTemplate");
  cb:SetPoint("RIGHT", poly, "TOPRIGHT", -13, -140);
  cb:SetScript("OnClick", function(self)
    changes.Poly = changes.Poly or { };
    changes.Poly.raid = changes.Poly.raid or { };
    changes.Poly.raid.bool = ctruth(self:GetChecked());
  end);

  cb = CreateFrame("CheckButton", "SC_Main_Poly_BrokenCheck", poly,
    "OptionsCheckButtonTemplate");
  cb:SetPoint("RIGHT", poly, "TOPRIGHT", -13, -170);
  cb:SetScript("OnClick", function(self)
    changes.Poly = changes.Poly or { };
    changes.Poly.broken = ctruth(self:GetChecked());
  end);

  -- Sayings Frame
  local scroll = CreateFrame("ScrollFrame", "SC_Main_Poly_Messages",
    SC_Main_Poly, "UIPanelScrollFrameTemplate");
  scroll:SetWidth(300);
  scroll:SetHeight(75);
  scroll:SetPoint("BOTTOM", SC_Main_Poly, "BOTTOM", 0, 70);
  scroll:SetBackdrop({
      bgFile = "Interface/Tooltips/UI-Tooltip-Background",
      edgeFile = "Interface/Tooltips/UI-Tooltip-Border",
      tile = true,
      tileSize = 16,
      edgeSize = 12,
      insets = { left = 2, right = 2, top = 2, bottom = 2 }
    });
  scroll:SetBackdropColor(0.1, 0.1, 0.4);
  scroll:SetBackdropBorderColor(0.1, 0.1, 0.8);

  local fs = scroll:CreateFontString("SC_Main_Poly_ScrollFrameNote", "ARTWORK",
    "GameFontNormal");
  fs:SetText(SC_MAIN_POLY_SCROLLNOTE);
  fs:SetTextColor(1, 0, .2);
  fs:SetPoint("CENTER", scroll, "BOTTOM", 0, -7);

  local mod = CreateFrame("Frame", "SC_Main_Poly_MessagesFrame",
    SC_Main_Poly_Messages);
  mod:SetWidth(300);
  mod:SetHeight(120);
  mod:SetPoint("TOPLEFT", scroll, "TOPLEFT");

  scroll:SetScrollChild(mod);

  local slider = CreateFrame("Slider", "SC_Main_Poly_Slider", poly,
    "OptionsSliderTemplate");
  slider:SetWidth(300);
  slider:SetHeight(20);
  slider:SetPoint("BOTTOM", poly, "BOTTOM", 0, 170);
  local sf = slider:CreateFontString("SC_Main_Poly_SliderState", "ARTWORK",
    "GameFontNormal");
  sf:SetText(SC_MAIN_POLY_SLIDERSTATE1);
  sf:SetPoint("CENTER", slider, "BOTTOM", 0, -10);
  slider:SetScript("OnValueChanged", Poly_LoadMessages);

  local done = CreateFrame("Button", "SC_Main_Poly_Done", SC_Main_Poly,
    "OptionsButtonTemplate");
  done:SetText(DONE);
  done:SetPoint("CENTER", SC_Main_Poly, "BOTTOM", -50, 20);
  done:SetScript("OnClick", function()
      SettingsDone();
      SC_Main:Hide();
    end);

  local cancel = CreateFrame("Button", "SC_Main_Poly_Cancel", SC_Main_Poly,
    "OptionsButtonTemplate");
  cancel:SetText(CANCEL);
  cancel:SetPoint("CENTER", SC_Main_Poly, "BOTTOM", 50, 20);
  cancel:SetScript("OnClick", function()
      changes = { };
      SC_Main:Hide();
    end);

  local defaults = CreateFrame("Button", "SC_Main_Poly_Defaults", SC_Main_Poly,
   "OptionsButtonTemplate");
  defaults:SetText(DEFAULT);
  defaults:SetPoint("CENTER", SC_Main_Poly, "BOTTOM", -50, 45);
  defaults:SetScript("OnClick", Poly_Defaults);

  local add = CreateFrame("Button", "SC_Main_Poly_AddMsg", SC_Main_Poly,
    "OptionsButtonTemplate");
  add:SetText(SC_MAIN_POLY_ADDMSG);
  add:SetPoint("CENTER", SC_Main_Poly, "BOTTOM", 50, 45);
  add:SetScript("OnClick", Poly_Messages_AddShow);

  -- Making Edit Frames
  local edit = CreateFrame("Frame", "SC_Main_Poly_EditFrame", SC);
  edit:SetWidth(400);
  edit:SetHeight(100);
  edit:SetPoint("CENTER", UIParent, "CENTER");
  edit:SetBackdrop({
      bgFile = "Interface/TutorialFrame/TutorialFrameBackground",
      edgeFile = "Interface/DialogFrame/UI-DialogBox-Border",
      tile = true,
      tileSize = 32,
      edgeSize = 16,
      insets = { left = 5, right = 5, top = 5, bottom = 5}
    });
  edit:SetBackdropColor(0.1, 0.1, 0.4);
  edit:SetBackdropBorderColor(0.1, 0.1, 0.8);

  fs = edit:CreateFontString("SC_Main_Poly_EditFrameNote", "ARTWORK",
    "GameFontNormal");
  fs:SetWidth(350);
  fs:SetPoint("CENTER", edit, "TOP", 0, -5);

  fs = edit:CreateFontString("SC_Main_Poly_EditFrameNote2", "ARTWORK",
    "GameFontNormal");
  fs:SetText("%");
  fs:SetPoint("CENTER", edit, "RIGHT", -25, 15);

  local eb1 = CreateFrame("EditBox", "SC_Main_Poly_EditFrame_EditBox1", edit);
  eb1:SetWidth(300);
  eb1:SetHeight(32);
  eb1:SetBackdrop({
      bgFile = "Interface/DialogFrame/UI-DialogBox-Background",
      edgeFile = "Interface/DialogFrame/UI-DialogBox-Border",
      tile = true,
      tileSize = 5,
      edgeSize = 10,
      insets = {left = 7, right = 5, top = 5, bottom = 5}
    });
  eb1:SetAutoFocus(false);
  eb1:SetBackdropColor(0.7, 0.1, 0.7);
  eb1:SetBackdropBorderColor(0.3, 0.1, 0.8);
  eb1:SetPoint("TOPLEFT", edit, "TOPLEFT", 10, -20);
  eb1:SetFontObject("GameFontNormal");
  eb1:SetTextColor(1, 1, 1);
  eb1:SetMaxLetters(200);

  local eb2 = CreateFrame("EditBox", "SC_Main_Poly_EditFrame_EditBox2", edit);
  eb2:SetWidth(40);
  eb2:SetHeight(32);
  eb2:SetBackdrop({
      bgFile = "Interface/DialogFrame/UI-DialogBox-Background",
      edgeFile = "Interface/DialogFrame/UI-DialogBox-Border",
      tile = true,
      tileSize = 5,
      edgeSize = 10,
      insets = {left = 7, right = 5, top = 5, bottom = 5}
    });
  eb2:SetAutoFocus(false);
  eb2:SetBackdropColor(0.7, 0.1, 0.7);
  eb2:SetBackdropBorderColor(0.3, 0.1, 0.8);
  eb2:SetPoint("TOPRIGHT", edit, "TOPRIGHT", -35, -20);
  eb2:SetFontObject("GameFontNormal");
  eb2:SetTextColor(1, 1, 1);
  eb2:SetMaxLetters(3);

  local editclose = CreateFrame("Button", "SC_Main_Poly_EditFrameClose",
    edit, "UIPanelCloseButton");
  editclose:SetPoint("CENTER", edit, "TOPRIGHT", -13, -13);
  editclose:SetScript("OnClick", function()
    SC_Main_Poly_EditFrame:Hide();
  end);

  local editcancel = CreateFrame("Button", "SC_Main_Poly_EditFrameCancel",
    edit, "OptionsButtonTemplate");
  editcancel:SetText(CANCEL);
  editcancel:SetPoint("CENTER", edit, "BOTTOM", 50, 15);
  editcancel:SetScript("OnClick", function()
    SC_Main_Poly_EditFrame:Hide();
    SC_Main:Show();
  end);

  local editdone = CreateFrame("Button", "SC_Main_Poly_EditFrameDone",
    edit, "OptionsButtonTemplate");
  editdone:SetText(DONE);
  editdone:SetPoint("CENTER", edit, "BOTTOM", -50, 15);
  editdone:SetScript("OnClick", Poly_Messages_EditSet);

  local editremove = CreateFrame("Button", "SC_Main_Poly_EditFrameRemove",
    edit, "OptionsButtonTemplate");
  editremove:SetText(SC_MAIN_POLY_EDITFRAME_REMOVE);
  editremove:SetPoint("CENTER", edit, "BOTTOM", 0, 40);
  editremove:SetScript("OnClick", Poly_EditRemove);

  edit:Hide();
  poly:Hide();
  CreateSettingsFrame = nil;
end


  --==============
  --Event Handlers
  --==============

function UNIT_TARGET(unit)
  if not (unit:find("party") or unit:find("raid")) then  
    return;
  end

  local target = UnitName(unit.."target");
  if target and config.Poly.sheep.bool and config.Poly.auto
     and target == sheep_name then
    PolyGroupTarget(UnitName(unit));
  end
end

function UNIT_SPELLCAST_SENT(unit, cast, rank, sheep)
  if unit == "player" then
    if cast:find(SC_DEBUFF_OTHER_POLY) then
      sheep_name = sheep;
      breaker = nil;
      if config.Poly.auto then
        PolySendMessages();
      end
    else
      casting = false;
    end
  end
end

function UNIT_SPELLCAST_FAILED(unit, spell, rank)
  if unit == "player" then
    casting = false;
  end
end

function UNIT_SPELLCAST_INTERRUPTED(unit, spell, rank)
  if unit == "player" then
    casting = false;
  end
end

do
  -- table with polymorph SpellIDs
  local is_poly = {
    [118] = true, -- rank 1 sheep
    [12824] = true, -- rank 2 sheep
    [12825] = true, -- rank 3 sheep
    [12826] = true, -- rank 4 sheep
    [28272] = true, -- pig
    [28271] = true, -- turtle
  };
  function COMBAT_LOG_EVENT_UNFILTERED(time,
                                       event,
                                       source_GUID,
                                       source_name,
                                       source_flags,
                                       dest_GUID,
                                       dest_name,
                                       dest_flags,
                                       param9,
                                       param10,
                                       param11,
                                       param12)
    if event == "SPELL_AURA_DISPELLED" and dest_GUID == sheep_GUID and 
        is_poly[param12] then
      --[[print(SC_MSG_POLY_BREAK:format(dest_name, param10));
      if config.Poly.broken then
        if GetNumRaidMembers() > 0 then
          SendChatMessage(SC_MSG_POLY_BREAK_ANN:format(player.name,
                source_name, param10), "RAID");
        elseif GetNumPartyMembers() > 0 then
          SendChatMessage(SC_MSG_POLY_BREAK_ANN:format(player.name,
                source_name, param10), "PARTY");
        end
      end --]]
      sheep_GUID = nil;
    elseif event == "SPELL_AURA_APPLIED" and casting and casting_GUID == dest_GUID
           and is_poly[param9] then
      casting = false;
      sheep_name = dest_name;
      sheep_GUID = dest_GUID;
      breaker = nil;
    elseif event == "SPELL_AURA_REMOVED" and sheep_GUID == dest_GUID and
           is_poly[param9] then
      if breaker then
        if breaker_with then
          print(SC_MSG_POLY_BREAK:format(breaker, breaker_with));
          if config.Poly.broken then
            if GetNumRaidMembers() > 0 then
              SendChatMessage(SC_MSG_POLY_BREAK_ANN_MELEE:format(player.name,
                    breaker), "RAID");
            elseif GetNumPartyMembers() > 0 then
              SendChatMessage(SC_MSG_POLY_BREAK_ANN_MELEE:format(player.name,
                    breaker), "PARTY");
            end
          end
        else
          print(SC_MSG_POLY_BREAK_MELEE:format(breaker));
        end
      end
      sheep_GUID = nil;
    elseif dest_GUID == sheep_GUID and source_GUID ~= player.GUID and
           event:find("_DAMAGE$") then
      if event:find("^SWING") then
        breaker_with = nil;
      else
        breaker_with = param10;
      end
      breaker = source_name or ""; -- possibly environmental
    end
  end
end

function PLAYER_ENTERING_WORLD()
  Player();
  if player.class ~= "Mage" then
    _G.SLASH_POLYM1 = nil;
    _G.SlashCmdList["POLYM"] = nil;
    UnregisterModule(module);
  end
end
