--[[

Spellcraft: This is a World of Warcraft Mage Utility AddOn.
Copyright (C) 2007  Patrick J. Donnelly (batrick@unm.edu)

Permission is NOT granted to modify/redistribute this software.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

--]]


--=====================================
--Setting up the environment and module
--=====================================

local module = { };
local _G = getfenv(0);
setmetatable(module, {__index = _G.Spellcraft});
setfenv(1, module);

module.name = "Merchant";
module.lname = SC_MODULE_MERCHANT;
module.version = 1.115;
module.info = SC_MODULE_MERCHANT_INFO;

RegisterModule(module);

--======
--Locals
--======

local locked = false;
local tab;
local LastNotification = 0;
local Reagents = {
  Druid = {
             {SC_REAGENT_DRUID_MAPLE, 10, 20, 9},
             {SC_REAGENT_DRUID_STRANGLE, 10, 30, 9},
             {SC_REAGENT_DRUID_ASH, 10, 40, 9},
             {SC_REAGENT_DRUID_HORN, 10, 50, 9},
             {SC_REAGENT_DRUID_IRON, 10, 60, 9},
             {SC_REAGENT_DRUID_FLINT, 20, 70, 70},
             {SC_REAGENT_DRUID_BERRIES, 10, 50, 9},
             {SC_REAGENT_DRUID_THORN, 10, 60, 9},
             {SC_REAGENT_DRUID_QUILL, 10, 70, 70}
           },

  Hunter = {},

  Mage = {
            {SC_REAGENT_MAGE_TELE, 10, 20, 70},
            {SC_REAGENT_MAGE_PORT, 10, 40, 70},
            {SC_REAGENT_MAGE_POWDER, 20, 56, 70}
          },

  Paladin = {
              {SC_REAGENT_PALADIN_KINGS, 200, 52, 70},
              {SC_REAGENT_PALADIN_DIVINITY, 10, 30, 70}
            },

  Priest = {
              {SC_REAGENT_PRIEST_HOLY, 20, 48, 11},
              {SC_REAGENT_PRIEST_SACRED, 40, 56, 70}
           },


  Shaman = {
              {SC_REAGENT_SHAMAN_ANKH, 5, 30, 70}
           },
    
  Rogue = {},

  Warlock = {
               {SC_REAGENT_WARLOCK_INFERNAL, 5, 50, 70},
               {SC_REAGENT_WARLOCK_DEMONIC, 5, 60, 70}
            },

  Warrior = {},
};

local BuyReagents,
      OnMerchant,
      MakeChangeReagents,
      CheckReagentEditBoxs,
      ChangeReagent,
      CheckReagentEditBoxSanity,
      ShowDefaultReagents,
      ChangeReagentShow,
      CheckMerchant,
      CheckNeedForReagents,
      SellGrayTrash,
      HasGrayTrash,
      SetMerchantLock,
      CreateReagentNRepair,
      CreateChangeReagents,
      CreateReagentOptions,
      NotifyPurchaseReagents;


--===========================
--UI Frame Creation Functions
--===========================

-- These functions are recycled immediately after use.

CreateReagentOptions = function()
  Player();
  local cr, player, k, m, fs, edit = SC_Main_Merchant_CR, player, -80, 50;
  for i,v in ipairs(Reagents[player.class]) do
    if player.level >= v[3] and player.level <= v[3] + v[4] then
      fs = cr:CreateFontString("SC_Main_Merchant_CR_ReagentN"..i, "ARTWORK",
          "ChatFontNormal");
      fs:SetPoint("LEFT", cr, "TOPLEFT", 10, k);

      edit = CreateFrame("EditBox", "SC_Main_Merchant_CR_ReagentN"..i.."_Edit",
          cr);
      edit:SetWidth(40);
      edit:SetHeight(32);
      edit:SetBackdrop({
        bgFile = "Interface/DialogFrame/UI-DialogBox-Background",
        edgeFile = "Interface/DialogFrame/UI-DialogBox-Border",
        tile = true,
        tileSize = 5,
        edgeSize = 10,
        insets = { left = 5, right = 5, top = 5, bottom = 5 }
      });
      edit:SetAutoFocus(false);
      edit:SetBackdropColor(0.7, 0.1, 0.7);
      edit:SetBackdropBorderColor(0.3, 0.1, 0.8);
      edit:SetPoint("CENTER", cr, "RIGHT", -50, m);
      edit:SetFontObject("GameFontNormal");
      edit:SetMaxLetters(3);

      k = k - 50;
      m = m - 50;
    end
  end
  CreateReagentOptions = nil;
end

CreateChangeReagents = function()
  local cr = CreateFrame("Frame", "SC_Main_Merchant_CR", SC_Main_Merchant);
  cr:SetWidth(350);
  cr:SetHeight(250);
  cr:SetPoint("BOTTOM", SC_Main_Merchant, "BOTTOM", 0, 10);
  cr:SetBackdrop({
    bgFile = "Interface/Tooltips/UI-Tooltip-Background",
    edgeFile = "Interface/Tooltips/UI-Tooltip-Border",
    tile = true,
    tileSize = 16,
    edgeSize = 12,
    insets = { left = 2, right = 2, top = 2, bottom = 2 }
  });
  cr:SetBackdropColor(0.1, 0.7, 0.7);
  cr:SetBackdropBorderColor(0.6, 0.6, 0.6);

  local fs = cr:CreateFontString("SC_Main_Merchant_CR_Query1", "ARTWORK",
    "ChatFontNormal");
  fs:SetText(SC_REAGENT_CHANGE2);
  fs:SetPoint("CENTER", cr, "TOP", 0, -10);

  CreateReagentOptions();

  fs = cr:CreateFontString("SC_Main_Merchant_CR_Suggestion", "ARTWORK",
    "ChatFontNormal");
  fs:SetText(SC_REAGENT_CHANGE3);
  fs:SetPoint("CENTER", cr, "BOTTOM", 0, 15);

  local button = CreateFrame("Button", "SC_Main_Merchant_CR_Default", cr,
    "OptionsButtonTemplate");
  button:SetText(DEFAULT);
  button:SetPoint("CENTER", cr, "TOP", 0, -30);
  button:SetScript("OnClick", function()
    ShowDefaultReagents();
  end);

  button = CreateFrame("Button", "SC_Main_Merchant_CR_Done", cr,
    "OptionsButtonTemplate");
  button:SetText(DONE);
  button:SetPoint("CENTER", cr, "BOTTOM", -50, 40);
  button:SetScript("OnClick", function()
    CheckReagentEditBoxs();
    SettingsDone();
    SC_Main:Hide();
  end);

  button = CreateFrame("Button", "SC_Main_Merchant_CR_Cancel", cr,
    "OptionsButtonTemplate");
  button:SetText(CANCEL);
  button:SetPoint("CENTER", cr, "BOTTOM", 50, 40);
  button:SetScript("OnClick", function()
    Spellcraft.changes = { };
    SC_Main:Hide();
  end);

  cr:SetScript("OnShow", function()
    MakeChangeReagents();
  end);

  cr:SetScript("OnHide", CheckReagentEditBoxs);

  CreateChangeReagents = nil;
end

CreateReagentNRepair = function()
  local rnr = CreateFrame("Frame", "SC_ReagentNRepair", SC);
  rnr:SetWidth(300);
  rnr:SetHeight(450);
  rnr:SetPoint("CENTER", UIParent, "CENTER");
  rnr:SetBackdrop({
    bgFile = "Interface/TutorialFrame/TutorialFrameBackground",
    edgeFile = "Interface/DialogFrame/UI-DialogBox-Border",
    tile = true,
    tileSize = 32,
    edgeSize = 16,
    insets = { left = 5, right = 5, top = 5, bottom = 5 }
  });

  local fs = rnr:CreateFontString("SC_ReagentNRepairQuery1", "ARTWORK",
    "ChatFontNormal");
  fs:SetText(SC_MERCHANT_GRAY);
  fs:SetPoint("CENTER", rnr, "TOP", 0, -10);

  fs = rnr:CreateFontString("SC_ReagentNRepairQuery2a", "ARTWORK",
    "ChatFontNormal");
  fs:SetText(SC_MERCHANT_REPAIR);
  fs:SetPoint("CENTER", rnr, "TOP", 0, -100);

  fs = rnr:CreateFontString("SC_ReagentNRepairQuery2b", "ARTWORK",
    "ChatFontNormal");
  fs:SetPoint("CENTER", rnr, "TOP", 0, -120);

  fs = rnr:CreateFontString("SC_ReagentNRepairQuery3a", "ARTWORK",
    "ChatFontNormal");
  fs:SetText(SC_MERCHANT_REAGENT);
  fs:SetPoint("CENTER", rnr, "TOP", 0, -200);

  fs = rnr:CreateFontString("SC_ReagentNRepairQuery3b", "ARTWORK",
    "ChatFontNormal");
  fs:SetPoint("CENTER", rnr, "TOP", 0, -220);

  fs = rnr:CreateFontString("SC_ReagentNRepairQuery4", "ARTWORK",
    "ChatFontNormal");
  fs:SetText(SC_REAGENT_CHANGE);
  fs:SetPoint("CENTER", rnr, "TOP", 0, -300);

  local grayaccept = CreateFrame("Button", "SC_ReagentNRepair_GrayAccept",
    rnr, "OptionsButtonTemplate");
  grayaccept:SetPoint("TOP", SC_ReagentNRepair, "TOP", 0, -40);
  grayaccept:SetText(YES);
  grayaccept:SetScript("OnClick", function()
    SellGrayTrash();
    OnMerchant();
  end);

  local repairaccept = CreateFrame("Button", "SC_ReagentNRepair_RepairAccept",
    rnr, "OptionsButtonTemplate");
  repairaccept:SetPoint("TOP", SC_ReagentNRepair, "TOP", 0, -140);
  repairaccept:SetText(YES);
  repairaccept:SetScript("OnClick", function()
    RepairAllItems();
    OnMerchant();
  end);

  local reagentaccept = CreateFrame("Button", "SC_ReagentNRepair_ReagentAccept",
    rnr, "OptionsButtonTemplate");
  reagentaccept:SetPoint("TOP", SC_ReagentNRepair, "TOP", 0, -240);
  reagentaccept:SetText(YES);
  reagentaccept:SetScript("OnClick", function()
    BuyReagents();
    OnMerchant();
  end);

  local changereagent = CreateFrame("Button", "SC_ReagentNRepair_ChangeReagent",
    rnr, "OptionsButtonTemplate");
  changereagent:SetPoint("TOP", SC_ReagentNRepair, "TOP", 0, -340);
  changereagent:SetText(YES);
  changereagent:SetScript("OnClick", function()
    SC_ReagentNRepair:Hide();
    ChangeReagentShow();
  end);

  local close = CreateFrame("Button", "SC_ReagentNRepair_Close", rnr,
    "UIPanelCloseButton");
  close:SetPoint("TOPRIGHT", SC_ReagentNRepair, "TOPRIGHT");
  close:SetScript("OnClick", function()
    SC_ReagentNRepair:Hide();
  end);

  local close2 = CreateFrame("Button", "SC_ReagentNRepair_Close2", rnr,
    "OptionsButtonTemplate");
  close2:SetPoint("BOTTOM", SC_ReagentNRepair, "BOTTOM", 0, 10);
  close2:SetText(CLOSE);
  close2:SetScript("OnClick", function()
    SC_ReagentNRepair:Hide();
  end);

  rnr:Hide();
  CreateReagentNRepair = nil;
end


--==================
--Merchant Functions
--==================

HasGrayTrash = function()
  for bag = 0, 4 do
    for slot = 1, GetContainerNumSlots(bag) do
      local name = GetContainerItemLink(bag, slot);
      if name and string.find(name, "ff9d9d9d") then 
        return true;
      end
    end
  end
  return false;
end

SellGrayTrash = function()
  for bag = 0, 4 do
    for slot = 1, GetContainerNumSlots(bag) do
      local name = GetContainerItemLink(bag, slot);
      if name and string.find(name, "ff9d9d9d") then 
        UseContainerItem(bag, slot);
        inform(SC_MSG_GRAYTRASH..name);
      end
    end
  end
end

local ReagentsNumber = { };
CheckNeedForReagents = function()
  if not Reagents[player.class] then return false end
  local player, reagents = player, Reagents[player.class];
  for i,v in ipairs(reagents) do
    ReagentsNumber[i] = 0;
  end
  for bag = 0, 4 do
    for slot = 1, GetContainerNumSlots(bag) do
      local link = GetContainerItemLink(bag, slot);
      if link then
        local _, quantity = GetContainerItemInfo(bag, slot);
        for i,v in ipairs(reagents) do
          if string.find(link, v[1]) then
            ReagentsNumber[i] = ReagentsNumber[i] + quantity;
          end
        end
      end
    end
  end

  local num = 0;
  for i,v in ipairs(ReagentsNumber) do
    if reagents[i][3] <= player.level and reagents[i][3] + reagents[i][4]
        >= player.level then
      ReagentsNumber[i] = config.Merchant.reagents[player.class][i] - v;
      if ReagentsNumber[i] < 0 then ReagentsNumber[i] = 0; end
    else
      ReagentsNumber[i] = 0;
    end
    num = num + ReagentsNumber[i];
  end

  return not (num == 0), ReagentsNumber;
end

CheckMerchant = function() -- returns whether the merchant can sell regeants
  if not Reagents[player.class] then return false end
  local reagent = { };
  for i = 1, GetMerchantNumItems() do
    local name, _, price = GetMerchantItemInfo(i);
    for j, v in ipairs(Reagents[player.class]) do
      if name == Reagents[player.class][j][1] then
        reagent[j] = {i, price};
      end
    end
  end
  if not next(reagent) then
    return false;
  else
    return true, reagent;
  end
end

BuyReagents = function()
  local b1, reagents = CheckMerchant();
  local b2, ReagentsNumber = CheckNeedForReagents();
  if not (b1 and b2) then return end
  for i,r in ipairs(reagents) do
    if ReagentsNumber[i] ~= 0 then
      BuyMerchantItem(r[1], ReagentsNumber[i]);
    end
  end
end

OnMerchant = function()
  if locked then return end
  local gray = HasGrayTrash();
  local cost = GetRepairAllCost();
  local repairb = CanMerchantRepair() and cost ~= 0;
  local reagentb, reagent = CheckNeedForReagents();
  local breagent, mreagent = CheckMerchant();

  if not (gray or repairb or (reagentb and breagent)) then return end

  local mreagentCost = 0;
  if breagent and reagentb then
    for i,r in ipairs(mreagent) do
      mreagentCost = mreagentCost + r[2] * reagent[i];
    end
  end

  if config.Merchant.silent then
    if gray then
      SellGrayTrash();
    end
    if repairb and GetMoney() > cost then
      RepairAllItems();
      inform(
            ("%s|cFFFFFF33%d %s|r|cFFCCCCFF %02d %s|r|cFFCC9933 %02d %s|r")
            :format(REPAIR_COST, cost / 10000, GOLD, cost % 10000 / 100,
            SILVER, cost % 100, COPPER));
    elseif repairb then
      error(SC_MSG_ERROR_MONEY);
    end
    if reagentb and breagent then
      BuyReagents();
      inform(SC_MSG_SILENT_REAGENT);
    end
    return;
  end

  if not SC_ReagentNRepair then CreateReagentNRepair(); end

  if not gray then
    SC_ReagentNRepairQuery1:SetTextColor(.5, .5, .5);
    SC_ReagentNRepair_GrayAccept:Disable();
    SC_ReagentNRepair_GrayAccept:SetTextColor(.5, .5, .5);
  else
    SC_ReagentNRepairQuery1:SetTextColor(1, 1, 1);
    SC_ReagentNRepair_GrayAccept:Enable();
    SC_ReagentNRepair_GrayAccept:SetTextColor(1, 1, 1);
  end
  if not repairb then
    SC_ReagentNRepairQuery2a:SetTextColor(.5, .5, .5);
    SC_ReagentNRepairQuery2b:Hide();
    SC_ReagentNRepair_RepairAccept:Disable();
    SC_ReagentNRepair_RepairAccept:SetTextColor(.5, .5, .5);
  else
    SC_ReagentNRepairQuery2b:SetText(string.format("%s |cFFFFFF33%d %s,|r|cFFCCCCFF %02d %s,|r|cFFCC9933 %02d %s|r", COSTS_LABEL , cost / 10000, GOLD, cost % 10000 / 100,
            SILVER, cost % 100, COPPER));
    SC_ReagentNRepairQuery2b:Show();
    SC_ReagentNRepairQuery2a:SetTextColor(1, 1, 1);
    SC_ReagentNRepair_RepairAccept:Enable();
    SC_ReagentNRepair_RepairAccept:SetTextColor(1, 1, 1);
  end
  if not (reagentb and breagent) then
    SC_ReagentNRepairQuery3a:SetTextColor(.5, .5, .5);
    SC_ReagentNRepairQuery3b:Hide();
    SC_ReagentNRepair_ReagentAccept:Disable();
    SC_ReagentNRepair_ReagentAccept:SetTextColor(.5, .5, .5);
  else
    SC_ReagentNRepairQuery3a:SetTextColor(1, 1, 1);
    SC_ReagentNRepairQuery3b:SetText(string.format("%s |cFFFFFF33%d %s,|r|cFFCCCCFF %02d %s,|r|cFFCC9933 %02d %s|r",COSTS_LABEL , mreagentCost / 10000, GOLD, mreagentCost % 10000 / 100,
            SILVER, mreagentCost % 100, COPPER));
    SC_ReagentNRepairQuery3b:Show();
    SC_ReagentNRepair_ReagentAccept:Enable();
    SC_ReagentNRepair_ReagentAccept:SetTextColor(1, 1, 1);
  end

  UIFrameFadeIn(SC_ReagentNRepair, 50, 0, 100);
  SC_ReagentNRepair:Show();--This seems to be necessary for some reason 
                           --(After a couple merchant updates, the frame stops
                           --showing)
end


--=======================
--Change Reagent Settings
--=======================

MakeChangeReagents = function()
  local player = player;
  if Reagents[player.class] == nil then return end
  for i,v in ipairs(Reagents[player.class]) do
    if v[3] <= player.level and v[3] + v[4] >= player.level then
      _G["SC_Main_Merchant_CR_ReagentN"..i]:SetText(v[1]);
      _G["SC_Main_Merchant_CR_ReagentN"..i.."_Edit"]:SetText(
	      config.Merchant.reagents[player.class][i]);
      _G["SC_Main_Merchant_CR_ReagentN"..i]:Show();
      _G["SC_Main_Merchant_CR_ReagentN"..i.."_Edit"]:Show();
    end
  end
end

CheckReagentEditBoxs = function()
  Spellcraft.changes = Spellcraft.changes or { };
  changes.Merchant = changes.Merchant or { };
  changes.Merchant.reagents = changes.Merchant.reagents or { };
  changes.Merchant.reagents[player.class] =
      changes.Merchant.reagents[player.class] or { };

  local x, y;
  for i in ipairs(Reagents[player.class]) do
    x, y = _G["SC_Main_Merchant_CR_ReagentN"..i.."_Edit"],
        _G["SC_Main_Merchant_CR_ReagentN"..i];
    if x then
      ChangeReagent(y:GetText(), x:GetText());
    end
  end
end

ChangeReagent = function(name, number)
  local b, num = CheckReagentEditBoxSanity(name, number);
  if b then
    for i,v in ipairs(Reagents[player.class]) do
      if v[1] == name then
        changes.Merchant.reagents[player.class][i] = num;
        break; 
      end
    end
  end
end

CheckReagentEditBoxSanity = function(reagent, num)
  local bool, number = numberSane(num);
  if not bool then
    error(string.format("%s %s: %d", reagent, SC_MSG_CREAGENT_ERROR, num));
    return false;
  else
    return true, number;
  end
end

ShowDefaultReagents = function()
  local x, y;
  for i,v in ipairs(Reagents[player.class]) do
    x, y = _G["SC_Main_Merchant_CR_ReagentN"..i.."_Edit"],
        _G["SC_Main_Merchant_CR_ReagentN"..i];
    if x and y then
      y:SetText(v[1]);
      x:SetText(v[2]);
    end
  end
end

ChangeReagentShow = function()
  if InCombatLockdown() then
    error(SC_MSG_SETTINGS_COMBAT);
  else
    if not SC_Main then MakeMainFrame(); end
    SC_Main:Show();
    _G["SC_MainTab"..tab]:Click();
  end
end


--============================
--Notification to Buy Reagents
--============================

NotifyPurchaseReagents = function()
  local time, reagents, notify = tonumber(date("%j%H")),
      config.Merchant.reagents[player.class], false;
  local bool, ReagentsNumber = CheckNeedForReagents();
  if bool then
    for i,v in ipairs(ReagentsNumber) do
      if reagents[i] - v <= reagents[i] * .3 then
        notify = true;
        break;
      end
    end
  end
  if LastNotification < time and notify then
    Notification(SC_MSG_REAGENTS_NEED);
    LastNotification = time;
  end
end

--================
--Module Functions
--================

function Load()
  -- Slash Commands
  _G.SLASH_CHANGEREAGENT1 = "/changereagent";
  _G.SlashCmdList["CHANGEREAGENT"] = ChangeReagentShow;

  -- Events
  SC_RegEvent(module, "PLAYER_ENTERING_WORLD");
  SC_RegEvent(module, "MERCHANT_SHOW");
  SC_RegEvent(module, "MERCHANT_CLOSED");
  SC_RegEvent(module, "UPDATE_LFG_TYPES");
end

function Defaults()
  config.Merchant = {load = true, silent = false, notify = true};
  config.Merchant.reagents = { };
  for i,v in pairs(Reagents) do
    config.Merchant.reagents[i] = { };
    for j,w in ipairs(Reagents[i]) do
      config.Merchant.reagents[i][j] = w[2];
    end
  end
end

function CreateSettingsFrame()
  if not SC_Main then MakeMainFrame(); end
  local merchant = CreateFrame("Frame", "SC_Main_Merchant", SC_Main);
  merchant:SetWidth(360);
  merchant:SetHeight(425);
  merchant:SetPoint("TOPLEFT", SC_Main, "TOPLEFT", 20, -50);
  merchant:SetBackdrop({
    bgFile = "Interface/Tooltips/UI-Tooltip-Background",
    edgeFile = "Interface/Tooltips/UI-Tooltip-Border",
    tile = true,
    tileSize = 16,
    edgeSize = 12,
    insets = { left = 2, right = 2, top = 2, bottom = 2 }
  });
  merchant:SetBackdropColor(0.1, 0.7, 0.1);
  merchant:SetBackdropBorderColor(0.6, 0.6, 0.1);
  merchant:SetScript("OnShow", function()
    SC_Main_Current:SetText(SC_MAIN_MERCHANT);
  end);


  local fs = merchant:CreateFontString("SC_Merchant_Notifications", "ARTWORK",
      "GameFontNormal");
  fs:SetText(SC_MAIN_MERCHANT_OPTION1);
  fs:SetWidth(250);
  fs:SetPoint("LEFT", merchant, "TOPLEFT", 10, -20);
  fs:SetJustifyH("LEFT");

  local cb = CreateFrame("CheckButton", "SC_Merchant_Edit1", merchant,
      "OptionsCheckButtonTemplate");
  cb:SetPoint("RIGHT", merchant, "TOPRIGHT", -10, -20);
  cb:SetChecked(config.Merchant.silent);
  cb:SetScript("OnClick", function(self)
    changes.Merchant = changes.Merchant or { };
    changes.Merchant.silent = ctruth(self:GetChecked());
  end);

  fs = merchant:CreateFontString("SC_Merchant_LowReagentsNotify", "ARTWORK",
      "GameFontNormal");
  fs:SetText(SC_MAIN_MERCHANT_OPTION2);
  fs:SetWidth(250);
  fs:SetPoint("LEFT", merchant, "TOPLEFT", 10, -50);
  fs:SetJustifyH("LEFT");

  local cb = CreateFrame("CheckButton", "SC_Merchant_Edit2", merchant,
      "OptionsCheckButtonTemplate");
  cb:SetPoint("RIGHT", merchant, "TOPRIGHT", -10, -50);
  cb:SetChecked(config.Merchant.notify);
  cb:SetScript("OnClick", function(self)
    local check = ctruth(self:GetChecked());
    changes.Merchant = changes.Merchant or { };
    changes.Merchant.notify = check;
    if check then
      SC_RegEvent(module, "PLAYER_UPDATE_RESTING");
    else
      SC_UnRegEvent(module, "PLAYER_UPDATE_RESTING");
    end
  end);

  tab = RegisterMainTab(merchant, SC_MAIN_MERCHANT);

  CreateChangeReagents();

  merchant:Hide();
  
  CreateSettingsFrame = nil;
end


  --==============
  --Event Handlers
  --==============

function PLAYER_ENTERING_WORLD(...)
  Player();
  if not Reagents[player.class] then
    error("unknown class, removing Merchant");
    _G.SLASH_CHANGEREAGENT1 = nil;
    _G.SlashCmdList["CHANGEREAGENT"] = nil;
    UnregisterModule(module);
  end
end

function MERCHANT_SHOW()
  locked = false;
  OnMerchant();
end

function MERCHANT_CLOSED()
  locked = true;
  if not SC_ReagentNRepair then return end
  SC_ReagentNRepair:Hide();
end

function PLAYER_UPDATE_RESTING()
  if IsResting() then
    NotifyPurchaseReagents();
  end
end

function BAG_UPDATE()
  if IsResting() then
    NotifyPurchaseReagents();
  end
end

function UPDATE_LFG_TYPES()
  -- I don't really remember why I use this : /
  if config.Merchant.notify then
    SC_RegEvent(module, "PLAYER_UPDATE_RESTING");
    SC_RegEvent(module, "BAG_UPDATE");
  end
  SC_UnRegEvent(module, "UPDATE_LFG_TYPES");
  UPDATE_LFG_TYPES = nil;
end
