--[[

Spellcraft: This is a World of Warcraft Mage Utility AddOn.
Copyright (C) 2007  Patrick J. Donnelly (batrick@unm.edu)

Permission is NOT granted to modify/redistribute this software.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

--]]


--[[

This module is attachable to the Minimap button. When you hover your mouse over
Minimap, the buff button will show (AI or AB texture). When clicked the button
will bring down a small drop down menu showing all the groups in your raid
or all the members in you party. Left clicking a party member will cause you
to cast AI on them. Right clicking casts AB. Left clicking on a group will
cause you to cast AI on a member with AI having the least time left.

A tooltip will show group members that use mana and the timer on their AI/AB.

This module also notifies the player when the buff of a player in a group being
watched is about to expire.

--]]

--=====================================
--Setting up the environment and module
--=====================================

local module = { };
local _G = getfenv(0);
setmetatable(module, {__index = _G.Spellcraft});
setfenv(1, module);

module.name = "Buffs";
module.lname = SC_MODULE_BUFFS;
module.version = 1.111;
module.info = SC_MODULE_BUFFS_INFO;

RegisterModule(module);


--======
--Locals
--======

local GroupButtons = { };
local PlayerButtons = { };
local Watch = { };
local Raid = { };


--=============
--Buff Checking
--=============

local function RaidInfo() -- Reuse tables later
  local name, subgroup, fileName, online, dead;
  local group = { };
  for i = 1, GetNumRaidMembers() do
    name, _, subgroup, level, _, fileName, _, online, dead =
        GetRaidRosterInfo(i); 
    group[subgroup] = group[subgroup] or { };
    table.insert(group[subgroup], {"raid"..i, name, level,
        standardName(fileName), online, dead});
  end
  return group;
end

local function UpdateRaid()
  Raid = RaidInfo();
end

local function IsManaUser(unitid)
  local _, c = UnitClass(unitid);
  return not (c == "WARRIOR" or c == "ROGUE");
end

local function GetIntTime(unitid)
  local i, time = 1;
  while true do
    local name, _, icon, _, _, timeLeft = UnitBuff(unitid, i, 1);
    if icon == "Interface\\Icons\\Spell_Holy_MagicalSentry" or
       icon == "Interface\\Icons\\Spell_Holy_ArcaneIntellect" then
      return math.ceil((timeLeft or 0) / 60); -- Just return seconds?
    elseif not name then
      return 0;
    end
    i = i + 1;
  end
end

local function GetGroup(raidindex)
  local _, _, subgroup = GetRaidRosterInfo(raidindex);
  return subgroup;
end

local function GetBestPlayerRaid()
  local best, time, hold = "", 61;
  for i = 1, GetNumRaidMembers() do
    hold = GetIntTime("raid"..i);
    if hold < time and IsManaUser("raid"..i) then
      best = "raid"..i;
      time = hold;
    end
  end
  return best, time;
end

local function GetBestPlayerGroup(group)
  local best, time = "NONE", 61;
  for _,v in pairs(Raid[group] or { }) do
    local int = GetIntTime(v[1]);
    if int < time and IsManaUser(v[1]) then
      best = v[1];
      time = int;
    end
  end
  return best, time;
end

local function GetBestPlayerParty()
  local best, time, hold = "NONE", 61;
  for i = 1, GetNumPartyMembers() do
    hold = GetIntTime("party"..i);
    if hold < time and IsManaUser("party"..i) then
      best = "party"..i;
      time = hold;
    end
  end
  return best, time;
end

local function GetBestGroup()
  local best_group, best_number, best_time, hold_number, hold_time, avg, hold
      = 0, 0, 61, 0, 0, 5;
  for i,v in pairs(Raid) do
    for _,w in pairs(v) do
      if IsManaUser(w[1]) then
        hold = GetIntTime(w[1]);
        hold_time = hold_time + hold;
        hold_number = hold_number + hold == 0 and 1 or 0;
      else
        avg = avg - 1;
      end
    end
    hold_time = hold_time / avg; -- Average
    if hold_time < best_time then
      best_group = i;
      best_time = hold_time;
      best_number = hold_number;
    elseif hold_time == best_time and hold_number > best_number then
      best_group = i;
      best_time = hold_time;
      best_number = hold_number;
    end
    hold_time = 0;
    hold_number = 0;
    avg = 5;
  end
  return best_group, best_time, best_number;
end


--======
--Frames
--======

local function GetWatchCheck()

end

local function CreateWatch()
  -- This Frame allows the player to specify only certain groups are to be
  -- buffed/watched.
  local watch = CreateFrame("Frame", "SC_Buffs_Watch", UIParent);
  watch:SetHeight(200);
  watch:SetWidth(200);
  watch:SetPoint("CENTER", UIParent, "CENTER");
  watch:SetBackdrop({
    bgFile = "Interface/TutorialFrame/TutorialFrameBackground",
    edgeFile = "Interface/DialogFrame/UI-DialogBox-Border",
    tile = true,
    tileSize = 32,
    edgeSize = 16,
    insets = { left = 5, right = 5, top = 5, bottom = 5 }
  });

  local fs = watch:CreateFontString("SC_Buffs_Watch_FS1", "ARTWORK",
    "ChatFontNormal");
  fs:SetText("Lol wut?");
  fs:SetPoint("CENTER", on, "TOP", 0, -15);

  local cb = CreateFrame("CheckButton", "SC_Buffs_Watch_CB1", watch,
    "OptionsCheckButtonTemplate");
  cb:SetPoint("CENTER", watch, "TOP", -50, -150);
  cb:SetScript("OnClick", GetWatchCheck);
  cb:SetChecked(false);

  cb = CreateFrame("CheckButton", "SC_Buffs_Watch_CB2", watch,
    "OptionsCheckButtonTemplate");
  cb:SetPoint("CENTER", watch, "TOP", -50, -100);
  cb:SetScript("OnClick", GetWatchCheck);
  cb:SetChecked(false);

  cb = CreateFrame("CheckButton", "SC_Buffs_Watch_CB3", watch,
    "OptionsCheckButtonTemplate");
  cb:SetPoint("CENTER", watch, "TOP", -50, -50);
  cb:SetScript("OnClick", GetWatchCheck);
  cb:SetChecked(false);

  cb = CreateFrame("CheckButton", "SC_Buffs_Watch_CB4", watch,
    "OptionsCheckButtonTemplate");
  cb:SetPoint("CENTER", watch, "TOP", -50, 0);
  cb:SetScript("OnClick", GetWatchCheck);
  cb:SetChecked(false);
  
  cb = CreateFrame("CheckButton", "SC_Buffs_Watch_CB5", watch,
    "OptionsCheckButtonTemplate");
  cb:SetPoint("CENTER", watch, "TOP", -50, 50);
  cb:SetScript("OnClick", GetWatchCheck);
  cb:SetChecked(false);

  cb = CreateFrame("CheckButton", "SC_Buffs_Watch_CB6", watch,
    "OptionsCheckButtonTemplate");
  cb:SetPoint("CENTER", watch, "TOP", -50, 100);
  cb:SetScript("OnClick", GetWatchCheck);
  cb:SetChecked(false);

  cb = CreateFrame("CheckButton", "SC_Buffs_Watch_CB7", watch,
    "OptionsCheckButtonTemplate");
  cb:SetPoint("CENTER", watch, "TOP", -50, 150);
  cb:SetScript("OnClick", GetWatchCheck);
  cb:SetChecked(false);

  cb = CreateFrame("CheckButton", "SC_Buffs_Watch_CB8", watch,
    "OptionsCheckButtonTemplate");
  cb:SetPoint("CENTER", watch, "TOP", -50, 200);
  cb:SetScript("OnClick", GetWatchCheck);
  cb:SetChecked(false);

  fs = watch:CreateFontString("SC_Buffs_Watch_FS2", "ARTWORK", "ChatFontNormal");
  fs:SetText("Receive warning how long before buff expires?");
  fs:SetPoint("CENTER", on, "TOP", 0, -15);

  local edit = CreateFrame("EditBox", "SC_Buffs_Watch_Edit", watch);
  edit:SetPoint("CENTER", watch, "BOTTOM", 0, 50);
  edit:SetText("hi");

  CreateWatch = nil;
end

local function PlayerTooltipShow(self)
  local whom = UnitName(self.player);
  GameTooltip:SetOwner(self, "ANCHOR_RIGHT");
  GameTooltip:SetText(string.format(SC_BUFFS_AI, whom));
  GameTooltip:Show();
end

local function GroupTooltipShow(self)
  GameTooltip:SetOwner(self, "ANCHOR_RIGHT");
  GameTooltip:SetText(string.format(SC_BUFFS_GROUP_TOOLTIP,
      self.group, UnitName(self.best) or "NONE"));
  GameTooltip:Show();
end

local function TooltipHide()
  GameTooltip:Hide();
end

local function CreatePlayerButton()
  -- These buttons will always have a player field associated with them.
  -- This will always be a unitid.
  local b = CreateFrame("Button", "SC_Buffs_Drop_Player"..(#PlayerButtons + 1),
    SC_Buffs, "SecureActionButtonTemplate");
  b:SetHeight(15);
  b:SetWidth(100);
  b:SetText(" ");
  b:SetTextColor(0, 1, 0);
  b:SetTextFontObject("GameFontHighlight");
  (b:GetFontString()):SetAllPoints(b);
  (b:GetFontString()):SetJustifyH("LEFT");

  b:SetScript("OnEnter", function()
    b:SetAttribute("unit", b.player);
    PlayerTooltipShow(b);
  end);
  b:SetScript("OnLeave", TooltipHide);
  b:SetScript("PostClick", ClearCursor);

  b:SetAttribute("type", "spell");
  b:SetAttribute("spell1", SC_SPELL_AI);

  table.insert(PlayerButtons, b);
  return b;
end

local function CreateSubDrop()
  local sub = CreateFrame("Frame", "SC_Buffs_SubDrop", SC_Buffs,
    "SecureFrameTemplate"); 
  sub:SetWidth(150);
  sub:SetHeight(100);
  sub:SetBackdrop({
    bgFile = "Interface/TutorialFrame/TutorialFrameBackground",
    edgeFile = "Interface/DialogFrame/UI-DialogBox-Border",
    tile = true,
    tileSize = 32,
    edgeSize = 16,
    insets = { left = 5, right = 5, top = 5, bottom = 5 }
  });
  sub:SetBackdropColor(.5, .5, .5);
  sub:SetBackdropBorderColor(.5, .5, .5);

  sub:Hide();
  CreateSubDrop = nil;
end

local function CreatePlayers()
  if not SC_Buffs_SubDrop then CreateSubDrop(); end
  local sub = SC_Buffs_SubDrop;
  sub:SetPoint("TOPLEFT", this, "TOPRIGHT");

  local place, k, button = -10, 1;

  for _,v in pairs(Raid[this.group]) do
    if IsManaUser(v[1]) then
      button = PlayerButtons[k] or CreatePlayerButton();
      button:SetParent(SC_Buffs_SubDrop);
      button.player = v[1];
      button:SetText(string.format("%s", UnitName(v[1])));
      button:ClearAllPoints();
      button:SetPoint("TOP", sub, "TOP", 0, place);
      button:Show();
      place = place - button:GetHeight();
      k = k + 1;
    end
  end
  sub:SetHeight(-place + 10);
  sub:Show();
  
  for i = k, #PlayerButtons do
    PlayerButtons[i]:Hide();
  end
end

local function CreateGroupButton()
  local b = CreateFrame("Button", "SC_Buffs_Drop_Group"..(#GroupButtons + 1),
    SC_Buffs_Drop, "SecureActionButtonTemplate");
  b:SetHeight(25);
  b:SetWidth(60);
  b:SetText(" ");
  b:SetTextColor(.4, 0, 1);
  b:SetTextFontObject("GameFontHighlight");
  (b:GetFontString()):SetAllPoints(b);
  (b:GetFontString()):SetJustifyH("LEFT");

  b:SetScript("OnEnter", function()
    b.first = Raid[b.group][1][1];
    b.best = GetBestPlayerGroup(b.group); --Will be a RaidID
    b:SetAttribute("unit1", b.best);
    b:SetAttribute("unit2", b.first);
    GroupTooltipShow();
  end);
  b:SetScript("OnLeave", TooltipHide);
  --[[b:SetScript("OnClick", function()
    if IsShiftKeyDown() then return end
    CreatePlayers();
  end);--]]
  --b:SetScript("PostClick", ClearCursor);
  b:SetAttribute("type1", "menu");
  b.showmenu = CreatePlayers;

  b:SetAttribute("type", "spell");

  b:SetAttribute("shift-spell1", SC_SPELL_AI);

  b:SetAttribute("spell2", SC_SPELL_AB);

  table.insert(GroupButtons, b);
  return b;
end

local function CreateGroups()
  local drop, place, k, button = SC_Buffs_Drop, -10, 1;

  for i in ipairs(Raid) do
    button = GroupButtons[i] or
        CreateGroupButton(string.format("%s %d", SC_GROUP, i));
    button:SetText(string.format("%s %d", SC_GROUP, i));
    button.group = i;
    button:ClearAllPoints();
    button:SetPoint("TOP", drop, "TOP", 0, place);
    place = place - button:GetHeight();
    k = k + 1;
  end
  drop:SetHeight(-place + 10);
  
  for i = k, #GroupButtons do
    GroupButtons[i]:Hide();
  end
end

local function CreateParty()
  local drop, place, button = SC_Buffs_Drop, -10;

  for i = 1, GetNumPartyMembers() do
    button = PlayerButtons[i] or CreatePlayerButton();
    button:SetParent(SC_Buffs_Drop);
    button.player = "party"..i;
    button:SetText(string.format("%s", UnitName(button.player)));
    button:SetPoint("TOP", drop, "TOP", 0, place);
    place = place - button:GetHeight();
  end
  drop:SetHeight(-place + 10);
end

local function SetUpGroups()
  if InCombatLockdown() then error("Cannot change in combat") return end

  UpdateRaid();

  if GetNumRaidMembers() > 0 then
    CreateGroups();
  elseif GetNumPartyMembers() > 0 then
    CreateParty();
  else
    error("You aren't in a party!");
  end
end

--Drop Frame
local function CreateDropFrame()
  local drop = CreateFrame("Frame", "SC_Buffs_Drop", SC_Buffs,
    "SecureFrameTemplate"); 
  drop:SetWidth(80);
  drop:SetHeight(300);
  drop:SetBackdrop({
    bgFile = "Interface/TutorialFrame/TutorialFrameBackground",
    edgeFile = "Interface/DialogFrame/UI-DialogBox-Border",
    tile = true,
    tileSize = 32,
    edgeSize = 16,
    insets = { left = 5, right = 5, top = 5, bottom = 5 }
  });
  drop:SetBackdropColor(.5, .5, .5);
  drop:SetBackdropBorderColor(.5, .5, .5);
  drop:SetPoint("TOPLEFT", SC_Buffs, "BOTTOMRIGHT");

  -- Setup Groups OnShow
  drop:SetScript("OnShow", SetUpGroups);

  drop:Hide();
  CreateDropFrame = nil;
end

local function BuffsUpdate()
  local buffs = SC_Buffs;
  buffs:ClearAllPoints();
  buffs:SetPoint("CENTER", UIParent, "CENTER",
                   config.Buffs.position.x, config.Buffs.position.y);
  buffs:SetUserPlaced(false);
end

local function Dragging()
  local buff, mini, position, minimum = SC_Buffs, SC_Minimap_Button, { }, { };

  buff:ClearAllPoints();
    
  config.Buffs.position.x, config.Buffs.position.y =
      GetCursorPosition();

  --Snap to Minimap?

  BuffsUpdate();
end

local function CreateBuffFrame()
  -- Start by making a buff button, not necessarily anchored to the Minimap
  local buff = CreateFrame("Button", "SC_Buffs", UIParent,
    "SecureActionButtonTemplate");
  buff:SetFrameStrata("LOW");
  buff:SetFrameLevel(5);
  buff:RegisterForClicks("LeftButtonDown");
  buff:RegisterForDrag("RightButton");
  buff.drag = false;
  buff:SetMovable(true);
  buff:SetWidth(25);
  buff:SetHeight(25);
  buff:SetNormalTexture(
       "Interface\\AddOns\\Spellcraft\\images\\SC.tga");
  buff:SetHighlightTexture("Interface/Minimap/UI-Minimap-ZoonButton-Highlight");
  buff:Hide();

  buff:SetAttribute("type", "spell");

  buff:SetAttribute("shift-spell1", SC_SPELL_AI);

  buff:SetAttribute("shift-spell2", SC_SPELL_AB);

  buff:SetScript("OnEnter", function()
    UpdateRaid();
    if GetNumRaidMembers() > 0 then
      buff.bestr, buff.bestrt = GetBestPlayerRaid();
      buff.bestg, buff.bestgt, buff.bestgn = GetBestGroup();
      buff:SetAttribute("unit2", Raid[buff.bestg][1]);
    else
      buff.bestr, buff.bestrt = GetBestPlayerParty();
    end
    buff:SetAttribute("unit1", buff.bestr);
    GameTooltip:SetOwner(buff, "ANCHOR_LEFT");
    GameTooltip:SetText(string.format(SC_BUFFS_TOOLTIP, InCombatLockdown()
        and string.format("|cffff0000%s|r", SC_BUFFS_LOCKED) or
        string.format("|c0000ff00%s|r", SC_BUFFS_UNLOCKED), buff.bestg or "",
        buff.bestgt or "", UnitName(buff.bestr), buff.bestrt));
    GameTooltip:Show();
    SC_RegEvent(module, "RAID_ROSTER_UPDATE");
  end);
  buff:SetScript("OnLeave", function()
    GameTooltip:Hide();
  end);
  buff:SetScript("OnHide", function()
    if SC_Buffs_Drop then SC_Buffs_Drop:Hide(); end
    if SC_Buffs_SubDrop then SC_Buffs_SubDrop:Hide(); end
    SC_UnRegEvent(module, "RAID_ROSTER_UPDATE");
    Raid = { };
    collectgarbage();--Clean up all the raid updates we made.
  end);
  buff:SetScript("OnDragStart", function() buff.drag = true; end);
  buff:SetScript("OnDragStop", function() buff.drag = false; end);
  buff:SetScript("OnUpdate", function()
    if buff.drag then
      Dragging();
    end
  end);
  buff:SetScript("OnClick", function()
    --Buff Drop Down Frame
    if IsShiftKeyDown() then return end
    if not SC_Buffs_Drop then
      CreateDropFrame();
    end
    local d, s = SC_Buffs_Drop, SC_Buffs_SubDrop;
    if d:IsShown() then
      d:Hide();
    else
      d:Show();
    end
    if s and s:IsShown() then s:Hide(); end
  end);
  buff:SetScript("PostClick", ClearCursor);
  buff:ClearAllPoints();
  buff:SetPoint("CENTER", _G[config.Buffs.position.anchor], "BOTTOMRIGHT",
      config.Buffs.position.x, config.Buffs.position.y);

  if config.Buffs.position.anchor == "SC_Minimap_Button" and
     Spellcraft.Minimap then
    Spellcraft.Minimap.AddMinimapElement(buff);
  end

  print("register for proper event, RAID_ROSTER_UPDATE??");
  print("buff tooltip is unclear. Shift+RightClick to drag??, let's do shift+rightlcick to do AB. Less likely for screwing up");

  CreateBuffFrame = nil;
end


--================
--Module Functions
--================

function Load()
  -- Events
  SC_RegEvent(module, "PLAYER_ENTERING_WORLD");
  print("register for combat lockdown event so buff frame can be hidden!");
  print("method to watch groups easily, command or something");
  print("find a cleaner frame backdrop")-- haha error owned
end

function Defaults()
  config.Buffs = {
    load = true,
    msg = true,
    position = {
      anchor = "SC_Minimap_Button",
      x = 25,
      y = -25
    }
  };
end


  --==============
  --Event Handlers
  --==============

function PLAYER_ENTERING_WORLD()
  Player();
  if player.class ~= "Mage" then
    UnregisterModule(module);
    return;
  end
  if not SC_Buffs then CreateBuffFrame(); end
end
