--[[---------------------------------------------------------------------
	Speedometer, v1.1
	by Phanx (Bherasha @ US Blackrock Horde)
	A simple text display of your current ranged attack speed.
	Optionally supports font selection through LibSharedMedia-3.0.
-----------------------------------------------------------------------]]

if select(2, UnitClass("player")) ~= "HUNTER" then return end

Speedometer = CreateFrame("Frame", "Speedometer", UIParent)

local Speedometer = Speedometer
local UnitRangedDamage = UnitRangedDamage
local floor = math.floor
local format = string.format
local len = string.len
local lower = string.lower
local match = string.match
local sub = string.sub
local upper = string.upper
local select = select
local tonumber = tonumber
local type = type

local db, paused
local defaultFont = "Fonts\\FRIZQT__.ttf"
local media = LibStub and LibStub:GetLibrary("LibSharedMedia-3.0", true)

local outlines = {
	["none"] = "NONE",
	["thin"] = "OUTLINE",
	["thick"] = "THICKOUTLINE",
}

local function move()
	this:StartMoving()
end

local function stop()
	this:StopMovingOrSizing()

	local s = this:GetEffectiveScale()
	db.x = this:GetLeft() * s
	db.y = this:GetTop() * s
end

local prefix = "|cff33ff99Speedometer:|r "
function Speedometer:Print(text, ...)
	ChatFrame1:AddMessage(prefix..format(text, ...))
end

function Speedometer:Update()
	self.text:SetFormattedText("%0.2f", UnitRangedDamage("player"))
end

function Speedometer:UNIT_RANGEDDAMAGE()
	self:Update()
end

function Speedometer:PLAYER_DEAD()
	self:Hide()
end

function Speedometer:PLAYER_ALIVE()
	self:Show()
end

function Speedometer:VARIABLES_LOADED()
	local defaults = {
		r = 1,
		g = 1,
		b = 1,
		a = 1,
		scale = 1,
		font = "Friz Quadrata TT",
		outline = "None",
		shadow = 1,
	}
	if not SpeedometerDB then
		SpeedometerDB = defaults
		db = SpeedometerDB
	else
		db = SpeedometerDB
		for k, v in pairs(defaults) do
			if not db[k] or type(db[k]) ~= type(defaults[k]) then
				db[k] = v
			end
		end
	end

	self:SetFrameStrata("BACKGROUND")
	self:SetScale(db.scale)
	self:SetAlpha(db.a)
	self:SetWidth(100)
	self:SetHeight(20)
	self:SetBackdrop({bgFile="Interface\\Tooltips\\UI-Tooltip-Background"})
	self:SetBackdropColor(1,1,1,0)
	self:SetClampedToScreen(true)
	self:SetMovable(true)
	self:RegisterForDrag("LeftButton")
	self:SetScript("OnDragStart", move)
	self:SetScript("OnDragStop", stop)

	if db.x and db.y then
		local s = self:GetEffectiveScale()
		self:ClearAllPoints()
		self:SetPoint("TOPLEFT", UIParent, "BOTTOMLEFT", db.x / s, db.y / s)
	else
		self:SetPoint("CENTER", UIParent, "CENTER")
	end

	if not db.lock then
		self:SetBackdropColor(1,1,1,1)
		self:EnableMouse(true)
	end

	self.text = self:CreateFontString(nil, "OVERLAY")
	self.text:SetWidth(100)
	self.text:SetHeight(20)
	self.text:SetPoint("CENTER", self, "CENTER")
	self.text:SetFont(media and media:Fetch("font", db.font) or defaultFont, 15, outlines[db.outline])
	self.text:SetShadowOffset(db.shadow, -db.shadow)
	self.text:SetTextColor(db.r, db.g, db.b)

	if UnitIsDeadOrGhost("player") then
		self:Hide()
	else
		self:Show()
	end

	self:UnregisterEvent("VARIABLES_LOADED")

	self:RegisterEvent("UNIT_RANGEDDAMAGE")
	self:RegisterEvent("PLAYER_DEAD")
	self:RegisterEvent("PLAYER_ALIVE")

	self:Update()

	if media then
		media.RegisterCallback(self, "LibSharedMedia_Registered", "LibSharedMedia_Update")
		media.RegisterCallback(self, "LibSharedMedia_SetGlobal", "LibSharedMedia_Update")
	end
end

function Speedometer:LibSharedMedia_Update(callback, type, handle)
	if media and type == "font" then
		self.text:SetFont(media:Fetch("font", db.font), 15, outlines[db.outline])
	end
end

Speedometer:SetScript("OnEvent", function(self, event, ...)
	if self[event] and type(self[event]) == "function" then
		self[event](self, ...)
	else
		error("No handler found for event "..event.."!")
	end
end)

Speedometer:RegisterEvent("VARIABLES_LOADED")

--[[---------------------------------------------------------------------
	Options
-----------------------------------------------------------------------]]

local ENABLED = "|cff99ff99enabled|r"
local DISABLED = "|cffff9999disabled|r"
local STATUS = "|cffffff99%s|r"

function Speedometer:CMD_ALPHA(v)
	v = tonumber(v)
	if v and v >= 0.1 and v <= 1 then
		db.a = v
		self:SetAlpha(v)
		self:Print("Alpha set to %s.", format(STATUS, format("%0.2f", v)))
	else
		self:Print("Invalid input. Alpha must be a number between 0.1 and 1.")
	end
end

local HELP_COLOR = "Color must be entered as an \"|cffff3333RR|r|cff33ff33GG|r|cff3333ffBB|r\" hex value, or as an \"|cffff3333R|r |cff33ff33G|r|cff3333ffB|r\" tuple using values in either the 0-1 or 0-255 range."
function Speedometer:CMD_COLOR(v)
	if not v or len(v) == 0 then
		self:Print("Current color is r = %s, g = %s, b = %s", db.r, db.g, db.b)
		return
	end
	local r, g, b
	if len(v) == 6 and match(v, "^%x*$") then
		r, g, b = tonumber(sub(v, 1, 2), 16) / 255, tonumber(sub(v, 3, 4), 16) / 255, tonumber(sub(v, 5, 6), 16) / 255
	else
		r, g, b = match(v, "^([%d%.]+) ([%d%.]+) ([%d%.]+)$")
		r, g, b = tonumber(r), tonumber(g), tonumber(b)
	end
	if not (r and g and b) then
		self:Print(HELP_COLOR)
		return
	end
	if r >= 0 and r <= 1 and g >= 0 and g <= 1 and b >= 0 and b <= 1 then
		-- good to go
	elseif r >= 0 and r <= 255 and g >= 0 and g <= 255 and b >= 0 and b <= 255 then
		r = r / 255
		g = g / 255
		b = b / 255
	else
		self:Print(HELP_COLOR)
		return
	end
	db.r = r
	db.g = g
	db.b = b
	self.text:SetTextColor(r, g, b)
	self:Print("Color set to %s.", format(STATUS, format("%02x%02x%02x", r * 255, g * 255, b * 255)))
end

local media_fonts
local media_fonts_text
function Speedometer:CMD_FONT(v)
	if media then
		if not media_fonts then
			media_fonts = media:List("font")
		end
		if not media_fonts_text then
			media_fonts_text = ""
			for i, font in ipairs(media_fonts) do
				media_fonts_text = media_fonts_text..", "..font
			end
			media_fonts_text = sub(media_fonts_text, 3)
		end
		v = lower(tostring(v))
		if v ~= "nil" and v ~= "" then
			local found = false
			for i, font in pairs(media_fonts) do
				if lower(font) == v then
					db.font = font
					self.text:SetFont(media:Fetch("font", font), 15, outlines[db.outline])
					found = true
					break
				end
			end
			if found then
				self:Print("Font changed to %s.", format(STATUS, db.font))
				return
			end
		end
		self:Print("Available fonts: %s.", media_fonts_text)
	else
		self:CMD_HELP()
	end
end

function Speedometer:CMD_OUTLINE(v)
	v = lower(tostring(v))
	if outlines[v] then
		db.outline = v
		local face, size = self.text:GetFont()
		self.text:SetFont(face, size, outlines[v])
		self:Print("Outline set to %s.", format(STATUS, v))
	else
		self:Print("Outline must be one of: None, Thin, Thick.")
	end
end

function Speedometer:CMD_SHADOW(v)
	v = tonumber(v)
	if v and v >= 0 and v <= 2 then
		db.shadow = v
		self.text:SetShadowOffset(0, 0)
		self.text:SetShadowOffset(v, -v)
		self:Print("Shadow offset set to %s.", format(STATUS, v))
	else
		self:Print("Invalid input. Shadow offset must be a number between 0 and 2 (inclusive).")
	end
end

function Speedometer:CMD_SCALE(v)
	v = tonumber(v)
	if v and v >= 0.1 and v <= 4 then
		db.scale = v
		self:SetScale(v)
		self:Print("Scale set to %s.", format(STATUS, v))
	else
		self:Print("Invalid input. Scale must be a number between 0.1 and 2. (inclusive)")
	end
end

function Speedometer:CMD_LOCK()
	if not db.lock then
		self:SetBackdropColor(1,1,1,0)
		self:EnableMouse(false)
		db.lock = true
		self:Print("Locked.")
	else
		self:SetBackdropColor(1,1,1,1)
		self:EnableMouse(true)
		db.lock = false
		self:Print("Unlocked.")
	end
end

function Speedometer:CMD_PAUSE()
	if not paused then
		paused = true
		Speedometer:UnregisterEvent("UNIT_RANGEDDAMAGE")
		Speedometer:UnregisterEvent("PLAYER_DEAD")
		Speedometer:UnregisterEvent("PLAYER_ALIVE")
		self:Hide()
		self:Print("Paused.")
	else
		paused = false
		Speedometer:RegisterEvent("UNIT_RANGEDDAMAGE")
		Speedometer:RegisterEvent("PLAYER_DEAD")
		Speedometer:RegisterEvent("PLAYER_ALIVE")
		self:Show()
		self:Update()
		self:Print("Resumed.")
	end
end

function Speedometer:CMD_HELP()
	self:Print("Use |cffffff99/speed|r with the following commands:")
	ChatFrame1:AddMessage(" -|cffffff99 alpha |r- set the transparency [".. format(STATUS, db.a) .."]")
	ChatFrame1:AddMessage(" -|cffffff99 color |r- set the text color [".. format(STATUS, format("%02x%02x%02x", db.r * 255, db.g * 255, db.b * 255)) .."]")
	if media then
		ChatFrame1:AddMessage(" -|cffffff99 font |r- set the font face [".. format(STATUS, db.font) .."]")
	end
	ChatFrame1:AddMessage(" -|cffffff99 outline |r- set the font outline [".. format(STATUS, db.outline) .."]")
	ChatFrame1:AddMessage(" -|cffffff99 scale |r- set the scale [".. format(STATUS, format("%0.2f", db.scale)) .."]")
	ChatFrame1:AddMessage(" -|cffffff99 shadow |r- set the shadow offset (0 = no shadow) [".. format(STATUS, db.shadow) .."]")
	ChatFrame1:AddMessage(" -|cffffff99 lock |r- toggle the locked status [".. ( db.lock and ENABLED or DISABLED ) .."]")
	ChatFrame1:AddMessage(" -|cffffff99 pause |r- temporarily hide the display [".. ( paused and ENABLED or DISABLED ) .."]")	
end

SLASH_SPEEDOMETER1 = "/speed"
SlashCmdList.SPEEDOMETER = function(input)
	local cmd, args = match(input, "^%s*(%w+)%s*(.*)$")
	local handler = cmd and Speedometer["CMD_"..upper(cmd)]
	if handler then
		handler(Speedometer, args)
	else
		Speedometer:CMD_HELP()
	end
end