--[[ SimpleMail
     Read your mail.
]]--

scriptErrors = 1
SimpleMail = {}

SimpleMail.DebugLevel = 0
SimpleMail.Version = "0.10"
-- I don't know whether this will change.
SimpleMail.MAILMAX = 50
-- The number of items that could possibly be attached to a single message:
SimpleMail.ITEMMAX = 12
SimpleMail.opts = {}

--[[
  Design notes:
  Requests for items are asynchronous.  Thus, after you send the
  request to get an item... NOTHING HAPPENS.  Until later.

  Thus, while check requests can just be iterated, get requests
  have to be handled over time in response to events.  This requires
  us to preserve state.

  State is:

  * Current slot to check
  * Number of slots left to check
  * Matching pattern, if any

  When an event comes in:
  * Check current slot.
    -- if it matches, fire a get request and decrement slots left
    -- if it doesn't, decrement current slot and slots left until
       either it matches, or you run out of slots

  Flags:
  -c -> print only totals
  -a -> only "Auction House" messages
  -A -> everything BUT "Auction House" messages
  -e -> only "Auction expired" messages
  -f -> look at sender only, not subject
  -j -> look at subject only, not sender
  -l -> only "Auction canceLed" messages
  -w -> only "Auction won" messages
  -o -> only "Outbid on" messages
  -s -> only "Auction successful" messages
  -a -> only "Auction House" messages
  -A -> everything BUT "Auction House" messages
  -C -> accept COD
]]--

--[[ Data structure:
  SimpleMail_Options = {
    ["Delay"] = N 		-- number of frames to delay when getting gold
  }
]]--

function SimpleMail.Debug(level, text, arg1, arg2, arg3, arg4, arg5, arg6, arg7,
arg8, arg9)
  if (level <= SimpleMail.DebugLevel) then
    ChatFrame1:AddMessage("SimpleMail: " .. format(text or 'nil', arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9), 0,1,0.4)
  end
end

-- Event stuff:

function SimpleMail.GetInfo(slot)
  SimpleMail.Debug(3, "getting info for slot %d", slot)
  SimpleMail.packageIcon, SimpleMail.stationeryIcon, SimpleMail.sender, SimpleMail.subject, SimpleMail.money, SimpleMail.CODAmount, SimpleMail.daysLeft, SimpleMail.hasItem, SimpleMail.wasRead, SimpleMail.wasReturned, SimpleMail.textCreated, SimpleMail.canReply, SimpleMail.isGM = GetInboxHeaderInfo(slot)
  SimpleMail.sender = SimpleMail.sender or "no sender"
  SimpleMail.subject = SimpleMail.subject or "no subject"
  SimpleMail.itemCount = 0
  SimpleMail.Debug(3, "slot %d hasitem %d", slot, SimpleMail.hasItem or 0)
  for i = 1,SimpleMail.ITEMMAX do
    SimpleMail.Debug(4, "checking sub-slot %d", i)
    local name, itemTexture, count, quality, canUse = GetInboxItem(slot, i)
    SimpleMail.Debug(4, "[%d]: %s, %d", i, name or "nil", count or "nil")
    if (name ~= nil) then
      SimpleMail.itemCount = SimpleMail.itemCount + 1
    end
  end
  if (string.find(SimpleMail.sender, 'Auction House$')) then
    SimpleMail.AuctionType = string.match(SimpleMail.subject, '^Auction ([^:]*):')
    if (string.find(SimpleMail.subject, '^Outbid')) then
      SimpleMail.AuctionType = 'outbid'
    end
    if (string.find(SimpleMail.subject, '^Sale Pending')) then
      SimpleMail.AuctionType = 'pending'
    end
  else
    SimpleMail.AuctionType = nil
  end
end

function SimpleMail.PrettyPrint(amount)
  local gold, silver, copper
  local money = ""

  if (amount < 0) then
    amount = amount * -1
    money = "-"
  end
  gold = math.floor(amount / 10000)
  silver = math.floor(amount / 100) % 100
  copper = amount % 100

  if (gold > 0) then
    money = money .. "|cFFFFEE00" .. gold .. "g"
  end
  if (silver > 0 or gold > 0) then
    money = money .. "|cFFFFFFFF" .. silver .. "s"
  end
  money = money .. "|cFFEE7700" ..  copper .. "c|r"
  return money
end

function SimpleMail.Check(slot)
  if (not SimpleMail.opts['c']) then
    local slottext, moneytext, itemtext
    if (SimpleMail.DebugLevel > 0) then
      slottext = format('[%d] ', slot)
    else
      slottext = ''
    end
    if (SimpleMail.money ~= 0) then
      moneytext = format(' (%s)', SimpleMail.PrettyPrint(SimpleMail.money))
    elseif (SimpleMail.CODAmount ~= 0) then
      moneytext = format(' (%s)', SimpleMail.PrettyPrint(-1 * SimpleMail.CODAmount))
    else
      moneytext = ''
    end
    if (SimpleMail.itemCount ~= 0) then
      itemtext = format(' [%d item%s]',
      			SimpleMail.itemCount,
			(SimpleMail.itemCount == 1) and '' or 's')
    else
      itemtext = ''
    end
    if (SimpleMail.AuctionType) then
      SimpleMail.Debug(0, '%s%s%s%s',
        slottext, SimpleMail.subject, moneytext, itemtext)
    else
      SimpleMail.Debug(0, '%sFrom %s: %s%s%s',
        slottext, SimpleMail.sender, SimpleMail.subject, moneytext, itemtext)
    end
  end
  SimpleMail.TotalItems = SimpleMail.TotalItems + SimpleMail.itemCount
  if (SimpleMail.money ~= nil) then
    SimpleMail.TotalMoney = SimpleMail.TotalMoney + SimpleMail.money
    SimpleMail.TotalMoney = SimpleMail.TotalMoney - SimpleMail.CODAmount
  end
  return 1
end

function SimpleMail.Get(slot)
  local doanother = 1
  SimpleMail.Debug(1, "get index %d from %s: %s", slot,
  	SimpleMail.sender, SimpleMail.subject)
  SimpleMail.Pending = 0
  -- we skip over CODs, usually.
  if (SimpleMail.CODAmount ~= 0 and not SimpleMail.opts['C']) then
    return 1
  end
  -- if there's more than one item, register for item updates, because
  -- we get no mailbox events from taking items.
  if (SimpleMail.itemCount > 1) then
    SimpleMail_Frame:RegisterEvent("BAG_UPDATE")
    TakeInboxItem(slot, SimpleMail.itemCount)
    SimpleMail.itemCount = SimpleMail.itemCount - 1
    SimpleMail.TotalItems = SimpleMail.TotalItems + 1
    SimpleMail.Pending = 1
    -- if you just removed the second-to-last item, the message is changed
    if (SimpleMail.itemCount == 1) then
      SimpleMail.Pending = SimpleMail.Pending + 1
    end
    SimpleMail.wasRead = 1
    SimpleMail.ExtraDelay = true
    return -1
  end
  -- otherwise, start on the reading part.
  if (SimpleMail.itemCount > 0 or SimpleMail.money > 0) then
    GetInboxText(slot)
    SimpleMail.Pending = SimpleMail.Pending + 1
    doanother = 0
  end
  -- This seems to be the special case that makes auction won
  -- messages get "got" before they're ready otherwise.
  if (SimpleMail.AuctionType == 'won' or SimpleMail.AuctionType == 'successful') then
    GetInboxInvoiceInfo(slot)
    if (not SimpleMail.wasRead) then
      SimpleMail.Pending = SimpleMail.Pending + 1
      doanother = 0
    end
  end
  if ((SimpleMail.itemCount > 0) and ((SimpleMail.CODAmount == 0) or SimpleMail.opts['C'])) then
    TakeInboxItem(slot)
    SimpleMail.TotalItems = SimpleMail.TotalItems + 1
    doanother = 0
    SimpleMail.Pending = SimpleMail.Pending + 1
  end
  if (SimpleMail.money > 0) then
    TakeInboxMoney(slot)
    doanother = 0
    SimpleMail.Pending = SimpleMail.Pending + 1
    SimpleMail.TotalMoney = SimpleMail.TotalMoney + SimpleMail.money
  end
  if (SimpleMail.money > 0 and SimpleMail.itemCount < 1) then
    SimpleMail.ExtraDelay = true
  else
    SimpleMail.ExtraDelay = false
  end
  SimpleMail.Debug(2, "Pending " .. SimpleMail.Pending .. ", doanother " ..  doanother)
  return doanother
end

SimpleMail.Functions = {
	["check"] = SimpleMail.Check,
	["get"] = SimpleMail.Get,
	-- Dummy values so the "usage" check works.
	["delay"] = SimpleMail.Check,
	["debug"] = SimpleMail.Check,
	["reset"] = SimpleMail.Check,
	["kick"] = SimpleMail.Check,
}

function SimpleMail.OnSlash(command)
  SimpleMail.OldOpts = SimpleMail.opts
  SimpleMail.opts = GetOpt.getopt('aAcCeEf:j:k#ln#oOpPsSwWv', command)

  if (SimpleMail.opts == nil) then
    cmd = "help"
    args = {}
  else
    cmd = table.remove(SimpleMail.opts.leftover_args, 1) or "help"
    args = SimpleMail.opts.leftover_args
  end

  SimpleMail.Debug(2, "SimpleMail (version %s) slashcmd: %s", SimpleMail.Version, cmd)
  SimpleMail.Debug(3, "cmd: '%s' args '%s' '%s' %s",
		   (cmd or "nil"),
		   (args[1] or "nil"),
		   (args[2] or "nil"),
		   (args[3] and '[...]' or ''))

  -- these commands can be used without the mailbox open.
  local func = SimpleMail.Functions[cmd]
  if (cmd == "help" or cmd == "usage" or func == nil) then
    SimpleMail.Debug(0, "usage:")
    SimpleMail.Debug(0, "/sm [flags] check [match] -- print headers")
    SimpleMail.Debug(0, "/sm [flags] get [match] -- get items/money")
    SimpleMail.Debug(0, "/sm reset -- reset state")
    SimpleMail.Debug(0, "/sm kick -- nudge SimpleMail if it seems stalled")
    SimpleMail.Debug(0, "flags:")
    SimpleMail.Debug(0, "* -a: only consider Auction House messages")
    SimpleMail.Debug(0, "* -A: exclude Auction House messages")
    SimpleMail.Debug(0, "* -c: print totals, rather than showing individual messages")
    SimpleMail.Debug(0, "* -C: accept CODs")
    SimpleMail.Debug(0, "* -e: only consider auction expirations")
    SimpleMail.Debug(0, "* -f <sender>: match sender")
    SimpleMail.Debug(0, "* -j <subject>: match subject")
    SimpleMail.Debug(0, "* -k <offset>: s(k)ip first <offset> items")
    SimpleMail.Debug(0, "* -l: only consider auctions cancelled")
    SimpleMail.Debug(0, "* -n <count>: at most <count> items")
    SimpleMail.Debug(0, "* -o: only consider auctions outbid")
    SimpleMail.Debug(0, "* -p: only consider pending sales")
    SimpleMail.Debug(0, "* -s: only consider auction successes")
    SimpleMail.Debug(0, "* -w: only consider auctions won")
    SimpleMail.Debug(0, "* -v: reject matches, accept non-matches")
    return
  elseif (cmd == "debug") then
    SimpleMail.Debug(0, "args[1] = '%s'", args[1] or "nil")
    if (args[1] == nil or args[1] == '') then
      if (SimpleMail.DebugLevel == 0) then
        SimpleMail.DebugLevel = 2
      else
        SimpleMail.DebugLevel = 0
      end
    else
      local new = tonumber(args[1])
      if (new == nil or new < 0) then
        SimpleMail.Debug(0, "/sm debug [level] -- set debug level (must be non-negative)")
	return
      else
        SimpleMail.DebugLevel = new
      end
    end
    SimpleMail.Debug(0, "Debug level set to " .. SimpleMail.DebugLevel)
    return
  elseif (cmd == "kick") then
    if (SimpleMail.Function) then
      -- restore flags, such as "auctions only", that were set previously
      SimpleMail.opts = SimpleMail.OldOpts
      SimpleMail.Pending = 1
      SimpleMail.Debug(0, "*snort* What, huh?  I wasn't sleeping!")
      SimpleMail.OnEvent("MAIL_INBOX_UPDATE")
    else
      SimpleMail.Debug(0, "I was already done!")
    end
    return
  elseif (cmd == "delay") then
    local delay = tonumber(args[1] or "-1")
    if (delay >= 0) then
      SimpleMail_Options["Delay"] = delay
      SimpleMail.Debug(0, "Delay set to %d frames.", delay)
      return
    else
      SimpleMail.Debug(0, "Couldn't find a postivive number in '%s'", args[1] or "<nil>")
    end
  elseif (cmd == "reset") then
    SimpleMail.Reset()
    SimpleMail.Debug(0, "Reset complete.")
    return
  end
  SimpleMail_Frame:UnregisterEvent("MAIL_INBOX_UPDATE")
  CheckInbox()
  SimpleMail.Items = GetInboxNumItems()
  if (SimpleMail.Items < 1) then
    SimpleMail.Debug(0, "Most SimpleMail commands operate only when the mailbox is open and has mail in it.")
    return
  end
  SimpleMail.CurrentSlot = SimpleMail.Items
  local count = SimpleMail.opts['n'] or 50
  local offset = SimpleMail.opts['k'] or 0
  SimpleMail.Offset = offset
  SimpleMail.Remaining = count
  SimpleMail.Match = match or ""
  SimpleMail.MatchInvert = SimpleMail.opts['v']
  SimpleMail.Debug(0, "Command %s, range %d-%d, match %s'%s'",
  	cmd,
	offset + 1, offset + count,
	(SimpleMail.MatchInvert and "(anything-but) ") or "",
	SimpleMail.Match or "<empty>")
  SimpleMail.Function = func
  SimpleMail.Count = 0
  SimpleMail.TotalMoney = 0
  SimpleMail.TotalCount = 0
  SimpleMail.TotalItems = 0
  -- And now we begin the handling cycle, by registering for mailbox
  -- events:
  SimpleMail.GetInfo(SimpleMail.CurrentSlot)
  SimpleMail_Frame:RegisterEvent("MAIL_INBOX_UPDATE")
  SimpleMail.Pending = 0
  SimpleMail.OnEvent("MAIL_INBOX_UPDATE")
end

function SimpleMail.OnUpdate()
  SimpleMail.OnEvent("DELAY_EVENT")
end

function SimpleMail.Reset()
  this:SetScript("OnUpdate", nil)
  SimpleMail_Frame:UnregisterEvent("BAG_UPDATE")
  SimpleMail_Frame:UnregisterEvent("MAIL_INBOX_UPDATE")
  SimpleMail.CurrentSlot = nil
  SimpleMail.CurrentSubSlot = nil
  SimpleMail.Remaining = nil
  SimpleMail.Match = nil
  SimpleMail.Function = nil
  SimpleMail.Count = 0
  SimpleMail.itemCount = 0
  SimpleMail.TotalCount = 0
  SimpleMail.TotalItems = 0
  SimpleMail.TotalMoney = 0
  SimpleMail.ExtraDelay = false
  CheckInbox()
  SimpleMail.Items = GetInboxNumItems()
end

function SimpleMail.OnEvent(event, arg1, arg2, arg3)
  if (SimpleMail.Pending > 1) then
    -- How many pending requests do we have?
    SimpleMail.Pending = SimpleMail.Pending - 1
    SimpleMail.Debug(1, "Disregarding pending %s.  Remaining: %d", event, SimpleMail.Pending)
    return
  end
  if (SimpleMail.ExtraDelay) then
    SimpleMail.ExtraDelay = false
    SimpleMail.Pending = SimpleMail_Options["Delay"] or 15
    this:SetScript("OnUpdate", SimpleMail.OnUpdate)
    return
  end
  SimpleMail.Debug(2, "Got event %s, blocking", event)
  this:SetScript("OnUpdate", nil)
  SimpleMail_Frame:UnregisterEvent("MAIL_INBOX_UPDATE")
  SimpleMail_Frame:UnregisterEvent("BAG_UPDATE")
  -- nothing to do
  if (SimpleMail.CurrentSlot == nil) then
    SimpleMail.Reset()
    return
  end
  if (SimpleMail.CurrentSlot <= SimpleMail.Offset) then
    SimpleMail.Reset()
    return
  end
  CheckInbox()
  SimpleMail.Items = GetInboxNumItems()
  if (SimpleMail.CurrentSlot > SimpleMail.Items) then
    SimpleMail.Reset()
    return
  end

  local header
  local doanother = 1

  repeat
    local match
    repeat
      match = true
      if (SimpleMail.CurrentSlot <= SimpleMail.Offset) then
	break
      end
      if (SimpleMail.sender == nil) then
	SimpleMail.Debug(0, "No sender for slot " .. SimpleMail.CurrentSlot .. " with " .. SimpleMail.Remaining .. " left.")
	SimpleMail.Reset()
	return
      end
      SimpleMail.TotalCount = SimpleMail.TotalCount + 1
      if (SimpleMail.opts['f'] and not string.find(SimpleMail.sender, SimpleMail.opts['f'])) then
	match = SimpleMail.opts['v'] or false
      end
      if (SimpleMail.opts['j'] and not string.find(SimpleMail.subject, SimpleMail.opts['j'])) then
	match = SimpleMail.opts['v'] or false
      end
      header = SimpleMail.sender .. ":" .. SimpleMail.subject
      if ((SimpleMail.opts['a'] and not SimpleMail.AuctionType) or
          (SimpleMail.opts['A'] and SimpleMail.AuctionType)) then
	match = false
      end
      if (SimpleMail.opts['C'] and SimpleMail.CODAmount == 0) then
	match = false
      end
      if (SimpleMail.opts['e'] and SimpleMail.AuctionType ~= 'expired') then
        match = false
      elseif (SimpleMail.opts['E'] and SimpleMail.AuctionType == 'expired') then
        match = false
      elseif (SimpleMail.opts['s'] and SimpleMail.AuctionType ~= 'successful') then
        match = false
      elseif (SimpleMail.opts['S'] and SimpleMail.AuctionType == 'successful') then
        match = false
      elseif (SimpleMail.opts['w'] and SimpleMail.AuctionType ~= 'won') then
        match = false
      elseif (SimpleMail.opts['W'] and SimpleMail.AuctionType == 'won') then
        match = false
      elseif (SimpleMail.opts['l'] and SimpleMail.AuctionType ~= 'cancelled') then
        match = false
      elseif (SimpleMail.opts['L'] and SimpleMail.AuctionType == 'cancelled') then
        match = false
      elseif (SimpleMail.opts['o'] and SimpleMail.AuctionType ~= 'outbid') then
        match = false
      elseif (SimpleMail.opts['O'] and SimpleMail.AuctionType == 'outbid') then
        match = false
      elseif (SimpleMail.opts['p'] and SimpleMail.AuctionType ~= 'pending') then
        match = false
      elseif (SimpleMail.opts['P'] and SimpleMail.AuctionType == 'pending') then
        match = false
      end
      if (match) then
        if (SimpleMail.Match ~= nil and
	    ((not string.find(header, SimpleMail.Match)) == (not SimpleMail.opts['v']))) then
	  SimpleMail.Debug(1, "'%s' did%s match '%s'",
	    header,
	    SimpleMail.opts['v'] and "n't" or "",
	    SimpleMail.Match)
	  match = false
        end
      else
	SimpleMail.Debug(3, "Ignoring '%s' due to flags.", header)
      end
      if (not match) then
	SimpleMail.CurrentSlot = SimpleMail.CurrentSlot - 1
        SimpleMail.GetInfo(SimpleMail.CurrentSlot)
      end
    until (match)
    if (SimpleMail.CurrentSlot <= SimpleMail.Offset) then
      break
    end
    doanother = SimpleMail.Function(SimpleMail.CurrentSlot)
    -- negative return means to come back to this slot -- e.g. for attachments
    if (doanother < 0) then
      SimpleMail.TotalCount = SimpleMail.TotalCount - 1
      return
    else
      SimpleMail.Remaining = SimpleMail.Remaining - 1
      SimpleMail.CurrentSlot = SimpleMail.CurrentSlot - 1
      SimpleMail.GetInfo(SimpleMail.CurrentSlot)
      SimpleMail.Count = SimpleMail.Count + 1
    end
  until (doanother <= 0) or (SimpleMail.Remaining < 1) or (SimpleMail.CurrentSlot <= SimpleMail.Offset)
  SimpleMail_Frame:RegisterEvent("MAIL_INBOX_UPDATE")
  if (SimpleMail.Remaining < 1 or SimpleMail.CurrentSlot <= SimpleMail.Offset) then
    SimpleMail.Debug(0, "Matched %d/%d messages.", SimpleMail.Count, SimpleMail.TotalCount)
    if (SimpleMail.TotalMoney ~= 0) then
      SimpleMail.Debug(0, "Total is %s.",
      	SimpleMail.PrettyPrint(SimpleMail.TotalMoney))
    end
    if (SimpleMail.TotalItems ~= 0) then
      SimpleMail.Debug(0, "Total of %d item%s.", SimpleMail.TotalItems,
      	SimpleMail.TotalItems == 1 and "" or "s")
    end
    SimpleMail.Reset()
  end
end

function SimpleMail.OnLoad()
  -- start out with empty set, to be corrected when variables are loaded
  SimpleMail_Options = { }
  SlashCmdList["SIMPLEMAIL"] = SimpleMail.OnSlash
  SimpleMail.Reset()
end

SLASH_SIMPLEMAIL1 = "/simplemail"
SLASH_SIMPLEMAIL2 = "/sm"
