-- $Date: 2008-07-08 21:07:00 -0400 (Tue, 08 Jul 2008) $

SBF = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceConsole-2.0", "AceDB-2.0")
SBF:RegisterDB("SBFDB", nil, "char")
local SML = LibStub and LibStub:GetLibrary('LibSharedMedia-3.0')

SBF._beta = false
SBF.minor = 8
SBF.revision = tonumber(("$Revision: 78120 $"):match("%d+"))

-- This is a bitmask
debugMask = {
  [1] = "General",
  [2] = "Buff",
  [4] = "Frame",
  [8] = "Buff name",
  [256] = "Table recycling",
}
SBF.debug = 0

local _G = getfenv(0)
local _GetPlayerBuff = GetPlayerBuff
local _GetPlayerBuffName = GetPlayerBuffName
local _GetPlayerBuffApplications = GetPlayerBuffApplications
local _GetPlayerBuffTimeLeft = GetPlayerBuffTimeLeft
local _GetPlayerBuffTexture = GetPlayerBuffTexture
local _GetWeaponEnchantInfo = GetWeaponEnchantInfo
local _GetPlayerBuffDispelType = GetPlayerBuffDispelType
local _GetCursorPosition = GetCursorPosition

SBF.ENCHANT = 0
SBF.TOTEM = 1
SBF.HELPFUL = 10
SBF.HARMFUL = 20

SBF.debugmsg = function(self, level, fmt, ...)
  if (self.debug > 0) and (bit.band(level, self.debug) == level) then
    ChatFrame3:AddMessage(format(format("|cff00ff00SBF:|r %s", fmt), ...))
  end
end

SBF.msg = function(self, fmt, ...)
  ChatFrame1:AddMessage(format(format("|cff00ff00SBF:|r %s", fmt), ...))
end

SBF.maxBuffs = { [10] = 32, [20] = 16 }

SBF.OnInitialize = function(self)
	self.frameDelay = 0.6
  self.updateDelay = 1.0
  self.timeLeftIncrement = 0.5
	self.alphaStep = 0.1
	self.tables = {}
  self.updateFrame = CreateFrame("Frame")
  self.flash = 0
  self.barsShowing = 0
  
	if self._beta then
		self.versionStr = self.version.." beta "..self.minor
	else
		self.versionStr = self.version.."."..self.minor
	end
	
	self.sortOptions = {
		self.NoSort,
		self.AscendingTimeSort,
		self.DescendingTimeSort,
		self.AscendingNameSort,
		self.DescendingNameSort,
	}

  self.player = UnitName("player")
	
	self:RegisterChatCommand("/sbf", {
		type = 'group',
		args = {
			options = {
        name = "Options", desc = SBF.strings.OPTIONS, type = 'execute',
        func = "OpenOptions",
			},
      hide = {
        name = "Hide", desc = SBF.strings.HIDE, type = 'execute',
        func = "DisableDefaultBuffFrame",
      },
      safecall = {
        name = "Safecall", desc = SBF.strings.SAFECALL, type = 'toggle',
        get = function()
          return self.db.account.safecall
        end,
        set = function(v)
          self.db.account.safecall = v
        end,
      },
			debug = {
        name = "Debug", desc = "Debug level bitmask", type = 'range',
        min = 0, max = 512,
        get = function()
          return SBF.debug
        end,
        set = function(v)
          SBF.debug = v
          for i,str in pairs(debugMask) do
            if (bit.band(v, i) == i) then
              SBF:msg("%s debugging enabled", str)
            end
          end
        end
			},
    },
	})
    
	self.none = "none"
	self.frames = {}
	self.buffs = {}
	self.debuffs = {}
	
	self.flashTime = 0
	self.flashState = 1
	self.alpha = 1
	self.lastBarUpdate = self.db.account.refresh
  
	self.classDispel = {
		MAGE = { self.strings.CURSE },
		PRIEST = { self.strings.DISEASE, self.strings.MAGIC },
		DRUID = { self.strings.CURSE, self.strings.POISON},
		PALADIN = { self.strings.POISON, self.strings.DISEASE, self.strings.MAGIC },
		SHAMAN = { self.strings.DISEASE, self.strings.POISON },
		WARRIOR = {},
		ROGUE = {},
		HUNTER = {},
		WARLOCK = {},
	}
	
	SML:Register("sound", SBF.strings.sounds[1], [[Sound\Spells\AntiHoly.wav]])
	SML:Register("sound", SBF.strings.sounds[2], [[Sound\interface\iTellMessage.wav]])
	SML:Register("sound", SBF.strings.sounds[3], [[Sound\interface\AuctionWindowOpen.wav]])
	SML:Register("sound", SBF.strings.sounds[4], [[Sound\interface\FriendJoin.wav]])
	SML:Register("sound", SBF.strings.sounds[5], [[Sound\Creature\Murloc\mMurlocAggroOld.wav]])
end

SBF.OpenOptions = function(self)
	local loaded, message = LoadAddOn("SBFOptions")
	if loaded then
    if not SBFOptions.minor then
      self:msg(self.strings.OLDOPTIONS)
      return
    end
    if (SBF.version ~= SBFOptions.version) or ((SBF.version == SBFOptions.version) and (SBF.minor ~= SBFOptions.minor)) then
      self:msg(self.strings.OPTIONSVERSION, SBFOptions.versionStr, SBF.versionStr)
      return
    end
    self:EndEvents()
    self:ClearBuffFrames()
    self:ClearBuffs()
		SBFOptions:ShowOptions()
	else
		self:msg(self.strings.CONFIGERROR, message)
	end
end

SBF.CloseOptions = function(self)
	collectgarbage()
	self.showingOptions = nil
 	self:GetExtents()
	self:SetRefresh()
  self:ForceGet()
end

SBF.OnDisable = function(self)
	self:msg(self.strings.DISABLE)
end

SBF.OnEnable = function(self)
	self:DisableDefaultBuffFrame()
	_,self.playerClass = UnitClass("player")
	self:DoSavedVars()
	self:CreateFrames()
  self:SetupFrames()
	self:SetupEnchants()

  if IsAddOnLoaded("ButtonFacade") then
    local bf = LibStub("AceAddon-3.0"):GetAddon("ButtonFacade")
    bf:GetModule("SBF"):Load()
  end

  self:SetRefresh()
 	self:GetExtents()
	self:RegisterEvent("PLAYER_AURAS_CHANGED", "ForceGet")
  self:RegisterEvent("PARTY_MEMBERS_CHANGED", "ForceGet")
  self:RegisterEvent("RAID_ROSTER_UPDATE", "ForceGet")
	self:RegisterEvent("UNIT_INVENTORY_CHANGED", "InventoryChanged")
  self:RegisterEvent("MINIMAP_UPDATE_TRACKING", "UpdateTracking")
  self:RegisterEvent("PLAYER_ENTERING_WORLD", "EnterWorld")
  self.updateFrame:SetScript("OnUpdate", self.Updates)

  self.hasTotems = (self.playerClass == "SHAMAN")
  if self.hasTotems then
    self:RegisterEvent("PLAYER_TOTEM_UPDATE", "TotemUpdate")
  end

  self:UpdateTracking()
  self:ForceGet()
end


SBF.ToggleActive = function(self)
  -- Don't allow the user to try and disable SBF
	self:msg(self.strings.DISABLE)
end

SBF.EndEvents = function(self)
  self:CancelScheduledEvent("SBFBuffUpdate")
  self:CancelScheduledEvent("SBFBarRefresh")
  self:CancelScheduledEvent("SBFTimerRefresh")
  self:CancelScheduledEvent("SBFTimeLeftRefresh")
end

SBF.SetRefresh = function(self)
  self.checkVisibility = false
  for k,v in ipairs(SBF.db.profile.frames) do
    if v.frameVisibility and (v.frameVisibility > 1) then
      self.checkVisibility = true
    end
  end
  if self.checkVisibility then
    self:ScheduleRepeatingEvent("SBFVisibility", self.FrameUpdate, self.frameDelay, self)
  else
    self:CancelScheduledEvent("SBFVisibility")
  end
  self:ScheduleRepeatingEvent("SBFBuffUpdate", self.Update, self.updateDelay, self)
  self:ScheduleRepeatingEvent("SBFTimerRefresh", self.UpdateTimers, self.updateDelay, self)
  self:ScheduleRepeatingEvent("SBFTimeLeftRefresh", self.UpdateTimeLeft, self.timeLeftIncrement, self)
end

SBF.Updates = function()
  if (SBF.barsShowing > 0) then
    SBF:UpdateBars(arg1)
  end
  if (SBF.flash > 0) then
    SBF:UpdateAlpha(arg1)
  end
end

SBF.ResetFrames = function(self, openOptions)
  for i=1,#self.db.profile.frames do
    self:PutTable(self.db.profile.frames[i])
    self.db.profile.frames[i] = nil
  end
  self:DoSavedVars()
  self:CreateFrames()
  if openOptions then
    SBFOptions:ShowOptions()
  end
end

SBF.InventoryChanged = function(self)
  if (arg1 == "player") then
    local update = false
    local itemLink, itemID
    for _,i in pairs(self.enchantID) do
      itemLink = GetInventoryItemLink("player", i)
      if itemLink then
        itemID = string.match(itemLink, "Hitem:(.-):")
        if self.enchants[i].itemID and (self.enchants[i].itemID ~= itemID) then
          update = true
          self.enchants[i].update = true
        end
      end
    end
    if update then
      self:ForceGet()
    end
  end
end

SBF.EnterWorld = function(self)
  if self.hasTotems then
    self:TotemUpdate()
  end
end

SBF.ForceGet = function(self)
  self.getBuffs = true
  self:Update()
end

SBF.RefreshBuffs = function(self)
  self:ClearBuffFrames()
  self:ClearBuffs()
	self.getBuffs = true
end

SBF.Update = function(self)
  if not self.reassignFrames then
    self:Safecall(self.UpdateEnchants)
  end
	if self.getBuffs or self.reassignFrames then
		self:Safecall(self.GetBuffs)
		self.getBuffs = false
		self.reassignFrames = false
	end
  self:Safecall(self.UpdateBuffs)
  if self.needSetup then
    self:SetupSlots()
    self.needSetup = false
  end
end

SBF.FrameUpdate = function(self)
	self:FrameVisibility()
end

SBF.UpdateAlpha = function(self, elapsed)
	self.flashTime = self.flashTime - elapsed
	if (self.flashTime < 0) then
		local overtime = -self.flashTime
		if self.flashState then
			self.flashState = nil
			self.flashTime = BUFF_FLASH_TIME_OFF
		else
			self.flashState = 1
			self.flashTime = BUFF_FLASH_TIME_ON
		end
		if (overtime < self.flashTime) then
			self.flashTime = self.flashTime - overtime
		end
	end

	if self.flashState then
		self.alpha = (BUFF_FLASH_TIME_ON - self.flashTime) / BUFF_FLASH_TIME_ON
	else
		self.alpha = self.flashTime / BUFF_FLASH_TIME_ON
	end
	self.alpha = (self.alpha * (1 - BUFF_MIN_ALPHA)) + BUFF_MIN_ALPHA
	if enchTimer then
		enchTimer = enchTimer - elapsed
	end
end

--
-- Error functions
--
SBF.perrCount = 0
SBF.perr = function(msg)
  SBF.perrCount = SBF.perrCount + 1
  if (SBF.perrCount < 10) then
    local m = string.match(msg, ".+: (.+)")
    local file, line = string.match(debugstack(2, 1, 0), "(.+):(%d+):")
    if file and line then
      self:msg(SBF.strings.ERRFMT, file, line, m)
    else
      self:msg(msg)
    end
  elseif (SBF.perrCount == 10) then
    for k,v in ipairs(SBF.strings.MANYFAULT) do
      self:msg(v)
    end
  end
end

SBF.Error = function(self, fmt, ...)
  local file,line = string.match(debugstack(2,1,0), ".+\\(.-):(%d+):")
  local f = string.format(self.strings.ERRFMT, file, line, fmt)
  self:msg(f, ...)
end

SBF.Safecall = function(self, func)
  if self.db.account.safecall then
    _G.xpcall(func, self.perr)
  else
    func()
  end
end

--
-- Buff Functions
-- 

-- Buff expired
SBF.ExpirationNotice = function(self, buff)
  local var = buff._var
	if var and var.textNotice and buff.warnExpire and buff.timeLeft and (buff.expFloor == 0) then
		local frame = getglobal(var.expireFrame or "ChatFrame1")
		frame:AddMessage(string.format("|cff00ff00%s|r %s", tostring(buff.name), self.strings.BUFFNOTICE))
	end
end

-- Buff about to expire
SBF.ExpiryWarning = function(self, buff)
  local var = buff._var

  if var and buff.warnExpire and not buff.warned and not buff.untilCancelled and not UnitIsDead("player") 
	and not UnitIsGhost("player") and (buff.expFloor == var.expireTime) then
    buff.warned = 1
    
    if not var.debuffTimerColour then
      buff._slot.timer.text:SetVertexColor(var.expireTimerColour.r, var.expireTimerColour.g, var.expireTimerColour.b)
    end
    if var.flashBuff then
      self.flash = self.flash + 1
      buff._slot.button:SetScript("OnUpdate", self.BuffButton_OnUpdate)
    end
    if var.textWarning then
			local frame = getglobal(var.expireFrame or "ChatFrame1")
			frame:AddMessage(string.format("|cff00ff00%s|r %s", tostring(buff.name), self.strings.BUFFEXPIRE))
		end
		if var.sctWarn then
			self:SCTExpireMessage(buff)
		end
		if var.soundWarning then
			local s = SML:Fetch("sound", var.sound)
			if s then
				PlaySoundFile(s)
			end
		end
	end
end

SBF.UpdateBuffs = function(self, f)
  if not self then
    self = SBF
  end
	if (#self.buffs == 0) or self.getBuffs then
    return
  end
  
  local var,f
  
	for i,frame in ipairs(self.frames) do
    if not f or (frame == f) then
      var = frame._var
      for index,slot in ipairs(frame.slots) do
        buff = slot.buff
        if not buff then
          break
        end
        self:GetBuffInfo(buff)

        if not self.showingOptions and self.db.profile.enableRFilters and not buff.isMirror then
          if (self:GetFrame(buff) ~= buff.frame) then
            self.reassignFrames = true
            return
          end 
        end
        
        if buff.last and (buff.expFloor > buff.last + 2) then
          self.needSetup = true
          self:BuffDuration(buff)
          if slot.button then
            slot.button:SetAlpha(var.opacity)
            if (buff.expFloor > var.expireTime) then
              self.flash = math.max(self.flash - 1, 0)
              slot.button:SetScript("OnUpdate", nil)
              buff.warned = nil
            end
          end
          if slot.bar then
            buff.setFast = false
            slot.bar:SetMinMaxValues(0, buff.duration)
            slot.bar.spark:Hide()
          end
          if slot.timer then
            buff._slot.timer.text:SetVertexColor(var.regularTimerColour.r, var.regularTimerColour.g, var.regularTimerColour.b)
          end
        end
        self:ExpiryWarning(buff)
      end
    end
	end
end

SBF.GetBuffInfo = function(self, buff, init)
  if not self.showingOptions then
    buff.last = buff.expFloor
    buff.lastCount = buff.count
    if (buff.buffType == self.ENCHANT) or (buff.buffType == self.TOTEM) then
      return 
    end
    if buff.index and buff.type then
      _GetPlayerBuff(buff.index, buff.type)
      if buff.untilCancelled then
        buff.timeLeft = buff.maxTime and 9999 or 0
      else
        buff.timeLeft = _GetPlayerBuffTimeLeft(buff.index)
      end
      buff.expFloor = floor(buff.timeLeft)
      buff.count = _GetPlayerBuffApplications(buff.index)

      if init then
        buff.icon = _GetPlayerBuffTexture(buff.index)
        if buff.count and (buff.count > 1) then
          buff.hadCount = true
        end
        if (buff.type == "HARMFUL") then
          buff.buffType = self.HARMFUL
          buff.dispelType = _GetPlayerBuffDispelType(buff.index) or self.none
        else
          buff.buffType = SBF.HELPFUL
        end
      end
    end
  end
end

SBF.IterateBuffs = function(self, buffList, buffType)
	local i,k,index,untilCancelled,buff,name,rank,exists

	for i,buff in ipairs(buffList) do
		buff.index = -1
	end
  local t = self.HELPFUL
  if (buffType == "HARMFUL") then
    t = self.HARMFUL
  end
	for i=1,self.maxBuffs[t] do
		exists = false
		index, untilCancelled = _GetPlayerBuff(i, buffType)
		if (index == 0) then
      break
    else
			name, rank = _GetPlayerBuffName(index)
			if not self:IsExcluded(name) then
				for k,buff in ipairs(buffList) do
					if (name == buff.name) and (buff.index == -1) then
						exists = true
						buff.index = index
            self:GetBuffInfo(buff, true)
            self:BuffDuration(buff)
            self:debugmsg(3, format("Refreshing buff |cff00ffaa%s|r", buff.name))
						break
					end
				end
 
				if not exists then
					buff = self:GetTable()
					buff.type = buffType
					buff.name = name
          buff.rank = tonumber(rank:match("%d+"))
					buff.index = index 
					buff.untilCancelled = (untilCancelled == 1)
          if buff.untilCancelled then
            buff.maxTime = self.db.profile.auraMaxTime
          end
          self:GetBuffInfo(buff, true)
          self:BuffDuration(buff)
          table.insert(buffList, buff)
				end
			end
		end
	end

	local clear
  repeat
    clear = true
    for i,buff in ipairs(buffList) do
      if (buff.index < 0) then
        clear = false
        self:EndBuff(table.remove(buffList, i))
      end
    end
  until clear
end

SBF.ClearBuffs = function(self)
  while (#self.buffs > 0) do
    self:EndBuff(table.remove(self.buffs), true)
  end
end

SBF.GetBuffs = function(self)
	if not self then
    self = SBF
  end
  if self.showingOptions then
		return
	end
  self:debugmsg(2, "Get Buffs")
	self:ClearBuffFrames()
	if self.getBuffs then
    self:RemoveEnchants()
    self:RemoveTotems()
    self:IterateBuffs(self.buffs, "HELPFUL")
    self:IterateBuffs(self.debuffs, "HARMFUL")
    self:AddTotems()
    self:AddEnchants()
    self:AddTrackingBuff()
  end
	self:AssignFrames()
  self:SetupSlots()
end

SBF.EndBuff = function(self, buff, quiet)
  if buff._slot then
    if buff._slot.button then
      buff._slot.button.buff = nil
    end
    buff._slot.buff = nil
  end
  if not buff.isMirror then
    if (buff.buffType < self.HARMFUL) and not quiet then
      SBF:ExpirationNotice(buff)
    end
    if (buff.buffType > self.ENCHANT) then
      self:debugmsg(1, format("Ending buff |cff00ffaa%s|r", buff.name))
      self:PutTable(buff)
    end
  end
end

SBF.AssignFrames = function(self)
  for i,buff in ipairs(self.buffs) do
    self:AssignFrame(buff)
  end
  for i,buff in ipairs(self.debuffs) do
    self:AssignFrame(buff)
  end
end

SBF.AssignFrame = function(self, buff)
  if not buff.isMirror then
    buff.frame = self:GetFrame(buff)
  end
  local frame = self.frames[buff.frame]
  -- !! Don't change this one to buff._var !! --
  local var = self.db.profile.frames[buff.frame]
  
  if not self.showingOptions then
    if (#frame.buffs == var.count) and (buff.frame > 2) then
      if (buff.buffType < self.HARMFUL) then
        frame = self.frames[1]
        var = self.db.profile.frames[1]
      else
        frame = self.frames[2]
        var = self.db.profile.frames[2]
      end
    elseif var.mirrorBuffs then
      if not buff.mirror then
        buff.mirror = self:CopyTable(buff)
      end
      buff.mirror.isMirror = true
      if (buff.buffType < self.HARMFUL) then
        buff.mirror.frame = 1
      else
        buff.mirror.frame = 2
      end
      self:AssignFrame(buff.mirror)
    elseif buff.mirror then
      self:PutTable(buff.mirror)
      buff.mirror = nil
    end
  end
  self:debugmsg(4, format("Assigning |cff00ffaa%s|r to frame |cff00aaff%d|r", buff.name, buff.frame))
  table.insert(frame.buffs, buff)
  buff._var = var
end

SBF.SetupSlots = function(self, f)
  local buff, var, colour
  self.barsShowing = 0
  for i,frame in ipairs(self.frames) do
    if not f or (i == frame) then
      self:SetupFrameSlots(frame)
    end
  end
end

SBF.SetupFrameSlots = function(self, frame)
  local var = frame._var
  if (var.sortType > 1) and self.sortOptions[var.sortType] then
    table.sort(frame.buffs, self.sortOptions[var.sortType])
  end
  for j,slot in ipairs(frame.slots) do
    buff = frame.buffs[j]
    if buff then
      slot.buff = buff
      buff._slot = slot

      buff.warnExpire = not buff.untilCancelled and (self:IsAlwaysWarn(buff.name) or (self:GetSpell(buff.name) >= var.warnTime))

      if slot.timer then
        if (var.timerFormat < 5) and not buff.untilCancelled then
          self:SetBuffTime(slot.timer, buff.timeLeft, var.timerFormat)
          if (buff.buffType == self.HARMFUL) and var.debuffTimerColour then
            colour = DebuffTypeColor[buff.dispelType]
          elseif buff.warnExpire and (buff.timeLeft <= var.expireTime) then
            colour = var.expireTimerColour
            if var.flashBuff then
              self.flash = self.flash + 1
              buff._slot.button:SetScript("OnUpdate", self.BuffButton_OnUpdate)
            end
          else
            colour = var.regularTimerColour
          end
          slot.timer.text:SetVertexColor(colour.r, colour.g, colour.b)
          slot.timer:Show()
        else
          slot.timer:Hide()
        end
      end

      if slot.button then
        buff._slot.button.buff = buff  -- so OnEnter can do the tooltip without backflips
        slot.button.icon:SetTexture(buff.icon)
        slot.button:SetAlpha(var.opacity)
        slot.button:Show()
        if var.countInName or not buff.hadCount then
          slot.button.count:Hide()
        elseif buff.hadCount then
          slot.button.count:Show()
          slot.button.count:SetFormattedText(buff.count)
        end
      end

      if slot.bar then
        slot.bar.spark:Hide()
        if var.countInName and buff.hadCount then
          slot.bar.text:SetFormattedText("%s (%d)", var.shortNames and self:ShortName(buff.name) or buff.name, buff.count)
        else
          slot.bar.text:SetFormattedText(var.shortNames and self:ShortName(buff.name) or buff.name)
        end
        if buff.untilCancelled then
          slot.bar:SetMinMaxValues(0, 9999)
        elseif var.fastBar and (buff.duration >= var.expireTime) and (buff.expFloor <= var.expireTime) then
          slot.bar:SetMinMaxValues(0, var.expireTime)
        else
          slot.bar:SetMinMaxValues(0, buff.duration)
        end
        slot.bar:SetValue(buff.timeLeft)
        slot.bar:Show()
        if not (buff.isTracking or buff.untilCancelled) then
          self.barsShowing = self.barsShowing + 1
        end
      end
    
      if (buff.buffType == self.HARMFUL) then
        if slot.bar then
          if var.debuffBarColour then
            slot.bar:SetStatusBarColor(DebuffTypeColor[buff.dispelType].r,DebuffTypeColor[buff.dispelType].g,DebuffTypeColor[buff.dispelType].b,var.barDebuffColour.a)
          else
            slot.bar:SetStatusBarColor(var.barDebuffColour.r,var.barDebuffColour.g,var.barDebuffColour.b,var.barDebuffColour.a)
          end
        end
        if slot.button then
          slot.button.border:Show()
          slot.button.border:SetVertexColor(DebuffTypeColor[buff.dispelType].r, DebuffTypeColor[buff.dispelType].g, DebuffTypeColor[buff.dispelType].b)
        end
      else
        if slot.bar then
          slot.bar:SetStatusBarColor(var.barBuffColour.r,var.barBuffColour.g,var.barBuffColour.b,var.barBuffColour.a)
        end
        if slot.button then
          slot.button.border:Hide()
        end
      end
    else
      if slot.bar then
        slot.bar:Hide()
      end
      if slot.timer then
        slot.timer:Hide()
      end
      if slot.button then
        slot.button:Hide()
      end
    end
  end
end


SBF.RemoveAlwaysWarn = function(self, name)
	for i,v in pairs(SBF.db.profile.alwaysWarnList) do
		if (v == name) then
			table.remove(SBF.db.profile.alwaysWarnList, i)
			return
		end
	end
end

SBF.AddAlwaysWarn = function(self, name)
	for i,v in pairs(SBF.db.profile.alwaysWarnList) do
		if (v == name) then
			return
		end
	end
	table.insert(SBF.db.profile.alwaysWarnList, name)
end

SBF.IsAlwaysWarn = function(self, name)
	for i,v in ipairs(self.db.profile.alwaysWarnList) do
		if (v == name) then
			return true
		end
	end
	return false
end

SBF.AddExclude = function(self, name)
	for i,v in pairs(SBF.db.profile.exclusionList) do
		if (v == name) then
			return
		end
	end
	table.insert(SBF.db.profile.exclusionList, name)
end

SBF.RemoveExclude = function(self, name)
	for i,v in pairs(SBF.db.profile.exclusionList) do
		if (v == name) then
			table.remove(SBF.db.profile.exclusionList, i)
			return
		end
	end
end

SBF.IsExcluded = function(self, name)
	for i,v in ipairs(self.db.profile.exclusionList) do
		if (v == name) then
			return true
		end
	end
	return false
end

SBF.ShowIn = function(self, name)
	for b,f in pairs(self.db.profile.buffFrameList) do
		if (name == b) then
			return f
		end
	end
	return nil
end

SBF.GetFrame = function(self, buff)
  local f = self:ShowIn(buff.name)
	if f and self.frames[f] then
    self:debugmsg(4, format("Assign buff |cff00ffaa%s|r to frame |cff00aaff%d|r via Show in Frame", buff.name, f))
    return f
	end
	if self.db.profile.enableFilters then
		local f, filter = self:DoFilters(buff)
		if f and self.frames[f] then
      self:debugmsg(4, format("Assign buff |cff00ffaa%s|r to frame |cff00aaff%d|r via filter |cff00ff33", buff.name, f, filter))
			return f
		end
	end
	if (buff.buffType == self.HARMFUL) then
    self:debugmsg(4, format("Assign buff |cff00ffaa%s|r to frame |cff00aaff2|r by default", buff.name))
		return 2
	end
  self:debugmsg(4, format("Assign buff |cff00ffaa%s|r to frame |cff00aaff1|r by default", buff.name))
	return 1
end

-- 
-- Buff button functions
-- 
SBF.BuffButtonDropDown_Init = function()
	local level = UIDROPDOWNMENU_MENU_LEVEL
	
	if this.buff or (UIDROPDOWNMENU_MENU_LEVEL > 1) then
		local info
		
		if (UIDROPDOWNMENU_MENU_LEVEL == 1) then
			info = UIDropDownMenu_CreateInfo()
			info.text			= SBF.menuBuff.name
			info.isTitle	= 1
			UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL)
			
			info = UIDropDownMenu_CreateInfo()
			info.text					= SBF.strings.BUFFFRAME
			info.hasArrow			= true
			UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL)

			info = UIDropDownMenu_CreateInfo()
			info.text			= SBF.strings.EXCLUDE
			info.arg1			= "exclude"
			info.func			= SBF.BuffButtonDropDown_OnClick
			UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL)

			if not this.buff.untilCancelled then
				info = UIDropDownMenu_CreateInfo()
				info.text			= SBF.strings.ALWAYSWARN
				info.arg1			= "alwayswarn"
				info.func			= SBF.BuffButtonDropDown_OnClick
				info.checked	= SBF:IsAlwaysWarn(this.buff.name)
				UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL)
			end
			
			info = UIDropDownMenu_CreateInfo()
			info.disabled = 1
			UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL)

			if (this.buff.buffType ~= SBF.HARMFUL) then
				info = UIDropDownMenu_CreateInfo()
				info.text			= SBF.strings.CANCEL
				info.arg1			= "cancelbuff"
				info.func			= SBF.BuffButtonDropDown_OnClick
				UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL)
			end
		elseif (UIDROPDOWNMENU_MENU_LEVEL == 2) then
			for i,frame in ipairs(SBF.frames) do
				info = UIDropDownMenu_CreateInfo()
				info.text = SBF.strings.SHOWATBUFFRAME..i
				info.arg1	= "frame"
				info.arg2	= i
				info.func = SBF.BuffButtonDropDown_OnClick
				info.checked = (SBF.db.profile.buffFrameList[SBF.menuBuff.name] == i)
				UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL)
			end
			info = UIDropDownMenu_CreateInfo()
			info.text = SBF.strings.NOFRAME
			info.arg1	= "clearframe"
			info.func = SBF.BuffButtonDropDown_OnClick
			UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL)
			return
		end
	end
end

SBF.BuffButtonDropDown_OnClick = function(arg1, arg2)
  if not SBF.menuBuff then
    SBF:Error(SBF.strings.ERRDROPDOWN_BUFF, arg1)
    return
  elseif not SBF.menuBuff.name then
    SBF:Error(SBF.strings.ERRDROPDOWN_NAME, arg1)
    return
  elseif not arg1 then
    SBF:Error(SBF.strings.ERRDROPDOWNopERATION, SBF.menuBuff.name)
    return
  end
  
  if (arg1 == "exclude") then
		SBF:AddExclude(SBF.menuBuff.name)
	elseif (arg1 == "alwayswarn") then
		if this.checked then
			SBF:RemoveAlwaysWarn(SBF.menuBuff.name)
		else
			SBF:AddAlwaysWarn(SBF.menuBuff.name)
		end
	elseif (arg1 == "cancelbuff") then
    if (SBF.menuBuff.buffType == SBF.ENCHANT) then
      SBF.menuBuff.cancel = SBF.enchantDebounce
    elseif SBF.menuBuff.totemSlot then
      DestroyTotem(SBF.menuBuff.totemSlot)
    else
      CancelPlayerBuff(SBF.menuBuff.name)
    end
	elseif (arg1 == "frame") then
		SBF.db.profile.buffFrameList[SBF.menuBuff.name] = arg2
	elseif (arg1 == "clearframe") then
		SBF.db.profile.buffFrameList[SBF.menuBuff.name] = nil
	end
	CloseDropDownMenus(1)
	SBF.getBuffs = true
end

SBF.BuffButton_OnUpdate = function()
  local buff = this.buff
  local var = buff._var
  if buff and var and not SBF.showingOptions and not buff.untilCancelled 
  and (floor(buff.timeLeft) <= (var.expireTime or 0)) then
    this:SetAlpha(SBF.alpha)
    if this.overlay then
      this.overlay:SetAlpha(1)
    end
    buff.update = false
  end
end

SBF.BuffButton_OnClick = function()
  if not SBF.showingOptions then
    local var = this.buff._var
    if this.buff.isTracking then
      ToggleDropDownMenu(1, nil, MiniMapTrackingDropDown, this, 0, -5)
		elseif (arg1 == "RightButton") and not IsShiftKeyDown() and not var.disableRightClick then
			if (this.buff.buffType == SBF.ENCHANT) then
				this.buff.cancel = SBF.enchantDebounce
			elseif this.buff.totemSlot then
        DestroyTotem(this.buff.totemSlot)
      else
        CancelPlayerBuff(this.buff.index)
			end
		elseif (arg1 == "LeftButton") then
      if IsShiftKeyDown() and this.dropDown then
        SBF.menuBuff = this.buff
        ToggleDropDownMenu(1, nil, this.dropDown)
      elseif IsAltKeyDown() then
        for k,v in pairs(this.buff) do
          ChatFrame3:AddMessage(tostring(k).."=>"..tostring(v))
        end
      end
		end
	end
end

SBF.ShowTooltip = function()
  GameTooltip:SetOwner(this, "ANCHOR_TOP")
  if this.buff then
    local var = this.buff._var
    if not SBF.showingOptions and not var.noTooltips then
      if (this.buff.buffType == SBF.ENCHANT) then
        if this.buff.showItem then
          GameTooltip:SetInventoryItem("player", this.buff.invID)
        else
          GameTooltip:SetText(this.buff.name)
        end
      elseif this.buff.index and this.buff.type then
        GameTooltip:SetPlayerBuff(this.buff.index, this.buff.type)
      elseif this.buff.totemSlot then
        GameTooltip:SetTotem(this.buff.totemSlot)
      end
    end
  else
    GameTooltip:SetText(SBF.strings.BUFFERROR)
  end
end

SBF.HideTooltip = function(self)
	GameTooltip:Hide()
end

SBF.UpdateTimeLeft = function(self)
  local update = false
  for index,frame in ipairs(self.frames) do
    for i,buff in ipairs(frame.buffs) do
      if buff.timeLeft and not buff.untilCancelled then
        buff.last = buff.expFloor
        -- For frames with bars, this update is done with more precision by SBF:UpdateBars
        if not frame._var.showBars then
          buff.timeLeft = max(0, buff.timeLeft - self.timeLeftIncrement)
        end
        buff.expFloor = floor(buff.timeLeft)
        if (buff.expFloor > buff.last + 2) then
          update = true
        end
      end
    end
  end
  if update then
    self:UpdateBuffs()
  end
end

-- Much faster (every OnUpdate cycle) update for frames with bars so that they animate smoothly
SBF.UpdateBars = function(self, delta)
  if not self.getBuffs then
    local var
    for i,frame in ipairs(self.frames) do
      var = frame._var
      -- This update only needs to happen this frequently for frames with no bars.  Let SBF:UpdateTimeLeft take care of the rest
      if var.showBars then
        for j,slot in ipairs(frame.slots) do
          if slot.buff and slot.buff.timeLeft and not slot.buff.untilCancelled then
            slot.buff.timeLeft = max(0, slot.buff.timeLeft - delta)
            slot.bar:SetValue(slot.buff.timeLeft)
            if var.fastBar and (slot.buff.duration > var.warnTime) and (slot.buff.expFloor <= var.expireTime) then
              if not slot.buff.setFast then
                slot.bar:SetMinMaxValues(0, var.expireTime)
                slot.bar.spark:Show()
                slot.buff.setFast = true
              end
              slot.bar.spark:SetPoint("CENTER", slot.bar, "LEFT", (slot.buff.timeLeft / var.expireTime * var.barWidth), 0)
            end
          end
        end
      end
    end
  end
end

SBF.UpdateTimers = function(self)
  if not self.getBuffs then
    for i,frame in ipairs(self.frames) do
      for j,slot in ipairs(frame.slots) do
        if slot.buff and slot.buff.expFloor and slot.timer then
          self:SetBuffTime(slot.timer, slot.buff.expFloor, frame._var.timerFormat)
        end
      end
    end
  end
end


--
-- Tracking icon
--
SBF.AddTrackingBuff = function(self)
  if not self.db.profile.showTracking then
    return
  end
  if self.db.profile.trackFirst then
    table.insert(self.buffs, 1, self.trackingBuff)
  else
    table.insert(self.buffs, self.trackingBuff)
  end
end

SBF.UpdateTracking = function(self)
  if not self.db.profile.showTracking then
    return
  end
  MiniMapTracking:Hide()
  if not self.trackingBuff then
    self.trackingBuff = {duration = 9999, untilCancelled = true, buffType = self.ENCHANT, type = "ENCHANT", isTracking = true}
  end
  if self.db.profile.auraMaxTime then
    self.trackingBuff.timeLeft = 9999
    self.trackingBuff.expFloor = 9999
  else
    self.trackingBuff.timeLeft = 0
    self.trackingBuff.expFloor = 0
  end
  
  local name, texture, active
  self.trackingBuff.name = self.strings.NOTRACKING
  self.trackingBuff.icon = [[Interface\Minimap\Tracking\None]]
  for i=1,GetNumTrackingTypes() do 
    name, texture, active = GetTrackingInfo(i)
    if active then
      self.trackingBuff.name = name
      self.trackingBuff.icon = texture
    end
  end
  if self.trackingBuff._slot then
    self.trackingBuff._slot.button.icon:SetTexture(self.trackingBuff.icon)
    if self.trackingBuff._slot.bar then
      self.trackingBuff._slot.bar.text:SetFormattedText(self.trackingBuff.name)
    end
  end
end

-- 
-- Enchant functions
-- 
SBF.enchantID = {16, 17}
SBF.SetupEnchants = function(self)
	if not self.enchants then
		self.enchants = {}
	end
	for _,i in pairs(self.enchantID) do
		if not self.enchants[i] then
			self.enchants[i] = self:GetTable()
			self.enchants[i].buffType = self.ENCHANT
			self.enchants[i].type = "ENCHANT"
			self.enchants[i].invID = i
			self.enchants[i].cancelNum = i-15
			self.enchants[i].cancel = 0
		end
	end
end


SBF.AddEnchants = function(self)
	for index,enchant in pairs(self.enchants) do
		if enchant.hasBuff and enchant.name and not self:IsExcluded(v.name) then
			self:InsertEnchant(index)
		end
	end
end

SBF.InsertEnchant = function(self, invID)
	for k,v in ipairs(self.buffs) do
		if (v.invID == invID) then
			return
		end
	end
  self.enchants[invID].frame = self:GetFrame(self.enchants[invID])
	if SBF.db.profile.enchantsFirst then
    table.insert(self.buffs, 1, self.enchants[invID])
  else
    table.insert(self.buffs, self.enchants[invID])
  end
end

SBF.RemoveEnchants = function(self)
  local removed
  while true do
    removed = false
    for k,v in ipairs(self.buffs) do
      if v.invID then
        v._slot = nil
        table.remove(self.buffs, k)
        removed = true
      end
    end
    if not removed then
      return
    end
  end
end


SBF.GetEnchantName = function(self, invID)
  SBFTooltip:SetOwner(UIParent,"ANCHOR_NONE") 
	SBFTooltip:ClearLines()
  SBFTooltip:SetInventoryItem("player", invID)
	local name, lines
  lines = SBFTooltip:NumLines()
  self:debugmsg(8, "Tooltip has %d lines", lines)
	if true then
    for i=2,lines do
      line = getglobal("SBFTooltipTextLeft"..i):GetText()
      if line then
        self:debugmsg(8, "Tooltip line %d -> %s", i, line)
        name = string.match(line, "^(.-) %(")
        if name then
          self:debugmsg(8, "Found enchant name: %s", name)
          return name, false
        end
      end
    end
  end
	local l = GetInventoryItemLink("player", invID)
	if l then
		local name = GetItemInfo(l)
		if name then
      return name, true
    end
	end
  return string.format("Temporary Enchant %d", invID), true
end

SBF.enchantDebounce = 15
SBF.UpdateEnchants = function(self)
  if not self then
    self = SBF
  end
  local needGet = false
  
	if not self.showingOptions then
		self.enchants[16].last = self.enchants[16].timeLeft
		self.enchants[17].last = self.enchants[17].timeLeft
		self.enchants[16].hasBuff, self.enchants[16].timeLeft, self.enchants[16].count, 
		self.enchants[17].hasBuff, self.enchants[17].timeLeft, self.enchants[17].count = _GetWeaponEnchantInfo()
		
		for k,v in pairs(self.enchants) do
			if v.hasBuff and (v.cancel == 0) then
				v.timeLeft = v.timeLeft/1000
        v.expFloor = floor(v.timeLeft)
				if v.update or not v.name or not v.last or (v.expFloor > v.last + 2) then
          v.icon = GetInventoryItemTexture("player", v.invID)
          v.name, v.showItem = self:GetEnchantName(v.invID)
          if v.name then
            self:BuffDuration(v)
            v.itemID = string.match(GetInventoryItemLink("player", v.invID), "Hitem:(.-):")
            v.frame = self:GetFrame(v)
            v._var = self.db.profile.frames[v.frame]
            v.warnExpire = (self:IsAlwaysWarn(v.name) or (self:GetSpell(v.name) >= v._var.warnTime))
            self.getBuffs = true
          end
          v.update = false
        end
			else
				if v.last then
          local var = v._var
          if var.sctWarn then
						if SCT then
              local frame = SCT:Get("SHOWFADE", SCT.FRAMES_TABLE) or 1
              local txt = string.format("-[%s]", tostring(v.name))
              if (frame == SCT.MSG) then
                SCT:DisplayMessage(txt, var.sctColour)
              else
                SCT:DisplayText(txt, var.sctColour, var.sctCrit, "event", frame)
              end
						elseif MikSBT then
							MikSBT.DisplayMessage(string.format("-[%s]", tostring(v.name)), MikSBT.DISPLAYTYPE_NOTIFICATION, false, 
                var.sctColour.r*255, var.sctColour.g*255, var.sctColour.b*255)
						elseif (getglobal("SHOW_COMBAT_TEXT") == "1") then
							CombatText_AddMessage(string.format(AURA_END, tostring(v.name)), COMBAT_TEXT_SCROLL_FUNCTION, 0.1, 1.0, 0.1, nil, nil)
						elseif Parrot then
							Parrot:GetModule("Display"):CombatText_AddMessage("-("..tostring(v.name)..")", nil, var.sctColour.r, var.sctColour.g, var.sctColour.b, 
								nil, nil, nil, v.icon)
						end
					end
					self.getBuffs = true
				end
				if (v.cancel > 0) then
					if (v.cancel == self.enchantDebounce) then
						CancelItemTempEnchantment(v.cancelNum)
					end
					v.cancel = v.cancel - 1
				end
        v.hasBuff = false
				v.last = nil
				v.name = nil
				v.count = nil
				v.warned = nil
				v.timeLeft = nil
        v.itemID = nil
        v.filterName = nil
        self:PutTimer(v.timer)
        self:PutBar(v.bar)
        self:PutButton(v.button) 
			end
		end
	end
end

-- 
-- Support Functions
-- 

SBF.SCTExpireMessage = function(self, buff)
	if not buff.frame and not buff.name then
		self:msg(SBF.strings.INVALIDBUFF)
		return
	end
	local v = buff._var
	if v then
    if SCT then
      local frame = SCT:Get("SHOWFADE", SCT.FRAMES_TABLE) or 1
      local txt = string.format("[%s] %s", tostring(buff.name), self.strings.BUFFEXPIRE)
      if (frame == SCT.MSG) then
        SCT:DisplayMessage(txt, v.sctColour)
      else
        SCT:DisplayText(txt, v.sctColour, v.sctCrit, "event", frame, nil, nil, buff.icon)
      end
    elseif MikSBT then
      MikSBT.DisplayMessage(string.format("%s %s", tostring(buff.name), self.strings.BUFFEXPIRE), MikSBT.DISPLAYTYPE_NOTIFICATION, 
      v.sctCrit, v.sctColour.r*255, v.sctColour.g*255, v.sctColour.b*255)
    elseif (getglobal("SHOW_COMBAT_TEXT") == "1") then
      CombatText_AddMessage(string.format("<%s> %s", tostring(buff.name), self.strings.BUFFEXPIRE), CombatText_StandardScroll, 
        v.sctColour.r, v.sctColour.g, v.sctColour.b, (v.sctCrit and "crit"), nil)
    elseif Parrot then
      Parrot:GetModule("Display"):ShowMessage(string.format("%s %s", tostring(buff.name), self.strings.BUFFEXPIRE), nil, v.sctCrit, 
        v.sctColour.r, v.sctColour.g, v.sctColour.b, nil, nil, nil, buff.icon)
    end
  end
end

SBF.ShortName = function(self, name)
	if not self.strTmp then
		self.strTmp = CreateFrame("Button")
	end
	self.strTmp:SetFormattedText("")
	for word in string.gmatch(name, "[^%s]+") do 
		self.strTmp:SetFormattedText("%s%s", self.strTmp:GetText() or "", string.match(word, "^."))
		if string.find(word, "[:]") then
			self.strTmp:SetFormattedText("%s:", self.strTmp:GetText())
		end
	end
	return self.strTmp:GetText()
end

SBF.GetSpell = function(self, name)
	if name and self.db.account.spells and self.db.account.spells[name] then
		return self.db.account.spells[name].duration, self.db.account.spells[name].untilCancelled, self.db.account.spells[name].buffType
	end
  return 0,nil,nil
end

SBF.SetSpell = function(self, buff, force)
	if buff.name then
    if not self.db.account.spells then
      self.db.account.spells = {}
    end
    
    if not self.db.account.spells[buff.name] then
      self.db.account.spells[buff.name] = {}
    end
    self.db.account.spells[buff.name].duration = buff.duration
    self.db.account.spells[buff.name].untilCancelled = buff.untilCancelled
    self.db.account.spells[buff.name].buffType = buff.buffType
  end
end

SBF.BuffDuration = function(self, buff)
	if self.showingOptions then
    return
  end
  buff.duration = self:GetSpell(buff.name)
  if not self.db.account.spells[buff.name] or (buff.timeLeft > buff.duration) then
    buff.duration = ceil(buff.timeLeft)
    self:SetSpell(buff)
  end
  if buff.untilCancelled then
    buff.duration = 9999
  end
end

SBF.SetBuffTime = function(self, timer, timeLeft, timerFormat)
	if (timerFormat == 1) then
		if (timeLeft >= 86400) then
			timer.text:SetFormattedText(DAY_ONELETTER_ABBR, ceil(timeLeft/ 86400))
		elseif (timeLeft >= 3600) then
			timer.text:SetFormattedText(HOUR_ONELETTER_ABBR, ceil(timeLeft/ 3600))
		elseif (timeLeft >= 60) then
			timer.text:SetFormattedText(MINUTE_ONELETTER_ABBR, ceil(timeLeft / 60))
		else
			timer.text:SetFormattedText(SECOND_ONELETTER_ABBR, timeLeft)
		end
	elseif (timerFormat == 2) then
		timer.text:SetFormattedText(self.timerFormat2, floor(timeLeft/60), mod(timeLeft, 60))
	elseif (timerFormat == 3) then
		timer.text:SetFormattedText("%d", floor(timeLeft))
	elseif (timerFormat == 4) then
    if (math.floor(timeLeft) <= 60) then
      timer.text:SetFormattedText("%d", math.floor(timeLeft))
    else
      timer.text:SetFormattedText("")      
    end
	end
end

SBF.timerFormat2 = "%02d:%02d"

SBF.EmptyFunc = function()
end

SBF.DisableDefaultBuffFrame = function(self)
  BuffFrame:ClearAllPoints()
  BuffFrame:SetPoint("BOTTOM", UIParent, "TOP", 0, 100)
	BuffFrame:UnregisterAllEvents()
	BuffFrame:SetScript("OnUpdate", nil)
	BuffFrame:Hide()

  TemporaryEnchantFrame:ClearAllPoints()
  TemporaryEnchantFrame:SetPoint("BOTTOM", UIParent, "TOP", 0, 100)
	TemporaryEnchantFrame:SetScript("OnUpdate", nil)
	TemporaryEnchantFrame:Hide()
  TempEnchant1:SetScript("OnUpdate", nil)
  TempEnchant1:Hide()
  TempEnchant2:SetScript("OnUpdate", nil)
  TempEnchant2:Hide()

  -- Just try to re-show the frame now, random addon!
  BuffFrame.SetPoint = self.EmptyFunc
  BuffFrame.Show = self.EmptyFunc
  -- In this case, the GM ticket frame would be the guilty one to re-anchor it.
  TemporaryEnchantFrame.SetPoint = self.EmptyFunc
  TemporaryEnchantFrame.Show = self.EmptyFunc
end

-- 
-- Functions that do visual changes
-- 
SBF.ClearFrame = function(self, frame)
  while (#frame.slots > 0) do
    self:PutSlot(table.remove(frame.slots))
  end
end

SBF.FillFrame = function(self, frame)
  local slot
  for i=1,frame._var.count do
    slot = self:GetSlot(true, true, frame._var.showBars)
    slot.index = i
    table.insert(frame.slots, slot)
  end
end

SBF.SetupFrames = function(self)
  for index,frame in ipairs(self.frames) do
    self:SetupFrame(frame)
    frame:Show()
  end
end

SBF.SetupFrame = function(self, frame)
  if not frame.slots then
    frame.slots = self:GetTable()
  end
  self:ClearFrame(frame)
  self:FillFrame(frame)
  
	local var = frame._var
  local countColour = var.countColour
	local durationColour = var.regularTimerColour
	for j,slot in ipairs(frame.slots) do
    slot.button:SetScale(var.iconScale)
    slot.button.count:SetVertexColor(countColour.r, countColour.g, countColour.b)
    slot.button:SetAlpha(var.opacity)
    slot.button:Show()

    if slot.timer then
			local outline = nil
			if var.timerOutline then
				outline = "OUTLINE"
			end
			slot.timer:SetScale(var.timerScale)
			slot.timer.text:SetJustifyH(var.timerJustify)
			slot.timer.text:SetFont(SML:Fetch("font", var.timerFont), var.timerFontSize, outline)
			slot.timer:ClearAllPoints()
      if slot.bar then
        slot.timer:SetParent(slot.bar)
        slot.timer:SetPoint(var.barTimerPosition, slot.bar, var.barTimerPosition, var.timerPositionX , var.timerPositionY)
      elseif slot.button then
        slot.timer:SetParent(slot.button)
        slot.timer:SetPoint("TOP", slot.button, "BOTTOM", var.timerPositionX , var.timerPositionY)
      end
      slot.timer:Show()
    end

    if slot.bar then
      slot.bar:SetParent(slot.button)
      slot.bar:ClearAllPoints()
      if var.leftBars then
        slot.bar:SetPoint("RIGHT", slot.button, "LEFT", -var.barOffset, 0)
      else
        slot.bar:SetPoint("LEFT", slot.button, "RIGHT", var.barOffset, 0)
      end
      slot.bar:SetWidth(var.barWidth)
      slot.bar:SetStatusBarTexture(SML:Fetch("statusbar", var.barTexture))
      slot.bar.text:SetWidth(var.barWidth - 3 or 122)
      slot.bar.text:SetJustifyH(var.barNameJustify)
      slot.bar.text:SetFont(SML:Fetch("font", var.barNameFont), var.barNameFontSize)
      slot.bar.text:SetTextColor(var.barNameColour.r,var.barNameColour.g,var.barNameColour.b)
      slot.bar:SetBackdropColor(var.barBackdropColour.r,var.barBackdropColour.g,var.barBackdropColour.b,var.barBackdropColour.a)
      slot.bar:Show()
    end
	end
	self:ArrangeFrame(frame)
end


SBF.ArrangeFrame = function(self, frame)
	local var = frame._var
  local	spacingX = var.spacingX
	local spacingY = var.spacingY
	
	if var.showBars then
		spacingX = spacingX + var.barWidth
    frame.showBars = true
  else
    frame.showBars = false
	end
	
	if (#frame.slots > 0) then
		local rowCount, inRows, reverse, bottom, lastAnchor, offX, offY, buffCount, rowMin, leftovers
		local curSlot, slot, lastSlot, lastRowAnchor
		
		buffCount = var.count
		rowCount = var.rowCount
		inRows = var.rows
		reverse = var.reverse
		bottom = var.bottom
		rowMin = math.floor(buffCount/rowCount)
		leftovers = buffCount - (rowMin * rowCount)
	
		lastSlot = nil
    curSlot = 1
		for i=1,rowCount do
			num = rowMin
			if (leftovers > 0) then
				num = num + 1
				leftovers = leftovers - 1
			end
			for j=1,num do
        slot = frame.slots[curSlot]
        if not slot.button then
        else
          slot.button:ClearAllPoints()
          slot.button:SetParent(frame)
          
          if (j == 1) then
            if (curSlot == 1) then
              slot.button:SetPoint("CENTER", frame)
            else
              if inRows then
                offX = 0
                offY = 32 + spacingY
                if not bottom then
                  offY = offY * -1
                end
              else
                offY = 0
                offX = 32 + spacingX
                if reverse then
                  offX = offX * -1
                end
              end
              slot.button:SetPoint("CENTER", lastRowAnchor, "CENTER", offX, offY)
            end
            lastRowAnchor = slot.button
          else
            if inRows then
              offX = 32 + spacingX
              if reverse then
                offX = offX * -1
              end
              offY = 0
            else
              offX = 0
              offY = 32 + spacingY
              if not bottom then
                offY = offY * -1
              end
            end
            slot.button:SetPoint("CENTER", lastSlot.button, "CENTER", offX, offY)
          end
          if slot.timer then
            slot.timer:SetFrameLevel(slot.button:GetFrameLevel()+1)
          end
          lastSlot = slot
          curSlot = curSlot + 1
          if (curSlot > #frame.slots) then
            break
          end
        end
			end
			if (curSlot > #frame.slots) then
				break
			end
		end
	end
end

SBF.GetExtents = function(self)
  for frame,_ in ipairs(self.frames) do
    self:GetExtent(frame)
  end
end

SBF.GetExtent = function(self, i)
	local frame = SBF.frames[i]
  
  if not frame then
    self:msg(SBF.strings.ERRBUFF_EXTENT, i)
    return
  end
  
  if (#frame.slots == 0) then
    return
  end
	
  local bar = frame.slots[1].bar
	local button = frame.slots[1].button
	local buffCount = frame._var.count
	local inRows = frame._var.rows
	local reverse = frame._var.reverse
	local bottom = frame._var.bottom
	local showBars = frame._var.showBars
  local leftBars = showBars and frame._var.leftBars
	local spacingX = frame._var.spacingX
	local spacingY = frame._var.spacingY

	local scale
  if button then
    scale = button:GetEffectiveScale() or 1
  elseif bar then
    scale = bar:GetEffectiveScale() or 1
  end

	local height = 20
	local width = 20
	if bar then
    bar:Show()
		width = width + frame._var.barWidth + frame._var.barOffset
	end
	
	self:PutTable(frame.extent)
	frame.extent = self:GetTable()

	local rowCount, columnCount, columnHeight, rowWidth
	if inRows then
		rowCount = ceil(buffCount/frame._var.rowCount)
		columnCount = frame._var.rowCount
		columnHeight = ((columnCount * height) + (columnCount - 1) * (spacingY + 12)) * scale
		rowWidth = ((rowCount * width) + (rowCount - 1) * (spacingX + 12)) * scale
	else
		rowCount = frame._var.rowCount
		columnCount = ceil(buffCount/rowCount)
		columnHeight = ((columnCount * height) + (columnCount - 1) * (spacingY + 12)) * scale
		rowWidth = ((rowCount * width) + (rowCount - 1) * (spacingX + 12)) * scale
  end

  if bottom then
    frame.extent.bottom = ceil(button:GetBottom() * scale)
    frame.extent.top = ceil(frame.extent.bottom + columnHeight)
  else
    frame.extent.top = ceil(button:GetTop() * scale)
    frame.extent.bottom = ceil(frame.extent.top - columnHeight)
  end

  if reverse then
    frame.extent.right = ceil(button:GetRight() * scale)
    if bar and not leftBars then
      frame.extent.right = frame.extent.right + frame._var.barWidth * scale
    end
    frame.extent.left = ceil(frame.extent.right - rowWidth)
  else
    frame.extent.left = ceil(button:GetLeft() * scale)
    if bar and leftBars then
      frame.extent.left = frame.extent.left - frame._var.barWidth * scale
    end
    frame.extent.right = ceil(frame.extent.left + rowWidth)
  end
end

SBF.NoSort = function(a,b)
	if not b or not b.index then return true end
	if not a or not a.index then return false end
	return a.index < b.index
end

SBF.AscendingNameSort = function(a,b)
	if not b or not b.name then return false end
	if not a or not a.name then return true end
	return a.name < b.name
end

SBF.DescendingNameSort = function(a,b)
	if not b or not b.name then return true end
	if not a or not a.name then return false end
	return a.name > b.name
end

SBF.AscendingTimeSort = function(a,b)
	if not b or not b.timeLeft then return false end
	if not a or not a.timeLeft then return true end
	return a.timeLeft < b.timeLeft
end

SBF.DescendingTimeSort = function(a,b)
	if not b or not b.timeLeft then return true end
	if not a or not a.timeLeft then return false end
	return a.timeLeft > b.timeLeft
end


--
-- Frame Management
-- 
SBF.GetSlot = function(self, button, timer, bar)
  local slot = self:GetTable()
  if button then
    slot.button = self:GetButton()
  end
  if timer then
    slot.timer = self:GetTimer()
  end
  if bar then
    slot.bar = self:GetBar()
  end
  return slot
end

SBF.PutSlot = function(self, slot)
  if slot then
    self:PutButton(slot.button)
    slot.button = nil
    self:PutTimer(slot.timer)
    slot.timer = nil
    self:PutBar(slot.bar)
    slot.bar = nil
    self:PutTable(slot, true)
  end
end

SBF.buttons = {}
SBF.buttonCount = 0
SBF.GetButton = function(self)
	if (#self.buttons > 0) then
    return table.remove(self.buttons)
  end
  self.buttonCount = self.buttonCount + 1
	return CreateFrame("Button", "SBFButton"..self.buttonCount, UIParent, "SBFBuffButtonTemplate")
end

SBF.PutButton = function(self, button)
	if button then
    button.buff = nil
    button:SetScript("OnUpdate", nil)
    button.count:Show()
    button:ClearAllPoints()
    button:Hide()
    table.insert(self.buttons, button)
  end
end

SBF.timers = {}
SBF.timerCount = 0
SBF.GetTimer = function(self)
	if (#self.timers > 0) then
    return table.remove(self.timers)
  end
  self.timerCount = self.timerCount + 1
	return CreateFrame("Frame", "SBFBuffTimer"..self.timerCount, UIParent, "SBFBuffTimerTemplate")
end

SBF.PutTimer = function(self, timer)
	if timer then
		timer.buff = nil
    timer:ClearAllPoints()
		timer:Hide()
    table.insert(self.timers, timer)
	end
end

SBF.bars = {}
SBF.barCount = 0
SBF.GetBar = function(self)
	if (#self.bars > 0) then
    return table.remove(self.bars)
  end
	self.barCount = self.barCount + 1
	bar = CreateFrame("StatusBar", "SBFBuffBar"..self.barCount, UIParent, "SBFBuffBarTemplate")
	return bar
end

SBF.PutBar = function(self, bar)
	if bar then
    bar.buff = nil
		bar:ClearAllPoints()
		bar:Hide()
    table.insert(self.bars, bar)
	end
end

SBF.ClearBuffFrames = function(self, f)
  for i,frame in ipairs(self.frames) do
    if not f or (frame == f) then
      for i,slot in ipairs(frame.slots) do
        slot.buff = nil
      end
      while (#frame.buffs > 0) do
        table.remove(frame.buffs)
      end
    end
	end
end

SBF.FrameVisibility = function(self)
	for i,frame in ipairs(self.frames) do
		if self.showingOptions or not frame._var or not frame._var.frameVisibility or (frame._var.frameVisibility == 1) then
				frame:Show()
				frame:SetAlpha(1)
		elseif (frame._var.frameVisibility == 2) then
				frame:Hide()
		elseif (frame._var.frameVisibility == 3) then
			if InCombatLockdown() then
				frame:Show()
			else
				frame:Hide()
			end
		elseif (frame._var.frameVisibility == 4) then
			if InCombatLockdown() then
				frame:Hide()
			else
				frame:Show()
			end
		elseif (frame._var.frameVisibility == 5) then
			if self:MouseIsOver(i) then
				frame:Show()
			else
				frame:Hide()
			end
		end
	end
end

SBF.MouseIsOver = function(self, frame)
	local x, y = _GetCursorPosition()
	local extent = self.frames[frame].extent
	if extent then
		return (x >= extent.left) and (x <= extent.right) and (y >= extent.bottom) and (y <= extent.top)
	end
	return false
end

SBF.framePool = {}
SBF.GetBuffFrame = function(self)
	for i,frame in ipairs(self.framePool) do
		if not frame.inUse then
			frame.inUse = true
			return frame
		end
	end
	i = #self.framePool + 1
	f = CreateFrame("Frame", "SBFFrame"..i, UIParent, "SBFBuffFrameTemplate")
	table.insert(self.framePool, f)
	f.inUse = true
	return f
end

SBF.PutBuffFrame = function(self, f)
	f.tab1:Hide()
	f.tab2:Hide()
	if f.buffs then
		while (#f.buffs > 0) do
      self:EndBuff(table.remove(f.buffs), true)
		end
	end
  if f.slots then
		while (#f.slots > 0) do
      self:PutSlot(table.remove(f.slots))
		end
  end
  f._var = nil
  f:ClearAllPoints()
	f:Hide()
	f.inUse = false
end

SBF.CreateFrames = function(self)
	local f

	while (#self.frames > 0) do
    self:PutBuffFrame(table.remove(self.frames))
	end
	
	for i,var in pairs(self.db.profile.frames) do
    f = self:GetBuffFrame()
    f.id = i
    f._var = var
    f.tab1.label:SetFormattedText(self.strings.DRAGTAB, i)
    f.tab2.label:SetFormattedText(self.strings.DRAGTAB, i)

    if not var.point then
      if (i == 1) then
        self.db.profile.frames[i].point = {"TOPRIGHT", -65, -250}
      elseif (i == 2) then
        self.db.profile.frames[i].point = {"TOPRIGHT", -110, -250}
      else
        self.db.profile.frames[i].point = {"CENTER", 0, 0}
      end
    end
    f:Show()
    f:ClearAllPoints()
    f:SetPoint(var.point[1], UIParent, var.point[1], var.point[2], var.point[3])
    self.frames[i] = f
	end
	self:FrameVisibility()
end

--
-- Saved Vars
-- 
SBF:RegisterDefaults('server',{
	spells = {},
})

SBF:RegisterDefaults('profile',{
	frames = {},
	alwaysWarnList = {},
	exclusionList = {},
	buffFrameList = {},
	filters = {},
})

SBF.ValidateFrameVars = function(self, i)
	
	if not self.db.profile.frames[i] then
		self.db.profile.frames[i] = {}
	end
	v = self.db.profile.frames[i]
	
	if (i == 1) then
		v.count = v.count or 20
		v.point = v.point or {"TOPRIGHT", -65, -250}
	elseif (i == 2) then
		v.count = v.count or 16
		v.point = v.point or {"TOPRIGHT", -110, -250}
	else
		v.count = v.count or 10
		v.point = v.point or {"CENTER", 0, 0}
	end
	v.rows = v.rows
	v.iconScale = v.iconScale or 1
	v.timerScale = v.timerScale or 1
	v.timerPositionY = v.timerPositionY or v.timerPosition or 7
	v.timerPositionX = v.timerPositionX or 0
	v.opacity = v.opacity or 1
	v.spacingX = v.spacingX or v.spacing or 0
	v.spacingY = v.spacingY or 0
	v.timerFormat = v.timerFormat or 2
	v.reverse = v.reverse
	v.bottom = v.bottom
	v.rowCount = v.rowCount or 1
	v.textWarning = v.textWarning
	v.soundWarning = v.soundWarning
	v.flashBuff = v.flashBuff
	v.sctWarn = v.sctWarn
	v.sound = v.sound or "None"
	v.warnTime = v.warnTime or 0
	v.expireTime = v.expireTime or 30
	v.disableRightClick = v.disableRightClick
	v.timerJustify = v.timerJustify or "CENTER"
	v.frameVisibility = v.frameVisibility or 1
	v.barOffset = v.barOffset or 0
	v.barWidth = v.barWidth or 200
	v.barTexture = v.barTexture or "Blizzard"
	v.barNameJustify = v.barNameJustify or "LEFT"
	v.barTimerPosition = v.barTimerPosition or "RIGHT"
	v.barNameFont = v.barNameFont or "Friz Quadrata TT"
	v.barNameFontSize = v.barNameFontSize or 12
	v.timerFontSize = v.timerFontSize or 10
  if self.db.profile.expireFrame and not v.expireFrame then
    v.expireFrame = self.db.profile.expireFrame    
  else
    v.expireFrame = v.expireFrame or "ChatFrame1"
  end

	v.barBuffColour = v.barBuffColour or {r = 0.0, g = 0.7, b = 1.0 }
	v.barDebuffColour = v.barDebuffColour or {r = 0.7, g = 0.0, b = 0.0}
	v.regularTimerColour = v.regularTimerColour or {r=NORMAL_FONT_COLOR.r, g=NORMAL_FONT_COLOR.g, b=NORMAL_FONT_COLOR.b}
	v.expireTimerColour = v.expireTimerColour or {r=NORMAL_FONT_COLOR.r, g=NORMAL_FONT_COLOR.g, b=NORMAL_FONT_COLOR.b}
	v.countColour = v.countColour or {r=HIGHLIGHT_FONT_COLOR.r, g=HIGHLIGHT_FONT_COLOR.g, b=HIGHLIGHT_FONT_COLOR.b}
	v.sctColour = v.sctColour or {r=GREEN_FONT_COLOR.r, g=GREEN_FONT_COLOR.g, b=GREEN_FONT_COLOR.b}
	v.barBackdropColour= v.barBackdropColour or {r=GRAY_FONT_COLOR.r, g=GRAY_FONT_COLOR.g, b=GRAY_FONT_COLOR.b, a = 1.0}
	v.barNameColour= v.barNameColour or {r=NORMAL_FONT_COLOR.r, g=NORMAL_FONT_COLOR.g, b=NORMAL_FONT_COLOR.b}
	
  if not v.barBackdropColour.a then
    v.barBackdropColour.a = 1
  end
  
	if (v.timerFont == "SBFFontTiny") then
		v.timerFont = "Friz Quadrata TT"
	elseif (v.timerFont == "SBFFontTinyOutline") then
		v.timerFont = "Friz Quadrata TT"
		v.timerOutline = true
	else
		v.timerFont = v.timerFont or "Friz Quadrata TT"
	end
	v.timerPosition = nil
	v.spacing = nil
  v.sortType = v.sortType or 1

  if (type(v.sortType) == "string") then
    if (v.sortType == SBF.strings.sort[5]) then
      v.sortType = 5
    elseif (v.sortType == SBF.strings.sort[4]) then
      v.sortType = 4
    elseif (v.sortType == SBF.strings.sort[3]) then
      v.sortType = 3
    elseif (v.sortType == SBF.strings.sort[2]) then
      v.sortType = 2
    else
      v.sortType = 1
    end
  end
	
  if SCT and (SCT:Get("SHOWFADE", SCT.FRAMES_TABLE) == SCT.MSG) then
    v.sctCrit = false
  end
	return v
end

SBF.DoSavedVars = function(self)
	if not self.db.profile.frames then
		self.db.profile.frames = {}
	end

  if self.db.profile.frames[0] then
    self:PutTable(self.db.profile.frames[0])
		self.db.profile.frames[0] = nil
  end

	self:ValidateFrameVars(1)
	self:ValidateFrameVars(2)
	for i = 3,#self.db.profile.frames do
		self:ValidateFrameVars(i)
	end
  self.db.profile.expireFrame = nil

	if SBFVar then
    if SBFVar.spells then
      SBFVar.spells = nil
    end
    if SBFVar[self.player] then
      SBFVar[self.player] = nil
    end
  end
	
  if (self.db.account.safecall == nil) then
    self.db.account.safecall = true
  end
  
	self.db.account.refresh = nil
	
	if (type(self.db.account.spells) ~= "table") then
    self.db.account.spells = {}
  end
  
  for k,v in pairs(self.db.account.spells) do
    if (v.buffType == 1) then
      v.buffType = 10
    end
    if (v.buffType == 2) then
      v.buffType = 20
    end
  end
  
  if not self.db.profile.alwaysWarnList then
		self.db.profile.alwaysWarnList = {}
	end

	if not self.db.profile.exclusionList then
		self.db.profile.exclusionList = {}
	end

	if not self.db.profile.buffFrameList then
		self.db.profile.buffFrameList = {}
	end
	
	if not self.db.profile.filters then
		self.db.profile.filters = {}
	end
	
end

--
-- Filter Functions
--
SBF.DoFilters = function(self, buff)
	for i,f in ipairs(self.db.profile.filters) do
		for frame,expr in string.gmatch(f, "(..?):(.+)") do
			if self:ParseFilter(expr, buff) then
				return tonumber(frame), expr
			end
		end
	end
	return nil,nil
end

SBF.ParseFilter = function(self, str, buff)
	if string.find(str, "[%(%)]") then
		local rc, subrc, m, l, op
		local p,n = 0,1
		for i=1,string.len(str)+1 do
			l = string.byte(str,i)
			if (l == 40) then
				p = p + 1
				if (p == 1) then
					n = i + 1
				end
			elseif (l == 41) then
				p = p - 1
				if (p == 0) then
					m = i-1
				end
			elseif (p == 0) and ((l == 38) or (l == 124) or (l == nil)) then
				if not m then
					m = i-1
				end
				subrc = SBF:ParseFilter(string.sub(str, n, m), buff)
				if not op then
					rc = subrc
				elseif (op == 38) then -- &
					rc = rc and subrc
				elseif (op == 124) then	-- |
					rc = rc or subrc
				end
				op = l
				m = nil
				n = i + 1
			end
		end
		return rc
	else
		return self:EvaluateExpression(str, buff)
	end
end

-- /dump self:EvaluateExpression("n~Elixir", {name = "Elixir of Fortitude", timeLeft = 1234, duration = 1800})

SBF.EvaluateExpression = function(self, ex, buff)
  local rc, explicitH, o, expr = false, false
  local op, expr, c, m, d, a, subrc, alpha, beta
	if not buff.filterName then
		buff.filterName = string.gsub(string.lower(tostring(buff.name)), " ", "")
	end
	if not self.exCache then
		self.exCache = {}
	end
	if not self.exCache[ex] then
		self.exCache[ex] = string.gsub(ex, " ", "").."."
	end
	for expr,op in string.gmatch(self.exCache[ex], "(.-)([|&.])") do
		subrc = false
		c,m,d = string.match(expr, "(%a+)(%p?%p?)(.*)") 
		if c and m and d then
      alpha = string.byte(c, 1)
      beta = string.byte(c, 2)
			-- Temporary item enchants
			if (c == "e") then
				subrc = (buff.buffType == self.ENCHANT)
			-- Act on buff name
			elseif (c == "n") then
				if m and d then
					d = string.lower(d)
					if (m == "=") then
						subrc = (d == buff.filterName)
					elseif (m == "!=") then
						subrc = not (d == buff.filterName)
					elseif (m == "~") then
						subrc = (string.find(buff.filterName, d) ~= nil)
					elseif (m == "!~") then
						subrc = (string.find(buff.filterName, d) == nil)
					end
				end
			-- tracking buff
			elseif (c == "tr") then
				if buff.isTracking then
          subrc = true
				end
			-- totems
			elseif (c == "to") then
				if (buff.buffType == self.TOTEM) then
          subrc = true
				end
			-- tooltip text
			elseif (c == "tt") then
				if m and d then
					subrc = self:ScanBuffTooltip(buff, d)
					if (m == "!~") then
						subrc = not subrc
					end
				end
			-- group status
			elseif (alpha == self.strings.g) then
        if not beta then
          subrc = false
				elseif (beta == self.strings.s) then
					subrc = not UnitInRaid("player") and (GetNumPartyMembers() == 0)
				elseif (beta == self.strings.g) then
					subrc = UnitInRaid("player") or (GetNumPartyMembers() > 0)
				elseif (beta == self.strings.p) then
					subrc = (GetNumPartyMembers() > 0)
				elseif (beta == self.strings.r) then
					subrc = UnitInRaid("player")
				end
        if (m == "!") then
          subrc = not subrc
        end
			-- debuff flag
			elseif (alpha == self.strings.h) and (buff.buffType == self.HARMFUL) then
				if not beta then
					subrc = true
				elseif (beta == self.strings.c) then
					subrc = self:BuffSchool(buff, self.strings.CURSE)
				elseif (beta == self.strings.d) then
					subrc = self:BuffSchool(buff, self.strings.DISEASE)
				elseif (beta == self.strings.m) then
					subrc = self:BuffSchool(buff, self.strings.MAGIC)
				elseif (beta == self.strings.p) then
					subrc = self:BuffSchool(buff, self.strings.POISON)
				elseif (beta == self.strings.u) then
					subrc = self:BuffSchool(buff)
				elseif (beta == self.strings.a) then
					for k,v in pairs(self.classDispel[self.playerClass]) do
						subrc = subrc or self:BuffSchool(buff, v)
					end
				end
        if (m == "!") then
          subrc = not subrc
        end
				explicitH = true
			-- untilCancelled flag
			elseif (c == "a") then
				subrc = (buff.untilCancelled and true)
				if (m == "!") then
					subrc = not subrc
				end
			-- Math comparisons
			elseif m and d then
				d = tonumber(d)
				-- Buff duration, seconds
				if (c == "d") and not buff.untilCancelled then
					a = buff.duration
				-- Buff duration, minutes
				elseif (c == "D") and not buff.untilCancelled then
					a = buff.duration/60
				-- Buff time remaining, seconds
				elseif (c == "r") and self.db.profile.enableRFilters then
					a = buff.expFloor
				-- Buff time remaining, minutes
				elseif (c == "R") and self.db.profile.enableRFilters then
					a = buff.expFloor/60
				else
					a = nil
					subrc = false
				end
				if a then
					if (m == "<") then
					 subrc = (a < d)
					elseif (m == "<=") then
					 subrc = (a <= d)
					elseif (m == ">") then
					 subrc = (a > d)
					elseif (m == ">=") then
					 subrc = (a >= d)
					end
				end
			end
			if not o then
				rc = subrc
			elseif (o == "|") then
				rc = rc or subrc
			elseif (o == "&") then
				rc = rc and subrc
			end
			o = op
		end
	end
	if not explicitH then
		rc = rc and (buff.buffType ~= self.HARMFUL)
	end
	return rc
end

SBF.BuffSchool = function(self, buff, str)
	if self.showingOptions or not buff or not buff.index then
		return false
	end
	if (buff.dispelType == str) then
		return true
	end
	_GetPlayerBuff(buff.index, "HARMFUL")
	SBFTooltip:SetPlayerBuff(buff.index)
	local text = SBFTooltipTextRight1:GetText()
	if str then
		if text and string.find(text, str) then
			return true
		end
	elseif not SBFTooltipTextRight1:GetText() then
		return true
	end
	return false
end

SBF.ScanBuffTooltip = function(self, buff, str)
  if self.showingOptions or not buff or not buff.index then
    return false
  end
	local line
  if buff.buffType == self.HELPFUL then
    _GetPlayerBuff(buff.index, "HELPFUL") 
  else
    _GetPlayerBuff(buff.index, "HARMFUL") 
  end
	SBFTooltip:SetPlayerBuff(buff.index)
	for i=1,SBFTooltip:NumLines() do
		line = getglobal("SBFTooltipTextLeft"..i):GetText()
		if string.find(string.lower(line), str) then
			return true
		end
		line = getglobal("SBFTooltipTextRight"..i):GetText()
		if line and string.find(string.lower(line), str) then
			return true
		end
	end
	return false
end

-- Table Management
SBF.tablesCreated = 0
SBF.tablesIn = 0
SBF.tablesOut = 0

SBF.TableStats = function(self)
  self:msg("Created %d tables.  Tables in = %d, tables out = %d", SBF.tablesCreated, SBF.tablesIn, SBF.tablesOut)
end

SBF.GetTable = function(self)
	if (#self.tables == 0) then
		self.tablesCreated = self.tablesCreated + 1
		self.tablesOut = self.tablesOut + 1
		return {}
	end
	self.tablesOut = self.tablesOut + 1
	self.tablesIn = self.tablesIn - 1
	return table.remove(self.tables, 1)
end

SBF.PutTable = function(self, t, topOnly)
	if not t then
		return
	end
  if t.buffType and t.name then
    if t.isMirror then
      self:debugmsg(256, format("PutTable called for |cff00ffaa%s (mirror)", t.name))
    else
      self:debugmsg(256, format("PutTable called for |cff00ffaa%s", t.name))
    end
  end
	for k,v in pairs(t) do
		if (type(v) == "table") and not topOnly and not v.IsObjectType and (string.byte(k, 1) ~= 95) then
			self:PutTable(v)
		end
		t[k] = nil
	end
	self.tablesOut = self.tablesOut - 1
	self.tablesIn = self.tablesIn + 1
	table.insert(self.tables, t)
end

SBF.CopyTable = function(self, src)
	local dst = self:GetTable()
	for k,v in pairs(src) do
		if (type(v) == "table") and not v.IsObjectType and (string.byte(k, 1) ~= 95) then
			dst[k] = self:CopyTable(v)
		else
			dst[k] = v
		end
	end
	return dst
end