local L = AceLibrary("AceLocale-2.2"):new("SanityItemCache")

SanityItemCache = AceLibrary("AceAddon-2.0"):new("AceConsole-2.0", "AceEvent-2.0", "AceDB-2.0", "AceDebug-2.0")
SanityItemCache:RegisterDB("SanityItemCacheDB", "SanityItemCacheDBPC")

local default_player_setup = {
	Bags = {},
	BagInfo = {},
	ItemCategories = {},
	CategoryCategories = {},
	ItemTags = {},
	Money = 0
}

function SanityItemCache_CreateTempTableFuncs()
	local tempTables = {}
	
	local clearFunc = function(table)
		for k in pairs(table) do table[k] = nil end
	end
	
	local getFunc = function (name, clear)	
		local n = name:lower()
		tempTables[n] = tempTables[n] or {}
		t = tempTables[n]
		if clear then
			clearFunc(t)
		end
		return t
	end
	return getFunc, clearFunc
end

local get_table, clear_table = SanityItemCache_CreateTempTableFuncs()

local function translate_index(self, k)
	-- We really don't want to blow our stack on valid numeric indexes
	if k == 1 or k == 2 or k == 3 or k == 4 or k == 5 then
		return rawget(self, k)
	end
	
	local keys = SanityItemCache.translationKeys
	local v = SanityItemCache.translationKeys[k]
	if v then
		return rawget(self, v)
	else
		v = rawget(self, k)
		if v then
			return v
		end
	end
	
	if rawget(self, 3) ~= nil then
		if not rawget(self, "gotItemInfo") then
			local name, idstr, itemQuality, itemLevel, itemMinLevel, itemType, subType, stack, equipLoc, texture = GetItemInfo(rawget(self, 3));
			rawset(self, "Name", name or "Unknown")
			rawset(self, "ID", idstr or -1)
			rawset(self, "Quality", itemQuality or 1)
			rawset(self, "iLevel", itemLevel or 0)
			rawset(self, "MinLevel", itemMinLevel or 0)
			rawset(self, "Type", itemType or "Unknown")
			rawset(self, "SubType", subType or "Unknown")
			rawset(self, "Stack", stack or 1)
			rawset(self, "EquipLoc", SANITY_CATEGORYNAMES[equipLoc] or equipLoc or L["INVALID"])
			rawset(self, "Texture", texture or "Interface\\Icons\\INV_Misc_QuestionMark")
			rawset(self, "gotItemInfo", true)
		end
		return rawget(self, k)
	end
	return nil
end

local function newindex(table, key, value)
	if key == "readonly" or not table["readonly"] then
		rawset(table, key, value)
		return
	end
	SanityItemCache:Debug("Invalid set: " .. key .. " - item is readonly.")
end

SanityItemCache:RegisterDefaults("realm", { chars = {} })

----[[ Start Category Handling ]]----

function SanityItemCache:SetItemCategory(itemName, categoryName)
	self.db.profile.ItemCategories[itemName] = categoryName
	self:ReprepItem(itemName)
end

function SanityItemCache:DeleteCategory()
end

function SanityItemCache:SetCategoryCategory(itemName, categoryName)
end

function SanityItemCache:GetExistingCategories()
	local names = get_table("categoryNames")
	if #names > 0 then
		return names
	end

	local temp = get_table("categoryNamesTemp")

	clear_table(names)
	clear_table(temp)
	
	local i = 0
	for k,v in pairs(SanityItemCache.db.profile.ItemCategories) do
		if v then
			temp[v] = 1
		end
	end
	for k,v in pairs(SanityItemCache.db.profile.CategoryCategories) do
		if v then
			temp[v] = 1
		end
	end

	for k,v in pairs(temp) do
		tinsert(names, k)
	end
	return names
end
----[[ Start Tagging ]]----

function SanityItemCache:ItemHasTag(item, tag)
	if self.db.profile.ItemTags[item] then
		return self.db.profile.ItemTags[item][tag] ~= nil
	end
	return false
end

function SanityItemCache:GetCategoryFor(item)
	local name = self.db.profile.ItemCategories[item]
	if name then
		local catName = self.db.profile.CategoryCategories[name]
		if catName then return catName end
	end
	return name
end

function SanityItemCache:AddItemTag(item, tag)
	self.db.profile.ItemTags[item] = self.db.profile.ItemTags[item] or {}
	self.db.profile.ItemTags[item][tag] = 1
	self:ReprepItem(item)
	self:GetTagList(true)
end

function SanityItemCache:RemoveItemTag(item, tag, noRefresh)
	if not self.db.profile.ItemTags[item] then return end
	local tTags = self.db.profile.ItemTags[item] or {}
	local ct = 1
	local removedTag = true
	while removedTag == true do
		removedTag = false
		for k,v in pairs(tTags) do
			if k == tag then
				tremove(tTags, ct)
				removedTag = true
				break
			end
			ct = ct + 1
		end
	end
	self:ReprepItem(item)
	self:GetTagList(true)
end

function SanityItemCache:GetItemsForTag(tag)
	if not self.db.profile.ItemTags then 
		return
	end
	local result = get_table("itemsForTag")
	clear_table(result)
	local resultCount = 0
	for k,v in pairs(self.db.profile.ItemTags) do
		for k2, v2 in pairs(v) do
			if k2 == tag then
				tinsert(result, k)
			end
		end
	end
	return result
end

function SanityItemCache:GetTagsFor(item)
	if not self.db.profile.ItemTags[item] then 
		return ""
	end
	local s = ""
	local tags = get_table("tagsForItem", true)
	for k,v in pairs(self.db.profile.ItemTags[item]) do
		tinsert(tags, k)
	end
	table.sort(tags)
	for k,v in pairs(tags) do
		if s == "" then
			s = v
		else
			s = s .. ", " .. v
		end
	end
	return s
end

function SanityItemCache:UntagItem(item, tag)
	if not self.db.profile.ItemTags or not self.db.profile.ItemTags[item] then return end
	local ct = 1
	for k, v in pairs(self.db.profile.ItemTags[item]) do
		if strlower(k) == strlower(tag) then
			tremove(self.db.profile.ItemTags[item], ct)
			return
		end
		ct = ct + 1
	end
end

function SanityItemCache:KillTag(tag)
	if not self.db.profile.ItemTags then return end
	for k,v in pairs(self.db.profile.ItemTags) do
		self:UntagItem(k, tag)
	end
	self:GetTagList(true)
end

function SanityItemCache:GetTagList(force)
	if self.tagList and not self.tagListDirty and not force then return self.tagList end
	local tempTagList = get_table("tempTagList")
	clear_table(tempTagList)
	
	for k,v in pairs(self.db.profile.ItemTags) do
		for k1,v1 in pairs(v) do
			tempTagList[k1] = 1
		end
	end
	
	self.tagList = get_table("tagList")
	clear_table(self.tagList)
	for k,v in pairs(tempTagList) do
		tinsert(self.tagList, tostring(k))
	end
	table.sort(self.tagList)
	self.tagListDirty = false
	return self.tagList
end

----[[ End Tagging ]]----

function SanityItemCache:OnInitialize()
	self.char = UnitName("player")
	self:RegisterEvent("BANKFRAME_OPENED")
	self:RegisterEvent("BANKFRAME_CLOSED")
	self:RegisterEvent("ADDON_LOADED")
	self:RegisterEvent("PLAYER_LOGOUT")
	self.translationKeys = {["Count"] = 1, ["Bag"] = 2, ["Link"] = 3, ["Soulbound"] = 4, ["Slot"] = 5}
	self.translateMeta = {__index = translate_index, __newindex = newindex}
	self.BankIsVisible = false
	self.filterTypes = self.filterTypes or {}
	
	SanityItemCache_ItemTooltip = CreateFrame("GameTooltip", "SanityItemCache_ItemTooltip", WorldFrame, "GameTooltipTemplate")
	SanityItemCache_ItemTooltip:SetOwner(WorldFrame, "ANCHOR_NONE")

	self.db.profile.ItemCategories = self.db.profile.ItemCategories or {}
	self.db.profile.CategoryCategories = self.db.profile.CategoryCategories or {}
	self.db.profile.ItemTags = self.db.profile.ItemTags or {}

	if not self.db.realm.chars[self.char] then
		self.db.realm.chars[self.char] = {}
		for k,v in pairs(default_player_setup) do
			self.db.realm.chars[self.char][k] = {}
		end
	end
	local upgraded = false
	for k,v in pairs(self.db.realm.chars) do
		if v.BagItems then
			upgraded = true
			v.BagItems = nil
			v.MailItems = nil
			v.EquipmentItems = nil
			v.BankItems = nil
			v.Bags = {}
			v.BagInfo = {}
		end
		if v.Bags and type(v.Bags) == "table" and #v.Bags > 0 then
			for k2,v2 in pairs(v.Bags) do
				if type(v2) ~= "table" then
					v.BagItems = nil
					v.MailItems = nil
					v.EquipmentItems = nil
					v.BankItems = nil
					v.Bags = {}
					v.BagInfo = {}
					upgraded = true
					break
				end
			end
		end
		v.BagInfo = v.BagInfo or {}
	end
	if upgraded then
		SanityItemCache:Print("Please note that Sanity has undergone a database reorganization and your old item data is no longer available. Please check item locations for each character you wish to have item data for.")
	end
	self.INVENTORY_BAG_ID = SANITY_INVENTORY_BAG_ID
	self.MAILBOX_BAG_ID = SANITY_MAILBOX_BAG_ID
	SanityItemCache:PLAYER_MONEY()
	self.CacheInited = true
end


do
	-- private table... 0 = bags, 1 = bank, 2 = guild bank
	local bagTypes = {}
	for _,v in ipairs(SANITY_PLAYER_BAGS) do bagTypes[v] = 0 end
	for _,v in ipairs(SANITY_BANK_BAGS) do bagTypes[v] = 1 end
	for _,v in ipairs(SANITY_GUILD_BAGS) do bagTypes[v] = 2 end
	
	function SanityItemCache:GetBagNameFor(bagID)
		local type = bagTypes[bagID]
		if type == 0 then return L["Bags"] end
		if type == 1 then return L["Bank"] end
		if type == 2 then return L["Guild Bank"] end
		
		if bagID == SANITY_INVENTORY_BAG_ID then return L["Wearing"] end
		if bagID == SANITY_MAILBOX_BAG_ID then return L["Mailbox"] end
		if bagID == KEYRING_CONTAINER then return L["Keyring"] end
	
		return L["INVALID"]
	end
end

function SanityItemCache:NukeCharacter(char)
	local ct = 1
	for tchar,dummy in pairs(self.db.realm.chars) do
		if tchar ~= char then
			tremove(self.db.realm.chars, ct)
			return
		end
		ct = ct + 1
	end
end

function SanityItemCache:ADDON_LOADED()
	local upgraded = false
	for k,v in pairs(self.db.realm.chars) do
		if v.BagItems then
			upgraded = true
			v.BagItems = nil
			v.MailItems = nil
			v.EquipmentItems = nil
			v.BankItems = nil
			v.Bags = {}
			v.BagInfo = {}
		end
		if v.Bags and type(v.Bags) == "table" and #v.Bags > 0 then
			for k2,v2 in pairs(v.Bags) do
				if type(v2) ~= "table" then
					v.BagItems = nil
					v.MailItems = nil
					v.EquipmentItems = nil
					v.BankItems = nil
					v.Bags = {}
					v.BagInfo = {}
					upgraded = true
					break
				end
			end
		end
		v.BagInfo = v.BagInfo or {}
	end
	if upgraded then
		SanityItemCache:Print("Please note that Sanity has undergone a database reorganization and your old item data is no longer available. Please check item locations for each character you wish to have item data for.")
	end
	
	for char,dummy in pairs(self.db.realm.chars) do
		for k,v in pairs(dummy.Bags) do
			for k2,v2 in pairs(v) do
				setmetatable(v2, self.translateMeta)
			end
		end
	end
	SanityItemCache:ScanAll()
	self:RegisterEvent("BAG_UPDATE")
	self:RegisterEvent("UNIT_INVENTORY_CHANGED")
	self:RegisterEvent("PLAYER_MONEY")
end

function SanityItemCache:PLAYER_LOGOUT()
	for charName, char in pairs(self.db.realm.chars) do
		for k,v in pairs(char.Bags) do
			local t = {}
			for k2,v2 in pairs(v) do
				tinsert(t, {
					[1] = v2[1],
					[2] = v2[2],
					[3] = v2[3],
					[4] = v2[4],
					[5] = v2[5]
				})
			end
			char.Bags[k] = t
		end
	end
end

function SanityItemCache:ItemIsUsable(item)
	if UnitName("player") == item["Owner"] then
		if item["Bag"] == SANITY_MAILBOX_BAG_ID then return false end		-- We can't use items in the mail
		if item["Bag"] == SANITY_INVENTORY_BAG_ID then return true end	-- We can always use items we're wearing
		if item["Bag"] >= 0 and item["Bag"] <= NUM_BAG_SLOTS then return true end	-- We can always use items in our bags
		return self.BankIsVisible
	end
	return false
end

function SanityItemCache:BANKFRAME_OPENED()
	self.BankIsVisible = true
	self:ScheduleScan(BANK_CONTAINER)
	for i = NUM_BAG_SLOTS+1, NUM_BAG_SLOTS+NUM_BANKBAGSLOTS do
		self:ScheduleScan(i)
	end
	-- inventory slots 68-74 are bank bag slots (no constant avail?)
	self:ScheduleEquipScan(68, 74, SANITY_BANK_SLOTS_ID)
end

function SanityItemCache:BANKFRAME_CLOSED()
	self.BankIsVisible = false
end

function SanityItemCache:UNIT_INVENTORY_CHANGED()
	self:ScheduleEquipScan()
end

function SanityItemCache:ScanAll()
	SanityItemCache:ScanBags()
	SanityItemCache:UNIT_INVENTORY_CHANGED()
end

function SanityItemCache:BAG_UPDATE()
	if not tonumber(arg1) then return end
	self:ScheduleScan(arg1)
	if self.BankIsVisible then
		self:ScheduleScan(BANK_CONTAINER)
		for i = NUM_BAG_SLOTS+1, NUM_BAG_SLOTS+NUM_BANKBAGSLOTS do
			self:ScheduleScan(i)
		end
	end
	-- inventory slots 68-74 are bank bag slots (no constant avail?)
	self:ScheduleEquipScan(68, 74, SANITY_BANK_SLOTS_ID)
end

function SanityItemCache:ReprepItem(tItem)
  itemName = self:GetItemName(tItem)
  for char, data in pairs(self.db.realm.chars) do 
    for k, bag in pairs(data.Bags) do 
      for k, item in pairs(bag) do 
        if item["Name"] == itemName and item["Prepped"] then 
          self:PrepareItem(item, true) 
        end 
      end 
    end 
  end
end 

function SanityItemCache:PLAYER_MONEY()
	self.db.realm.chars[self.char].Money = GetMoney()
end

-- To prevent problems with massive numbers of updates all at once (think zoning), we'll jsut schedule a scan and then do it once whatever it is has decided that we're good to stop scanning
function SanityItemCache:ScheduleScan(id)
	self:ScheduleEvent("SanityItemCacheScanBag" .. id, SanityItemCache.ScanBag, 0.05, self, id)
end

function SanityItemCache:ScheduleEquipScan(min,max,bagID)
	if not bagID then bagID = SANITY_INVENTORY_BAG_ID end
	self:ScheduleEvent("SanityItemCacheScanEquip", function() 
		local bagStatusTable = get_table("SanityItemCacheScanEquipStatus")
		if not bagStatusTable.bag_slots then
			bagStatusTable.bag_slots = 0
			bagStatusTable.filled_slots = 0
		end
		SanityItemCache.db.realm.chars[self.char].Bags[bagID], SanityItemCache.db.realm.chars[self.char].BagInfo[bagID] = SanityItemCache:ScanEquipment(min,max,bagID), bagStatusTable
	end, 0.05)
end

function SanityItemCache:ScanBag(bagID)
	self.countDirty = true
	local a, b = self:Scan(bagID)
	if a then
		self.db.realm.chars[self.char].Bags[bagID], self.db.realm.chars[self.char].BagInfo[bagID] = a, b
	end
	self:UpdateItemCount()
end

function SanityItemCache:GetBagData(char)
	if not self.CacheInited then return 0,0 end
	return self:CountSlots(SANITY_PLAYER_BAGS, char)
end

function SanityItemCache:GetBankData(char)
	if not self.CacheInited then return 0,0 end
	return self:CountSlots(SANITY_BANK_BAGS, char)
end

function SanityItemCache:CountSlots(bags, char)
	if not char then char = self.char end
	local c, cf = 0, 0
	for k, bag in ipairs(bags) do
		if self.db.realm.chars[char].BagInfo[bag] then
			c = c + self.db.realm.chars[char].BagInfo[bag].bag_slots
			cf = cf + self.db.realm.chars[char].BagInfo[bag].filled_slots
		end
	end
	return cf, c
end

function SanityItemCache:ScanBags()
	for i = 0, 4 do
		self:ScheduleScan(i)
	end
	self:ScheduleScan(KEYRING_CONTAINER)
end

function SanityItemCache:MatchClause(item,clause)
	local type, value = strsplit(":", clause, 2)
	if (type == "true" or type == "false") then return type end -- dont touch something we've done before
	if not value then 
		value = type
		type = "name"
	end
	type = strtrim(type):lower()
	value = strtrim(value)
	if (value:sub(1,1) == "\"" and value:sub(-1,-1)=="\"") then value = value:sub(2,-2) end
	local o = false

	self.filterTypes = self.filterTypes or {}
	local filter = self.filterTypes[type]
	if filter ~= nil then o = filter(item, value) end

	return o and "true" or "false" -- make sure we return "true" or "false"
end

function SanityItemCache:MatchExpr(item,filter,search)
  self:PrepareItem(item)
  local expr = filter:trim()
  if expr == "" then return false end
  if search and search:trim() ~= "" then
    expr = "("..expr..") name:\""..search:trim().."\""
  end
  -- match full expressions
  expr = expr:gsub("%a+:\"[^\"]+\"",function(m) return self:MatchClause(item,m) end)
  expr = expr:gsub("%a+:[%a%d%.]+",function(m) return self:MatchClause(item,m) end)
  
  -- match remaining standalone words (either * or a name search)
  expr = expr:gsub("%a+",function(m) return self:MatchClause(item,m) end)
  
  -- match our * clause
  expr = expr:gsub("*","true")
  
  -- remove duplicate spacing
  expr = expr:gsub("%s+"," ") 
  
  -- remove spacing around ,
  expr = expr:gsub("%s?,%s?",",") 
  
  -- remove spacing inside brackets
  expr = expr:gsub("[(]%s?","(")
  expr = expr:gsub("%s?[)]",")")
  
  -- replace operators
  expr = expr:gsub(" "," and ")
  expr = expr:gsub(","," or ")
  expr = expr:gsub("!"," not ")
  
  -- load the lua string to a function
  local func, error = loadstring("return "..expr)
  
  --[[ 
  if we have a function, run and return
  else fail silently (may want to raise some sort of error here maybe - but not for the moment as SanityBags attempts to evaluate a filter expression whilst it's being typed.
  ]]
  if func then return func() else return false end
end

function SanityItemCache:Scan(bag)
	if GetContainerNumSlots(bag) == 0 then return nil end

	local items = get_table("bag_items_" .. bag)
	clear_table(items)
	
	local bagData = get_table("bag_items_data_" .. bag)
	bagData.bag_slots = 0
	bagData.filled_slots = 0
	
	bagData.bag_slots = GetContainerNumSlots(bag)
	for slot = 1, bagData.bag_slots do
		local link = GetContainerItemLink(bag, slot)
		if link then
			if bag ~= KEYRING_CONTAINER then
				bagData.filled_slots = (bagData.filled_slots or 0) + 1
			end
			local n, _, itemQuality, itemLevel, itemMinLevel, itemType, subType, stack, equipLoc, texture  = GetItemInfo(link);
			if itemQuality then
				SanityItemCache_ItemTooltip:ClearLines()
				if bag == BANK_CONTAINER then
					SanityItemCache_ItemTooltip:SetInventoryItem("player", BankButtonIDToInvSlotID(slot, nil));
				else
					SanityItemCache_ItemTooltip:SetBagItem(bag, slot)
				end

				local soulbound = 0
				if bag ~= KEYRING_CONTAINER then
					for i=1,SanityItemCache_ItemTooltip:NumLines() do
						local obj = getglobal("SanityItemCache_ItemTooltipTextLeft" .. i)
						if obj:GetText() == L["Soulbound"] or obj:GetText() == L["Quest Item"] then
							soulbound = 1
						end
					end
				else
					soulbound = 1
				end
				
				local texture, count, locked = GetContainerItemInfo(bag, slot);
				local _,_,itemID = string.find(link, "H(item:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+)")
				local item = get_table("bag_table_" .. bag .. "_item_" .. slot)
				clear_table(item)
				tinsert(item, count)
				tinsert(item, bag)
				tinsert(item, itemID)
				tinsert(item, soulbound)
				tinsert(item, slot)
				setmetatable(item, self.translateMeta)
				item = self:PrepareItem(item)
				tinsert(items, item)
			end
		end
	end
	return items, bagData
end

function SanityItemCache:ScanEquipment(min, max, bagID)
	-- inventory 1-19, bags 20-23
	if not min then min = 1 end
	if not max then max = 23 end
	if not bagID then bagID = SANITY_INVENTORY_BAG_ID end
	
	local items = get_table("equip_items_"..bagID)
	clear_table(items)
	for i = min,max do
		SanityItemCache_ItemTooltip:SetInventoryItem("player", i)
		local soulbound = 0
		for i=1,SanityItemCache_ItemTooltip:NumLines() do
			if SanityItemCache_ItemTooltipTextLeft2:GetText() == L["Soulbound"] or SanityItemCache_ItemTooltipTextLeft2:GetText() == L["Quest Item"] then
				soulbound = 1
			end
		end
		local link = GetInventoryItemLink("player", i)
		if link then
			local name = GetItemInfo(link)
			local _,_,itemID = string.find(link, "H(item:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+)")
			
			local item = get_table("equip_item_slot_" .. i)
			clear_table(item)
			
			if i <= 23 and i >= 20 or i >= 68 then
				-- do not count bag "stacks" they actually indicate count of contents
				tinsert(item, 1)
			else
				tinsert(item, GetInventoryItemCount("player", i))
			end
			tinsert(item, bagID)
			tinsert(item, itemID)
			tinsert(item, soulbound)
			tinsert(item, i)
			setmetatable(item, self.translateMeta)
			item = self:PrepareItem(item)
			tinsert(items, item)
		end
	end
	return items
end

function SanityItemCache:ScanMailbox()
	local inbox_items = GetInboxNumItems();
	local items = get_table("mailbox_items")
	clear_table(items)
	
	local num = 0
	for i = 1, GetInboxNumItems() do
        for j = 1, ATTACHMENTS_MAX_SEND do
			num = num + 1
			-- alternatively, j + (i - 1) * ATTACHMENTS_MAX_SEND - but this creates a sparse tho predictable index
			local soulbound = 0
			local link = GetInboxItemLink(i,j)
			if link then
				local itemName, itemLink, itemRarity, itemLevel, itemMinLevel, itemType, itemSubType, itemStackCount, itemEquipLoc, invTexture = GetItemInfo(link)
				local _,_,itemID = string.find(link, "H(item:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+)")
				local name, itemTexture, count, quality, canUse = GetInboxItem(i,j);
				
				local item = get_table("mailbox_item_slot_" .. num)
				clear_table(item)
				tinsert(item, count)
				tinsert(item, SANITY_MAILBOX_BAG_ID)
				tinsert(item, itemID)
				tinsert(item, soulbound)
				tinsert(item, num)
				setmetatable(item, self.translateMeta)
				item = self:PrepareItem(item)
				tinsert(items, item)
			end
		end
	end
	local mailboxBagInfo = get_table("mailbox_bag_info")
	if not mailboxBagInfo.bag_slots then
		mailboxBagInfo.bag_slots = 0
		mailboxBagInfo.filled_slots = 0
	end
	self.db.realm.chars[self.char].Bags[SANITY_MAILBOX_BAG_ID], self.db.realm.chars[self.char].BagInfo[SANITY_MAILBOX_BAG_ID] = items, mailboxBagInfo
end

function SanityItemCache:GetTooltipString(link)
	if not link then return nil end
	local name = GetItemInfo(link)
	if not name then name = "Unknown" end
	self.TooltipDataCache = self.TooltipDataCache or {}
	if self.TooltipDataCache[name] then
		return self.TooltipDataCache[name]
	end
	SanityItemCache_ItemTooltip:ClearLines()
	SanityItemCache_ItemTooltip:SetOwner(WorldFrame, "ANCHOR_NONE")
	SanityItemCache_ItemTooltip:SetHyperlink(link)
	local s = nil
	for i=1,SanityItemCache_ItemTooltip:NumLines() do
		local lText = getglobal("SanityItemCache_ItemTooltipTextLeft" .. i)
		local rText = getglobal("SanityItemCache_ItemTooltipTextRIght" .. i)
		if lText and lText:GetText() then
			if s then s = s .. " " .. lText:GetText() 
			else s = lText:GetText() end
		end
		if rText and rText:GetText() then
			if s then s = s .. " " .. rText:GetText() 
			else s = rText:GetText() end
		end
	end
	self.TooltipDataCache[name] = s
	return s
end

function SanityItemCache:GetItemName(item)
  if type(item) == "string" then 
    return item 
  elseif type(item) == "table" and item.Name then 
    return item.Name 
  end
end

function SanityItemCache:PrepareItem(item, force)
  if item["Prepped"] and not force then return item end
  item.readonly = false  
  
  item.SetCategory = function(catName)
    SanityItemCache:SetItemCategory(item["Name"], cat)
  end
  item.AddTag = function(tag)
    SanityItemCache:AddItemTag(item["Name"], tag)
  end
  item.RemoveTag = function(tag)
    SanityItemCache:RemoveItemTag(item["Name"], tag)
  end
  
  item["Location"] = self:GetBagNameFor(item["Bag"])
  item["Stacks" ] = 1
  if item["Link"] then
    if item["Dangerous"] then
      item["Dangerous"] = false
    end    
    if not item["Type"] then
      item["Type"] = L["Unknown"]
      item["SubType"] = L["Unknown"]
      item["EquipLoc"] = L["Unknown"]
      item["Dangerous"] = true
    end
  end    

  item.Tags = SanityItemCache:GetTagsFor(item["Name"])
  
  if not item.invalid then
    item["Tooltip"] = SanityItemCache:GetTooltipString(item["Link"]) or ""
  else
    item["Tooltip"] = ""
  end
  local catName = SanityItemCache.db.profile.ItemCategories[item["Name"]]
  if catName then
    item["TypeSort"] = catName
  else
    if item["Type"] == item["SubType"] then
      item["TypeSort"] = item["SubType"]
      if item["EquipLoc"] and #item["EquipLoc"] > 0 then
        item["TypeSort"] = item["TypeSort"] .. " - " .. item["EquipLoc"]
      end
    elseif item["EquipLoc"] and #item["EquipLoc"] > 0 then
      item["TypeSort"] = item["Type"] .. " - " .. item["EquipLoc"]
    else
      item["TypeSort"] = item["Type"]
  end
  end
  local catCatName = SanityItemCache.db.profile.CategoryCategories[item["TypeSort"]]
  if catCatName then
    item["TypeSort"] = catCatName
  end
  item["TypeSort"] = item["TypeSort"] or "No Type"
  item["Owner"] = item["Owner"] or self.char
  item["Key"] = item["Name"] .. "|" .. (item["Location"] or "INVALID") .. "|" .. item["Owner"]
  item["Prepped"] = true
  item.readonly = true
  return item
end

function SanityItemCache:UpdateItemCount()
end
