﻿---------------------------------
--       Local Variables       --
---------------------------------
local core = assert(SacredBuff, 'SacredBuff must be installed and enabled to run SacredBuff Mounts!')
local L = AceLibrary('AceLocale-2.2'):new('SacredBuff')
local PT = AceLibrary('PeriodicTable-3.0')
local D = AceLibrary('Dewdrop-2.0')

local new = core.new
local newHash = core.newHash
local del = core.del

local outlandZones, isInOutlands, inCombat, timeSinceUpdate
local tconcat, mountCache, concatTable, mountTypes

---------------------------------
--      Addon Declaration      --
---------------------------------
SBMounts = AceLibrary('AceAddon-2.0'):new('AceEvent-2.0', 'AceDB-2.0')

SBMounts.options = {
	name = 'Mount Options', type = 'group',
	desc = 'A myriad of Mount options.',
	hidden = function()
		return SacredBuff.playerLevel < 30
    end,
	args = {--[[
		{
			name = 'Land', type = 'header',
			desc = 'Land Mounts', order = 1,
			hidden = function() end
		},
		{
			type = 'header', order = 99,
			hidden = function() end
		},
		{
			name = 'Flying', type = 'header',
			desc = 'Flying Mounts', order = 100,
			hidden = function() end
		}
	]]}
}



---------------------------------
--        Initialization       --
---------------------------------
function SBMounts:OnInitialize()

	self:RegisterDB('SBMDB', nil, 'char')
	
	self:ToggleActive(core:IsActive())

end

function SBMounts:OnEnable()
	tconcat = table.concat
	
	isInOutlands = self:InOutlands()
	
	mountTypes = newHash(
		'Normal60', 30,
		'Normal100', 60,
		"Ahn'Qiraj100", 60,
		'Flying60', 70,
		'Flying280', 70
	)

	mountCache = setmetatable(new(), newHash(
		'__mode', 'v',
		'__index', function(t, k)
			local mType, mQuality = select(3, k:find("([%a%']+)(%d+)"))
			
			local setString = PT:GetSetStringUncompressed('Misc.Mount.'..mType)
			if not concatTable then
				concatTable = new()
			end
			
			for itemId, speed in setString:gmatch('(%d+):(%d+)') do
				if speed == mQuality then
					tinsert(concatTable, itemId)
				end
			end
			
			local v = table.concat(concatTable, ',')
			
			concatTable = del(concatTable, true)
			
			if not v:find('(%d)') then
				v = nil
			end
			
			t[k] = v
			
			return t[k]
		end
	))
	
	self:RegisterBucketEvent('BAG_UPDATE', 1, 'RefreshMountTables')
	self:RegisterEvent('ZONE_CHANGED', 'CheckZone')
	self:RegisterEvent('PLAYER_ENTERING_WORLD', 'CheckZone')
	self:RegisterEvent('PLAYER_REGEN_ENABLED')
	self:RegisterEvent('PLAYER_REGEN_DISABLED')
	inCombat = InCombatLockdown() and true or nil
	
	if not self.mounts then
		self:RefreshMountTables()
	end
	
	if not self.button then
		self.button = self:CreateButton()
	end
	self:UpdateButtonAttributes()
end

function SBMounts:OnDisable()
	inCombat = nil
	isInOutlands = nil
	tconcat = nil
	outlandZones = nil
	mountTypes = del(mountTypes)
	mountCache = del(mountCache)
end


---------------------------------
--        Event Handlers       --
---------------------------------
function SBMounts:RefreshMountTables()
	if self.db.char.mount and GetItemCount(self.db.char.mount) == 0 then
		self.db.char.mount = nil
	end
	
	if not self.mounts then
		self.mounts = new()
	end
	
	local changes, itemString
	local level = core.playerLevel
	for k in next, mountTypes do
		if level >= mountTypes[k] then
			itemString = self:IterateMounts(k)
			if self.mounts[k] ~= itemString then
				self.mounts[k] = itemString
				changes = true
			end
		end
	end
	
	if changes then
		self:UpdateMountMenu()
	end
end

function SBMounts:CheckZone()
	isInOutlands = self:InOutlands()
end

function SBMounts:PLAYER_REGEN_DISABLED()
	inCombat = true
end

function SBMounts:PLAYER_REGEN_ENABLED()
	inCombat = nil
end


---------------------------------
--        Other Methods        --
---------------------------------
function SBMounts:InOutlands()
	if not IsInInstance() then
		if not outlandZones then
			outlandZones = strjoin(',', L['Twisting Nether'], GetMapZones(3))
		end
		return outlandZones:find(GetRealZoneText()) and true
	end
end

function SBMounts:IterateMounts(mountType)
	local setString = mountCache[mountType]
	
	if not concatTable then
		concatTable = new()
	end
	
	for itemId in setString:gmatch('(%d+)') do
		if GetItemCount(tonumber(itemId)) > 0 then
			tinsert(concatTable, itemId)
		end
	end
	
	local itemString = tconcat(concatTable, ',') or nil
	concatTable = del(concatTable)
	
	return itemString
end

function SBMounts:NewMountOption(k, outlandsOnly)
	local mountName = GetItemInfo(k)
	return newHash(
		'name', mountName,
		'type', 'toggle',
		'desc', mountName,
		'isRadio', true,
		'get', function()
			if self.db.char.mount and self.db.char.mount == k then
				return true
			end
		end,
		'set', function()
			if self.button and not InCombatLockdown() then
				if self.db.char.mount and self.db.char.mount == mountName then
					return
				end
				self.db.char.mount = k
				self:UpdateButtonAttributes()
				D:Close()
			end
		end
	)
end

function SBMounts:UpdateButtonAttributes()
	local mountID = self.db.char.mount or next(self.options.args)
	if not mountID then
		self.button:Hide()
		return
	end
	
	self.button:SetAttribute('macrotext', '/cast '..GetItemInfo(mountID))
	self.button.icon:SetTexture(select(10, GetItemInfo(mountID)))
	
	if not self.button:IsShown() then
		self.button:Show()
	end
end

function SBMounts:UpdateMountMenu()
	local options = self.options.args
	for k in next, options do
		if options[k].type ~= 'header' and GetItemCount(k) == 0 then
			k = del(k)
		end
	end
	
	concatTable = concatTable or new()
	
	for k in next, self.mounts do
		if self.mounts[k] ~= '' then
			tinsert(concatTable, self.mounts[k])
		end
	end
	
	local mounts = tconcat(concatTable, ',')
	concatTable = del(concatTable)
	
	for mount in mounts:gmatch('(%d+)') do
		mount = tonumber(mount)
		if not options[mount] then
			options[mount] = self:NewMountOption(mount)
		end
	end
end

local OnUpdate = function(self, elapsed)
	if not timeSinceLastUpdate then
		timeSinceLastUpdate = 1
	end
	timeSinceLastUpdate = timeSinceLastUpdate + elapsed
	if timeSinceLastUpdate > 1 then
		
		if UnitOnTaxi('player') or IsIndoors() or inCombat then
			if not self.notUsable then
				_G.SetDesaturation(self.icon, true)
				self.notUsable = true
			end
		elseif self.notUsable then
			_G.SetDesaturation(self.icon, nil)
			self.notUsable = nil
		end
		
		timeSinceLastUpdate = 0
	end
end

function SBMounts:CreateButton()
	local f = core:NewActionButton(-32, -32)
	
	D:Register(f,
		'dontHook', true,
		'children', function()
			D:FeedAceOptionsTable(self.options)
		end
	)
	
	core.frame:SetAttribute('addchild', f)
	
	f:SetAttribute('*type*', 'macro')
	
	f:SetAttribute('alt-type2', 'menu')
	f.menu = self.OnMenuShow
	
	f:SetScript('OnUpdate', OnUpdate)
	f:SetScript('OnHide', function()
		timeSinceLastUpdate = nil
	end)
	
	return f
end

function SBMounts:OnMenuShow()
	if InCombatLockdown() then
		return
	end
	
	local self = SBMounts
	if ( D:IsOpen(self.button) ) then
		D:Close()
		return
	end
	
	self:UpdateMountMenu()
	D:Open(self.button)
end
