--[[
Name: SphereInventory-1.0
Revision: $Rev: 10000 $
Developed by: Zirah
Website: http://thomas.weinert.info/zirah/
Description: inventory handling for sphere addons
Dependencies: AceLibrary, AceOO-2.0
]]

local MAJOR_VERSION = 'SphereInventory-2.0'
local MINOR_VERSION = '$Revision: 10000 $'

if not AceLibrary then error(MAJOR_VERSION .. ' requires AceLibrary') end
if not AceLibrary:IsNewVersion(MAJOR_VERSION, MINOR_VERSION) then return end

if not AceLibrary:HasInstance('AceOO-2.0') then error(MAJOR_VERSION .. ' requires AceOO-2.0') end
local AceOO = AceLibrary:GetInstance('AceOO-2.0')
local Mixin = AceOO.Mixin

local SphereInventory = Mixin {
    'InventoryScan',
    'InventoryScanBag',
    'InventoryParseLink',
    'InventoryCheckItem',
    'InventoryGetItemData',
    'InventoryGetItemCooldown',
    'InventoryGetMountData',
    'InventorySetItem',
}

function SphereInventory:InventoryScan()
    self.inventoryItems = {
        ['HEARTHSTONE'] = nil,
        ['MOUNT'] = nil,
        ['MOUNT_AQ'] = nil,
        ['MOUNT_FLY'] = nil,
        ['POTION_HP'] = nil,
        ['POTION_MP'] = nil,
        ['DRINK'] = nil,
        ['FOOD'] = nil,
    }
    if not self.PT then
        self.PT = AceLibrary:GetInstance("PeriodicTable-3.0");
    end
    
    for bagId=0, 4, 1 do
        self:InventoryScanBag(bagId);
    end
end

function SphereInventory:InventoryScanBag(bagId) 
    -- For each slot
	for slotId=1, GetContainerNumSlots(bagId), 1 do
        local itemLink = GetContainerItemLink(bagId, slotId);
        if itemLink ~= nil then
            local itemId = self:InventoryParseLink(itemLink);
            local _, itemCount = GetContainerItemInfo(bagId, slotId);
            self:InventoryCheckItem(itemId, itemCount);
        end
    end
end

function SphereInventory:InventoryCheckItem(itemId, itemCount)
    local _, _, _, _, itemMinLevel = GetItemInfo(itemId);
    -- hearthstone?  
    if itemId == 6948 then
        self:InventorySetItem('HEARTHSTONE', itemId, 1, false);
    -- is it a heal potion?
    elseif self.PT:ItemInSet(itemId, 'Consumable.Potion.Recovery.Healing.Basic') then
        if self.inventoryItems['POTION_HP'] == nil or self.inventoryItems['POTION_HP'].count <= 0 then
            self:InventorySetItem('POTION_HP', itemId, itemCount, false);
        elseif self.inventoryItems['POTION_HP'].id == itemId then
            self.inventoryItems['POTION_HP'].count = self.inventoryItems['POTION_HP'].count + itemCount;
        elseif self.db.profile.buttonPotionWeakest and self.inventoryItems['POTION_HP'].minLevel > itemMinLevel then
            self:InventorySetItem('POTION_HP', itemId, itemCount, false);
        elseif (not self.db.profile.buttonPotionWeakest) and self.inventoryItems['POTION_HP'].minLevel < itemMinLevel then
            self:InventorySetItem('POTION_HP', itemId, itemCount, false);
        end

    -- is it some conjured food?
    elseif self.PT:ItemInSet(itemId, 'Consumable.Food.Edible.Basic.Conjured') then
        if self.inventoryItems['FOOD'] == nil or self.inventoryItems['FOOD'].count <= 0 then
            self:InventorySetItem('FOOD', itemId, itemCount, true);  
        elseif self.inventoryItems['FOOD'].id == itemId then
            self.inventoryItems['FOOD'].count = self.inventoryItems['FOOD'].count + itemCount;
        elseif not self.inventoryItems['FOOD'].conjured then
            self:InventorySetItem('FOOD', itemId, itemCount, true); 
        elseif self.db.profile.buttonDrinkWeakest and self.inventoryItems['FOOD'].minLevel > itemMinLevel then
            self:InventorySetItem('FOOD', itemId, itemCount, false); 
        elseif (not self.db.profile.buttonDrinkWeakest) and self.inventoryItems['FOOD'].minLevel < itemMinLevel then
            self:InventorySetItem('FOOD', itemId, itemCount, false); 
        end

    -- is it some food?
    elseif self.PT:ItemInSet(itemId, 'Consumable.Food.Edible.Basic.Non-Conjured') then
        if self.inventoryItems['FOOD'] == nil or self.inventoryItems['FOOD'].count <= 0 then
            self:InventorySetItem('FOOD', itemId, itemCount, false);  
        elseif self.inventoryItems['FOOD'].id == itemId then
            self.inventoryItems['FOOD'].count = self.inventoryItems['FOOD'].count + itemCount;
        elseif self.db.profile.buttonDrinkWeakest and (not self.inventoryItems['FOOD'].conjured) and self.inventoryItems['FOOD'].minLevel > itemMinLevel then
            self:InventorySetItem('FOOD', itemId, itemCount, false);  
        elseif (not self.db.profile.buttonDrinkWeakest) and (not self.inventoryItems['FOOD'].conjured) and self.inventoryItems['FOOD'].minLevel < itemMinLevel then
            self:InventorySetItem('FOOD', itemId, itemCount, false);  
        end

    -- is it some food that has bonuses?
    elseif self.PT:ItemInSet(itemId, 'Consumable.Food.Edible.Bonus') then
        if self.inventoryItems['FOODBONUS'] == nil or self.inventoryItems['FOODBONUS'].count <= 0 then
            self:InventorySetItem('FOODBONUS', itemId, itemCount, false);  
        elseif self.inventoryItems['FOODBONUS'].id == itemId then
            self.inventoryItems['FOODBONUS'].count = self.inventoryItems['FOODBONUS'].count + itemCount;
        elseif self.db.profile.buttonDrinkWeakest and (not self.inventoryItems['FOODBONUS'].conjured) and self.inventoryItems['FOODBONUS'].minLevel > itemMinLevel then
            self:InventorySetItem('FOODBONUS', itemId, itemCount, false);  
        elseif (not self.db.profile.buttonDrinkWeakest) and (not self.inventoryItems['FOODBONUS'].conjured) and self.inventoryItems['FOODBONUS'].minLevel < itemMinLevel then
            self:InventorySetItem('FOODBONUS', itemId, itemCount, false);  
        end

    -- is it a Healthstone?
    elseif self.PT:ItemInSet(itemId, 'Consumable.Warlock.Healthstone') then
        if self.inventoryItems['HEALTHSTONE'] == nil or self.inventoryItems['HEALTHSTONE'].count <= 0 then
            self:InventorySetItem('HEALTHSTONE', itemId, itemCount, false);
        elseif self.inventoryItems['HEALTHSTONE'].id == itemId then
            self.inventoryItems['HEALTHSTONE'].count = self.inventoryItems['HEALTHSTONE'].count + itemCount;
        elseif self.db.profile.buttonPotionWeakest and self.inventoryItems['HEALTHSTONE'].minLevel > itemMinLevel then
            self:InventorySetItem('HEALTHSTONE', itemId, itemCount, false);
        elseif (not self.db.profile.buttonPotionWeakest) and self.inventoryItems['HEALTHSTONE'].minLevel < itemMinLevel then
            self:InventorySetItem('HEALTHSTONE', itemId, itemCount, false);
        end

    -- is it a Regeant?
    elseif self.PT:ItemInSet(itemId, 'Reagent.Class.Rogue') then
        if self.inventoryItems['REAGENT'] == nil or self.inventoryItems['REAGENT'].count <= 0 then
            self:InventorySetItem('REAGENT', itemId, itemCount, true);  
        elseif self.inventoryItems['REAGENT'].id == itemId then
            self.inventoryItems['REAGENT'].count = self.inventoryItems['REAGENT'].count + itemCount;
        elseif not self.inventoryItems['REAGENT'].conjured then
            self:InventorySetItem('REAGENT', itemId, itemCount, true); 
        end

    -- is it thistle tea?
    elseif self.PT:ItemInSet(itemId, 'Consumable.Buff.Energy.Self') then
        if self.inventoryItems['THISTLETEA'] == nil or self.inventoryItems['THISTLETEA'].count <= 0 then
            self:InventorySetItem('THISTLETEA', itemId, itemCount, true);  
        elseif self.inventoryItems['THISTLETEA'].id == itemId then
            self.inventoryItems['THISTLETEA'].count = self.inventoryItems['THISTLETEA'].count + itemCount;
        elseif not self.inventoryItems['THISTLETEA'].conjured then
            self:InventorySetItem('THISTLETEA', itemId, itemCount, true); 
        end

    -- is it a deadly poison?
    elseif self.PT:ItemInSet(itemId, 'Consumable.Weapon Buff.Poison.Deadly') then
        if self.inventoryItems['DEADLY_POISON'] == nil or self.inventoryItems['DEADLY_POISON'].count <= 0 then
            self:InventorySetItem('DEADLY_POISON', itemId, itemCount, false);
        elseif self.inventoryItems['DEADLY_POISON'].id == itemId then
            self.inventoryItems['DEADLY_POISON'].count = self.inventoryItems['DEADLY_POISON'].count + itemCount;
        elseif self.db.profile.buttonPoisonWeakest and (not self.inventoryItems['DEADLY_POISON'].conjured) and self.inventoryItems['DEADLY_POISON'].minLevel > itemMinLevel then
            self:InventorySetItem('DEADLY_POISON', itemId, itemCount, false);  
        elseif (not self.db.profile.buttonPoisonWeakest) and (not self.inventoryItems['DEADLY_POISON'].conjured) and self.inventoryItems['DEADLY_POISON'].minLevel < itemMinLevel then
            self:InventorySetItem('DEADLY_POISON', itemId, itemCount, false);  
        end

    -- is it a crippling poison?
    elseif self.PT:ItemInSet(itemId, 'Consumable.Weapon Buff.Poison.Crippling') then
        if self.inventoryItems['CRIPPLE_POISON'] == nil or self.inventoryItems['CRIPPLE_POISON'].count <= 0 then
            self:InventorySetItem('CRIPPLE_POISON', itemId, itemCount, false);
        elseif self.inventoryItems['CRIPPLE_POISON'].id == itemId then
            self.inventoryItems['CRIPPLE_POISON'].count = self.inventoryItems['CRIPPLE_POISON'].count + itemCount;
        elseif self.db.profile.buttonPoisonWeakest and (not self.inventoryItems['CRIPPLE_POISON'].conjured) and self.inventoryItems['CRIPPLE_POISON'].minLevel > itemMinLevel then
            self:InventorySetItem('CRIPPLE_POISON', itemId, itemCount, false);  
        elseif (not self.db.profile.buttonPoisonWeakest) and (not self.inventoryItems['CRIPPLE_POISON'].conjured) and self.inventoryItems['CRIPPLE_POISON'].minLevel < itemMinLevel then
            self:InventorySetItem('CRIPPLE_POISON', itemId, itemCount, false);  
        end

    -- is it a numbing poison?
    elseif self.PT:ItemInSet(itemId, 'Consumable.Weapon Buff.Poison.Mind Numbing') then
        if self.inventoryItems['NUMB_POISON'] == nil or self.inventoryItems['NUMB_POISON'].count <= 0 then
            self:InventorySetItem('NUMB_POISON', itemId, itemCount, false);
        elseif self.inventoryItems['NUMB_POISON'].id == itemId then
            self.inventoryItems['NUMB_POISON'].count = self.inventoryItems['NUMB_POISON'].count + itemCount;
        elseif self.db.profile.buttonPoisonWeakest and (not self.inventoryItems['NUMB_POISON'].conjured) and self.inventoryItems['NUMB_POISON'].minLevel > itemMinLevel then
            self:InventorySetItem('NUMB_POISON', itemId, itemCount, false);  
        elseif (not self.db.profile.buttonPoisonWeakest) and (not self.inventoryItems['NUMB_POISON'].conjured) and self.inventoryItems['NUMB_POISON'].minLevel < itemMinLevel then
            self:InventorySetItem('NUMB_POISON', itemId, itemCount, false);  
        end

    -- is it a instant poison?
    elseif self.PT:ItemInSet(itemId, 'Consumable.Weapon Buff.Poison.Instant') then
        if self.inventoryItems['INSTANT_POISON'] == nil or self.inventoryItems['INSTANT_POISON'].count <= 0 then
            self:InventorySetItem('INSTANT_POISON', itemId, itemCount, false);
        elseif self.inventoryItems['INSTANT_POISON'].id == itemId then
            self.inventoryItems['INSTANT_POISON'].count = self.inventoryItems['INSTANT_POISON'].count + itemCount;
        elseif self.db.profile.buttonPoisonWeakest and (not self.inventoryItems['INSTANT_POISON'].conjured) and self.inventoryItems['INSTANT_POISON'].minLevel > itemMinLevel then
            self:InventorySetItem('INSTANT_POISON', itemId, itemCount, false);  
        elseif (not self.db.profile.buttonPoisonWeakest) and (not self.inventoryItems['INSTANT_POISON'].conjured) and self.inventoryItems['INSTANT_POISON'].minLevel < itemMinLevel then
            self:InventorySetItem('INSTANT_POISON', itemId, itemCount, false);  
        end

    -- is it a wounding poison?
    elseif self.PT:ItemInSet(itemId, 'Consumable.Weapon Buff.Poison.Wound') then
        if self.inventoryItems['WOUND_POISON'] == nil or self.inventoryItems['WOUND_POISON'].count <= 0 then
            self:InventorySetItem('WOUND_POISON', itemId, itemCount, false);
        elseif self.inventoryItems['WOUND_POISON'].id == itemId then
            self.inventoryItems['WOUND_POISON'].count = self.inventoryItems['WOUND_POISON'].count + itemCount;
        elseif self.db.profile.buttonPoisonWeakest and (not self.inventoryItems['WOUND_POISON'].conjured) and self.inventoryItems['WOUND_POISON'].minLevel > itemMinLevel then
            self:InventorySetItem('WOUND_POISON', itemId, itemCount, false);  
        elseif (not self.db.profile.buttonPoisonWeakest) and (not self.inventoryItems['WOUND_POISON'].conjured) and self.inventoryItems['WOUND_POISON'].minLevel < itemMinLevel then
            self:InventorySetItem('WOUND_POISON', itemId, itemCount, false);  
        end

    -- is it a bandage?
    elseif self.PT:ItemInSet(itemId, 'Consumable.Bandage.Basic') then
        if self.inventoryItems['BANDAGE'] == nil or self.inventoryItems['BANDAGE'].count <= 0 then
            self:InventorySetItem('BANDAGE', itemId, itemCount, false);
        elseif self.inventoryItems['BANDAGE'].id == itemId then
            self.inventoryItems['BANDAGE'].count = self.inventoryItems['BANDAGE'].count + itemCount;
        end

   -- is it an AV bandage?
    elseif self.PT:ItemInSet(itemId, 'Consumable.Bandage.Battleground.Alterac Valley') then   
        if self.inventoryItems['BGBANDAGE'] == nil or self.inventoryItems['BGBANDAGE'].count <= 0 then
            self:InventorySetItem('BGBANDAGE', itemId, itemCount, false);  
        elseif self.inventoryItems['BGBANDAGE'].id == itemId then
            self.inventoryItems['BGBANDAGE'].count = self.inventoryItems['BGBANDAGE'].count + itemCount;
        end

   -- is it an AB bandage?
    elseif self.PT:ItemInSet(itemId, 'Consumable.Bandage.Battleground.Arathi Basin') then   
        if self.inventoryItems['BGBANDAGE'] == nil or self.inventoryItems['BGBANDAGE'].count <= 0 then
            self:InventorySetItem('BGBANDAGE', itemId, itemCount, false);  
        elseif self.inventoryItems['BGBANDAGE'].id == itemId then
            self.inventoryItems['BGBANDAGE'].count = self.inventoryItems['BGBANDAGE'].count + itemCount;
        end

   -- is it a WSG bandage?
    elseif self.PT:ItemInSet(itemId, 'Consumable.Bandage.Battleground.Warsong Gulch') then   
        if self.inventoryItems['BGBANDAGE'] == nil or self.inventoryItems['BGBANDAGE'].count <= 0 then
            self:InventorySetItem('BGBANDAGE', itemId, itemCount, false);  
        elseif self.inventoryItems['BGBANDAGE'].id == itemId then
            self.inventoryItems['BGBANDAGE'].count = self.inventoryItems['BGBANDAGE'].count + itemCount;
        end

    -- is it a mount?
    elseif self.PT:ItemInSet(itemId, 'Misc.Mount.Normal') or self.PT:ItemInSet(itemId, 'Misc.Mount.Ahn\'Qiraj') or self.PT:ItemInSet(itemId, 'Misc.Mount.Flying') then
        -- normal/epic riding  mount
        if self.PT:ItemInSet(itemId, 'Misc.Mount.Normal') then
            if self.inventoryItems['MOUNTS'] == nil then
                self:InventorySetItem('MOUNTS', itemId, 1, false);  
            elseif self.inventoryItems['MOUNTS'].minLevel < itemMinLevel then
                self:InventorySetItem('MOUNTS', itemId, 1, false);  
            end
        end
        -- aq mount
        if self.PT:ItemInSet(itemId, 'Misc.Mount.Ahn\'Qiraj') then
            if self.inventoryItems['MOUNTS_AQ'] == nil then
                self:InventorySetItem('MOUNTS_AQ', itemId, false);  
            elseif self.inventoryItems['MOUNTS_AQ'].minLevel < itemMinLevel then
                self:InventorySetItem('MOUNTS_AQ', itemId, 1, false);  
            end        
        end
        -- flying mount 
        if self.PT:ItemInSet(itemId, 'Misc.Mount.Flying') then
            if self.inventoryItems['MOUNTS_FLY'] == nil then
                self:InventorySetItem('MOUNTS_FLY', itemId, 1, false);  
            elseif self.inventoryItems['MOUNTS_FLY'].minLevel < itemMinLevel then
                self:InventorySetItem('MOUNTS_FLY', itemId, 1, false);  
            end        
        end
    end
end

function SphereInventory:InventorySetItem(groupId, itemId, itemCount, conjured)
    local itemName, itemString, _, itemLevel, itemMinLevel, _, _, _, _, itemTexture = GetItemInfo(itemId);
    local itemTextureFile = 'WoWUnknownItem01';
    if itemTexture ~= nil then 
        _, _, itemTextureFile = string.find(itemTexture, '([^\\]+)$'); 
    end
    if self.playerData.level ~= nil and (itemMinLevel == nil or itemMinLevel <= self.playerData.level) then
        if self.inventoryItems[groupId] == nil then
            self.inventoryItems[groupId] = {};
        end
        self.inventoryItems[groupId]['id'] = itemId;
        self.inventoryItems[groupId]['name'] = itemName;
        self.inventoryItems[groupId]['count'] = itemCount;
        self.inventoryItems[groupId]['minLevel'] = itemMinLevel;
        self.inventoryItems[groupId]['texture'] = itemTextureFile;
        self.inventoryItems[groupId]['conjured'] = conjured;
        self.inventoryItems[groupId]['spell'] = GetItemSpell(itemName);
    end
end

function SphereInventory:InventoryParseLink(linkStr) 
	if (linkStr) then
		local _, _, id = string.find(linkStr,'|Hitem:(%d+)');
		if (id) then
			return tonumber(id);
		end
	end
end

function SphereInventory:InventoryGetItemCooldown(itemName)
    if itemName ~= nil and itemName then
        local startTime, duration, enable = GetItemCooldown(itemName);
        if enable and startTime ~= nil and startTime > 0 and duration ~= nil and duration > 1 then
            return math.floor(startTime - GetTime() + duration + 0.5);
        end
    end
    return 0;
end

function SphereInventory:InventoryGetItemData(groupId) 
    if self.inventoryItems ~= nil and self.inventoryItems[groupId] ~= nil then
        local cooldown = self:InventoryGetItemCooldown(self.inventoryItems[groupId].name); 
        return self.inventoryItems[groupId].count, self.inventoryItems[groupId].name, self.inventoryItems[groupId].texture, self.inventoryItems[groupId].spell, cooldown;
    else
        return nil;
    end
end

function SphereInventory:InventoryGetMountData()
    local currentMount = nil;
    if  self.inventoryItems == nil then
        self.inventoryItems = {};
    end
    if self.playerData.inAQ and self.inventoryItems['MOUNTS_AQ'] ~= nil then
        currentMount = 'MOUNTS_AQ';
    end
    if currentMount == nil and self.playerData.inOutlands and self.inventoryItems['MOUNTS_FLY'] ~= nil then
        currentMount = 'MOUNTS_FLY';
    end
    if currentMount == nil and self.inventoryItems['MOUNTS'] ~= nil then
        currentMount = 'MOUNTS';
    end
    if currentMount ~= nil and self.inventoryItems[currentMount] ~= nil then
        return self.inventoryItems[currentMount].name, self.inventoryItems[currentMount].texture, self.inventoryItems[currentMount].spell;
    end
    return nil;
end


AceLibrary:Register(SphereInventory, MAJOR_VERSION, MINOR_VERSION, SphereInventory.activate)
SphereInventory = AceLibrary(MAJOR_VERSION)