--[[ SLDataText - Copyright (C) 2008 Jeff "Taffu" Fancher | Version: 1.0 | License: GPL v3.0 ]]
--[[ XP.lua - Revised: 03/25/2008 ]]

local SLDataText = LibStub("AceAddon-3.0"):GetAddon("SLDataText")
local XP = SLDataText:NewModule("XP", "AceEvent-3.0", "AceTimer-3.0", "AceConsole-3.0")
local SML = LibStub("LibSharedMedia-3.0")
local TBL = SML:List("font")
local L = LibStub("AceLocale-3.0"):GetLocale("SLDataText")
local MODNAME = "XP"
local coreDB
local db

local defaults = {
    profile = {
        minimal = true,
        showRest = true,
        showPerc = false,
        basetext = "xp",
        fontSize = 16,
        fontFace = "Arial Narrow",
        useCoreFace = true,
        useCoreSize = true,
        pos = { anchorPoint = "RIGHT", anchor = "UIParent", anchorFrom = "RIGHT", x = -100, y = 0 },
        justify = "RIGHT",
        suppTT = false,
        overrideCombatHide = false,
    },
}

local options = {
    xp = {
        name = L["XP"],
        arg = MODNAME,
        type = "group",
        args = {
            lagDesc = {
                order = 1,
                type = "description",
                name = L["XP Desc"],
            },
            enabled = {
				order = 2,
				type = "toggle",
				name = L["Enable XP"],
				get = function() return SLDataText:GetModuleEnabled(MODNAME) end,
				set = function(info, value) SLDataText:SetModuleEnabled(MODNAME, value) end,
			},
            overrideCombatHide = {
                order = 3,
                type = "toggle",
                name = L["Override Combat Hide"],
                get = function() return db.overrideCombatHide end,
                set = function() db.overrideCombatHide = not db.overrideCombatHide XP:Refresh() end,
            },
            style = {
                order = 2,
                name = L["Style"],
                type = "group",
                args = {
                    btDesc = {
                        order = 0,
                        type = "description",
                        name = L["Base Text Desc"],
                    },
                    basetext = {
                        order = 1,
                        type = "input",
                        name = L["Base Text"],
                        get = function() return db.basetext end,
                        set = function(_, value) db.basetext = value XP:Refresh() end,
                    },
                    showRestDesc = {
                        order = 2,
                        type = "description",
                        name = L["Show Rest Desc"],
                    },
                    showRest = {
                        order = 3,
                        type = "toggle",
                        name = L["Show Rest"],
                        get = function() return db.showRest end,
                        set = function() db.showRest = not db.showRest XP:Refresh() end,
                    },
                    showPercDesc = {
                        order = 4,
                        type = "description",
                        name = L["Show Perc Desc"],
                    },
                    showPerc = {
                        order = 5,
                        type = "toggle",
                        name = L["Show Percentage"],
                        get = function() return db.showPerc end,
                        set = function() db.showPerc = not db.showPerc XP:Refresh() end,
                    },
                    suppTTDesc = {
                        order = 6,
                        type = "description",
                        name = L["suppTTDesc"],
                    },
                    suppTT = {
                        order = 7,
                        type = "toggle",
                        name = L["suppTT"],
                        get = function() return db.suppTT end,
                        set = function() db.suppTT = not db.suppTT XP:Refresh() end,
                    },
                    useCoreDesc = {
                        order = 9,
                        type = "description",
                        name = L["Use Core Desc"],
                    },
                    useCoreFace = {
                        order = 10,
                        type = "toggle",
                        name = L["Use Core Font Face"],
                        get = function() return db.useCoreFace end,
                        set = function() db.useCoreFace = not db.useCoreFace XP:Refresh() end,
                    },
                    useCoreSize = {
                        order = 11,
                        type = "toggle",
                        name = L["Use Core Font Size"],
                        get = function() return db.useCoreSize end,
                        set = function() db.useCoreSize = not db.useCoreSize XP:Refresh() end,
                    },
                    useSelfDesc = {
                        order = 12,
                        type = "description",
                        name = L["Use Self Desc"],
                    },
                    fontFace = {
                        order = 13,
                        type = "select",
                        values = SML:List("font"),
                        name = L["Font Face"],
                        get = function()
                            for k, v in pairs(TBL) do
                                if db.fontFace == v then
                                    return k
                                end
                            end
                        end,
                        set = function(_, font)
                            db.fontFace = TBL[font]
                            font = SML:Fetch("font", TBL[font])
                            XP:Refresh()
                        end,
                        disabled = function()
                            local istrue
                            if ( db.useCoreFace ) then istrue = true else istrue = false end
                            return istrue
                        end,
                    },
                    spacer = { order = 14, type = "description", name = "" },
                    fontSize = {
                        order = 15,
                        type = "range",
                        name = L["Font Size"],
                        min = 4, max = 36, step = 1,
                        get = function() return db.fontSize end,
                        set = function(_, value) db.fontSize = value XP:Refresh() end,
                        disabled = function()
                            local istrue
                            if ( db.useCoreSize ) then istrue = true else istrue = false end
                            return istrue
                        end,
                    },
                },
            },
            position = {
                order = 3,
                type = "group",
                name = L["Position"],
                args = {
                    justifyDesc = {
                        order = 1,
                        type = "description",
                        name = L["Justify Desc"],
                    },
                    justify = {
                        order = 2,
                        type = "select",
                        values = SLDataText.JustTbl,
                        name = L["Set Justify"],
                        get = function() return db.justify end,
                        set = function(_, value) db.justify = value XP:Refresh() end,
                    },
                    parentDesc = {
                        order = 3,
                        type = "description",
                        name = L["Parent Desc"],
                    },
                    parent = {
                        order = 4,
                        type = "input",
                        name = L["Parent Anchor"],
                        get = function() return db.pos.anchor end,
                        set = function(_, value) db.pos.anchor = value XP:Refresh() end,
                    },
                    anchorFromDesc = {
                        order = 5,
                        type = "description",
                        name = L["Anchor From Desc"],
                    },
                    anchorFrom = {
                        order = 6,
                        type = "select",
                        values = SLDataText.PointTbl,
                        name = L["Anchor From"],
                        get = function() return db.pos.anchorFrom end,
                        set = function(_, value) db.pos.anchorFrom = value XP:Refresh() end,
                    },
                    offsetDesc = { 
                        order = 7, 
                        type = "description", 
                        name = L["Offset Desc"],
                    },
                    posX = {
                        order = 8,
                        type = "input",
                        name = L["X Offset"],
                        get = function() return db.pos.x end,
                        set = function(_, value) db.pos.x = value XP:Refresh() end,
                    },
                    spacer2 = { order = 9, type = "description", name = "" },
                    posY = {
                        order = 10,
                        type = "input",
                        name = L["Y Offset"],
                        get = function() return db.pos.y end,
                        set = function(_, value) db.pos.y = value XP:Refresh() end,
                    },
                },
            },
        },
    },
}

function XP:OnInitialize()
    self.db = SLDataText.db:RegisterNamespace(MODNAME, defaults)
    db = self.db.profile
    coreDB = SLDataText.db.profile
    -- Check plugins enabled state
    self:SetEnabledState(SLDataText:GetModuleEnabled(MODNAME))
    -- Add Plugins Options
    SLDataText:InjectOptions(MODNAME, options)
end

function XP:OnEnable()
    -- Check frame or build frame
    if ( not self.frame ) then 
        self.frame = CreateFrame("Frame", "SLDTXP", UIParent)
        self.text = self.frame:CreateFontString(nil, "OVERLAY")
    end
    -- Check shown if enabled, show if not already
    if ( not self.frame:IsShown() ) then self.frame:Show() end
    -- Register applicable events
    self:RegisterEvent("PLAYER_ENTERING_WORLD", "Refresh")
    self:RegisterEvent("PLAYER_LEVEL_UP", "Refresh")
    self:RegisterEvent("PLAYER_XP_UPDATE", "Refresh")
    self:RegisterEvent("UPDATE_EXHAUSTION", "Refresh")
    self:RegisterEvent("PLAYER_UPDATE_RESTING", "Refresh")
end

function XP:OnDisable()
    -- Check shown if disabled, hide if not already
    if ( self.frame:IsShown() ) then self.frame:Hide() end
end

function XP:Refresh()
    -- Get XP Values
    self.curXP, self.maxXP = UnitXP("player"), UnitXPMax("player")
    self.xpToGo = (self.maxXP - self.curXP)
    self.curXPPerc = ((self.curXP / self.maxXP) * 100)
    self.curXPPercToGo = (100 - ((self.curXP / self.maxXP) * 100))
    -- Get Rest XP Values
    if ( GetXPExhaustion() ) then self.restXP = GetXPExhaustion() else self.restXP = 0 end
    self.restXPPerc = ((self.restXP / self.maxXP) * 100)
    
    if ( db.showPerc ) then
        self.xp = string.format("%s%s", floor(self.curXPPercToGo), "%")
    else
        self.xp = self.xpToGo
    end
    if ( db.showRest ) then
        if ( self.restXP > 0 ) then
            self.rxp = string.format(" (%s%s)", floor(self.restXPPerc), "%")
        else
            self.rxp = ""
        end
    else
        self.rxp = ""
    end
    
    -------------------------------------
    -- Tooltip --------------------------
    -------------------------------------
    if ( coreDB.isLocked and not db.suppTT and UnitLevel("player") ~= 70 ) then
        self.frame:SetScript("OnEnter", function(this)
            GameTooltip:SetOwner(this, "ANCHOR_CURSOR")
            GameTooltip:AddLine(string.format("%s%s|r", SLDataText.ColorTable["enUS"]["Yellow"], L["Experience Points"]))
            GameTooltip:AddLine("-------------------------", 1, 1, 1)
            GameTooltip:AddDoubleLine(string.format("%s%s|r", SLDataText.ColorTable["enUS"]["Yellow"], L["Current XP:"]), string.format("%s (%.0f%s)", self.curXP, self.curXPPerc, "%"), 1, 1, 1, 1, 1, 1)
            GameTooltip:AddDoubleLine(string.format("%s%s|r", SLDataText.ColorTable["enUS"]["Yellow"], L["Needed XP:"]), self.maxXP, 1, 1, 1, 1, 1, 1)
            GameTooltip:AddDoubleLine(string.format("%s%s|r", SLDataText.ColorTable["enUS"]["Yellow"], L["XP To Go:"]), string.format("%s (%.0f%s)", self.xpToGo, self.curXPPercToGo, "%"), 1, 1, 1, 1, 1, 1)
            if ( self.restXP > 0 ) then
                GameTooltip:AddLine("-------------------------", 1, 1, 1)
                GameTooltip:AddDoubleLine(string.format("%s%s|r", SLDataText.ColorTable["enUS"]["Yellow"], L["Rest XP:"]), string.format("%s (%.0f%s)", self.restXP, self.restXPPerc, "%"), 1, 1, 1, 1, 1, 1)
            end
            GameTooltip:SetScale(1)
            GameTooltip:Show()
        end)
        self.frame:SetScript("OnLeave", function()
            if ( GameTooltip:IsShown() ) then
                GameTooltip:Hide()
            end
        end)
    else
        self.frame:SetScript("OnEnter", nil)
        self.frame:SetScript("OnLeave", nil)
    end
    -------------------------------------
    
    self.xpsuffix = db.basetext
    -- Update Frame
    self:RefreshFrame()
end

function XP:RefreshFrame()
    local curFont, curFontSize
    if ( db.useCoreFace ) then curFont = SLDataText.db.profile.coreFontFace else curFont = db.fontFace end
    if ( db.useCoreSize ) then curSize = SLDataText.db.profile.coreFontSize else curSize = db.fontSize end
    self.text:SetFont(SML:Fetch("font", curFont), curSize)
    self.text:SetShadowColor(0, 0, 0, 1)
    self.text:SetShadowOffset(1.5, -1.5)
    self.text:SetJustifyH(db.justify)
    
    -- Set Text
    local color1, color2 = SLDataText.vColor, SLDataText.btColor
    if ( UnitLevel("player") == 70 ) then
        self.text:SetFormattedText("%s%s|r", color2, "Max Level")
    else
        self.text:SetFormattedText("%s%s%s|r%s%s|r", color1, self.xp, self.rxp, color2, self.xpsuffix)
    end

    -- Frame Properties
    self.frame:SetWidth(self.text:GetWidth())
    self.frame:SetHeight(self.text:GetHeight())
    self.frame:EnableMouse(true)
    
    -- Movement & Position
    if not coreDB.isLocked then
        self.frame:SetMovable(true)
        self.frame:RegisterForDrag("LeftButton")
        self.frame:SetScript("OnMouseDown", function()
            SLDataText:MoveFrame(self, db)
        end)
        self.frame:SetScript("OnMouseUp", function()
            SLDataText:StopFrame(self, db)
        end)
    else
        self.frame:SetMovable(false)
        self.frame:RegisterForDrag(nil)
        self.frame:SetScript("OnMouseDown", nil)
        self.frame:SetScript("OnMouseUp", nil)
    end
    
    if not self.isMoving then
        self.frame:ClearAllPoints()
        self.frame:SetPoint(db.justify, db.pos.anchor, db.pos.anchorFrom, db.pos.x, db.pos.y)
        self.text:ClearAllPoints()
        self.text:SetPoint(db.justify, self.frame, db.justify, 0, 0)
    end
end