--[[ SLDataText - Copyright (C) 2008 Jeff "Taffu" Fancher | Version: 1.0 | License: GPL v3.0 ]]
--[[ Tracking.lua - Revised: 03/28/2008 ]]

local SLDataText = LibStub("AceAddon-3.0"):GetAddon("SLDataText")
local Tracking = SLDataText:NewModule("Tracking", "AceEvent-3.0", "AceTimer-3.0", "AceConsole-3.0")
local SML = LibStub("LibSharedMedia-3.0")
local TBL = SML:List("font")
local L = LibStub("AceLocale-3.0"):GetLocale("SLDataText")
local MODNAME = "Tracking"
local coreDB
local tborder = "Interface\\Minimap\\MiniMap-TrackingBorder"

local db
local defaults = {
    profile = {
        iconOnly = true,
        textOnly = false,
        iconSize = 22,
        hideBorder = false,
        fontSize = 16,
        fontFace = "Arial Narrow",
        useCoreFace = true,
        useCoreSize = true,
        pos = { anchorPoint = "TOPLEFT", anchor = "Minimap", anchorFrom = "TOPLEFT", x = -10, y = -40 },
        justify = "LEFT",
        overrideCombatHide = false,
        suppTT = false,
    },
}

local options = {
    track = {
        name = L["Tracking"],
        arg = MODNAME,
        type = "group",
        args = {
            lagDesc = {
                order = 1,
                type = "description",
                name = L["Tracking Desc"],
            },
            enabled = {
				order = 2,
				type = "toggle",
				name = L["Enable Tracking"],
				get = function() return SLDataText:GetModuleEnabled(MODNAME) end,
				set = function(info, value) SLDataText:SetModuleEnabled(MODNAME, value) end,
			},
            overrideCombatHide = {
                order = 3,
                type = "toggle",
                name = L["Override Combat Hide"],
                get = function() return db.overrideCombatHide end,
                set = function() db.overrideCombatHide = not db.overrideCombatHide Tracking:Refresh() end,
            },
            style = {
                order = 2,
                name = L["Style"],
                type = "group",
                args = {
                    iconOnlyDesc = {
                        order = 0,
                        type = "description",
                        name = L["Icon Only Desc"],
                    },
                    iconOnly = {
                        order = 1,
                        type = "toggle",
                        name = L["Icon Only"],
                        get = function() return db.iconOnly end,
                        set = function() db.iconOnly = not db.iconOnly Tracking:Refresh() end,
                        disabled = function()
                            local istrue
                            if ( db.textOnly ) then istrue = true else istrue = false end
                            return istrue
                        end,
                    },
                    iconSize = {
                        order = 2,
                        type = "range",
                        name = L["Icon Size"],
                        min = 4, max = 36, step = 1,
                        get = function() return db.iconSize end,
                        set = function(_, value) db.iconSize = value Tracking:Refresh() end,
                        disabled = function()
                            local istrue
                            if ( not db.iconOnly or db.textOnly ) then istrue = true else istrue = false end
                            return istrue
                        end,
                    },
                    hideBorder = {
                        order = 3,
                        type = "toggle",
                        name = L["Hide Border"],
                        get = function() return db.hideBorder end,
                        set = function() db.hideBorder = not db.hideBorder Tracking:Refresh() end,
                    },
                    textOnlyDesc = {
                        order = 4,
                        type = "description",
                        name = L["Text Only Desc"],
                    },
                    textOnly = {
                        order = 5,
                        type = "toggle",
                        name = L["Text Only"],
                        get = function() return db.textOnly end,
                        set = function() db.textOnly = not db.textOnly Tracking:Refresh() end,
                        disabled = function()
                            local istrue
                            if ( db.iconOnly ) then istrue = true else istrue = false end
                            return istrue
                        end,
                    },
                    suppTTDesc = {
                        order = 6,
                        type = "description",
                        name = L["suppTTDesc"],
                    },
                    suppTT = {
                        order = 7,
                        type = "toggle",
                        name = L["suppTT"],
                        get = function() return db.suppTT end,
                        set = function() db.suppTT = not db.suppTT Tracking:Refresh() end,
                    },
                    useCoreDesc = {
                        order = 9,
                        type = "description",
                        name = L["Use Core Desc"],
                    },
                    useCoreFace = {
                        order = 10,
                        type = "toggle",
                        name = L["Use Core Font Face"],
                        get = function() return db.useCoreFace end,
                        set = function() db.useCoreFace = not db.useCoreFace Tracking:Refresh() end,
                    },
                    useCoreSize = {
                        order = 11,
                        type = "toggle",
                        name = L["Use Core Font Size"],
                        get = function() return db.useCoreSize end,
                        set = function() db.useCoreSize = not db.useCoreSize Tracking:Refresh() end,
                    },
                    useSelfDesc = {
                        order = 12,
                        type = "description",
                        name = L["Use Self Desc"],
                    },
                    fontFace = {
                        order = 13,
                        type = "select",
                        values = SML:List("font"),
                        name = L["Font Face"],
                        get = function()
                            for k, v in pairs(TBL) do
                                if db.fontFace == v then
                                    return k
                                end
                            end
                        end,
                        set = function(_, font)
                            db.fontFace = TBL[font]
                            font = SML:Fetch("font", TBL[font])
                            Tracking:Refresh()
                        end,
                        disabled = function()
                            local istrue
                            if ( db.useCoreFace ) then istrue = true else istrue = false end
                            return istrue
                        end,
                    },
                    spacer = { order = 14, type = "description", name = "" },
                    fontSize = {
                        order = 15,
                        type = "range",
                        name = L["Font Size"],
                        min = 4, max = 36, step = 1,
                        get = function() return db.fontSize end,
                        set = function(_, value) db.fontSize = value Tracking:Refresh() end,
                        disabled = function()
                            local istrue
                            if ( db.useCoreSize ) then istrue = true else istrue = false end
                            return istrue
                        end,
                    },
                },
            },
            position = {
                order = 3,
                type = "group",
                name = L["Position"],
                args = {
                    justifyDesc = {
                        order = 1,
                        type = "description",
                        name = L["Justify Desc"],
                    },
                    justify = {
                        order = 2,
                        type = "select",
                        values = SLDataText.JustTbl,
                        name = L["Set Justify"],
                        get = function() return db.justify end,
                        set = function(_, value) db.justify = value Tracking:Refresh() end,
                    },
                    parentDesc = {
                        order = 3,
                        type = "description",
                        name = L["Parent Desc"],
                    },
                    parent = {
                        order = 4,
                        type = "input",
                        name = L["Parent Anchor"],
                        get = function() return db.pos.anchor end,
                        set = function(_, value) db.pos.anchor = value Tracking:Refresh() end,
                    },
                    anchorFromDesc = {
                        order = 5,
                        type = "description",
                        name = L["Anchor From Desc"],
                    },
                    anchorFrom = {
                        order = 6,
                        type = "select",
                        values = SLDataText.PointTbl,
                        name = L["Anchor From"],
                        get = function() return db.pos.anchorFrom end,
                        set = function(_, value) db.pos.anchorFrom = value Tracking:Refresh() end,
                    },
                    offsetDesc = { 
                        order = 7, 
                        type = "description", 
                        name = L["Offset Desc"],
                    },
                    posX = {
                        order = 8,
                        type = "input",
                        name = L["X Offset"],
                        get = function() return db.pos.x end,
                        set = function(_, value) db.pos.x = value Tracking:Refresh() end,
                    },
                    spacer2 = { order = 9, type = "description", name = "" },
                    posY = {
                        order = 10,
                        type = "input",
                        name = L["Y Offset"],
                        get = function() return db.pos.y end,
                        set = function(_, value) db.pos.y = value Tracking:Refresh() end,
                    },
                },
            },
        },
    },
}

function Tracking:OnInitialize()
    -- Register plugins svars
    self.db = SLDataText.db:RegisterNamespace(MODNAME, defaults)
    db = self.db.profile
    coreDB = SLDataText.db.profile
    -- Check plugins enabled state
    self:SetEnabledState(SLDataText:GetModuleEnabled(MODNAME))
    -- Add Plugins Options
    SLDataText:InjectOptions(MODNAME, options)
end

function Tracking:OnEnable()
    -- Check frame or build frame
    if ( not self.frame ) then 
        self.frame = CreateFrame("Frame", "SLDTTracking", UIParent)
        self.text = self.frame:CreateFontString(nil, "OVERLAY")
        self.texture = self.frame:CreateTexture(nil, "ARTWORK")
        self.border = self.frame:CreateTexture(nil, "OVERLAY")
        -- Menu button
        self.button = CreateFrame("Button", nil, self.frame)
        self.menu = CreateFrame("Frame", nil, self.button, UIDropDownMenuTemplate)
        self.menu:SetClampedToScreen(true)
        -- Setup Menu Basics
        self.menu:SetID(1)
        self.menu:SetScript("OnLoad", function() MiniMapTrackingDropDown_OnLoad() end)
    end
    -- Check shown if enabled, show if not already
    if ( not self.frame:IsShown() ) then self.frame:Show() end
    if ( MiniMapTracking:IsShown() ) then MiniMapTracking:Hide() end
    -- Register applicable events
    self:RegisterEvent("PLAYER_ENTERING_WORLD", "Refresh")
    self:RegisterEvent("MINIMAP_UPDATE_TRACKING", "Refresh")
end

function Tracking:OnDisable()
    -- Check shown if disabled, hide if not already
    if ( self.frame:IsShown() ) then self.frame:Hide() end
end

function Tracking:Refresh()
    -- Gather tracking info
    local name, texture, active, category
    local anyActive = false
    local count = GetNumTrackingTypes()
    for id = 1, count do
        name, _, active, category = GetTrackingInfo(id)
        if active then
            anyActive = true
            self.tracktext = name
        end
    end
    
    if not anyActive then
        self.tracktext = L["None"]
        self.icon = GetTrackingTexture()
    else
        self.icon = GetTrackingTexture()
    end
    
    if ( db.iconOnly ) then self.tracktext = "" end
    
    -- Update Frame
    self:RefreshFrame()
end

function Tracking:RefreshFrame()
    local curFont, curFontSize
    if ( db.useCoreFace ) then curFont = SLDataText.db.profile.coreFontFace else curFont = db.fontFace end
    if ( db.useCoreSize ) then curSize = SLDataText.db.profile.coreFontSize else curSize = db.fontSize end
    self.text:SetFont(SML:Fetch("font", curFont), curSize)
    self.text:SetShadowColor(0, 0, 0, 1)
    self.text:SetShadowOffset(1.5, -1.5)
    self.text:SetJustifyH(db.justify)
    
    -- Set Text & Icon
    local color1 = SLDataText.vColor
    if ( db.iconOnly ) then
        self.text:SetText("")
        self.texture:SetTexture(self.icon)
        self.texture:SetHeight(db.iconSize)
        self.texture:SetWidth(db.iconSize)
        if ( not self.texture:IsShown() ) then self.texture:Show() end
        if ( not self.border:IsShown() ) then self.border:Show() end
    elseif ( db.textOnly ) then
        self.text:SetText(string.format("%s%s|r", color1, self.tracktext))
        if ( self.texture:IsShown() ) then self.texture:Hide() end
        if ( self.border:IsShown() ) then self.border:Hide() end
    else
        self.text:SetText(string.format("%s%s|r", color1, self.tracktext))
        self.texture:SetTexture(self.icon)
        self.texture:SetHeight(self.text:GetHeight()+2)
        self.texture:SetWidth(self.text:GetHeight()+2)
        if ( not self.texture:IsShown() ) then self.texture:Show() end
        if ( not self.border:IsShown() ) then self.border:Show() end
    end

    -- Frame Properties
    if ( db.textOnly ) then
        self.frame:SetWidth(self.text:GetWidth())
        self.frame:SetHeight(self.text:GetHeight())
    elseif ( db.iconOnly ) then
        self.frame:SetWidth(self.texture:GetWidth())
        self.frame:SetHeight(self.texture:GetHeight())
    else
        self.frame:SetWidth(self.text:GetWidth()+self.texture:GetWidth())
        self.frame:SetHeight(max(self.text:GetHeight(), self.texture:GetHeight()))
    end
    self.frame:EnableMouse(true)
    
    -- Movement & Position
    if ( not coreDB.isLocked ) then
        self.frame:SetMovable(true)
        self.frame:RegisterForDrag("LeftButton")
        self.frame:SetScript("OnMouseDown", function()
            SLDataText:MoveFrame(self, db)
        end)
        self.frame:SetScript("OnMouseUp", function()
            SLDataText:StopFrame(self, db)
        end)
    else
        self.frame:SetMovable(false)
        self.frame:RegisterForDrag(nil)
        self.frame:SetScript("OnMouseDown", nil)
        self.frame:SetScript("OnMouseUp", nil)
        ---------------------------------------------------------
        -- Tracking Menu ----------------------------------------
        ---------------------------------------------------------
        self.button:SetAllPoints(self.frame)
        self.button:SetScript("OnClick", function()
            ToggleDropDownMenu(1, nil, MiniMapTrackingDropDown, self.button, 0, 5)
        end)
    end
    
    if not self.isMoving then
        self.frame:ClearAllPoints()
        self.frame:SetPoint(db.justify, db.pos.anchor, db.pos.anchorFrom, db.pos.x, db.pos.y)
        self.text:ClearAllPoints()
        if ( db.textOnly ) then 
            self.text:SetPoint(db.justify, self.frame, db.justify, 0, 0)
        elseif ( not db.textOnly and db.justify == "LEFT" ) then
            self.text:SetPoint(db.justify, self.texture, "RIGHT", 0, 0)
        elseif ( not db.textOnly and db.justify == "RIGHT" ) then
            self.text:SetPoint(db.justify, self.frame, db.justify, 0, 0)
        elseif ( not db.textOnly and db.justify == "CENTER" ) then
            local twidth = self.text:GetWidth() + self.texture:GetWidth()
            local cp = twidth/2
            self.text:SetPoint(db.justify, self.frame, "RIGHT", cp, 0)
        end
        self.texture:ClearAllPoints()
        if ( db.iconOnly ) then 
            self.texture:SetPoint(db.justify, self.frame, db.justify, 0, 0)
        elseif ( not db.iconOnly and db.justify == "LEFT" ) then
            self.texture:SetPoint(db.justify, self.frame, db.justify, 0, 0)
        elseif ( not db.iconOnly and db.justify == "RIGHT" ) then
            self.texture:SetPoint(db.justify, self.text, "LEFT", 0, 0)
        elseif ( not db.iconOnly and db.justify == "CENTER") then
            self.texture:SetPoint("RIGHT", self.text, "LEFT", 0, 0)
        end
    end
    
    if ( not db.hideBorder ) then
        self.border:ClearAllPoints()
        self.border:SetTexture(tborder)
        self.border:SetWidth(self.texture:GetWidth()*2.4)
        self.border:SetHeight(self.texture:GetHeight()*2.4)
        self.border:SetPoint("CENTER", self.texture, "CENTER", self.texture:GetWidth()/2, self.texture:GetHeight()/2-self.texture:GetHeight()/2*2)
        if ( not db.textOnly ) then
            if ( not self.border:IsShown() ) then self.border:Show() end
        end
    else
        if ( self.border:IsShown() ) then self.border:Hide() end
    end
end