--[[ SLDataText - Copyright (C) 2008 Jeff "Taffu" Fancher | Last Revision: 1.02 | License: GPL v3.0 ]]
--[[ Memory.lua - Revised: 04/01/2008 ]]

local SLDataText = LibStub("AceAddon-3.0"):GetAddon("SLDataText")
local Memory = SLDataText:NewModule("Memory", "AceEvent-3.0", "AceTimer-3.0", "AceConsole-3.0")
local SML = LibStub("LibSharedMedia-3.0")
local TBL = SML:List("font")
local L = LibStub("AceLocale-3.0"):GetLocale("SLDataText")
local MODNAME = "Memory"
local coreDB

local db
local defaults = {
    profile = {
        showBlizz = false,
        precision = 1,
        basetext = "mb",
        numAddOnsShow = 10,
        fontSize = 16,
        fontFace = "Arial Narrow",
        useCoreFace = true,
        useCoreSize = true,
        pos = { anchorPoint = "RIGHT", anchor = "UIParent", anchorFrom = "RIGHT", x = -100, y = 60 },
        justify = "RIGHT",
        overrideCombatHide = false,
        suppTT = false,
    },
}

local options = {
    mem = {
        name = L["Memory"],
        arg = MODNAME,
        type = "group",
        args = {
            lagDesc = {
                order = 1,
                type = "description",
                name = L["Mem Desc"],
            },
            enabled = {
				order = 2,
				type = "toggle",
				name = L["Enable Memory"],
				get = function() return SLDataText:GetModuleEnabled(MODNAME) end,
				set = function(info, value) SLDataText:SetModuleEnabled(MODNAME, value) end,
			},
            overrideCombatHide = {
                order = 3,
                type = "toggle",
                name = L["Override Combat Hide"],
                get = function() return db.overrideCombatHide end,
                set = function() db.overrideCombatHide = not db.overrideCombatHide Memory:Refresh() end,
            },
            style = {
                order = 2,
                name = L["Style"],
                type = "group",
                args = {
                    btDesc = {
                        order = 0,
                        type = "description",
                        name = L["Base Text Desc"],
                    },
                    basetext = {
                        order = 1,
                        type = "input",
                        name = L["Base Text"],
                        get = function() return db.basetext end,
                        set = function(_, value) db.basetext = value Memory:Refresh() end,
                    },
                    precDesc = {
                        order = 2,
                        type = "description",
                        name = L["Precision Desc"]
                    },
                    precision = {
                        order = 3,
                        type = "range",
                        name = L["Decimal Precision"],
                        min = 0, max = 2, step = 1,
                        get = function() return db.precision end,
                        set = function(_, value) db.precision = value Memory:Refresh() end,
                    },
                    numAddShownDesc = {
                        order = 4,
                        type = "description",
                        name = L["Shown Addon Count Desc"],
                    },
                    numAddShown = {
                        order = 5,
                        type = "range",
                        min = 5, max = 150, step = 5,
                        name = L["Shown Addon Count"],
                        get = function() return db.numAddOnsShow end,
                        set = function(_, value) db.numAddOnsShow = value Memory:Refresh() end,
                    },
                    showBlizz = {
                        order = 6,
                        type = "toggle",
                        name = L["Show Blizz Memory"],
                        get = function() return db.showBlizz end,
                        set = function() db.showBlizz = not db.showBlizz Memory:Refresh() end,
                    },
                    suppTTDesc = {
                        order = 7,
                        type = "description",
                        name = L["suppTTDesc"],
                    },
                    suppTT = {
                        order = 8,
                        type = "toggle",
                        name = L["suppTT"],
                        get = function() return db.suppTT end,
                        set = function() db.suppTT = not db.suppTT Memory:Refresh() end,
                    },
                    useCoreDesc = {
                        order = 9,
                        type = "description",
                        name = L["Use Core Desc"],
                    },
                    useCoreFace = {
                        order = 10,
                        type = "toggle",
                        name = L["Use Core Font Face"],
                        get = function() return db.useCoreFace end,
                        set = function() db.useCoreFace = not db.useCoreFace Memory:Refresh() end,
                    },
                    useCoreSize = {
                        order = 11,
                        type = "toggle",
                        name = L["Use Core Font Size"],
                        get = function() return db.useCoreSize end,
                        set = function() db.useCoreSize = not db.useCoreSize Memory:Refresh() end,
                    },
                    useSelfDesc = {
                        order = 12,
                        type = "description",
                        name = L["Use Self Desc"],
                    },
                    fontFace = {
                        order = 13,
                        type = "select",
                        values = SML:List("font"),
                        name = L["Font Face"],
                        get = function()
                            for k, v in pairs(TBL) do
                                if db.fontFace == v then
                                    return k
                                end
                            end
                        end,
                        set = function(_, font)
                            db.fontFace = TBL[font]
                            font = SML:Fetch("font", TBL[font])
                            Memory:Refresh()
                        end,
                        disabled = function()
                            local istrue
                            if ( db.useCoreFace ) then istrue = true else istrue = false end
                            return istrue
                        end,
                    },
                    spacer = { order = 14, type = "description", name = "" },
                    fontSize = {
                        order = 15,
                        type = "range",
                        name = L["Font Size"],
                        min = 4, max = 36, step = 1,
                        get = function() return db.fontSize end,
                        set = function(_, value) db.fontSize = value Memory:Refresh() end,
                        disabled = function()
                            local istrue
                            if ( db.useCoreSize ) then istrue = true else istrue = false end
                            return istrue
                        end,
                    },
                },
            },
            position = {
                order = 3,
                type = "group",
                name = L["Position"],
                args = {
                    justifyDesc = {
                        order = 1,
                        type = "description",
                        name = L["Justify Desc"],
                    },
                    justify = {
                        order = 2,
                        type = "select",
                        values = SLDataText.JustTbl,
                        name = L["Set Justify"],
                        get = function() return db.justify end,
                        set = function(_, value) db.justify = value Memory:Refresh() end,
                    },
                    parentDesc = {
                        order = 3,
                        type = "description",
                        name = L["Parent Desc"],
                    },
                    parent = {
                        order = 4,
                        type = "input",
                        name = L["Parent Anchor"],
                        get = function() return db.pos.anchor end,
                        set = function(_, value) db.pos.anchor = value Memory:Refresh() end,
                    },
                    anchorFromDesc = {
                        order = 5,
                        type = "description",
                        name = L["Anchor From Desc"],
                    },
                    anchorFrom = {
                        order = 6,
                        type = "select",
                        values = SLDataText.PointTbl,
                        name = L["Anchor From"],
                        get = function() return db.pos.anchorFrom end,
                        set = function(_, value) db.pos.anchorFrom = value Memory:Refresh() end,
                    },
                    offsetDesc = { 
                        order = 7, 
                        type = "description", 
                        name = L["Offset Desc"],
                    },
                    posX = {
                        order = 8,
                        type = "input",
                        name = L["X Offset"],
                        get = function() return db.pos.x end,
                        set = function(_, value) db.pos.x = value Memory:Refresh() end,
                    },
                    spacer2 = { order = 9, type = "description", name = "" },
                    posY = {
                        order = 10,
                        type = "input",
                        name = L["Y Offset"],
                        get = function() return db.pos.y end,
                        set = function(_, value) db.pos.y = value Memory:Refresh() end,
                    },
                },
            },
        },
    },
}

function Memory:OnInitialize()
    self.db = SLDataText.db:RegisterNamespace(MODNAME, defaults)
    db = self.db.profile
    coreDB = SLDataText.db.profile
    -- Check plugins enabled state & add options
    self:SetEnabledState(SLDataText:GetModuleEnabled(MODNAME))
    SLDataText:InjectOptions(MODNAME, options)
end

function Memory:OnEnable()
    if ( not self.frame ) then 
        self.frame = CreateFrame("Frame", "SLDTMemory", UIParent)
        self.text = self.frame:CreateFontString(nil, "OVERLAY")
    end
    if ( not self.frame:IsShown() ) then self.frame:Show() end
    
    self:RegisterEvent("PLAYER_ENTERING_WORLD", "Refresh")
    self:ScheduleRepeatingTimer("Refresh", 5)
end

function Memory:OnDisable()
    if ( self.frame:IsShown() ) then self.frame:Hide() end
end

function Memory:Refresh()
    UpdateAddOnMemoryUsage()
    local mem = collectgarbage("count")
    local realmem = 0
    local count = GetNumAddOns()
    
    for i = 1, count do realmem = realmem + GetAddOnMemoryUsage(i) end
    if ( db.showBlizz ) then
        self.mem = format("%."..db.precision.."f", mem/1000)
    else
        self.mem = format("%."..db.precision.."f", realmem/1000)
    end
    self.memsuffix = db.basetext    
    
    if ( coreDB.isLocked and not db.suppTT ) then
        local topAddOns = {}
        for i = 1, db.numAddOnsShow do topAddOns[i] = { value = 0, name = "" } end
        
        self.frame:SetScript("OnEnter", function(this)
            GameTooltip:SetOwner(this, "ANCHOR_CURSOR")
            GameTooltip:AddLine(format("%s%s|r", SLDataText.ColorTable["enUS"]["Yellow"], L["Memory Stats"]))
            GameTooltip:AddLine("-------------------------", 1, 1, 1)
            local numAddOns = GetNumAddOns()
            local addonMemTbl = {}
            
            for i = 1, db.numAddOnsShow, 1 do topAddOns[i].value = 0 end
            
            for i = 1, numAddOns, 1 do
                local name, title, _, enabled, _, _, _ = GetAddOnInfo(i)
                local addonMem = GetAddOnMemoryUsage(i)
                if ( enabled ) then
                    table.insert(addonMemTbl, { title, addonMem })
                    for j = 1, db.numAddOnsShow, 1 do
                        if ( addonMem > topAddOns[j].value ) then
                            for k = db.numAddOnsShow, 1, -1 do
                                if ( k == j ) then
                                    topAddOns[k].value = addonMem
                                    topAddOns[k].name = title
                                    break
                                elseif ( k ~= 1 ) then
                                    topAddOns[k].value = topAddOns[k-1].value
                                    topAddOns[k].name = topAddOns[k-1].name
                                end
                            end
                            break
                        end
                    end
                end
            end
            for i = 1, db.numAddOnsShow, 1 do
                if ( topAddOns[i].value == 0 ) then
                    break
                end
                local size, name = topAddOns[i].value, topAddOns[i].name
                if ( size > 1000 ) then
                    size = format("%.1f %s", size / 1000, "mb")
                    GameTooltip:AddDoubleLine(name, size, 1, 1, 1, 1, 1, 1)
                else
                    size = format("%.1f %s", size, "kb")
                    GameTooltip:AddDoubleLine(name, size, 1, 1, 1, 1, 1, 1)
                end
            end
            
            GameTooltip:SetScale(1)
            GameTooltip:Show()
        end)
        self.frame:SetScript("OnLeave", function()
            if ( GameTooltip:IsShown() ) then
                GameTooltip:Hide()
            end
        end)
    else
        self.frame:SetScript("OnEnter", nil)
        self.frame:SetScript("OnLeave", nil)
    end
    
    self:RefreshFrame()
end

Memory.isMoving = false

function Memory:RefreshFrame()
    local curFont, curFontSize
    if ( db.useCoreFace ) then curFont = SLDataText.db.profile.coreFontFace else curFont = db.fontFace end
    if ( db.useCoreSize ) then curSize = SLDataText.db.profile.coreFontSize else curSize = db.fontSize end
    self.text:SetFont(SML:Fetch("font", curFont), curSize)
    self.text:SetShadowColor(0, 0, 0, 1)
    self.text:SetShadowOffset(1.5, -1.5)
    self.text:SetJustifyH(db.justify)
    
    local color1, color2 = SLDataText.vColor, SLDataText.btColor
    self.text:SetFormattedText("%s%s|r%s%s|r", color1, self.mem, color2, self.memsuffix)

    self.frame:SetWidth(self.text:GetWidth())
    self.frame:SetHeight(self.text:GetHeight())
    self.frame:EnableMouse(true)
    
    if not coreDB.isLocked then
        self.frame:SetMovable(true)
        self.frame:RegisterForDrag("LeftButton")
        self.frame:SetScript("OnMouseDown", function()
            SLDataText:MoveFrame(self, db)
        end)
        self.frame:SetScript("OnMouseUp", function()
            SLDataText:StopFrame(self, db)
        end)
    else
        self.frame:SetMovable(false)
        self.frame:RegisterForDrag(nil)
        self.frame:SetScript("OnMouseDown", nil)
        self.frame:SetScript("OnMouseUp", nil)
    end
    
    if not self.isMoving then
        self.frame:ClearAllPoints()
        self.frame:SetPoint(db.justify, db.pos.anchor, db.pos.anchorFrom, db.pos.x, db.pos.y)
        self.text:ClearAllPoints()
        self.text:SetPoint(db.justify, self.frame, db.justify, 0, 0)
    end
end