--[[ SLDataText - Copyright (C) 2008 Jeff "Taffu" Fancher | Last Revision: 1.02 | License: GPL v3.0 ]]
--[[ Friends.lua - Revised: 04/01/2008 ]]

local SLDataText = LibStub("AceAddon-3.0"):GetAddon("SLDataText")
local Friends = SLDataText:NewModule("Friends", "AceEvent-3.0", "AceTimer-3.0", "AceConsole-3.0")
local SML = LibStub("LibSharedMedia-3.0")
local Tablet = LibStub("Tablet-2.0")
local TBL = SML:List("font")
local L = LibStub("AceLocale-3.0"):GetLocale("SLDataText")
local MODNAME = "Friends"
local coreDB
local charDB

local db
local defaults = {
    profile = {
        basetext = L["Friends:"],
        fontSize = 16,
        fontFace = "Arial Narrow",
        useCoreFace = true,
        useCoreSize = true,
        pos = { anchorPoint = "RIGHT", anchor = "UIParent", anchorFrom = "RIGHT", x = -100, y = -80 },
        justify = "RIGHT",
        suppTT = false,
        overrideCombatHide = false,
        ttPoint = "TOP",
        ttfPoint = "BOTTOM",
    },
}

local options = {
    friends = {
        name = L["Friends"],
        arg = MODNAME,
        type = "group",
        args = {
            desc = {
                order = 1,
                type = "description",
                name = L["Friends Desc"],
            },
            enabled = {
				order = 2,
				type = "toggle",
				name = L["Enable Friends"],
				get = function() return SLDataText:GetModuleEnabled(MODNAME) end,
				set = function(info, value) SLDataText:SetModuleEnabled(MODNAME, value) end,
			},
            overrideCombatHide = {
                order = 3,
                type = "toggle",
                name = L["Override Combat Hide"],
                get = function() return db.overrideCombatHide end,
                set = function() db.overrideCombatHide = not db.overrideCombatHide Friends:Refresh() end,
            },
            style = {
                order = 2,
                name = L["Style"],
                type = "group",
                args = {
                    btDesc = {
                        order = 0,
                        type = "description",
                        name = L["Base Text Desc"],
                    },
                    basetext = {
                        order = 1,
                        type = "input",
                        name = L["Base Text"],
                        get = function() return db.basetext end,
                        set = function(_, value) db.basetext = value Friends:Refresh() end,
                    },
                    suppTTDesc = {
                        order = 7,
                        type = "description",
                        name = L["suppTTDesc"],
                    },
                    suppTT = {
                        order = 8,
                        type = "toggle",
                        name = L["suppTT"],
                        get = function() return db.suppTT end,
                        set = function() db.suppTT = not db.suppTT Friends:Refresh() end,
                    },
                    useCoreDesc = {
                        order = 9,
                        type = "description",
                        name = L["Use Core Desc"],
                    },
                    useCoreFace = {
                        order = 10,
                        type = "toggle",
                        name = L["Use Core Font Face"],
                        get = function() return db.useCoreFace end,
                        set = function() db.useCoreFace = not db.useCoreFace Friends:Refresh() end,
                    },
                    useCoreSize = {
                        order = 11,
                        type = "toggle",
                        name = L["Use Core Font Size"],
                        get = function() return db.useCoreSize end,
                        set = function() db.useCoreSize = not db.useCoreSize Friends:Refresh() end,
                    },
                    useSelfDesc = {
                        order = 12,
                        type = "description",
                        name = L["Use Self Desc"],
                    },
                    fontFace = {
                        order = 13,
                        type = "select",
                        values = SML:List("font"),
                        name = L["Font Face"],
                        get = function()
                            for k, v in pairs(TBL) do
                                if db.fontFace == v then
                                    return k
                                end
                            end
                        end,
                        set = function(_, font)
                            db.fontFace = TBL[font]
                            font = SML:Fetch("font", TBL[font])
                            Friends:Refresh()
                        end,
                        disabled = function()
                            local istrue
                            if ( db.useCoreFace ) then istrue = true else istrue = false end
                            return istrue
                        end,
                    },
                    spacer = { order = 14, type = "description", name = "" },
                    fontSize = {
                        order = 15,
                        type = "range",
                        name = L["Font Size"],
                        min = 4, max = 36, step = 1,
                        get = function() return db.fontSize end,
                        set = function(_, value) db.fontSize = value Friends:Refresh() end,
                        disabled = function()
                            local istrue
                            if ( db.useCoreSize ) then istrue = true else istrue = false end
                            return istrue
                        end,
                    },
                },
            },
            position = {
                order = 3,
                type = "group",
                name = L["Position"],
                args = {
                    justifyDesc = {
                        order = 1,
                        type = "description",
                        name = L["Justify Desc"],
                    },
                    justify = {
                        order = 2,
                        type = "select",
                        values = SLDataText.JustTbl,
                        name = L["Set Justify"],
                        get = function() return db.justify end,
                        set = function(_, value) db.justify = value Friends:Refresh() end,
                    },
                    parentDesc = {
                        order = 3,
                        type = "description",
                        name = L["Parent Desc"],
                    },
                    parent = {
                        order = 4,
                        type = "input",
                        name = L["Parent Anchor"],
                        get = function() return db.pos.anchor end,
                        set = function(_, value) db.pos.anchor = value Friends:Refresh() end,
                    },
                    anchorFromDesc = {
                        order = 5,
                        type = "description",
                        name = L["Anchor From Desc"],
                    },
                    anchorFrom = {
                        order = 6,
                        type = "select",
                        values = SLDataText.PointTbl,
                        name = L["Anchor From"],
                        get = function() return db.pos.anchorFrom end,
                        set = function(_, value) db.pos.anchorFrom = value Friends:Refresh() end,
                    },
                    offsetDesc = { 
                        order = 7, 
                        type = "description", 
                        name = L["Offset Desc"],
                    },
                    posX = {
                        order = 8,
                        type = "input",
                        name = L["X Offset"],
                        get = function() return db.pos.x end,
                        set = function(_, value) db.pos.x = value Friends:Refresh() end,
                    },
                    spacer2 = { order = 9, type = "description", name = "" },
                    posY = {
                        order = 10,
                        type = "input",
                        name = L["Y Offset"],
                        get = function() return db.pos.y end,
                        set = function(_, value) db.pos.y = value Friends:Refresh() end,
                    },
                },
            },
            tooltip = {
                order = 4,
                type = "group",
                name = L["Tooltip"],
                args = {
                    ttDesc = {
                        order = 1,
                        type = "description",
                        name = L["Tooltip Desc"],
                    },
                    ttPoint = {
                        order = 2,
                        type = "select",
                        values = SLDataText.PointTbl,
                        name = L["Anchor"],
                        get = function() return db.ttPoint end,
                        set = function(_, value) db.ttPoint = value Friends:Refresh() end,
                    },
                    ttfPoint = {
                        order = 3,
                        type = "select",
                        values = SLDataText.PointTbl,
                        name = L["Anchor From"],
                        get = function() return db.ttfPoint end,
                        set = function(_, value) db.ttfPoint = value Friends:Refresh() end,
                    },
                },
            },
        },
    },
}

function Friends:OnInitialize()
    self.db = SLDataText.db:RegisterNamespace(MODNAME, defaults)
    db = self.db.profile
    coreDB = SLDataText.db.profile
    charDB = SLDataText.db.char
    if ( not charDB.Friends ) then charDB.Friends = {} end
    -- Check plugins enabled state & add options
    self:SetEnabledState(SLDataText:GetModuleEnabled(MODNAME))
    SLDataText:InjectOptions(MODNAME, options)
end

function Friends:OnEnable()
    if ( not self.frame ) then 
        self.frame = CreateFrame("Frame", "SLDTFriends", UIParent)
        self.text = self.frame:CreateFontString(nil, "OVERLAY")
    end
    if ( not self.frame:IsShown() ) then self.frame:Show() end
    
    self:RegisterEvent("PLAYER_ENTERING_WORLD", "Refresh")
    self:RegisterEvent("FRIENDLIST_UPDATE", "Refresh")
    ShowFriends()
end

function Friends:OnDisable()
    if ( self.frame:IsShown() ) then self.frame:Hide() end
end

function Friends:Refresh()
    self.List = nil
    local friendstotal = GetNumFriends()
    local friendsonline = 0
    
    for i = 0, friendstotal do
        local name, lvl, class, area, online, status = GetFriendInfo(i)
        if ( online ) then
            friendsonline = friendsonline + 1
            if ( not self.List or self.List == nil ) then self.List = {} end
            local client = GetLocale() 
            local classcolor = SLDataText.ColorTable[client][class]
            local coloredclass = string.format("%s%s|r", classcolor, class)
            local coloredname
            if ( status == "" ) then
                coloredname = string.format("%s%s|r", classcolor, name)
            else
                coloredname = string.format("%s %s%s|r", status, classcolor, name)
            end
            table.insert(self.List, { coloredname, lvl, coloredclass, area, name })
        end
    end
    
    self.fonline = friendsonline
    
    if ( coreDB.isLocked and not db.suppTT and self.fonline > 0 ) then
        self.frame:SetScript("OnEnter", function() Tablet:Open(self.frame) end)
    else
        self.frame:SetScript("OnEnter", nil)
    end
    if ( not Tablet:IsRegistered(self.frame) ) then
        Tablet:Register(self.frame,
            "children", function()
                self:UpdateTablet()
            end,
            "point", function()
                return db.ttPoint
            end,
            "relativePoint", function()
                return db.ttfPoint
            end,
            "maxHeight", 500,
            "clickable", true,
            "hideWhenEmpty", true
        )
    end
    
    self:RefreshFrame()
end

function Friends:UpdateTablet()
    if ( self.fonline > 0 ) then
        ShowFriends()
        local header = Tablet:AddCategory()
        header:AddLine('text', L["Friend List"], 'size', 14)
        
        local col = {}
        table.insert(col, L["Name"])
        table.insert(col, L["Level"])
        table.insert(col, L["Class"])
        table.insert(col, L["Area"])
        
        local cat = Tablet:AddCategory("columns", #col)
        local header = {}
        for i = 1, #col do
            if ( i == 1 ) then
                header['text'] = col[i]
                header['justify'] = "CENTER"
            else
                header['text'..i] = col[i]
                header['justify'..i] = "CENTER"
            end
        end
        cat:AddLine(header)
        for _, val in ipairs(self.List) do
            local line = {}
            for i = 1, #col do
                if ( i == 1 ) then
                    line['text'] = val[i]
                    line['justify'] = "CENTER"
                    line['func'] = function() self:ListSendTell(val[5]) end
                elseif ( i == 3 ) then
                    line['text'..i] = val[i]
                    line['justify'..i] = "CENTER"
                else
                    line['text'..i] = val[i]
                    line['justify'..i] = "CENTER"
                    line['text'..i..'R'] = 1
                    line['text'..i..'G'] = 1
                    line['text'..i..'B'] = 1
                end
            end
            cat:AddLine(line)
        end
        
        Tablet:SetHint(L["Friends Hint"])
    end
end

function Friends:ListSendTell(name)
    if ( not name ) then return end
    if ( IsAltKeyDown() ) then
        InviteUnit(name)
    else
        SetItemRef("player:"..name, "|Hplayer:"..name.."|h["..name.."|h", "LeftButton")
    end
end

function Friends:RefreshFrame()
    local curFont, curFontSize
    if ( db.useCoreFace ) then curFont = SLDataText.db.profile.coreFontFace else curFont = db.fontFace end
    if ( db.useCoreSize ) then curSize = SLDataText.db.profile.coreFontSize else curSize = db.fontSize end
    self.text:SetFont(SML:Fetch("font", curFont), curSize)
    self.text:SetShadowColor(0, 0, 0, 1)
    self.text:SetShadowOffset(1.5, -1.5)
    self.text:SetJustifyH(db.justify)
    
    local color1, color2 = SLDataText.vColor, SLDataText.btColor
    self.text:SetFormattedText("%s%s|r %s%s|r", color2, db.basetext, color1, self.fonline)   
    
    self.frame:SetWidth(self.text:GetWidth())
    self.frame:SetHeight(self.text:GetHeight())
    
    if not coreDB.isLocked then
        self.frame:SetMovable(true)
        self.frame:RegisterForDrag("LeftButton")
        self.frame:EnableMouse(true)
        self.frame:SetScript("OnMouseDown", function()
            SLDataText:MoveFrame(self, db)
        end)
        self.frame:SetScript("OnMouseUp", function()
            SLDataText:StopFrame(self, db)
        end)
    else
        self.frame:SetMovable(false)
        self.frame:RegisterForDrag(nil)
        self.frame:EnableMouse(true)
        self.frame:SetScript("OnMouseDown", nil)
        self.frame:SetScript("OnMouseUp", nil)
    end
    
    if not self.isMoving then
        self.frame:ClearAllPoints()
        self.frame:SetPoint(db.justify, db.pos.anchor, db.pos.anchorFrom, db.pos.x, db.pos.y)
        self.text:ClearAllPoints()
        self.text:SetPoint(db.justify, self.frame, db.justify, 0, 0)
    end
end