--[[ SLDataText - Copyright (C) 2008 Jeff "Taffu" Fancher | Last Revision: 1.02 | License: GPL v3.0 ]]
--[[ Durability.lua - Revised: 04/03/2008 ]]

local SLDataText = LibStub("AceAddon-3.0"):GetAddon("SLDataText")
local Durability = SLDataText:NewModule("Durability", "AceEvent-3.0", "AceTimer-3.0", "AceConsole-3.0")
local SML = LibStub("LibSharedMedia-3.0")
local TBL = SML:List("font")
local L = LibStub("AceLocale-3.0"):GetLocale("SLDataText")
local MODNAME = "Durability"
local coreDB

local db
local defaults = {
    profile = {
        basetext = "AP:",
        fontSize = 16,
        fontFace = "Arial Narrow",
        useCoreFace = true,
        useCoreSize = true,
        pos = { anchorPoint = "RIGHT", anchor = "UIParent", anchorFrom = "RIGHT", x = -100, y = -60 },
        justify = "RIGHT",
        suppTT = false,
        overrideCombatHide = false,
    },
}

local options = {
    dura = {
        name = L["Durability"],
        arg = MODNAME,
        type = "group",
        args = {
            desc = {
                order = 1,
                type = "description",
                name = L["Durability Desc"],
            },
            enabled = {
				order = 2,
				type = "toggle",
				name = L["Enable Durability"],
				get = function() return SLDataText:GetModuleEnabled(MODNAME) end,
				set = function(info, value) SLDataText:SetModuleEnabled(MODNAME, value) end,
			},
            overrideCombatHide = {
                order = 3,
                type = "toggle",
                name = L["Override Combat Hide"],
                get = function() return db.overrideCombatHide end,
                set = function() db.overrideCombatHide = not db.overrideCombatHide Durability:Refresh() end,
            },
            style = {
                order = 2,
                name = L["Style"],
                type = "group",
                args = {
                    btDesc = {
                        order = 0,
                        type = "description",
                        name = L["Base Text Desc"],
                    },
                    basetext = {
                        order = 1,
                        type = "input",
                        name = L["Base Text"],
                        get = function() return db.basetext end,
                        set = function(_, value) db.basetext = value Durability:Refresh() end,
                    },
                    suppTTDesc = {
                        order = 6,
                        type = "description",
                        name = L["suppTTDesc"],
                    },
                    suppTT = {
                        order = 7,
                        type = "toggle",
                        name = L["suppTT"],
                        get = function() return db.suppTT end,
                        set = function() db.suppTT = not db.suppTT Durability:Refresh() end,
                    },
                    useCoreDesc = {
                        order = 9,
                        type = "description",
                        name = L["Use Core Desc"],
                    },
                    useCoreFace = {
                        order = 10,
                        type = "toggle",
                        name = L["Use Core Font Face"],
                        get = function() return db.useCoreFace end,
                        set = function() db.useCoreFace = not db.useCoreFace Durability:Refresh() end,
                    },
                    useCoreSize = {
                        order = 11,
                        type = "toggle",
                        name = L["Use Core Font Size"],
                        get = function() return db.useCoreSize end,
                        set = function() db.useCoreSize = not db.useCoreSize Durability:Refresh() end,
                    },
                    useSelfDesc = {
                        order = 12,
                        type = "description",
                        name = L["Use Self Desc"],
                    },
                    fontFace = {
                        order = 13,
                        type = "select",
                        values = SML:List("font"),
                        name = L["Font Face"],
                        get = function()
                            for k, v in pairs(TBL) do
                                if db.fontFace == v then
                                    return k
                                end
                            end
                        end,
                        set = function(_, font)
                            db.fontFace = TBL[font]
                            font = SML:Fetch("font", TBL[font])
                            Durability:Refresh()
                        end,
                        disabled = function()
                            local istrue
                            if ( db.useCoreFace ) then istrue = true else istrue = false end
                            return istrue
                        end,
                    },
                    spacer = { order = 14, type = "description", name = "" },
                    fontSize = {
                        order = 15,
                        type = "range",
                        name = L["Font Size"],
                        min = 4, max = 36, step = 1,
                        get = function() return db.fontSize end,
                        set = function(_, value) db.fontSize = value Durability:Refresh() end,
                        disabled = function()
                            local istrue
                            if ( db.useCoreSize ) then istrue = true else istrue = false end
                            return istrue
                        end,
                    },
                },
            },
            position = {
                order = 3,
                type = "group",
                name = L["Position"],
                args = {
                    justifyDesc = {
                        order = 1,
                        type = "description",
                        name = L["Justify Desc"],
                    },
                    justify = {
                        order = 2,
                        type = "select",
                        values = SLDataText.JustTbl,
                        name = L["Set Justify"],
                        get = function() return db.justify end,
                        set = function(_, value) db.justify = value Durability:Refresh() end,
                    },
                    parentDesc = {
                        order = 3,
                        type = "description",
                        name = L["Parent Desc"],
                    },
                    parent = {
                        order = 4,
                        type = "input",
                        name = L["Parent Anchor"],
                        get = function() return db.pos.anchor end,
                        set = function(_, value) db.pos.anchor = value Durability:Refresh() end,
                    },
                    anchorFromDesc = {
                        order = 5,
                        type = "description",
                        name = L["Anchor From Desc"],
                    },
                    anchorFrom = {
                        order = 6,
                        type = "select",
                        values = SLDataText.PointTbl,
                        name = L["Anchor From"],
                        get = function() return db.pos.anchorFrom end,
                        set = function(_, value) db.pos.anchorFrom = value Durability:Refresh() end,
                    },
                    offsetDesc = { 
                        order = 7, 
                        type = "description", 
                        name = L["Offset Desc"],
                    },
                    posX = {
                        order = 8,
                        type = "input",
                        name = L["X Offset"],
                        get = function() return db.pos.x end,
                        set = function(_, value) db.pos.x = value Durability:Refresh() end,
                    },
                    spacer2 = { order = 9, type = "description", name = "" },
                    posY = {
                        order = 10,
                        type = "input",
                        name = L["Y Offset"],
                        get = function() return db.pos.y end,
                        set = function(_, value) db.pos.y = value Durability:Refresh() end,
                    },
                },
            },
        },
    },
}

function Durability:OnInitialize()
    self.db = SLDataText.db:RegisterNamespace(MODNAME, defaults)
    db = self.db.profile
    coreDB = SLDataText.db.profile
    -- Check plugins enabled state & add options
    self:SetEnabledState(SLDataText:GetModuleEnabled(MODNAME))
    SLDataText:InjectOptions(MODNAME, options)
end

function Durability:OnEnable()
    if ( not self.frame ) then 
        self.frame = CreateFrame("Frame", "SLDTDurability", UIParent)
        self.text = self.frame:CreateFontString(nil, "OVERLAY")
    end
    if ( not self.frame:IsShown() ) then self.frame:Show() end
    
    self:RegisterEvent("PLAYER_ENTERING_WORLD", "Refresh")
    --self:RegisterEvent("UPDATE_INVENTORY_ALERTS", "Refresh")
    self:RegisterEvent("UPDATE_INVENTORY_DURABILITY", "Refresh")
end

function Durability:OnDisable()
    if ( self.frame:IsShown() ) then self.frame:Hide() end
end

Durability.SlotInfo = {
    [1] = { slot = "HeadSlot", name = L["Head"], equip = false, value = 0, max = 0, perc = 100 },
    [2] = { slot = "ShoulderSlot", name = L["Shoulder"], equip = false, value = 0, max = 0, perc = 100 },
    [3] = { slot = "ChestSlot", name = L["Chest"], equip = false, value = 0, max = 0, perc = 100 },
    [4] = { slot = "WaistSlot", name = L["Waist"], equip = false, value = 0, max = 0, perc = 100 },
    [5] = { slot = "WristSlot", name = L["Wrist"], equip = false, value = 0, max = 0, perc = 100 },
    [6] = { slot = "HandsSlot", name = L["Hands"], equip = false, value = 0, max = 0, perc = 100 },
    [7] = { slot = "LegsSlot", name = L["Legs"], equip = false, value = 0, max = 0, perc = 100 },
    [8] = { slot = "FeetSlot", name = L["Feet"], equip = false, value = 0, max = 0, perc = 100 },
    [9] = { slot = "MainHandSlot", name = L["Main Hand"], equip = false, value = 0, max = 0, perc = 100 },
    [10] = { slot = "SecondaryHandSlot", name = L["Off Hand"], equip = false, value = 0, max = 0, perc = 100 },
    [11] = { slot = "RangedSlot", name = L["Ranged"], equip = false, value = 0, max = 0, perc = 100 },
}

function Durability:Refresh()
    for i = 1, 11 do
        local slotID = GetInventorySlotInfo(self.SlotInfo[i].slot)
        local itemLink = GetInventoryItemLink("player", slotID)
        if ( itemLink ~= nil ) then 
            self.SlotInfo[i].equip = true
        else
            self.SlotInfo[i].equip = false
        end
        
        local value, maximum = GetInventoryItemDurability(slotID)
        if ( value ~= nil ) then self.SlotInfo[i].value = value else self.SlotInfo[i].value = 0 end
        if ( maximum ~= nil ) then self.SlotInfo[i].max = maximum else self.SlotInfo[i].max = 0 end
        self.SlotInfo[i].perc = floor((self.SlotInfo[i].value/self.SlotInfo[i].max)*100)
        
        local minVal = 100
        for i = 1, 11 do
            if ( self.SlotInfo[i].perc < minVal ) then minVal = self.SlotInfo[i].perc end
        end
        
        self.durability = format("%.0f%s", minVal, "%")
    end
    
    if ( coreDB.isLocked and not db.suppTT ) then
        self.frame:SetScript("OnEnter", function()
            GameTooltip:SetOwner(this, "ANCHOR_CURSOR")
            GameTooltip:AddLine(format("%s%s|r", SLDataText.ColorTable["enUS"]["Yellow"], L["Durability Stats"]))
            GameTooltip:AddLine("-------------------------", 1, 1, 1)
            for i = 1, 11 do
                if ( self.SlotInfo[i].equip == true ) then
                    local slot = self.SlotInfo[i].name
                    local value = self.SlotInfo[i].perc
                    if ( value <= 50 ) then
                        value = format("%s%.0f%s|r", SLDataText.ColorTable["enUS"]["Yellow"], value, "%")
                    elseif ( value <= 25 ) then
                        value = format("%s%.0f%s|r", SLDataText.ColorTable["enUS"]["Red"], value, "%")
                    else
                        value = format("%.0f%s", value, "%")
                    end
                    if ( value ~= -1 ) then
                        GameTooltip:AddDoubleLine(slot, value, 1, 1, 1, 1, 1, 1)
                    end
                end
            end
            GameTooltip:Show()
        end)
        self.frame:SetScript("OnLeave", function()
            if GameTooltip:IsShown() then
                GameTooltip:Hide()
            end
        end)
    else
        self.frame:SetScript("OnEnter", nil)
        self.frame:SetScript("OnLeave", nil)
    end
    
    self:RefreshFrame()
end

function Durability:RefreshFrame()
    local curFont, curFontSize
    if ( db.useCoreFace ) then curFont = SLDataText.db.profile.coreFontFace else curFont = db.fontFace end
    if ( db.useCoreSize ) then curSize = SLDataText.db.profile.coreFontSize else curSize = db.fontSize end
    self.text:SetFont(SML:Fetch("font", curFont), curSize)
    self.text:SetShadowColor(0, 0, 0, 1)
    self.text:SetShadowOffset(1.5, -1.5)
    self.text:SetJustifyH(db.justify)
    
    local color1, color2 = SLDataText.vColor, SLDataText.btColor
    self.text:SetFormattedText("%s%s|r %s%s|r", color2, db.basetext, color1, self.durability)  

    self.frame:SetWidth(self.text:GetWidth())
    self.frame:SetHeight(self.text:GetHeight())
    self.frame:EnableMouse(true)
    
    if not coreDB.isLocked then
        self.frame:SetMovable(true)
        self.frame:RegisterForDrag("LeftButton")
        self.frame:SetScript("OnMouseDown", function()
            SLDataText:MoveFrame(self, db)
        end)
        self.frame:SetScript("OnMouseUp", function()
            SLDataText:StopFrame(self, db)
        end)
    else
        self.frame:SetMovable(false)
        self.frame:RegisterForDrag(nil)
        self.frame:SetScript("OnMouseDown", nil)
        self.frame:SetScript("OnMouseUp", nil)
    end
    
    if not self.isMoving then
        self.frame:ClearAllPoints()
        self.frame:SetPoint(db.justify, db.pos.anchor, db.pos.anchorFrom, db.pos.x, db.pos.y)
        self.text:ClearAllPoints()
        self.text:SetPoint(db.justify, self.frame, db.justify, 0, 0)
    end
end