--[[ SLDataText - Copyright (C) 2008 Jeff "Taffu" Fancher | Last Revision: 1.02 | License: GPL v3.0 ]]
--[[ Clock.lua - Revised: 04/01/2008 ]]

local SLDataText = LibStub("AceAddon-3.0"):GetAddon("SLDataText")
local Clock = SLDataText:NewModule("Clock", "AceEvent-3.0", "AceTimer-3.0", "AceConsole-3.0")
local SML = LibStub("LibSharedMedia-3.0")
local TBL = SML:List("font")
local L = LibStub("AceLocale-3.0"):GetLocale("SLDataText")
local MODNAME = "Clock"
local coreDB

local db
local defaults = {
    profile = {
        is24Hour = false,
        isServerTime = false,
        isBoth = false,
        fontSize = 16,
        fontFace = "Arial Narrow",
        useCoreFace = true,
        useCoreSize = true,
        seperator = ":",
        showSeconds = false,
        pos = { anchorPoint = "RIGHT", anchor = "Minimap", anchorFrom = "BOTTOM", x = 0, y = -5 },
        justify = "RIGHT",
        overrideCombatHide = false,
    },
}

local options = {
    clock = {
        name = L["Clock"],
        arg = MODNAME,
        type = "group",
        args = {
            clockDesc = {
                order = 0,
                type = "description",
                name = L["Clock Desc"],
            },
            enabled = {
				order = 1,
				type = "toggle",
				name = L["Enable Clock"],
				get = function() return SLDataText:GetModuleEnabled(MODNAME) end,
				set = function(info, value) SLDataText:SetModuleEnabled(MODNAME, value) end,
			},
            overrideCombatHide = {
                order = 3,
                type = "toggle",
                name = L["Override Combat Hide"],
                get = function() return db.overrideCombatHide end,
                set = function() db.overrideCombatHide = not db.overrideCombatHide Clock:Refresh() end,
            },
            style = {
                order = 2,
                name = L["Style"],
                type = "group",
                args = {
                    showSecDesc = {
                        order = 1,
                        type = "description",
                        name = L["Show Sec Desc"],
                    },
                    showSec = {
                        order = 2,
                        type = "toggle",
                        name = L["Show Seconds"],
                        get = function() return db.showSeconds end,
                        set = function() db.showSeconds = not db.showSeconds Clock:Refresh() end,
                    },
                    is24hourDesc = {
                        order = 3,
                        type = "description",
                        name = L["24 Hour Desc"],
                    },
                    is24hour = {
                        order = 4,
                        type = "toggle",
                        name = L["24 Hour"],
                        get = function() return db.is24Hour end,
                        set = function() db.is24Hour = not db.is24Hour Clock:Refresh() end,
                    },
                    isServerTimeDesc = {
                        order = 5,
                        type = "description",
                        name = L["Server Time Desc"],
                    },
                    isServerTime = {
                        order = 6,
                        type = "toggle",
                        name = L["Server Time"],
                        get = function() return db.isServerTime end,
                        set = function() db.isServerTime = not db.isServerTime Clock:Refresh() end,
                        disabled = function()
                            local istrue
                            if ( db.isBoth ) then istrue = true else istrue = false end
                            return istrue
                        end,
                    },
                    isBothDesc = {
                        order = 7,
                        type = "description",
                        name = L["Both Desc"],
                    },
                    isBoth = {
                        order = 8,
                        type = "toggle",
                        name = L["Show Both"],
                        get = function() return db.isBoth end,
                        set = function() db.isBoth = not db.isBoth Clock:Refresh() end,
                    },
                    useCoreDesc = {
                        order = 9,
                        type = "description",
                        name = L["Use Core Desc"],
                    },
                    useCoreFace = {
                        order = 10,
                        type = "toggle",
                        name = L["Use Core Font Face"],
                        get = function() return db.useCoreFace end,
                        set = function() db.useCoreFace = not db.useCoreFace Clock:Refresh() end,
                    },
                    useCoreSize = {
                        order = 11,
                        type = "toggle",
                        name = L["Use Core Font Size"],
                        get = function() return db.useCoreSize end,
                        set = function() db.useCoreSize = not db.useCoreSize Clock:Refresh() end,
                    },
                    useSelfDesc = {
                        order = 12,
                        type = "description",
                        name = L["Use Self Desc"],
                    },
                    fontFace = {
                        order = 13,
                        type = "select",
                        values = SML:List("font"),
                        name = L["Font Face"],
                        get = function()
                            for k, v in pairs(TBL) do
                                if db.fontFace == v then
                                    return k
                                end
                            end
                        end,
                        set = function(_, font)
                            db.fontFace = TBL[font]
                            font = SML:Fetch("font", TBL[font])
                            Clock:Refresh()
                        end,
                        disabled = function()
                            local istrue
                            if ( db.useCoreFace ) then istrue = true else istrue = false end
                            return istrue
                        end,
                    },
                    spacer = { order = 14, type = "description", name = "" },
                    fontSize = {
                        order = 15,
                        type = "range",
                        name = L["Font Size"],
                        min = 4, max = 36, step = 1,
                        get = function() return db.fontSize end,
                        set = function(_, value) db.fontSize = value Clock:Refresh() end,
                        disabled = function()
                            local istrue
                            if ( db.useCoreSize ) then istrue = true else istrue = false end
                            return istrue
                        end,
                    },
                },
            },
            position = {
                order = 3,
                type = "group",
                name = L["Position"],
                args = {
                    justifyDesc = {
                        order = 1,
                        type = "description",
                        name = L["Justify Desc"],
                    },
                    justify = {
                        order = 2,
                        type = "select",
                        values = SLDataText.JustTbl,
                        name = L["Set Justify"],
                        get = function() return db.justify end,
                        set = function(_, value) db.justify = value Clock:Refresh() end,
                    },
                    parentDesc = {
                        order = 3,
                        type = "description",
                        name = L["Parent Desc"],
                    },
                    parent = {
                        order = 4,
                        type = "input",
                        name = L["Parent Anchor"],
                        get = function() return db.pos.anchor end,
                        set = function(_, value) db.pos.anchor = value Clock:Refresh() end,
                    },
                    anchorFromDesc = {
                        order = 5,
                        type = "description",
                        name = L["Anchor From Desc"],
                    },
                    anchorFrom = {
                        order = 6,
                        type = "select",
                        values = SLDataText.PointTbl,
                        name = L["Anchor From"],
                        get = function() return db.pos.anchorFrom end,
                        set = function(_, value) db.pos.anchorFrom = value Clock:Refresh() end,
                    },
                    offsetDesc = { 
                        order = 7, 
                        type = "description", 
                        name = L["Offset Desc"],
                    },
                    posX = {
                        order = 8,
                        type = "input",
                        name = L["X Offset"],
                        get = function() return db.pos.x end,
                        set = function(_, value) db.pos.x = value Clock:Refresh() end,
                    },
                    spacer2 = { order = 9, type = "description", name = "" },
                    posY = {
                        order = 10,
                        type = "input",
                        name = L["Y Offset"],
                        get = function() return db.pos.y end,
                        set = function(_, value) db.pos.y = value Clock:Refresh() end,
                    },
                },
            },
        },
    },
}

function Clock:OnInitialize()
    self.db = SLDataText.db:RegisterNamespace(MODNAME, defaults)
    db = self.db.profile
    coreDB = SLDataText.db.profile
    -- Check module enabled state & add options
    self:SetEnabledState(SLDataText:GetModuleEnabled(MODNAME))
    SLDataText:InjectOptions(MODNAME, options)
end

function Clock:OnEnable()
    if ( not self.frame ) then 
        self.frame = CreateFrame("Frame", "SLDTClock", UIParent)
        self.text = self.frame:CreateFontString(nil, "OVERLAY")
    end
    if ( not self.frame:IsShown() ) then self.frame:Show() end
    if ( GameTimeFrame:IsShown() ) then GameTimeFrame:Hide() end
    
    self:RegisterEvent("PLAYER_ENTERING_WORLD", "Refresh")
    self:ScheduleRepeatingTimer("Refresh", 1)
end

function Clock:OnDisable()
    if ( self.frame:IsShown() ) then self.frame:Hide() end
    if ( not GameTimeFrame:IsShown() ) then GameTimeFrame:Show() end
end

function Clock:Refresh()
    local hour, minutes, seconds = date("%H"), date("%M"), date("%S")
    local newHour
    
    if ( db.is24Hour ) then newHour = hour self.timeOfDay = "" else
        local fixHour = (hour-hour)+hour
        if ( fixHour > 11 and fixHour ~= 24 ) then
            if ( fixHour > 12 ) then newHour = fixHour-12 else newHour = fixHour end
            self.timeOfDay = "pm"
        else
            if ( fixhour == 0 ) then newhour = 12 else newHour = fixHour end
            self.timeOfDay = "am"
        end
    end
    if ( db.showSeconds ) then
        self.localTime = format("%d"..db.seperator.."%s"..db.seperator.."%s", newHour, minutes, seconds)
    else
        self.localTime = format("%d"..db.seperator.."%s", newHour, minutes, timeOfDay)
    end
    
    local serverHour, serverMin = GetGameTime()
    local newSHour
    if ( serverMin < 10 ) then serverMin = string.format("0%s", serverMin) end
    if ( db.is24Hour  )then newSHour = serverHour self.timeOfDayServ = "" else
        local fixHour = (serverHour-serverHour)+serverHour
        if ( fixHour > 11 and fixHour ~= 24 ) then
            if ( fixHour > 12 ) then newSHour = fixHour-12 else newSHour = fixHour end
            self.timeOfDayServ = "pm"
        else
            if ( fixHour == 0 ) then newSHour =12 else newSHour = fixHour end
            self.timeOfDayServ = "am"
        end
    end
    self.serverTime = format("%d"..db.seperator.."%s", newSHour, serverMin)
    
    self:RefreshFrame()
end

Clock.isMoving = false

function Clock:RefreshFrame()
    local curFont, curFontSize
    if ( db.useCoreFace ) then curFont = SLDataText.db.profile.coreFontFace else curFont = db.fontFace end
    if ( db.useCoreSize ) then curSize = SLDataText.db.profile.coreFontSize else curSize = db.fontSize end
    self.text:SetFont(SML:Fetch("font", curFont), curSize)
    self.text:SetShadowColor(0, 0, 0, 1)
    self.text:SetShadowOffset(1.5, -1.5)
    self.text:SetJustifyH(db.justify)
    
    local color1, color2 = SLDataText.vColor, SLDataText.btColor
    if ( db.isServerTime ) then
        self.text:SetFormattedText("%s%s|r%s%s|r", color1, self.serverTime, color2, self.timeOfDayServ)
    elseif ( db.isBoth ) or ( db.isBoth and db.isServerTime ) then
        self.text:SetFormattedText("%s%s|r%s%s|r | %s%s|r%s%s|r", color1, self.localTime, color2, self.timeOfDay, color1, self.serverTime, color2, self.timeOfDayServ)
    else
        self.text:SetFormattedText("%s%s|r%s%s|r", color1, self.localTime, color2, self.timeOfDay)
    end    
    
    self.frame:SetWidth(self.text:GetWidth())
    self.frame:SetHeight(self.text:GetHeight())
    
    if not coreDB.isLocked then
        self.frame:SetMovable(true)
        self.frame:RegisterForDrag("LeftButton")
        self.frame:EnableMouse(true)
        self.frame:SetScript("OnMouseDown", function()
            SLDataText:MoveFrame(self, db)
        end)
        self.frame:SetScript("OnMouseUp", function()
            SLDataText:StopFrame(self, db)
        end)
    else
        self.frame:SetMovable(false)
        self.frame:RegisterForDrag(nil)
        self.frame:EnableMouse(false)
        self.frame:SetScript("OnMouseDown", nil)
        self.frame:SetScript("OnMouseUp", nil)
    end
    
    if not self.isMoving then
        self.frame:ClearAllPoints()
        self.frame:SetPoint(db.justify, db.pos.anchor, db.pos.anchorFrom, db.pos.x, db.pos.y)
        self.text:ClearAllPoints()
        self.text:SetPoint(db.justify, self.frame, db.justify, 0, 0)
    end
end