--[[ RECIPE BOOK TRADESKILLS :
Contains most of the functions used for dealing with collecting and storing personal tradeskill data
See RecipeBookDBFunctions for:
	SetPlayerTradeskillInfo(who, skill, key, value)
	AddDBItem(id, skill)
	AddMaterialsData(id, {materials list})
	AddKnownData(id, player, difficulty, faction code)

]]--

--============================== VARIABLES ==============================--
RBTradeskill = {};
local RB_PlayerTradeskills = {};

-- Default values for tradeskill data
RECIPEBOOK_TRADESKILL_DEFAULTS = {rank = {1, "number", 0}, maxrank = {2, "number", 0}, spec = {3, "number", 0}, subspec = {4, "number", 0}, lastupd = {5, "string", "Not yet updated"}, show = {6, "boolean", true}, numskills = {7, "number", 0},};

--============================== FUNCTIONS ==============================--
--[[ ForceSkillUpdate(skill) --> Sets RB_PlayerTradeskills[skil] to nil, forcing an update on next open. ]]--
function RBTradeskill:ForceSkillUpdate(skill)
	if RB_PlayerTradeskills[skill] then 
		RB_PlayerTradeskills[skill] = nil 
		return true;
	else return false;
	end;
end

--[[ SkillWindowOpen(event, [updated]) --> Scans through the tradeskill window, collecting and storing all the data. ]]--
function RBTradeskill:SkillWindowOpen(event, update)
	--set up functions for crafts/noncrafts
	local numSkills, getLine, getLink, numReagents, getRLine, getRLink, getGLink, craft;
	local total = 0;
	local known = 0;
	if (string.find(event, "CRAFT"))and RecipeBook.Parms.AddOnsLoaded["Blizzard_CraftUI"] then -- Enchanting window
		RecipeBook:Debug("Beginning craft scan.");
		numSkills = GetNumCrafts;
		getLine = GetCraftDisplaySkillLine;
		getLink = GetCraftItemLink;
		numReagents = GetCraftNumReagents;
		getRLine = GetCraftReagentInfo;
		getRLink = GetCraftReagentItemLink;
		getGLink = GetCraftRecipeLink;
		craft = true;
	elseif RecipeBook.Parms.AddOnsLoaded["Blizzard_TradeSkillUI"] then -- All other tradeskills
	    RecipeBook:Debug("Beginning tradeskill scan.");
	    numSkills = GetNumTradeSkills;
		getLine = GetTradeSkillLine;
		getLink = GetTradeSkillItemLink;
		numReagents = GetTradeSkillNumReagents;
		getRLine = GetTradeSkillReagentInfo;
		getRLink = GetTradeSkillReagentItemLink;
		getGLink = GetTradeSkillRecipeLink;
	else
	    return;
	end

	total = numSkills();
	local skill, rank, maxrank = getLine();
	skill = RBDB:Skill_TextToDB(skill);
	local fac = ((RecipeBook.Globals.Faction == FACTION_ALLIANCE) and 10 or 20); -- Set player faction

	-- Is this tradeskill being tracked?
	if skill == 0 or not RBDB:GetSkillIsTracked(RecipeBook.Globals.Player, fac, skill) then
		RecipeBook:Debug("Not tracking this tradeskill.");
		return false;
	-- This tradeskill has already been completely scanned, including reagents, and nothing new has been added.
	else
		known = RBDB:GetPlayerTradeskillInfo(RecipeBook.Globals.Player, fac, skill, "numskills");
		RecipeBook.Parms.LastTSUpdate = time();
		if type(known) == "number" and known >= total then
			RBDB:SetPlayerTradeskillInfo(RecipeBook.Globals.Player, fac, skill, "lastupd", string.gsub(date(), " %d+:%d+:%d+", "")); -- Update the "last update" field.
			-- RB_PlayerTradeskills[skill] = {ID = id, Count = total, Items = {}, Completed = true};
		    RecipeBook:Debug("Already scanned all these items. Exiting.");
		    return true;
		-- This tradeskill has fewer scanned items than the tradeskill window suggests it should, or has not been scanned.
		else
			update = true;
			if RB_PlayerTradeskills[skill] ~= nil then
			    if RB_PlayerTradeskills[skill].ErrorTime == nil then RB_PlayerTradeskills[skill].ErrorTime = 0 end;
			    if RB_PlayerTradeskills[skill].Reagents == nil then RB_PlayerTradeskills[skill].Reagents = {} end;
			    if RB_PlayerTradeskills[skill].Items == nil then RB_PlayerTradeskills[skill].Items = {} end;
			else
				RB_PlayerTradeskills[skill] = {Reagents = {}, Items = {}, ErrorTime = 0};
			end
		end
	end

   	RBDB:SetPlayerTradeskillInfo(RecipeBook.Globals.Player, fac, skill, "rank", tonumber(rank));
	RBDB:SetPlayerTradeskillInfo(RecipeBook.Globals.Player, fac, skill, "maxrank", tonumber(maxrank));
	if RBDB:GetPlayerTradeskillInfo(RecipeBook.Globals.Player, fac, skill, "show") ~= false then
		RBDB:SetPlayerTradeskillInfo(RecipeBook.Globals.Player, fac, skill, "show", true);
	end

	local diff, id, link;

	-- Create table of reagents for an index item
	local name,diff, id, rid;
	local count = 0;
	local incomplete = 0;
	local reagenTable = function(index)
	    local reagT = {};
	    for j = 1, numReagents(index) do
			local link = getRLink(index,j);
			if link then
				local _,_,num = getRLine(index,j);
				reagT[RBDB:Link_TextToDB(link)] = num;
				RB_PlayerTradeskills[skill].Reagents[getLink(index)] = nil;
			else --This link doesn't exist; reagents aren't loaded.
                RB_PlayerTradeskills[skill].Reagents[getLink(index)] = true;
                return {};
			end;
	    end
	    -- All reagents loaded, return reagent table.
		return reagT;
	end

	for i = 1, total do
	    -- Difficulty or 'header'
	    if craft then
       		rname, _, diff = GetCraftInfo(i);
		else
			rname, diff = GetTradeSkillInfo(i);
		end
		if diff == "header" then
		    count = count + 1; -- Increment count for header fields
		-- Not a header
		else
		    what = getLink(i);
			recipe = getGLink(i);
		    -- Item is already in the tradeskill item table, therefore has been completely scanned.
		    if RB_PlayerTradeskills[skill].Items[recipe] then
				count = count + 1; -- Increment count.
 		    else
		    -- Item is not already in the tradeskill item table
		        id = RBDB:Link_TextToDB(what);
				rid = RBDB:Link_TextToDB(recipe);
		        if id then
					local iname;
					if id == rid then iname = GetSpellInfo(id);
					else iname = GetItemInfo(what);
					end
					RBDB:AddDBItem(id, rid, iname, rname, skill); -- Potentially new item to the DB
					RBDB:AddKnownData(id, rid, RecipeBook.Globals.Player, diff, fac); -- Add known data for this player.
					
				    local reagenT = reagenTable(i);
				    -- This item has materials available
	            	count = count + 1; -- Increment count only if item is aGetdded.
					if next(reagenT) then
			            RB_PlayerTradeskills[skill].Items[recipe] = true;
		            	RBDB:AddMaterialsData(id, rid, reagenT);
					else
					    incomplete = incomplete + 1;
	            	end
				end
		    end
		end
	end
	
	if (count < total or count < 1 ) and ((time() - RB_PlayerTradeskills[skill].ErrorTime > 300) or RB_PlayerTradeskills[skill].ErrorMsg ~= "Tradeskill") then
		RBOutput:Print(string.format(RECIPEBOOK_ERR_TRADESKILLNOTSCANNED, RBDB:Skill_DBToText(skill)), "info");
		RB_PlayerTradeskills[skill].ErrorTime = time();
		RB_PlayerTradeskills[skill].ErrorMsg = "Tradeskill";
	elseif incomplete > 0 and ((time() - RB_PlayerTradeskills[skill].ErrorTime > 300) or RB_PlayerTradeskills[skill].ErrorMsg ~= "Reagents") then
	    RBOutput:Print(string.format(RECIPEBOOK_ERR_REAGENTSNOTSCANNED, RBDB:Skill_DBToText(skill)), "info");
	    RB_PlayerTradeskills[skill].ErrorTime = time();
		RB_PlayerTradeskills[skill].ErrorMsg = "Reagents";
	elseif count >= total and incomplete == 0 and not RB_PlayerTradeskills[skill].Completed then
	    RBOutput:Print(string.format(RECIPEBOOK_INFO_SCANCOMPLETED, RBDB:Skill_DBToText(skill)), "info");
		RB_PlayerTradeskills[skill].Completed = true;
	end

	RecipeBook:Debug("Scanned "..count);
	RecipeBook:Debug("Archived "..count-incomplete);
	RBDB:SetPlayerTradeskillInfo(RecipeBook.Globals.Player, fac, skill, "lastupd", string.gsub(date(), " %d+:%d+:%d+", ""));
	RBDB:SetPlayerTradeskillInfo(RecipeBook.Globals.Player, fac, skill, "numskills", count-incomplete);
 end


	
--[[ SkillWindowUpdate(event) --> Updates semi-intelligently to save memory and cycles ]]--
function RBTradeskill:SkillWindowUpdate(event)
	if not RBOptions:GetOption("TrackMe") then return end;
	if time() - RecipeBook.Parms.LastTSUpdate < 2 then return end;
	RBTradeskill:SkillWindowOpen(event, true)
end

--[[ SkillWindowClose() --> Resets data when a window closes. ]]--
function RBTradeskill:SkillWindowClose(event)
	RBDB:UpdateNameList();
end


--[[ SpecialtyScan() --> Scans through the player's known skills for specializations.
	Returns : new specialties added/changed or false ]]--
function RBTradeskill:SpecialtyScan()
	local specT = {};
	local _, _, _, numSpells = GetSpellTabInfo(1);
	for i = 1, numSpells, 1 do
		local name = GetSpellName(i, BOOKTYPE_SPELL);
		local found = false;
		if name then
			local fac = ((RecipeBook.Globals.Faction == FACTION_ALLIANCE) and 10 or 20); -- Set player faction
			for skill,v in pairs(RECIPEBOOK_SPECIALS) do
				local s = RBDB:Skill_TextToDB(skill);
			    for i, ts in ipairs(v) do
			        if ts == name then
						spec = RBDB:GetPlayerSpecializations(RecipeBook.Globals.Player, fac, s);
			            if spec ~= i then
			                RBDB:SetPlayerTradeskillInfo(RecipeBook.Globals.Player, fac, s, "spec", i);
							table.insert(specT, name);
						end
			            found = true;
			            break;
			        end
			    end
			    -- if found then break end;
			end
			if not found then
				for skill,v in pairs(RECIPEBOOK_SUBSPECIALS) do
					local s = RBDB:Skill_TextToDB(skill);
				    for j, ts in ipairs(v) do
				        if ts == name then
							_, spec = RBDB:GetPlayerSpecializations(RecipeBook.Globals.Player, fac, s);
			            	if spec ~= j then
			            	    RBDB:SetPlayerTradeskillInfo(RecipeBook.Globals.Player, fac, s, "subspec", j);
								table.insert(specT, name);
							end
				            found = true;
				            break;
				        end
				    end
				end
				-- if found then break end;
			end
		end
	end

	return #specT > 0 and RECIPEBOOK_ADDED_SPECIALS.. table.concat(specT, ", ") or false;
end

--[[ ProfessionScan() --> Scans through known professions for this character and populates the database with them, if tracking is enabled.
	Returns: Nothing ]]--
function RBTradeskill:ProfessionScan()
    local pind = 0;
	for i=1,10 do
	    local skill, _, expo = GetSkillLineInfo(i);
		if ( skill == TRADE_SKILLS ) then
		    if not expo then ExpandSkillHeader(i) end;
			pind = i;
			break;
		end
	end
	local skillT = {};
	local now = string.gsub(date(), " %d+:%d+:%d+", "");
	local fac = ((RecipeBook.Globals.Faction == FACTION_ALLIANCE) and 10 or 20); -- Set player faction
	for i = 1,7 do
		local skill, header, expo , rank, _, _, maxrank = GetSkillLineInfo(i+pind);
		if header and not expo then ExpandSkillHeader(i+pind) end; -- expand lines!
		skill = RBDB:Skill_TextToDB(skill);
		if not header and skill > 0 then
		    skillT[skill] = {Rank = rank, MaxR = maxrank}; -- All skills known
			if RBDB:GetSkillIsTracked(RecipeBook.Globals.Player, fac, skill) then
			    RBDB:SetPlayerTradeskillInfo(RecipeBook.Globals.Player, fac, skill, "rank", rank);
			    RBDB:SetPlayerTradeskillInfo(RecipeBook.Globals.Player, fac, skill, "maxrank", maxrank);
				RBDB:SetPlayerTradeskillInfo(RecipeBook.Globals.Player, fac, skill, "lastupd", now);
			end
		end
	end
	if next(skillT) ~= nil then
		for what, _ in pairs(RBDB:GetPlayerTradeskillInfo(RecipeBook.Globals.Player, fac, "all")) do
		    if not skillT[what] then
				RBDB:DeleteTradeskill(RecipeBook.Globals.Player, fac, what) end; -- Tradeskill no longer known.
		end
	end

end

function RBTradeskill:ReputationScan()
	for i = 1, GetNumFactions() do
  		name, _, rank, _, _, _, _, _, header = GetFactionInfo(i);
		local fac = ((RecipeBook.Globals.Faction == FACTION_ALLIANCE) and 10 or 20); -- Set player faction
  		if not header then
  		    RBDB:SetPlayerReputation(RecipeBook.Globals.Player, fac, name, rank);
		end
	end
end

--[[ ParseSkillupMessage(msg) --> Updates skill rank on a chat window skillup message. ]]--
function RBTradeskill:ParseSkillupMessage(msg)
	local skill, rank, t0;
	if not msg then return end;
    string.gsub(msg, RECIPEBOOK_CHAT_SKILLUP, function(a,b) skill = a; rank = tonumber(b); end); -- skill and rank from message
    skill = RBDB:Skill_TextToDB(skill);
	local fac = ((RecipeBook.Globals.Faction == FACTION_ALLIANCE) and 10 or 20); -- Set player faction
    if not RBDB:GetSkillIsTracked(RecipeBook.Globals.Player, fac, skill) then return end; -- Not a tracked tradeskill.
    
	RBDB:SetPlayerTradeskillInfo(RecipeBook.Globals.Player, fac, skill, "rank", rank);
	
	local banked = RBDB:GetBankedForSkill(skill, rank)
	if banked then
	    for id, who in pairs(banked) do
			if who then
		        local name, link = GetItemInfo(id);
				if not name then link = "Unsafe item [id: "..id.."]" end;
		    	RBOutput:Print(format(RECIPEBOOK_INFO_CANLEARNONSKILLUP, link, table.concat(who, ", ")), "output");
			end
		end
	end
end
