--[[
RecipeBookSkill: Contains functions and variables related to the RecipeBook SkillBrowseFrame.
]]--

--------------------------------------------------  DROPDOWN MENUS --------------------------------------------------
-------------------------- Tradeskill Sort Menu ------------------------
--[[ This is an expandable table for sort dropdown buttons.  To add a button, decide what your sort type will be indexed as (SortIndex).
	Add your button to RECIPEBOOK_SORTBUTTONS - you will want to use SortIndex as your button name
	RECIPEBOOK_SORTBUTTONS["SortIndex"] = { button information };
	You will need a "func" parameter which 
		1) Calls RBDB:SetOption("Browse", "Sort", SortIndex)
		2) Calls RBSkill:RefreshData;
	Then call table.insert(RECIPEBOOK_SORTMENU,  #RECIPEBOOK_SORTMENU - 1, "Your Button Name") from within your AddOn.
]]--
RECIPEBOOK_SORTBUTTONS = {};
RECIPEBOOK_SORTBUTTONS["Alphabetical"] = { text = RECIPEBOOK_BROWSE_DD_SORTALPHA, value = "Alphabetical" };
RECIPEBOOK_SORTBUTTONS["Difficulty"] = { text = RECIPEBOOK_BROWSE_DD_SORTDIFF, value = "Difficulty" };
RECIPEBOOK_SORTBUTTONS["Item Level"] = { text = RECIPEBOOK_BROWSE_DD_SORTILVL, value = "Item Level", header = RECIPEBOOK_BROWSE_FS_INFOITEMLEVEL, display = "iLevel"};
RECIPEBOOK_SORTBUTTONS["Minimum Level"] = { text = RECIPEBOOK_BROWSE_DD_SORTMLVL, value = "Minimum Level", header = RECIPEBOOK_BROWSE_FS_INFOMINLEVEL, display = "mLevel" };
RECIPEBOOK_SORTBUTTONS["Item Type"] = { text = RECIPEBOOK_BROWSE_DD_SORTITYPE, value = "Item Type"};

RECIPEBOOK_SORTMENU = { "Alphabetical", "Difficulty", "Item Level", "Minimum Level", "Item Type"};

-------------------------- Player Choose Menu ------------------------
local RECIPEBOOK_PLAYERMENU = {};
local RECIPEBOOK_SELECTBUTTONS = {};


-------------------------------------------------- VARIABLES --------------------------------------------------
RBSkill = {};
RBSkill.Last = {nil, nil};
RBSkill.TSData = {};
RBSkill.TrackData = {};
RBSkill.AddData = {};
RBSkill.AddSort = {};

local RBTRADESKILL_DISPLAY_ORDER = {"First Tradeskill", "Second Tradeskill", RECIPEBOOK_TRADESKILLS["Cooking"],RECIPEBOOK_TRADESKILLS["Fishing"], RECIPEBOOK_TRADESKILLS["First Aid"]};
RBSKILLBROWSEFRAME_NUMITEMS = 16;

-------------------------------------------------- FUNCTIONS --------------------------------------------------
--[[ Initialize() ]]--
function RBSkill:Initialize()
	for i,prefix in pairs({RECIPEBOOK_BROWSE_FS_PERSONAL, RECIPEBOOK_BROWSE_FS_SHARED}) do
	    for j, postfix in pairs({FACTION_ALLIANCE, FACTION_HORDE}) do
	    	local name = prefix .. "-" .. postfix;
	    	RECIPEBOOK_PLAYERMENU[string.upper(name)] = {};
	    	RECIPEBOOK_SELECTBUTTONS[string.upper(name)] = { text = name, value = name};
		end
	end
end

-------------------------- Tradeskill Sorting and List Generation ------------------------

--[[ MakeTradeskillTable(who) --> Generates a list of tradeskills known by a particular player.  There is no sort or filter applied to this very basic data 
RBSkill.TSData = {
	["Who"] = {Currently-shown Player, Faction}
	["What"] = Current skill
	["Skills"] = {
		[1] = {
			["ID"] = Skill 1 ID
			["Data"] = { Skill data }
			["Items"] = { Items data}
			["Display"] = {Data to be displayed, run through RBSkill:Sort() and RBSkill:Filter()}
		[2] = Skill 2
		[3] = Cooking
		[4] = Fishing
		[5] = First Aid
	}
]]--
function RBSkill:MakeTradeskillTable(who)
	-- Make who valid
	if not who then who = RecipeBook.Globals.Player;
	else who = RBOutput:Capitalize(who);
	end
	local fac = RBDB:GetPlayerFaction(who);
	--  Base data
	RBSkill.TSData = {};
	RBSkill.TSData["Who"] = {who, RBDB:Faction_DBToText(fac)}; -- {name, faction, shared}
	RBSkill.TSData["What"] = 1; 
	local _, what = unpack(RBSkill.Last);
	if what and what > 0 then RBSkill.TSData["What"] = what end;
	RBSkill.TSData["Skills"] = {};
	-- Get all tradeskill data
	local itemT = {};
	local all = RBOptions:GetOption("Browse", "Unknown"); -- Are we including items that are unknown?

	for skill, data in pairs(RBDB:GetPlayerTradeskillInfo(who, fac, "all", nil)) do
		itemT = {};
		for rid, info in pairs(RBDB:GetPlayerSkillItems(who, fac, skill, all)) do
			local id, diff = unpack(info);
			table.insert(itemT, {id, rid, diff}); -- item id, recipe id, difficulty; the first two will pin us down precisely to any given item.
		end
		local name = RBDB:Skill_DBToText(skill);
		local match = 0;
		for i,d in ipairs(RBTRADESKILL_DISPLAY_ORDER) do
			if d == name then
			    RBSkill.TSData["Skills"][i] = {["Name"] = name, ["Data"] = data, ["Items"] = itemT};
				match = i;
			 end
		end
		if match < 1 then
			if RBSkill.TSData["Skills"][1] == nil then match = 1; else match = 2 end
			RBSkill.TSData["Skills"][match] = {["Name"] = name, ["Data"] = data, ["Items"] = itemT};
		end
		RBSkill.TSData["Skills"][match]["Display"] = RBSkill:Sort(RBSkill.TSData["Skills"][match]["Items"]);
	end
	RBSkill:MakeDisplayList();
end

function RBSkill:MakeDisplayList()
	local what = RBSkill.TSData["What"];
	if not what then what = 1; RBSkill.TSData["What"] = 1; end;
	if not RBSkill.TSData["Skills"] or not RBSkill.TSData["Skills"][what] or not RBSkill.TSData["Skills"][what]["Items"] then return end;
	local text = RBUI_SkillFrame_EDIT_Filter:GetText();
	if text == RECIPEBOOK_BROWSE_FS_SEARCH then text = "" end;
	RBSkill.TSData["Skills"][what]["Display"] = RBSkill:Filter(RBSkill:Sort(RBSkill.TSData["Skills"][what]["Items"]), text);
end


--[[ AddExternalSort : Adds external data for RecipeBook to sort by. ]]--
function RBSkill:AddExternalSort(text, value, field, header)
	-- Create your button with the important data.
	RECIPEBOOK_SORTBUTTONS[value] = {["text"] = text, ["value"] = value, ["header"] = header, ["display"] = field};	
	-- Add your button to the SortMenu
	table.insert(RECIPEBOOK_SORTMENU, value);
end


--[[ Sort(table of items) --> Takes the table generated by RBSkill:MakeTradeskillTable ({ {id, rid, diff}, {id, rid, diff}...}) and produces functional data which is sorted. ]]--
function RBSkill:Sort(items, skill)
	-- items = table of {id, rid, diff} for a skill
	local itemT = {}; -- this is the table we'll be manipulating.
	-- Generate the item table.
	local info = {};
	local name, itemName, link, rare, level, minlvl, itype, stype, loc, tex;
	
	local function enchantWhere(name)
		local where = string.match(name, (ENSCRIBE.." ([%w%s]+) - .*"))
		if not where then where = "UNKNOWN" end;
		-- Weapons
		if where == ENCHSLOT_2HWEAPON or where == ENCHSLOT_WEAPON then
			return ENCHSLOT_WEAPON, where;
		-- Armor
		else
			iwhere = "INVTYPE_"..(string.upper(where));
			if iwhere then return ARMOR, iwhere end;
			
			return STRING_SCHOOL_UNKNOWN, where;
		end	
	end

	local sortBy = RBOptions:GetOption("Browse", "Sort");
	
	for x, data in ipairs(items) do
		local cached = true;
		-- if this is an enchantment, all bets are off.
		if data[1] == data[2] then
			name, _, tex = GetSpellInfo(data[1]);
			itemName = name;
			link = GetSpellLink(data[1]);
			-- We need to fill in the rest of this info; will work on that.
			rare = 1;
			level = 0;
			minlvl = 0;
			itype = "Enchant";
			stype, loc = enchantWhere(name);			
		else
			itemName, link, rare, level, minlvl, itype, stype, _, loc, tex = GetItemInfo(data[1]);
			if not itemName then cached = false end; -- No name.
			_, name = RBDB:Item_DBToText(data[1], data[2]); -- We are using the recipe name for display, not the item name.
		end
		info = {};
		info.Name = name; -- Name
		if itemName ~= nil then info.itemName = itemName else info.itemName = name end; -- for searching.
		info.Info = "";
		info.ID = data[1]; -- Item ID
		info.Link = link; -- Link for the item
		info.RID = data[2]; -- Recipe ID
		info.RLink = "enchant:"..data[2];  -- Recipe Link
		info.Diff = data[3]; -- Difficulty to make item
		info.Rare = rare; --The quality of the item (0-6, Poor-Artifact)
		info.iLevel = level; -- The Blizzard item level
		info.mLevel = minlvl; -- The minimum level needed for the item
		info.iType = itype;
		info.sType = stype; 
		info.Loc = loc;  -- The item type
		info.Texture = tex; -- The texture for the item
		info.Cached = cached; -- Is the item cached?
		if RBSkill.AddData[sortBy] then info = RBSkill.AddData[sortBy](info) end;
		if RECIPEBOOK_SORTBUTTONS[sortBy] == nil then
			RBOptions:SetOption("Browse", "Sort", "Alphabetical");
			sortBy = RBOptions:GetOption("Browse", "Sort");
		end			
		if RECIPEBOOK_SORTBUTTONS[sortBy]["display"]~= nil then 
			local text = info[RECIPEBOOK_SORTBUTTONS[sortBy]["display"]];
			if type(text) == "number" and text == 0 then text = "" end;
			info.Info = text;
		end
		table.insert(itemT, info);
	end
	
	--[[ 	Valid sorts by default:
		Alphabetical
		Difficulty
		Item Level
		Minimum Level
		Item Type
	]]--
	-- Default sort is alphabetical
	if sortBy == "Alphabetical" then
		table.sort(itemT, function(a,b) return RBSkill:SortBy(a, b, "Name", "zNoData", true) end);
		return itemT;
	-- Sort by difficulty (easy, moderate, difficult) - not tracking skill requirements
	elseif sortBy == "Difficulty" then
		table.sort(itemT, function(a,b) return RBSkill:SortBy(a, b, "Diff", 0) end);
		return itemT;
	-- sort by 
	elseif sortBy == "Item Level" then
		table.sort(itemT, function(a,b) return RBSkill:SortBy(a, b, "iLevel", 0) end);
		return itemT;
	elseif sortBy == "Minimum Level" then
		table.sort(itemT, function(a,b) return RBSkill:SortBy(a, b, "mLevel", 0) end);
		return itemT;
	elseif sortBy == "Item Type" then
		table.sort(itemT, function(a,b) return RBSkill:SortBy(a, b, {"iType", "sType", "Loc"}, {"zNotype", "zNotype", "zNoLoc"}, false) end );
		return itemT;
	-- Expansible for additional functions (if someone has more data to work with)
	elseif RBSkill.AddSort[sortBy] then
		 return RBSkill.AddSort[sortBy](itemT, sortBy);
	-- If the sort is not available then we default to alphabetical.
	else 
		table.sort(itemT, function(a,b) return RBSkill:SortBy(a, b, "Name") end);
		return itemT;
	end
end

--[[ A sorting algorithm that will recurse through a series of fields on which to sort. Always ends with an alphabetical sort BECAUSE the lua table.sort algorithm is not stable.]]--
function RBSkill:SortBy(a, b, indices, defaults, ascending) 
	if type(indices) ~= "table" then indices = {indices} end;
	if type(defaults) ~= "table" then defaults = {defaults} end;


	local index = table.remove(indices, 1);
	local default = table.remove(defaults, 1);
	if a[index] == nil then a[index] = default end;
	if b[index] == nil then b[index] = default end;
	
	if ascending then 
		if next(indices) ~= nil then 
			return (a[index] < b[index]) or ( (a[index] == b[index]) and RBSkill:SortBy(a, b, indices, defaults, ascending) ); 
		else 
			return (a[index] < b[index]) or ( (a[index] == b[index]) and (a["Name"] < b["Name"]) ); 
		end; 
	else
		if next(indices) ~= nil then 
			return (a[index] > b[index]) or ( (a[index] == b[index]) and RBSkill:SortBy(a, b, indices, defaults, ascending) ); 
		else 
			return (a[index] > b[index]) or ( (a[index] == b[index]) and (a["Name"] < b["Name"]) ); 
		end; 
	end
end


--[[ Filter(item table, filter text)  --> Takes a table from Sort() and filters it for items which match the filter text, returning a separate display table.   
	Since this is always called, if text is nil then recreate the original table.  
	Yes we could just return it but LUA is quite particular about pointers and I don't want to risk interfering with the original tradeskill table.
]]--
function RBSkill:Filter(items, text)
	-- We receive a sorted table which has been processed by RBSkill:Sort.  Ti, his is a basic assumption, as is the assumption that items.Name is non-nil.
	-- Rendering a subtable for returning, without altering the original items table.  This needs to be fast for real-time sorting As You Type.
	local itemT = {};
	for x, data in ipairs(items) do
		if text == nil or text == "" then 
			table.insert(itemT, data);
		elseif string.find(string.lower(data["Name"]), string.lower(text), 1, true) then
			table.insert(itemT, data);
		elseif string.find(string.lower(data["itemName"]), string.lower(text), 1, true) then
			table.insert(itemT, data);
		end
	end
	
	return itemT;
end

-------------------------- Tradeskill Display Frame ------------------------
--[[ ShowFrame([who], [what]) --> Opens the RecipeBook Browse window.  If provided parameters then starts with that character.  If not, uses the last opened character. ]]--
function RBSkill:ShowBrowseFrame()
	local who, what = unpack(RBSkill.Last);
	if not who then who = RecipeBook.Globals.Player end;
	
	if not RBDB:GetPlayerFaction(who) then return end; -- Verifying that who is in our list of gettable people.

	RBSkill:MakeTradeskillTable(who);
	RBSkill.TSData["What"] = 1;
	if what then
	    what = string.lower(what);
	    for i,v in pairs(RBSkill.TSData["Skills"]) do
	        if string.lower(v["Name"]) == what then RBSkill.TSData["What"] = i end;
		end
	end
	PanelTemplates_SetTab(RBUI_MainFrame, 2);
	RBUI_MainFrame:Show()
end

--[[ OnShow() --> Initializes data ]]--
function RBSkill:OnShow()
	if RBSkill.TSData["Who"] == nil then
		local who, what = unpack(RBSkill.Last);
		if not who then who = RecipeBook.Globals.Player end;
		if not what then what = 1 end;
		RBSkill:MakeTradeskillTable(who);
	end
	-- Set up player selection menu
	for i,prefix in pairs({RECIPEBOOK_BROWSE_FS_PERSONAL, RECIPEBOOK_BROWSE_FS_SHARED}) do
	    for j, postfix in pairs({FACTION_ALLIANCE, FACTION_HORDE}) do
	        local name = prefix .. "-" .. postfix;
	        RECIPEBOOK_PLAYERMENU[string.upper(name)] = {};
	        for k, who in pairs(RBDB:GetAllInFaction(postfix, prefix)) do
	        	table.insert(RECIPEBOOK_PLAYERMENU[string.upper(name)], who);
	    		RECIPEBOOK_SELECTBUTTONS[who] = { text = who, value = who };
			end
		end
	end
	RBUI:SubFrame_LoadOptions("RBUI_SkillFrame");
	RBUI:SubFrame_CheckBoxUpdate("RBUI_SkillFrame");
	RBSkill:OnUpdate();
end

--[[ OnHide() --> Tidies up ]]--
function RBSkill:OnHide()
	RBUI:SubFrame_CommitOptions("RBUI_SkillFrame", false);
	-- Set up last tradeskill.
	local what = RBSkill.TSData["What"];
	if what == nil then what = 1 end;
	RBSkill.Last = {RBSkill.TSData["Who"][1], what};
	-- Empty list.
	RBSkill.TSData = {};
    RBSkillTrackFrame:Hide();
    RBUI_SkillFrame_ScrollFrame:Hide();
end

--[[Refresh()  --> Makes a new tradeskill table.]]--
function RBSkill:Refresh(fromClick)
	RBUI:SubFrame_CommitOptions("RBUI_SkillFrame", false);
	local what = RBSkill.TSData["What"];
	if what == nil then 
		what = 1;
		RBSkill.TSData["What"] = 1;
	end;
	RBSkill.Last = {RBSkill.TSData["Who"][1], what};
	RBSkill:MakeTradeskillTable(RBSkill.TSData["Who"][1]);
	-- If this is not being called from RBSkill:UpdateData then back to the top of the scrollframe.
	if not fromClick then RBUI_SkillFrame_ScrollFrameScrollBar:SetValue(0) end;
	RBSkill:OnShow();
end

--[[ OnUpdate() --> Updates the browse frame when player or tradeskill selection changes ]]--
function RBSkill:OnUpdate()
	RBUI:SubFrame_CommitOptions("RBUI_SkillFrame", false);
	-- Reload tradeskill tracking frame
	if RBSkillTrackFrame:IsVisible() then RBSkill:TrackOnShow() end;
	
	-- Set the button text for player selection
    RBUI_SkillFrame_TXT_PlayerInfo:SetText(RBSkill.TSData["Who"][1].." ("..RBSkill.TSData["Who"][2]..", "..(RBSkill.TSData["Who"][3] and RECIPEBOOK_BROWSE_FS_SHARED or RECIPEBOOK_BROWSE_FS_PERSONAL)..") "..RECIPEBOOK_BROWSE_FS_CLICK);
	local button, skillText, specText, rankText, timeText, icon, checkBox, checkText;
    -- Activate tradeskill buttons and label accordingly.
    for i,v in ipairs(RBTRADESKILL_DISPLAY_ORDER) do
		-- Set up last tradeskill.
        button = getglobal("RBUI_SkillFrame_BUT_Skill"..i);
        skillText = getglobal("RBUI_SkillFrame_BUT_Skill"..i.."_TXT-Profession");
        specText = getglobal("RBUI_SkillFrame_BUT_Skill"..i.."_TXT-Spec");
        rankText = getglobal("RBUI_SkillFrame_BUT_Skill"..i.."_TXT-Rank");
        timeText = getglobal("RBUI_SkillFrame_BUT_Skill"..i.."_TXT-Timestamp");
        icon = getglobal("RBUI_SkillFrame_BUT_Skill"..i.."_TXT-Icon");
        checkBox = getglobal("RBUI_SkillFrame_BUT_Skill"..i.."_CBX-Show");

		button:Show();
		button:Enable();
		local fac = RBDB:GetPlayerFaction(RBSkill.TSData["Who"][1]);
        -- Data exists for this tradeskill
        if RBSkill.TSData["Skills"][i] ~= nil then
            -- Profession name
        	skillText:SetText(RBSkill.TSData["Skills"][i]["Name"]);
        	-- Specialization
        	local spec = RBDB:Specialty_DBToText(RBSkill.TSData["Skills"][i]["Name"], RBSkill.TSData["Skills"][i]["Data"][RBDB:TradeskillIndex_TextToDB("spec")], false);
        	local subspec = RBDB:Specialty_DBToText(RBSkill.TSData["Skills"][i]["Name"], RBSkill.TSData["Skills"][i]["Data"][RBDB:TradeskillIndex_TextToDB("subspec")], true);
        	local txt;
        	if spec ~= RECIPEBOOK_DBERROR_NOSPECID then
        	    txt = spec;
				if subspec ~= RECIPEBOOK_DBERROR_NOSPECID then
				    txt = txt.." ("..subspec..")";
				end
			else txt = RECIPEBOOK_BROWSE_FS_NOSPEC;
			end
			specText:SetText(txt);
			-- Rank and max rank
           	local rank, maxrank = RBSkill.TSData["Skills"][i]["Data"][RBDB:TradeskillIndex_TextToDB("rank")], RBSkill.TSData["Skills"][i]["Data"][RBDB:TradeskillIndex_TextToDB("maxrank")];
			txt = (rank and rank or "Unknown") .. "/" .. (maxrank and maxrank or "Unknown")
        	rankText:SetText(txt);
        	-- Timestamp
        	txt = RBSkill.TSData["Skills"][i]["Data"][RBDB:TradeskillIndex_TextToDB("lastupd")];
			if txt then timeText:SetText(txt) end;
			-- Icon
			icon:SetTexture("Interface\\AddOns\\RecipeBook\\Icons\\"..string.gsub(RBDB:Skill_DelocalizeText(RBSkill.TSData["Skills"][i]["Name"]), "%s", ""))
			-- Show Checkbox
			checkBox:Show();
			checkBox:Enable();
			checkBox:SetChecked(RBDB:GetSkillIsShown(RBSkill.TSData["Who"][1], fac, RBSkill.TSData["Skills"][i]["Name"]));
			if i == RBSkill.TSData["What"] then
				button:LockHighlight();
			else
			    button:UnlockHighlight();
			end
		else
		    -- No data exist AND this is a secondary profession (cooking, fishing, first aid) - display the texture
			if not string.find(RBTRADESKILL_DISPLAY_ORDER[i], "Tradeskill") then
	        	skillText:SetText(RBTRADESKILL_DISPLAY_ORDER[i]);
				icon:SetTexture("Interface\\Addons\\RecipeBook\\Icons\\"..string.gsub(RBDB:Skill_DelocalizeText(RBTRADESKILL_DISPLAY_ORDER[i]), "%s", ""))
				-- If this is tracked but blank, mark "No data" and enable the Show checkbox.  Else mark "Not Tracked"
				if RBDB:GetSkillIsTracked(RBSkill.TSData["Who"][1], fac, RBTRADESKILL_DISPLAY_ORDER[i]) then
        			rankText:SetText(RECIPEBOOK_BROWSE_FS_NODATA);
        			checkBox:Show();
					checkBox:Enable();
					checkBox:SetChecked(RBDB:GetSkillIsShown(RBSkill.TSData["Who"][1], fac, RBTRADESKILL_DISPLAY_ORDER[i]) and true or 0);
				else
					checkBox:Hide();
					rankText:SetText(RECIPEBOOK_BROWSE_FS_NOTRACKED);
				end
			-- No data exist AND this is a primary profession - display blank texture
			else
				skillText:SetText(RECIPEBOOK_BROWSE_FS_NOTRADESKILL);
				icon:SetTexture("Interface\\Addons\\RecipeBook\\Icons\\Default")
				rankText:SetText(" ");
		    	checkBox:Hide();
			end
			-- Set up the blank data stuff
        	specText:SetText(" ");
			timeText:SetText(" ")
			-- Disable button
        	button:Disable();
		end
	end
	RBSkill:MakeDisplayList();
	RBSkill:ScrollFrame_ScrollBarUpdate();
end

--[[ ProfessionOnClick(Button clicked) --> Displays data for the selected profession ]]--
function RBSkill:ProfessionOnClick()
	local id = string.match(this:GetName(), "RBUI_SkillFrame_BUT_Skill(%d+)");
	RBSkill.TSData["What"] = tonumber(id);
	RBSkill.Last = {RBSkill.TSData["Who"], RBSkill.TSData["What"]};
	RBUI_SkillFrame_ScrollFrameScrollBar:SetValue(0)
	RBSkill:OnUpdate();
end

--[[ ShowCheckboxClick(checked) --> toggles deprecation status for a tradeskill ]]--
function RBSkill:ShowCheckboxClick(checked)
	local id = string.match(this:GetName(), "(%d+)_CBX%-Show");
	local what;
	id = tonumber(id);
	if RBSkill.TSData["Skills"][id] ~= nil then
	    what = RBSkill.TSData["Skills"][id]["Name"];
	else
	    what = RBTRADESKILL_DISPLAY_ORDER[id];
	end

	-- RBDB:SetSkillIsShown(RBSkill.TSData["Who"][1], what, (checked == 1 and true or false));
	RBDB:SetSkillIsShown(RBSkill.TSData["Who"][1], RBDB:GetPlayerFaction(RBSkill.TSData["Who"][1]), what, (checked == 1 and true or false));
end

-------------------------- Doing Things with the  Display Frame ------------------------
--[[ DeleteAlt() : Deletes all data for the displayed alt; if it is not the current alt then removes completely from DB.]]--
function RBSkill:DeleteAlt()
	local who = RBSkill.TSData["Who"][1];
	if who == RecipeBook.Globals.Player then
		for skill, data in pairs(RBSkill.TSData["Skills"]) do
			-- RBDB:DeleteTradeskill(who, data["Name"]);
			RBDB:DeleteTradeskill(who, RBDB:Faction_TextToDB(RecipeBook.Globals.Faction, false), RBDB:Skill_TextToDB(data["Name"]));
		end
	else
		RBDB:DeletePlayer(who, RBDB:GetPlayerFaction(who));
	end
	RBSkill:Refresh();
end

--[[ ShowCheckboxClick(checked) --> toggles deprecation status for a tradeskill ]]--
function RBSkill:ShowCheckboxClick(checked)
	local id = tonumber(string.match(this:GetName(), "(%d+)_CBX%-Show"));
	local what;
	if RBSkill.TSData["Skills"][id] ~= nil then
	    what = RBSkill.TSData["Skills"][id]["Name"];
	else
	    what = RBTRADESKILL_DISPLAY_ORDER[id];
	end
	what = RBDB:Skill_TextToDB(what);
	RBDB:SetSkillIsShown(RBSkill.TSData["Who"][1], RBDB:GetPlayerFaction(RBSkill.TSData["Who"][1]), what, (checked == 1 and true or false));
end

-------------------------- Player Selection Dropdown ------------------------
--[[ DDChosePlayer_Initialize() ]]--
function RBSkill:DDChoosePlayer_Initialize()
	local level = UIDROPDOWNMENU_MENU_LEVEL;
	
	if level == 1 then 
		local info = UIDropDownMenu_CreateInfo();
		-- Title Menu: 
		info.text = RECIPEBOOK_CFG_FS_SELECTFACTION;
		info.isTitle = 1;
		info.notCheckable = 1;
		UIDropDownMenu_AddButton(info);
		
		-- Faction and player buttons
		for faction, p in pairs(RECIPEBOOK_PLAYERMENU) do
			if #p > 0 then -- This button needs displayed.
				local info = UIDropDownMenu_CreateInfo();
				local data = RECIPEBOOK_SELECTBUTTONS[faction];
				info.text = data.text;
				info.value = data.value;
				info.owner = this:GetParent();
				info.notCheckable = 1;
				info.hasArrow = true;
				UIDropDownMenu_AddButton(info, level);
			end
		end
	else
		local fac = string.upper(UIDROPDOWNMENU_MENU_VALUE);
		for f, players in pairs(RECIPEBOOK_PLAYERMENU) do 
			if f == fac and #players > 0 then
				-- Make buttons for all the players.
				for i, button in ipairs(players) do
					local info = UIDropDownMenu_CreateInfo();
					local data = RECIPEBOOK_SELECTBUTTONS[button];
					info.text = data.text;
					info.value = data.value;
					info.owner = this:GetParent();
					info.notCheckable = 1;
					info.hasArrow = nil;
					info.func = RBSkill.SelectPlayer;
					info.arg1 = self;
					info.arg2 = data.value;
					UIDropDownMenu_AddButton(info, 2);
				end
			end
		end
	end

	PlaySound("igMainMenuOpen");
end

--[[ SelectPlayer() --> Setup the Browse frame for a new player or refresh the current. --]]
function RBSkill:SelectPlayer(who)
	RecipeBook:Debug("Select Player.")
	RBSkill:MakeTradeskillTable(who);
	RBSkill.TSData["What"] = 1;
	ToggleDropDownMenu(1, nil, RBUI_SkillFrame_DD_ChoosePlayer, "RBUI_SkillFrame_BUT_PlayerInfo");
	RBUI_SkillFrame_ScrollFrameScrollBar:SetValue(0)
	RBSkill:OnUpdate();
end

-------------------------- Doing Things with the Items Display ------------------------
-- [[ ItemOnEnter() --> Displays tooltip for an item on mouseover ]]--
function RBSkill:ItemOnEnter(parent)
	local frame = this;
	if not parent then frame = this:GetParent() end;
	RecipeBook:SetDefaultButtonData(RecipeBook.Parms.Button);
	local id, rid = frame.iID, frame.rID;
	-- A potentially unsafe item
	local info = GetItemInfo(id);
	GameTooltip:SetOwner(RBUI_SkillFrame, "ANCHOR_BOTTOMRIGHT", -6, 448);
	if not frame.cached then
		GameTooltip:ClearLines();
		GameTooltip:AddLine(RECIPEBOOK_BROWSE_TT_UNCONFIRMEDHEADER);
		for x, line in pairs(RECIPEBOOK_BROWSE_TT_UNCONFIRMEDITEM) do
			GameTooltip:AddLine(line, 1, 1, 1, 1, 1);
		end
		GameTooltip:Show();
		return;
	end
	-- A safe item.
	if frame.rID == frame.iID or parent then 
		GameTooltip:SetHyperlink("enchant:"..frame.rID);
	else
		GameTooltip:SetHyperlink(RBOutput:MakeLink(frame.iID));
	end
	GameTooltip:Show();
end

--[[ ItemOnClick(Button clicked) --> Displays data on a particular item ]]--
function RBSkill:ItemOnClick(button, parent)
	local frame = this;
	if not parent then frame = this:GetParent() end;
	
	local id = frame.iID;
	local rid = frame.rID;
	local skill = frame.skill;

	if not frame.cached or (button == "RightButton" and GetItemInfo(id) == nil) then
		GameTooltip:SetHyperlink(RBOutput:MakeLink(id));
		GameTooltip:Show();
		return RBSkill:UpdateItemInfo(id);
	end

	local mats = RBDB:GetMaterialsInfo(id, skill, rid);
	if mats[1] then mats = mats[1][2] end; -- RID guarantees only one item returned; we care about the mats table.  If nothing returned, should not cause problems.
	local link = RBDB:Link_DBToText(id, rid);
	
	local function showMats()
		-- RecipeBook:Debug("showMats");
		local mlink = RBDB:Link_DBToText(id, rid, "gold");
		if ( ChatFrameEditBox:IsVisible() ) then
			ChatFrameEditBox:Insert(mlink);
		else
			local data = {};
		    for a, b in pairs(mats) do
		        tinsert(data, RBDB:Link_DBToText(a, 0, "mat").." x"..b);
			end
			RBOutput:Print(RECIPEBOOK_TXT_SUPPLIES..mlink..": "..table.concat(data, ", "), "output", nil, 0.25,0.5,1);
		end
	end
	
	local function showRef()
		if parent then 
			id = "enchant:" .. rid;
			link = RBDB:Link_DBToText(id, rid, "gold");
		elseif rid == id then id = "enchant:"..id;
		else id = RBOutput:MakeLink(id);
		end
		
		SetItemRef(id, link, button);
	end

	if IsAltKeyDown() then
		if parent then showRef();
		else showMats();
		end;
	elseif parent and IsShiftKeyDown() then
		showMats();
	else
		showRef();
	end
end

--[[ UpdateItem(frame) --> Sequentially updates an item until it has info, then calls RBSkill:Refresh() to update the data ]]--
function RBSkill:UpdateItemInfo(id)
	if GetItemInfo(id) == nil then 
		RecipeBook:ScheduleEvent(RBSkill.UpdateItemInfo, 0, self, id)
	else
		RBSkill:Refresh(true);
	end
end


--[[ ScrollFrame_ScrollBarUpdate() --> Scrolls ]]--
function RBSkill:ScrollFrame_ScrollBarUpdate()
	local hasItems = true;
	local length;
	local what = RBSkill.TSData["What"];
	if not what then RBSkill.TSData["What"] = 1; what = 1; end;
	if (RBSkill.TSData["Skills"] == nil) or (next(RBSkill.TSData["Skills"]) == nil) then
		RBTradeskill:ProfessionScan();
		RBSkill:MakeTradeskillTable(RecipeBook.Globals.Player);
		RBSkill.TSData["What"] = 1; 
		what = 1;
	end
	if RBSkill.TSData["Skills"][what] == nil or RBSkill.TSData["Skills"][what]["Display"] == nil then
	    hasItems = false;
	    length = 1;
	else
		length = #RBSkill.TSData["Skills"][what]["Display"];
	end;

	local line, index, button, text, icon, tex;
	local skillID;
	local offset = FauxScrollFrame_GetOffset(RBUI_SkillFrame_ScrollFrame);
	FauxScrollFrame_Update(RBUI_SkillFrame_ScrollFrame,length+1,RBSKILLBROWSEFRAME_NUMITEMS,20);
	local infoText  = RECIPEBOOK_SORTBUTTONS[RBOptions:GetOption("Browse", "Sort")]["header"];
	if infoText == nil then infoText = "" end;
	RBUI_SkillFrame_Flavor:SetText(infoText);
	for line = 1,RBSKILLBROWSEFRAME_NUMITEMS,1 do
		index = offset + line;
		button = getglobal("RBUI_SkillFrame_ScrollFrameItem"..line);
		text = getglobal("RBUI_SkillFrame_ScrollFrameItem"..line.."_Text");
		info = getglobal("RBUI_SkillFrame_ScrollFrameItem"..line.."_Info");
		icon = getglobal("RBUI_SkillFrame_ScrollFrameItem"..line.."_Icon");
		if hasItems then
			if index <= length then
				text:SetText(RBSkill.TSData["Skills"][what]["Display"][index]["Name"]);
				info:SetText(RBSkill.TSData["Skills"][what]["Display"][index]["Info"]);
				
				local diff = RBSkill.TSData["Skills"][what]["Display"][index]["Diff"];
				if diff ~= nil then
					-- text:SetTextColor(RBDB:Difficulty_DBToRGB(diff))
					text:SetTextColor(RBDB:Difficulty_DBToRGB(diff))
				end;
				button.iID = RBSkill.TSData["Skills"][what]["Display"][index]["ID"];
				button.rID = RBSkill.TSData["Skills"][what]["Display"][index]["RID"];
				button.cached = RBSkill.TSData["Skills"][what]["Display"][index]["Cached"];
				-- button.itemSkill = RBDB:Skill_TextToDB(RBSkill.TSData["Skills"][what]["Name"]);
				button.skill = RBDB:Skill_TextToDB(RBSkill.TSData["Skills"][what]["Name"]);
				if not button.cached then 
					tex = "Interface\\GossipFrame\\AvailableQuestIcon";
					-- text:SetText("? "..text:GetText());
					-- text:SetTextColor(1,0,0); --Make it red :)
				else
				tex = RBSkill.TSData["Skills"][what]["Display"][index]["Texture"];
				end
				if tex == nil then tex = "Interface\\Icons\\Spell_Holy_GreaterHeal" end;
				icon:SetNormalTexture(tex);
			end;
		else
		    text:SetText(RECIPEBOOK_TXT_TRADESKILLNOTTRACKED);
		    text:SetTextColor(1,0,0);
		end
		button:SetID(line);
		if index > length then
			button:Hide();
		else
			button:Show();
		end
	end
end


-------------------------- Sort Selection Dropdown ------------------------
--[[Initialize() --> Needed to make the initialize work correctly. ]]--
function RBSkill:DDSortType_Initialize()
	local level = 1;
	local info = UIDropDownMenu_CreateInfo();
	
	for i, button in ipairs(RECIPEBOOK_SORTMENU) do
		local data = RECIPEBOOK_SORTBUTTONS[button];
		info.text = data.text;
		info.value = data.value;
		if data.func then info.func = func;
		else info.func = RBSkill.SetSortType;
		end
		info.owner = this:GetParent();
		info.checked = nil;
		UIDropDownMenu_AddButton(info, level);
	end
end

--[[ SetSortType() --> Reorders the items according to sort type. --]]
function RBSkill:SetSortType()
	UIDropDownMenu_SetSelectedValue(this.owner, this.value);
	local stype = this.value;
	RBOptions:SetOption("Browse", "Sort", stype);
	RBSkill:MakeDisplayList();
	RBSkill:ScrollFrame_ScrollBarUpdate();
end

--============================== TRADESKILL TRACKING FRAME ==============================--
--[[ OnShow --> Makes all the pretty buttons show up with text ]]--
function RBSkill:TrackOnShow()
	for i, v in ipairs(RecipeBook.TSTable) do
		getglobal("RBSkillTrackFrame_Skill"..i.."_TXT-Icon"):SetTexture("Interface\\AddOns\\RecipeBook\\Icons\\"..string.gsub(RBDB:Skill_DelocalizeText(v), "%s", ""))
		getglobal("RBSkillTrackFrame_Skill"..i.."_TXT-Skill"):SetText(v);
		-- color if player knows?
     	getglobal("RBSkillTrackFrame_Skill"..i):Show()
	end
	RBSkill:TrackOnUpdate();
end

--[[OnUpdate --> Makes checks on pretty buttons. ]]--
function RBSkill:TrackOnUpdate()
	if RBUI_SkillFrame:IsVisible() then who = RBSkill.TSData["Who"][1];
	elseif RBUI_OptionsFrame:IsVisible() then who = "Get Defaults";
	end
	for i, v in ipairs(RecipeBook.TSTable) do
		local track = RBDB:GetSkillIsTracked(who, RBDB:GetPlayerFaction(who), i);
		RBSkill.TrackData[i] = track;
		local checkBox = getglobal("RBSkillTrackFrame_Skill"..i.."_TXT-CBX");
		if track then
	    	checkBox:SetTexture("Interface\\Buttons\\UI-CheckBox-Check");
		else
			checkBox:SetTexture("Interface\\Buttons\\UI-CheckBox-Up");
		end
	end
end

--[[TrackingButton_OnClick --> Sets the track status for a particular professsion ]]--
function RBSkill:TrackButton_OnClick(button)
	local id = string.match(this:GetName(), "RBSkillTrackFrame_Skill(%d+)");
	id = tonumber(id);
	if RBSkill.TrackData[id] then
		RBSkill.TrackData[id] = false;
	else RBSkill.TrackData[id] = true;
	end
	local checkBox = getglobal("RBSkillTrackFrame_Skill"..id.."_TXT-CBX");
	if RBSkill.TrackData[id] then
    	checkBox:SetTexture("Interface\\Buttons\\UI-CheckBox-Check");
	else
		checkBox:SetTexture("Interface\\Buttons\\UI-CheckBox-Up");
	end
end

--[[ ApplyTrackInfo() --> Commits tracking changes. ]]--
function RBSkill:TrackApplyTrackInfo()
	local who, fac;
	if RBUI_SkillFrame:IsVisible() then
		who = RBSkill.TSData["Who"][1];
		fac = RBDB:GetPlayerFaction(who);
		for i,v in ipairs(RBSkill.TrackData) do
		    RBDB:SetSkillIsTracked(who, fac, i, v);
		end
	else
		for i,v in ipairs(RBSkill.TrackData) do
		    RBOptions:SetOption("TrackSkills", i, v);
		end
	end
	RBDB:UpdateTrackedSkills(who, fac);
end

--[[ ResetDefaults() --> Resets changes to defaults. ]]--
function RBSkill:TrackResetDefaults()
	RBSkill.TrackData = RBOptions:GetOption("TrackSkills");
	RBSkill:TrackApplyTrackInfo();
	RBSkill:TrackOnUpdate();
end



