﻿--[[ RecipeBookShare: Sending, receiving, and parsing data 
Conventions:
source = who is sending the data
target = who is receiving the data
alt = whose data is being sent
skill = what skill is being sent
item = what item is being sent
msg = what message is being parsed
]]--

--============================== VARIABLES ==============================--
RB_Receiving = {};
RB_Receiving.ReceiveQueue = {};
RB_Receiving.AutoReceiveQueue = {};
RB_Receiving.ReceivePromptQueue = {};
RB_Receiving.AskBlock = {};
RB_Receiving.AskAccept = {};

RECIPEBOOK_SENDQUEUEFRAME_SENDELAY = 0.05;
RECIPEBOOK_SENDINITIATEFRAME_SENDELAY = 10;
RECIPEBOOK_REQUESTINGSEND_SENDELAY = 30;
RECIPEBOOK_REQUESTINGUPDATE_SENDELAY = 30;

UnitPopupMenus["RECIPEBOOK"] = {"RECIPEBOOK_REQUEST", "RBSHARE_SELF", "RBSHARE_ALL", "CANCEL"};
UnitPopupButtons["RECIPEBOOK"] = { text = RECIPEBOOK_MENU_MAIN, dist = 0, nested = 1};
UnitPopupButtons["RECIPEBOOK_REQUEST"] = { text = RECIPEBOOK_MENU_REQUEST, dist = 0,};
UnitPopupButtons["RBSHARE_SELF"] = { text = RECIPEBOOK_MENU_SHARESELF, dist = 0 };
UnitPopupButtons["RBSHARE_ALL"] = { text = RECIPEBOOK_MENU_SHAREALL, dist = 0 };

RBShare = {};
--============================== FUNCTIONS ==============================--
function RBShare:Initialize()
	table.insert(UnitPopupMenus["PLAYER"], #UnitPopupMenus["PLAYER"]-1, "RECIPEBOOK");
	table.insert(UnitPopupMenus["FRIEND"], #UnitPopupMenus["FRIEND"]-1, "RECIPEBOOK");
	table.insert(UnitPopupMenus["PARTY"], #UnitPopupMenus["PARTY"]-1, "RECIPEBOOK");
	
end

function RBShare:StaticPopups()
	-- You have declined a player's data
   	StaticPopupDialogs["RECIPEBOOK_BLOCKPLAYER"] = {
		text = RECIPEBOOK_POPUP_BLOCKPLAYER,
		button1 = TEXT(YES),
		button2 = TEXT(NO),
		OnAccept = function(data)
			local _, alt = unpack(data);
		    RBOptions:SetShareStatus(alt, 0);
		end,
		OnHide = function(data)
			local _, alt = unpack(data);
			RBShare:NextPromptQueue(alt);
		end,
		whileDead = 1;
		timeout = RECIPEBOOK_REQUESTINGSEND_SENDELAY;
	};
	-- You have accepted a player's data
   	StaticPopupDialogs["RECIPEBOOK_ACCEPTPLAYER"] = {
		text = RECIPEBOOK_POPUP_ACCEPTPLAYER,
		button1 = TEXT(YES),
		button2 = TEXT(NO),
		OnAccept = function(data)
			local _, alt = unpack(data);
		    RBOptions:SetShareStatus(alt, 1);
		end,
		OnHide = function(data)
			local _, alt = unpack(data);
			RBShare:NextPromptQueue(alt);
		end,
		whileDead = 1;
		timeout = RECIPEBOOK_REQUESTINGSEND_SENDELAY;
	};
	-- Someone wants to send data to you.
	StaticPopupDialogs["RECIPEBOOK_REQUESTING_SEND"] = {
		text = " ";
		button1 = TEXT(ACCEPT),
		button2 = TEXT(DECLINE),
		timeout = RECIPEBOOK_REQUESTINGSEND_SENDELAY,
		whileDead = 1;
		OnShow = function(data)
		    RB_Receiving.sessionAccepted = false;
		    RB_Receiving.buttonPressed = false;
			this.timeleft = RECIPEBOOK_REQUESTINGSEND_SENDELAY;
		end,
		OnAccept = function(data)
			local source, alt, msg, dist = unpack(data);
			RBShare:AcceptSession(source, alt, msg, dist);
		    RB_Receiving.sessionAccepted = true;
		    RB_Receiving.buttonPressed = true;
		end,
		OnCancel = function()
		    RB_Receiving.sessionAccepted = false;
		    RB_Receiving.buttonPressed = true;
		end,
		OnHide = function(data)
	    	local source, alt, msg, dist = unpack(data);
	    	local txt = RECIPEBOOK_SHARERROR_MANUALDECLINE;
	    	if not RB_Receiving.buttonPressed then
		        txt = RECIPEBOOK_SHARERROR_TIMEOUTDECLINE;
	    	end
		    if not RB_Receiving.sessionAccepted then
	    	    RBShare:DeclineSession(source, alt, string.format(txt, RecipeBook.Globals.Player, alt, source), dist);
			    local dialog = StaticPopup_Show("RECIPEBOOK_BLOCKPLAYER", alt);
   				if dialog then dialog.data = data end;
			else
			    local dialog = StaticPopup_Show("RECIPEBOOK_ACCEPTPLAYER", alt);
    			if dialog then dialog.data = data end;
			end
			RB_Receiving.sessionAccepted = nil;
		end,

		OnUpdate = function(elapsed, dialog)
			local source, alt = unpack(dialog.data);
			local text = getglobal(this:GetName().."Text");
			local timeleft = ceil(this.timeleft);
			text:SetText(string.format(RECIPEBOOK_POPUP_REQUESTING_SEND, source, alt, timeleft, GetText("SECONDS", nil, timeleft)));
			StaticPopup_Resize(this, "RECIPEBOOK_REQUESTING_SEND");
		end,
	};
	-- Someone wants to send data to you.
	StaticPopupDialogs["RECIPEBOOK_REQUESTING_UPDATE"] = {
		text = " ";
		button1 = TEXT(ACCEPT),
		button2 = TEXT(DECLINE),
		timeout = RECIPEBOOK_REQUESTINGUPDATE_SENDELAY,
		whileDead = 1;
		OnShow = function(data)
			this.timeleft = RECIPEBOOK_REQUESTINGUPDATE_SENDELAY;
		end,
		OnAccept = function(data)
			local target, _ = unpack(data);
			RBShare:InitiateSession(target, RecipeBook.Globals.Player);
		end,
		OnCancel = function(data)
			local target, _, dist = unpack(data);
		    RBShare:SendUnqueuedMessage(target, string.format(RECIPEBOOK_SHARE_NOUPDATE, string.format(RECIPEBOOK_UPDATERROR_MANUALDECLINE, RecipeBook.Globals.Player, target)), dist);
		end,
		OnUpdate = function(elapsed, dialog)
			local target = unpack(dialog.data);
			local text = getglobal(this:GetName().."Text");
			local timeleft = ceil(this.timeleft);
			text:SetText(string.format(RECIPEBOOK_POPUP_REQUESTING_UPDATE, target, timeleft, GetText("SECONDS", nil, timeleft)));
			StaticPopup_Resize(this, "RECIPEBOOK_REQUESTING_UPDATE");
		end,
	};
end
--============================== Utility and Control Functions ==============================--
--[[ NextPromptQueue(who, alt) --> Opens a prompt queue for who/alt if one is waiting ]]--
function RBShare:NextPromptQueue(alt)
	RB_Receiving.ReceivePromptQueue[alt] = nil;
	-- RB_Receiving.ReceivePromptQueue[alt] = {source, msg, dist};
    local newAlt, dataT = next(RB_Receiving.ReceivePromptQueue);
	if newAlt ~= nil and dataT ~= nil then
		table.insert(dataT, 2, newAlt);
		local dialog = StaticPopup_Show("RECIPEBOOK_REQUESTING_SEND", dataT[1], newAlt, dataT);
		if dialog then dialog.data = dataT end;
	end
end

--[[ IntitiateSession (target, alt) --> Asks whether data will be accepted for a particular alt. ]]--
-- RECIPEBOOK_SHARETRIGGER_INITIATE
function RBShare:InitiateSession(target, alt)
	if target == RecipeBook.Globals.Player then 
	    return RBOutput:Print(RECIPEBOOK_SHARERROR_SENDSELF, "error");
	-- Recurse
	elseif alt == "all" then
		if target == "Guild" then
			-- You can only send one alt at a time to the guild.
			RBShare:InitiateSession(target, RecipeBook.Globals.Player);
		else
			for i, name in ipairs(RBDB:GetAllInFaction(10)) do
			    RBShare:InitiateSession(target, name);
			end
			for i, name in ipairs(RBDB:GetAllInFaction(20)) do
				RBShare:InitiateSession(target, name);
			end
		end
		return;
	end
	
	if target == "Guild" then
	-- Send queue for tradeskills; there is no accept.
		if IsInGuild() then target = "GUILD";
		else return RBOutput:Print(RECIPEBOOK_SHARERROR_NOTINGUILD, "error");
		end
	
		if RecipeBook:GuildOnline() then
			if RBShare_SendQueueFrame.Outbound[target] == nil then RBShare_SendQueueFrame.Outbound[target] = {} end;
			RBShare_SendQueueFrame.Outbound[target][alt] = {};
			RBShare:QueueTradeskills(target, alt, false);
		else 
			return RBOutput:Print(RECIPEBOOK_SHARERROR_NOGUILDONLINE, "error");
		end
	else
		if RBShare_SendQueueFrame.Outbound[target] == nil then RBShare_SendQueueFrame.Outbound[target] = {} end;
		RBShare_SendQueueFrame.Outbound[target][alt] = {};
		RBShare:SendQueuedMessage(target, string.format(RECIPEBOOK_SHARE_INITIATE, alt));
		RBShare_SendInitiateFrame.Pending[target] = RBShare_SendInitiateFrame.timeSinceLastUpdate + RECIPEBOOK_SENDINITIATEFRAME_SENDELAY;
		if not RBShare_SendInitiateFrame:IsVisible() then RBShare_SendInitiateFrame:Show() end;
	end
end

--[[ DoCancel(source, message): Receives a cancel request and acts approprieately. ]]--
function RBShare:DoCancel(source, msg)
	RecipeBook:Debug("Running cancel for: "..source);
	if msg then RecipeBook:Debug("Reason: "..msg) end;
	local success = false;
	for i, data in ipairs(RBShare_SendQueueFrame.Messages) do
		if data[1] == source then table.remove(RBShare_SendQueueFrame.Messages, i) end
	end

	local err = string.match(msg, "<E:([%w%s]+)>");
	
	-- Clear receiving from data
	if RB_Receiving.ReceiveQueue[source] ~= nil then
		if err then RBOutput:Print(err, "error") end;
		RB_Receiving.ReceiveQueue[source] = nil;
	end
	RB_Receiving.AutoReceiveQueue[source] = nil;
	-- Clear Pending data (this is where the Success is needed, if player is offline)
	if RBShare_SendInitiateFrame.Pending[source] ~= nil then
		if err then RBOutput:Print(err, "error") end;
		RBShare_SendInitiateFrame.Pending[source] = nil;
		success = true;
	end
	-- Clear AutoSend data
	if RBShare_SendQueueFrame.AutoSend[source] ~= nil then
		RBShare_SendQueueFrame.AutoSend[source] = nil;
	end
	-- Clear sending frame data	
	if RBShare_SendQueueFrame.Outbound[source] ~= nil then 
		if err then RBOutput:Print(err, "error") end;
		RBShare_SendQueueFrame.Outbound[source] = nil;
	end

	return success;
end

--[[ SendCancel(source, alt, dist, error) : Sends a cancel request and does a cancel ]]--
function RBShare:SendCancel(target, alt, dist, err)
	if err then
		RBShare:SendUnqueuedMessage(target, string.format(RECIPEBOOK_SHARE_CANCELERR, alt, err));
	else
		RBShare:SendUnqueuedMessage(target, string.format(RECIPEBOOK_SHARE_CANCEL, alt));
		err = "";
	end;

	return RBShare:DoCancel(target, string.format(RECIPEBOOK_SHARE_CANCELERR, alt, err));
end

--[[ Conditional Accept: Accepts/Declines an incoming share (parse alt out) ]]--
function RBShare:ConditionalAccept(source, msg, dist)
	local alt = string.match(msg, "<P:([^%s>]+)>");
	
	local function doReturn(opt)
		-- AutoDecline
		if opt == 0 then
			return RBShare:DeclineSession(source, alt, string.format(RECIPEBOOK_SHARERROR_AUTODECLINE, RecipeBook.Globals.Player, alt), dist);
		-- AutoAccept
		elseif opt > 0 then
			return RBShare:AcceptSession(source, alt, msg, dist);
		-- Need to prompt
		else
			RBShare:SendUnqueuedMessage(source, RECIPEBOOK_SHARE_PROMPTING, dist); -- We are prompting: do not timeout
			RB_Receiving.ReceivePromptQueue[alt] = {source, msg, dist};
			if not StaticPopup_Visible("RECIPEBOOK_REQUESTING_SEND") then
				local dialog = StaticPopup_Show("RECIPEBOOK_REQUESTING_SEND", source, alt, {source, alt, msg, dist});
				if dialog then dialog.data = {source, alt, msg, dist} end
				return nil;
			end
		end
	end
	
	local o = RBOptions:GetShareStatus(alt);
	-- Listed in share list
	if type(o) == "number" then
		return doReturn(o);
	-- Guildwide share?  Parser should have already verified you are accepting these.
	elseif dist == "GUILD" then
		return doReturn(1);
	-- Special circumstances
	else
		o = nil;
		-- Check friends first
		ShowFriends();
		for n = 1, GetNumFriends() do
			if GetFriendInfo(n) == alt then 
				o = RBOptions:GetShareStatus("_Friend");
				break;
			end
		end
		-- Check for auto-accept status
		if o ~= nil and o > 0 then return doReturn(o);
		-- Not auto-accepted yet?  Check guild status.
		elseif IsInGuild() then
			GuildRoster();
			for n = 1, GetNumGuildMembers() do
				if GetGuildRosterInfo(n) == alt then
					-- This is a guildmate; use guild status
					return doReturn(RBOptions:GetShareStatus("_Guild"));
				end
			end
			-- This person is a friend but not a guildmate (and you are guilded); use friends
			if o ~= nil then return doReturn(o) end;
			-- This person is neither a friend nor a guildmate (and you are guilded); use default
			return doReturn(RBOptions:GetShareStatus("_Default"));
		-- You are not guilded and this is a friend; use friend
		elseif o ~= nil then return doReturn(o);
		--You are not guilded and this is not a friend; use default
		else return doReturn(RBOptions:GetShareStatus("_Default"));
		end
	end
end

--[[ ConditionalTradeskill((source, alt, skill, msg, dist): 	Accepts/Declines a particular tradeskill (behind the scenes) ]]--
function RBShare:ConditionalTradeskill(source, alt, skill, msg, dist)
	-- Not tracking tradeskill?  Decline automatically
	if not RBDB:GetSkillIsTracked(alt, RBDB:GetPlayerFaction(alt), tonumber(skill)) then
		return RBShare:DeclineTradeskill(source, alt, skill, dist);
	else
		--We should have an open queue if we're getting tradeskills in, but let's check anyway.
		if RB_Receiving.ReceiveQueue[source] and RB_Receiving.ReceiveQueue[source][alt] then
			return RBShare:AcceptTradeskill(source, alt, skill, msg, dist);
		-- It may be an AutoUpdate item.item as well.
		elseif RB_Receiving.AutoReceiveQueue[source] and RB_Receiving.AutoReceiveQueue[source][alt] then
			return RBShare:AcceptTradeskill(source, alt, skill, msg, dist);
		elseif dist == "GUILD" then 
			if RB_Receiving.AutoReceiveQueue[source] == nil then RB_Receiving.AutoReceiveQueue[source] = {} end;
			RB_Receiving.AutoReceiveQueue[source][alt] = {};
			return RBShare:AcceptTradeskill(source, alt, skill, msg, dist);
		else
			RecipeBook:Debug("Bad tradeskill.");
			return; -- Otherwise, no such item.
		end
	end
end

--[[ ConditionalUpdate(source, message) --> Given an update request from a player, decide whether to honor it. ]]--
function RBShare:ConditionalUpdate(source, msg, dist)
	local alt = string.match(msg, "<P:([^%s>]+)>");
	local o = RBOptions:GetShareStatus(source);
	-- AutoAccept this player, therefore share back
	if o == 0 then
		return RBShare:SendUnqueuedMessage(string.format(RECIPEBOOK_SHARE_NOUPDATE, string.format(RECIPEBOOK_SHARERROR_AUTODECLINE, RecipeBook.Globals.Player, source)), dist);
	-- You AutoUpdate to this player, they may request an update
	elseif RBDB:AltUpdatesTo(RecipeBook.Globals.Player, RBDB:Faction_TextToDB(RecipeBook.Globals.Faction, false), source) then
		RBShare:InitiateSession(source, RecipeBook.Globals.Player);
	-- Prompt.
	else
		if not StaticPopup_Visible("RECIPEBOOK_REQUESTING_UPDATE") then
			local dialog = StaticPopup_Show("RECIPEBOOK_REQUESTING_UPDATE", source, "", {source, msg, dist});
			if dialog then dialog.data = {source, msg, dist} end
			return nil;
		end
	end
end

--[[ ConditionalUpdate(source, message) --> Given an update request from a player, decide whether to honor it. ]]--
function RBShare:ConditionalAutoUpdate(source, msg, dist)
	local alt = string.match(msg, "<P:([^%s>]+)>");
	local o = RBOptions:GetShareStatus(alt);
	-- AutoAccept this player
	if o ~= nil and o > 0 then
		return RBShare:AcceptSession(source, alt, msg, dist, true);
	-- Anything else.
	else
		return RBShare:DeclineSession(source, alt, string.format(RECIPEBOOK_SHARERROR_AUTODECLINE, RecipeBook.Globals.Player, alt), dist, true);
	end
end


--[[ AcceptSession(source, alt, message, distribution method) --> Sets up a tradeskill session. ]]--
function RBShare:AcceptSession(source, alt, msg, dist, auto)
	if auto then
	    if RB_Receiving.AutoReceiveQueue[source] then
			RB_Receiving.AutoReceiveQueue[source][alt] = {};
		else
			RB_Receiving.AutoReceiveQueue[source] = {};
			RB_Receiving.AutoReceiveQueue[source][alt] = {};
		end
	else
	    if RB_Receiving.ReceiveQueue[source] then
			RB_Receiving.ReceiveQueue[source][alt] = {};
		else
			RB_Receiving.ReceiveQueue[source] = {};
			RB_Receiving.ReceiveQueue[source][alt] = {};
		end
	end
	RBShare:SendUnqueuedMessage(source, string.format(RECIPEBOOK_SHARE_ACCEPT, alt), dist);
end

--[[ AcceptSkill(source, alt, skill, msg, distribution method) : Opens a tradeskill session. ]]--
function RBShare:AcceptTradeskill(source, alt, skill, msg, dist)
	-- Auto Session
	if RB_Receiving.AutoReceiveQueue[source] and RB_Receiving.AutoReceiveQueue[source][alt] then
		RB_Receiving.AutoReceiveQueue[source][alt][skill] = {};
	-- Requested session
	elseif RB_Receiving.ReceiveQueue[source] and RB_Receiving.ReceiveQueue[source][alt] then
		RB_Receiving.ReceiveQueue[source][alt][skill] = {};
	-- Something has slipped past the parser
	else
		return;
	end
    local _, _, rank, maxrank, spec, subspec = string.find(msg, "<R:(%d+)/(%d+)%-(%d+):(%d+)>");
	local fac = string.match(msg, "<F:(%d+)>");
	if fac and (math.fmod(fac, 10) == 0) then 
		fac = tonumber(fac) + 1;  -- Add 1 to faction to represent shared data if not already marked as such.
	else 
		fac = RecipeBook.Globals.Faction;
	end
	
    RBDB:AddNewPlayer(alt, fac, true);
    RBDB:SetPlayerTradeskillInfo(alt, fac, skill, "rank", tonumber(rank));
    RBDB:SetPlayerTradeskillInfo(alt, fac, skill, "maxrank", tonumber(maxrank));
    RBDB:SetPlayerTradeskillInfo(alt, fac, skill, "spec", tonumber(spec));
    RBDB:SetPlayerTradeskillInfo(alt, fac, skill, "subspec", tonumber(subspec));
    RBDB:SetPlayerTradeskillInfo(alt, fac, skill, "lastupd", string.gsub(date(), " %d+:%d+:%d+", ""));
    -- Return accept.
	if dist ~= "GUILD" then 
		RBShare:SendUnqueuedMessage(source, string.format(RECIPEBOOK_SHARE_ACCEPTSKILL, alt, skill), dist);
	end
end

--[[ Decline Session (source, msg, dist): Declines a send request ]]--
function RBShare:DeclineSession(source, alt, reason, dist, auto)
	if not auto then 
		RBShare:SendUnqueuedMessage(source, string.format(RECIPEBOOK_SHARE_DECLINE, alt, reason), dist);
	else
		RBShare:SendUnqueuedMessage(source, string.format(RECIPEBOOK_SHARE_AUDECLINE, alt, reason), dist);
	end
end

--[[ Decline Tradeskill(source, alt, reason, dist) : Declines a particular tradeskill.  ]]--
function RBShare:DeclineTradeskill(source, alt, skill, dist)
	return RBShare:SendUnqueuedMessage(source, string.format(RECIPEBOOK_SHARE_DECLINESKILL, alt, skill), dist);
end

--[[ QueueUpdate(elapsed) : Keeps track of the countdown timer and queues appropriately for each frame. ]]--
function RBShare:QueueUpdate(elapsed)
	this.timeSinceLastUpdate = this.timeSinceLastUpdate + elapsed;
	local name = string.gsub(string.upper(this:GetName()), "RBSHARE", "RECIPEBOOK");
	if name  == "RECIPEBOOK_SENDQUEUEFRAME" and this.timeSinceLastUpdate > getglobal(name.."_SENDELAY") then
    	this.timeSinceLastUpdate = 0;
		RBShare:NextQueued();
	elseif name == "RECIPEBOOK_RECEIVEQUEUEFRAME" then
	    -- Nothing right now.
	elseif name == "RECIPEBOOK_SENDINITIATEFRAME" then
		for who, delay in pairs(this.Pending) do
		    if delay < this.timeSinceLastUpdate then
				if RBShare_SendQueueFrame.AutoSend[who] then
					RBShare:DoCancel(who, "AutoSend Delay Timeout");
				else
					RBShare:DoCancel(who, "Delay Timeout");
					RBOutput:Print(string.format(RECIPEBOOK_SHARERROR_TIMEOUT, who), "error");
				end
		    end
		end
		if next(this.Pending) == nil then
			this:Hide();
		end;
	else
	-- Do nothing
	end
end

--[[ SendQueuedMessage(target, message) --> Queues an outbound message; ]]--
function RBShare:SendQueuedMessage(target, msg, dist)
	if not RBShare_SendQueueFrame:IsVisible() then
	    RBShare_SendQueueFrame:Show();
	end
	table.insert(RBShare_SendQueueFrame.Messages, {target, msg, dist});
end

--[[ NextQueued() --> Sends the next item in the outbound queue. ]]--
function RBShare:NextQueued()
	if #RBShare_SendQueueFrame.Messages < 1 then
	    RBShare_SendQueueFrame:Hide();
	    return;
	end

	-- RecipeBook:Debug(#RBShare_SendQueueFrame.Messages .. " messages to send ");
	
	target, msg, dist = unpack(table.remove(RBShare_SendQueueFrame.Messages, 1));
	if (target and msg) then
	    if not dist then dist = "WHISPER" end;
		RBShare:SendRBMessage(target, msg, dist, true);
	end
	
	-- RecipeBook:Debug("Messages to send: "..#RBShare_SendQueueFrame.Messages);
	
	if #RBShare_SendQueueFrame.Messages < 1 then
	    RBShare_SendQueueFrame:Hide();
	    return;
	end
end

--[[ SendUnqueuedMessage(who, msg) --> Sends an immediate message. ]]--
function RBShare:SendUnqueuedMessage(target, msg, dist)
	if dist == "GUILD" then
	    -- We don't send unqueued messages to Guild channel.
	else
		RBShare:SendRBMessage(target, msg, dist, false);
	end
end

--[[ SendRBMessage(who, msg) --> Sends a message, with debug if Verbose mode is on. ]]--
function RBShare:SendRBMessage(target, msg, dist, queued)
	if RecipeBook.Parms.Debug.Verbose then
		local prefix = (dist == "GUILD" and "Guild send to " or (queued and "Queued send to " or "Immediate send to "));
		local color = (dist == "GUILD" and "green" or (queued and "yellow" or "cyan"));
		RBOutput:Print(prefix .. target .. ": "..RB_HEXCOLOR[color]..msg..RB_HEXCOLOR["end"], "output", "RBFrame");
	end
	SendAddonMessage(RECIPEBOOK_SHARETRIGGER, msg, dist, target);
end

--[[ QueueTradeskills(target. alt) --> Queues tradeskill requests for who ]]--
function RBShare:QueueTradeskills(target, alt, auto)
	RecipeBook:Debug("Queue Tradeskills to: ".. target);
	if auto and (RBShare_SendQueueFrame.AutoSend[target] == nil or RBShare_SendQueueFrame.AutoSend[target][alt] == nil) then 
		if target == "GUILD" then
			if RBShare_SendQueueFrame.AutoSend[target] == nil then RBShare_SendQueueFrame.AutoSend[target] = {} end;
			RBShare_SendQueueFrame.AutoSend[target][alt] = {};
		else
			-- RecipeBook:Debug("No valid AutoSend target.");
			return;
		end
	elseif (not auto) and (RBShare_SendQueueFrame.Outbound[target] == nil or RBShare_SendQueueFrame.Outbound[target][alt] == nil) then 
		if target == "GUILD" then
			if RBShare_SendQueueFrame.Outbound[target] == nil then RBShare_SendQueueFrame.Outbound[target] = {} end;
			RBShare_SendQueueFrame.Outbound[target][alt] = {};
		else
			-- RecipeBook:Debug("No valid Send target.");
			return;
		end
	end;
	
	local fac = RBDB:GetPlayerFaction(alt);
	for skill, _ in pairs(RBDB:GetPlayerTradeskillInfo(alt, fac, "all")) do
		-- Add to sending frame 
		if auto then RBShare_SendQueueFrame.AutoSend[target][alt][skill] = true;
		else RBShare_SendQueueFrame.Outbound[target][alt][skill] = true;
		end

		local r, mr, s, ss = unpack(RBDB:GetPlayerTradeskillInfo(alt, fac, skill));
		if r then
			if target == "GUILD" then
				RBShare:SendQueuedMessage(target, string.format(RECIPEBOOK_SHARE_OPENSKILL, skill, alt, fac, r, mr, s, ss), "GUILD");
				RBShare:DoGuildSend(alt, skill);
			else
				RBShare:SendQueuedMessage(target, string.format(RECIPEBOOK_SHARE_OPENSKILL, skill, alt, fac, r, mr, s, ss));
			end;
		end
	end
	if target == "GUILD" then
		RBShare_SendQueueFrame.Outbound["GUILD"] = nil;
		RBShare_SendQueueFrame.AutoSend["GUILD"] = nil;
	end
end

--[[ QueueItems(who, tradeskill) --> Queues items to be sent for who's tradeskill ]]--
function RBShare:QueueItems(target, alt, skill, dist)
	local lastitem = 0;
    for rid, data in pairs(RBDB:GetPlayerSkillItems(alt, nil, tonumber(skill), false)) do
		local id, diff = unpack(data);
        local iname, aname = RBDB:Item_DBToText(id, rid);
        RBShare:SendQueuedMessage(target, string.format(RECIPEBOOK_SHARE_ITEM, alt, skill, id, iname, diff, rid, aname), dist);
        lastitem = id;
    end
    if dist ~= "GUILD" then 
		if RBShare_SendQueueFrame.Outbound[target] then RBShare_SendQueueFrame.Outbound[target][alt][skill] = lastitem;
		elseif RBShare_SendQueueFrame.AutoSend[target] then RBShare_SendQueueFrame.AutoSend[target][alt][skill] = lastitem; 
		end
	end;
    RBShare:SendQueuedMessage(target, string.format(RECIPEBOOK_SHARE_LASTITEM, alt, skill, lastitem), dist);
end

--[[ QueueMaterials(who, tradeskill) --> Queues materials to be sent for the passed item ]]--
function RBShare:QueueMaterials(target, item, rid, skill, dist)
    -- Queue materials for item
    local text = string.format(RECIPEBOOK_SHARE_MATS, skill, item, rid);
	local minfo = RBDB:GetMaterialsInfo(item, skill, rid);
	if minfo[1] == nil then return end; -- No mats data stored.
    for mat, num in pairs(minfo[1][2]) do -- RID guarantees only one item returned in table: {rid, {matsinfo}};
        text = text .. string.format(RECIPEBOOK_SHARET_MAT, mat, num);
    end

    RBShare:SendQueuedMessage(target, text, dist);
end

--[[ DoGuildSend(alt, tradeskill) --> Queues tradeskills to be sent to the Guild. Does not depend on parser responses. ]]--
function RBShare:DoGuildSend(alt, skill)
	RBShare:QueueItems("GUILD", alt, skill, "GUILD");
    for rid, info in pairs(RBDB:GetPlayerSkillItems(alt, nil, skill, false)) do
    	RBShare:QueueMaterials("GUILD", info[1], rid, skill, "GUILD")
    end
    RBShare:SendQueuedMessage("GUILD", string.format(RECIPEBOOK_SHARE_TERMINATE, alt, skill, RBDB:GetPlayerTradeskillInfo(alt, RBDB:GetPlayerFaction(alt), skill, "numskills")), "GUILD")
	if RBShare_SendQueueFrame.Outbound["GUILD"] then RBShare_SendQueueFrame.Outbound["GUILD"][alt][skill] = nil;
	else RBShare_SendQueueFrame.AutoSend["GUILD"][alt][skill] = nil;
	end     
end

--[[ RequestUpdate(who) --> Requests an update to existing RecipeBook data from a player whose data you already have. ]]--
function RBShare:RequestUpdate(target)
	if RBDB:IsValidWho(target, RBDB:GetPlayerFaction(target)) then
	    RBShare:SendUnqueuedMessage(target, RECIPEBOOK_SHARE_REQUEST, "WHISPER");
	else
	    return RBOutput:Print(string.format(RECIPEBOOK_ERROR_NOREQUEST, target), "error")
	end
end

function RBShare:ProcessItemInfo(source, alt, skill, item, rid, msg, dist)
	local iname = string.match(msg, "<X:([^>]+)>");
	local aname = string.match(msg, "<Y:([^>]+)>");
	-- Record the item and difficulty.
	RBDB:AddDBItem(item, rid, iname, aname, skill);
	-- Send request/ack only if not a whole-guild send.
	if dist ~= "GUILD" then
		local matT = RBDB:GetMaterialsInfo(item, skill, rid);
		-- Request mats if we don't have any.
		if matT[1] == nil then 
			RBShare:SendUnqueuedMessage(source, string.format(RECIPEBOOK_SHARE_SENDMATS, skill, item, rid), dist);
		-- Otherwise acknowledge the item and continue.
		else
			RBShare:SendUnqueuedMessage(source, string.format(RECIPEBOOK_SHARE_SENDNOMATS, skill, item), dist);
		end
	end
end


--[[ UnitPopupOnClick : Hooked function for a unit popup to do RB functions. ]]--
function RBShare_UnitPopupOnClick()
	local dropdownFrame = getglobal(UIDROPDOWNMENU_INIT_MENU);
	local button = this.value;
	local name = dropdownFrame.name;
	
	if button == "RBSHARE_SELF" then
	    RBShare:InitiateSession(name, RecipeBook.Globals.Player)
	    ToggleDropDownMenu(1, nil, dropdownFrame, "cursor");
	elseif button == "RBSHARE_ALL" then
		RBShare:InitiateSession(name, "all")
	    ToggleDropDownMenu(1, nil, dropdownFrame, "cursor");
	elseif button == "RECIPEBOOK_REQUEST" then
	    RBShare:RequestUpdate(name);
	    ToggleDropDownMenu(1, nil, dropdownFrame, "cursor");
	end

	return;
end

--[[ UnitPopupHideButtons : Hooked function to hide buttons on a RB popup for non-friendly players. ]]--
function RBShare_UnitPopupHideButtons()
	local dropdownFrame = getglobal(UIDROPDOWNMENU_INIT_MENU);
	local coop = dropdownFrame.unit and UnitCanCooperate("player", dropdownFrame.unit)
	for index, value in ipairs(UnitPopupMenus[dropdownFrame.which]) do
		if (value == "RECIPEBOOK_SHARE" and not coop) then
		    UnitPopupShown[UIDROPDOWNMENU_MENU_LEVEL][index] = 0;
			return;
		end
	end
end


